/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.fukurou.process;

/**
 * HybsProcess は、バッチ処理サブクラスの共通インターフェースです。
 * HybsProcess を用いて、順次、バッチプロセスを実行することができます。
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public interface HybsProcess {

	/**
	 * 引数形式を解析する Argument オブジェクトに、引数を設定します。
	 * Argument の文字列から、引数かプロパティをセットします。
	 * ［プロパティ］のキー部の大文字･小文字は、厳格に判定しています。
	 * Argument の文字列には、タイプがあります。
	 *
	 * ［コメント］  ： # で始まる引数で、使用されません。(登録もされません。)
	 * ［引数］      ： #,-,= 以外で始まる通常の文字列。登録の順番が指定されます。
	 * ［プロパティ］： - で始まり、キーと値を=で区切っているパラメータです。順序は無関係。
	 *
	 * @param   arg	引数
	 */
	void putArgument( String arg ) ;

	/**
	 * Argument の文字列から、プロパティをセットします。
	 * ［プロパティ］のキー部の大文字･小文字は、厳格に判定しています。
	 * このメソッドは、引数 や コメントの判断を行いません。プロパティ のみ
	 * 設定されるものとして、処理します。
	 * プロパティの key=val が初めから分割されている場合の簡易メソッドです。
	 *
	 * @param   key キー
	 * @param   val 値
	 */
	void putArgument( String key,String val );

	/**
	 * プロセスの初期化を行います。初めに一度だけ、呼び出されます。
	 * 初期処理(ファイルオープン、ＤＢオープン等)に使用します。
	 * 引数の ParamProcess は、データベース接続、ログファイル、
	 * エラー時メール送信などの基本設定を管理しているインターフェース
	 * です。
	 *
	 * @param   paramProcess データベースの接続先情報などを持っているオブジェクト
	 */
	void init( ParamProcess paramProcess ) ;

	/**
	 * プロセスの終了を行います。最後に一度だけ、呼び出されます。
	 * 終了処理(ファイルクローズ、ＤＢクローズ等)に使用します。
	 *
	 * @param   isOK トータルで、OKだったかどうか [true:成功/false:失敗]
	 */
	void end( boolean isOK ) ;

	/**
	 * ディスプレイにメッセージを表示します。
	 *
	 * @param	msg	表示するメッセージ
	 */
	void println( final String msg ) ;

	/**
	 * ログファイルにメッセージを表示します。
	 *
	 * @param	msg	表示するメッセージ
	 */
	void logging( final String msg ) ;

	/**
	 * ディスプレイ出力する LoggerProcess オブジェクトをセットします。
	 *
	 * @param logger LoggerProcessオブジェクト
	 */
	void setLoggerProcess( final LoggerProcess logger );

	/**
	 * プロセスの処理結果のレポート表現を返します。
	 * 処理プログラム名、入力件数、出力件数などの情報です。
	 * この文字列をそのまま、標準出力に出すことで、結果レポートと出来るような
	 * 形式で出してください。
	 *
	 * @return   処理結果のレポート
	 */
	String report() ;

	/**
	 * このクラスの使用方法を返します。
	 *
	 * @return	このクラスの使用方法
	 */
	String usage() ;

}
