/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.fukurou.model;


/**
 * [PN],[OYA] などの [] で指定されたカラムで表されたフォーマットデータに対して、
 * DBTableModelオブジェクトを適用して 各カラムに実データを割り当てるオブジェクトです。
 *
 * @og.group 画面表示
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class ArrayDataModel implements DataModel<String> {
	private final String[] names ;
	private String[] values = null;

	/**
	 * 引数に名前配列を指定したコンストラクター
	 *
	 * @param	nms	名前配列
	 * @throws  IllegalArgumentException 引数の名前配列が null の場合
	 */
	public ArrayDataModel( final String[] nms ) {
		if( nms == null ) {
			String errMsg = "引数の名前配列に、null は設定できません。";
			throw new IllegalArgumentException( errMsg );
		}

		int size = nms.length ;
		names = new String[size] ;
		System.arraycopy( nms,0,names,0,size );
	}

	/**
	 * row にあるセルの設定値を置き換えます。
	 *
	 * @param   vals  新しい配列値。
	 * @param   row   値が変更される行(無視されます)
	 */
	public void setValues( final String[] vals, final int row ) {
		int size = vals.length;
		values = new String[size];
		System.arraycopy( vals,0,values,0,size );
	}

	/**
	 * カラム名に対応する カラム番号を返します。
	 *
	 * 特殊なカラムが指定された場合は、負の値を返します。
	 * 例えば、[KEY.カラム名]、[I]、[ROW.ID] など、特定の負の値を返します。
	 * また、カラム名が元のデータモデルに存在しない場合も、負の値か、
	 * Exception を返します。負の値なのか、Exception なのかは、
	 * 実装に依存します。
	 *
	 * @param	columnName	値が参照されるカラム名
	 *
	 * @return  指定されたセルのカラム番号。存在しなければ、-1
	 * @throws  IllegalArgumentException 引数のカラム名が null の場合
	 */
	public int getColumnNo( final String columnName ) {
		if( columnName == null ) {
			String errMsg = "引数のカラム名に、null は設定できません。";
			throw new IllegalArgumentException( errMsg );
		}

		int address = -1;
		for( int i=0; i<names.length; i++ ) {
			if( columnName.equalsIgnoreCase( names[i] ) ) {
				address = i;
				break;
			}
		}

		return address;
	}

	/**
	 * カラム名配列に対応する カラム番号配列を返します。
	 *
	 * これは、#getColumnNo( String ) に対する 複数のカラム名を検索した
	 * 場合と同じです。
	 *
	 * @param	clmNms 	値が参照されるカラム名配列
	 *
	 * @return  指定されたセルのカラム番号配列。
	 */
	public int[] getColumnNos( final String[] clmNms ) {
		if( clmNms == null ) {
			return new int[0];
		}

		int[] clmNos = new int[clmNms.length];
		for( int j=0; j<clmNms.length; j++ ) {
			int address = -1;
			for( int i=0; i<names.length; i++ ) {
				if( clmNms[j].equalsIgnoreCase( names[i] ) ) {
					address = i;
					break;
				}
			}
			clmNos[j] = address;
		}

		return clmNos;
	}

	/**
	 * カラム名配列を返します。
	 *
	 * @return	カラム名配列
	 */
	public String[] getNames() {
		return names.clone();
	}

	/**
	 * row にあるセルの属性値を配列で返します。
	 *
	 * @param   row     値が参照される行(無視されます)
	 *
	 * @return  指定されたセルの属性値
	 */
	public String[] getValues( final int row ) {
		return values.clone();
	}

	/**
	 * row および clm にあるセルの属性値をStringに変換して返します。
	 *
	 * @param   row     値が参照される行(無視されます)
	 * @param   clm     値が参照される列
	 *
	 * @return  指定されたセルの値
	 *
	 */
	public String getValue( final int row, final int clm) {
		return values[clm];
	}

	/**
	 * clm のNativeタイプを返します。
	 * Nativeタイプはorg.opengion.fukurou.model.NativeTypeで定義されています。
	 *
	 * @og.rev 4.1.1.2 (2008/02/28) 新規追加
	 * @og.rev 5.1.8.0 (2010/07/01) NativeType#getType(String) のメソッドを使用するように変更。
	 *
	 * @param  clm      値が参照される列
	 *
	 * @return Nativeタイプ
	 * @see org.opengion.fukurou.model.NativeType
	 */
	public NativeType getNativeType( final int clm ) {
//		return StringUtil.getNativeType( values[clm] );
		return NativeType.getType( values[clm] );
	}
}
