/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.io;

import org.odftoolkit.odfdom.OdfFileDom;
import org.odftoolkit.odfdom.doc.table.OdfTableCell;
import org.odftoolkit.odfdom.dom.element.office.OfficeAnnotationElement;
import org.odftoolkit.odfdom.dom.element.text.TextPElement;
import org.opengion.hayabusa.db.DBColumn;

/**
 * Calcファイルの書き出しクラスです。
 *
 * このクラスでは、通常の出力クラスと異なり、以下のように出力されます。
 *  ①データ部分には、X(文字側)または9(数値型)をリソース定義の桁数分出力
 *  ②各セルのコメント情報として{&#064;ANO.カラム名_行番号}を出力
 *
 * この出力結果は、通常、Calc帳票システムの雛形を作成するための、元情報として
 * 利用することを想定しています。
 *
 * @og.group ファイル出力
 *
 * @version  5.0
 * @author	 Hiroki Nakamura
 * @since    JDK6.0,
 */
public class TableWriter_CalcDefAno extends TableWriter_CalcDef {
	/** このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "6.4.2.0 (2016/01/29)" ;

	/**
	 * デフォルトコンストラクター
	 *
	 * @og.rev 6.4.2.0 (2016/01/29) PMD refactoring. Each class should declare at least one constructor.
	 */
	public TableWriter_CalcDefAno() {
		super();		// これも、自動的に呼ばれるが、空のメソッドを作成すると警告されるので、明示的にしておきます。
	}

	/**
	 * テキストコンテンツ用のセルを生成する。
	 *
	 * @param	contentDom	OdfFileDomオブジェクト
	 * @param	content		コンテンツ
	 * @param	col			DBColumnオブジェクト
	 * @param	isCellTypeNumber	[true:数字型/false:文字型]
	 * @param	isNumberList		[true:数字リスト=999/false:通常]
	 *
	 * @return	テキストコンテンツ用のセル
	 */
	@Override
	protected OdfTableCell createTextCell( final OdfFileDom contentDom, final String content, final DBColumn col, final Boolean isCellTypeNumber, final Boolean isNumberList ) {
		String val = null;
		if( isNumberList ) { val = "999"; }
		else {
			final int sizeX = col.getSizeX();
			final int sizeY = col.getSizeY();
			final String fillStr = isCellTypeNumber ? "9" : "X";
			final StringBuilder buf = new StringBuilder( BUFFER_MIDDLE );	// 6.1.0.0 (2014/12/26) refactoring
			for( int i=0; i<sizeX; i++ ) {
				buf.append( fillStr );
			}
			if( sizeY > 0 ) {
				buf.append( '.' );		// 6.0.2.5 (2014/10/31) char を append する。
				for( int i=0; i<sizeY; i++ ) {
					buf.append( fillStr );
				}
			}
			val = buf.toString();
		}

		final OdfTableCell cell = super.createTextCell( contentDom, val, isCellTypeNumber, isNumberList );
		final OfficeAnnotationElement anotation = cell.newOfficeAnnotationElement();
		final TextPElement text = anotation.newTextPElement();
		text.setTextContent( "{@ANO." + content + "}" );

		return cell;
	}
}
