/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.io;

import java.io.PrintWriter;

import org.opengion.fukurou.util.StringUtil;
import org.opengion.hayabusa.db.DBTableModel;

/**
 * CSV形式ダブルクォートファイル(CSV)形式書き込みクラスです。
 * 標準と異なるのは、文字列のみ、ダブルクオート処理を行い、数字型は、ダブルクオートも
 * ゼロカンマも付けません。
 *
 * DefaultTableWriter を継承していますので，ラベル，名前，データの出力部のみ
 * オーバーライドして，可変長CSV形式ファイルの出力機能を実現しています。
 *
 * @og.rev 5.6.9.4 (2013/10/31) 新規作成
 * @og.group ファイル出力
 *
 * @version  4.0
 * @author	 Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class TableWriter_CSV3 extends TableWriter_Default {
	/** このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "6.4.2.0 (2016/01/29)" ;

	/**
	 * デフォルトコンストラクター
	 *
	 * @og.rev 6.4.2.0 (2016/01/29) PMD refactoring. Each class should declare at least one constructor.
	 */
	public TableWriter_CSV3() {
		super();		// これも、自動的に呼ばれるが、空のメソッドを作成すると警告されるので、明示的にしておきます。
	}

	/**
	 * DBTableModel から データを作成して,PrintWriter に書き出します。
	 *
	 * @param	writer PrintWriterオブジェクト
	 */
	@Override
	public void writeDBTable( final PrintWriter writer )  {
		super.setSeparator( CSV_SEPARATOR );	// 3.5.6.0 (2004/06/18)
		super.writeDBTable( writer );
	}

	/**
	 * PrintWriter に DBTableModelのテーブル情報を書き込みます。
	 * このクラスでは，データを ダブルコーテーション(")で囲みます。
	 * PrintWriter に DBTableModelのテーブル情報を書き込みます。
	 *
	 * @og.rev 6.0.1.2 (2014/08/08) カラム飛ばしできる機能を追加
	 * @og.rev 6.0.4.0 (2014/11/28) データ出力用のレンデラー
	 *
	 * @param	table  DBTableModelオブジェクト
	 * @param	writer PrintWriterオブジェクト
	 */
	@Override
	protected void writeData( final DBTableModel table,final PrintWriter writer ) {
		final int numberOfRows =	table.getRowCount();
		final boolean useNumber = isUseNumber();
		final boolean useRenderer = isUseRenderer();	// 5.2.1.0 (2010/10/01)

		for( int row=0; row<numberOfRows; row++ ) {
			if( useNumber ) {
				writer.print( quotation( String.valueOf( row+1 ) ) );
				writer.print( CSV_SEPARATOR );
			}

			for( int i=0; i<numberOfColumns; i++ ) {
				if( i != 0 ) { writer.print( CSV_SEPARATOR ); }
				final int clm = clmNo[i];
				if( clm < 0 ) { continue; }			// 6.0.1.2 (2014/08/08) カラム飛ばし

				String val = table.getValue(row,clm);
				if( dbType[i] == NVAR ) {
					val = StringUtil.getReplaceEscape( val );
				}
				// 5.2.1.0 (2010/10/01) useRenderer 対応
				else if( useRenderer ) {
					// 6.0.4.0 (2014/11/28) データ出力用のレンデラー
					val = dbColumn[clm].getWriteValue( val );
				}

				// 数字型には、ダブルクオートは付けません。
				if( dbType[i] == NUMBER ) {
						writer.print( val );
				}
				else {
					writer.print( quotation( val ) );
				}
			}
			writer.println();
		}
	}

	/**
	 * データを書き込む場合の,区切り文字をセットします。
	 * このクラスでは，CSV 固定の為,区切り文字のセットは無効になります。
	 *
	 * @param	sprt 区切り文字
	 */
	@Override
	public void setSeparator( final String sprt ) {
		super.setSeparator( CSV_SEPARATOR ) ;	// 3.5.6.0 (2004/06/18)
	}
}
