/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.db;

/**
 * データのコード情報を取り扱うSelectionクラスの、NULL時オブジェクトです。
 *
 * 以前は、Selection オブジェクトが null の場合に、NullPointerException で
 * いきなりエラーで停止していましたが、この、NULLセレクションを作成することで、
 * 取りあえず、どういう状況なのか、判るようにします。
 *
 * @og.rev 5.7.3.0 (2014/02/07) 新規追加
 * @og.rev 5.7.7.1 (2014/06/13) Selectionオブジェクトの基本実装とします。
 * @og.group 選択データ制御
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class Selection_NULL implements Selection {
	private final String	initMsg ;

	/**
	 * デフォルトコンストラクター
	 * 継承元のクラスから、呼び出させるように、作成しておきます。
	 *
	 * @og.rev 5.7.7.1 (2014/06/13) 新規追加
	 */
	public Selection_NULL() { initMsg = null; }

	/**
	 * 引数に初期メッセージを指定して作成する、コンストラクター
	 *
	 * @og.rev 5.7.3.0 (2014/02/07) 新規追加
	 * @og.rev 6.2.6.0 (2015/06/19) type別Selectionの場合、ラベルリソースを使用する為、言語を引数で渡す。
	 * @og.rev 6.3.4.0 (2015/08/01) Selection_NULL の引数から、lang 属性を削除します。
	 *
	 * @param	strCode	初期メッセージ文字列
//	 * @param	lang  言語(今は未使用)
	 */
//	public Selection_NULL( final String strCode,final String lang ) {
	public Selection_NULL( final String strCode ) {
		initMsg = strCode ;
	}

	/**
	 * 初期値が選択済みの 選択肢(オプション)を返します。
	 * 無条件で、初期メッセージを返します。
	 *
	 * @og.rev 5.7.3.0 (2014/02/07) 新規追加
	 *
	 * @param   selectValue  選択されている値
	 * @param   seqFlag  シーケンスアクセス機能 [true:ON/false:OFF]
	 * @param   useShortLabel ラベル(短)をベースとしたオプション表示を行うかどうか(常にfalse)。
	 *
	 * @return  オプションタグ
	 * @og.rtnNotNull
	 * @og.rtnNotNull
	 */
	@Override
	public String getOption( final String selectValue,final boolean seqFlag, final boolean useShortLabel ) {
		return initMsg + " value=[" + selectValue + "]";
	}

	/**
	 * 初期値が選択済みの 選択肢(オプション)を返します。
	 * このオプションは、引数の値を初期値とするオプションタグを返します。
	 * ※ このクラスでは実装されていません。→ 実装を入れます。
	 *
	 * @og.rev 6.0.4.0 (2014/11/28) 実装を入れます。
	 * @og.rev 6.2.2.4 (2015/04/24) メソッド変更。旧 #getRadio( String , String , boolean )
	 *
	 * @param   name         ラジオの name
	 * @param   selectValue  選択されている値
	 * @param   useLabel     ラベル表示の有無 [true:有/false:無]
	 *
	 * @return  オプションタグ
	 * @og.rtnNotNull
	 */
	@Override
	public String getOption( final String name,final String selectValue,final boolean useLabel ) {
		return initMsg + " name=[" + name + "] value=[" + selectValue + "]";
	}

	/**
	 * 選択肢(value)に対するラベルを返します。
	 * 選択肢(value)が、存在しなかった場合は、選択肢そのものを返します。
	 * getValueLabel( XX ) は、getValueLabel( XX,false ) と同じです。
	 *
	 * @param   selectValue 選択肢の値
	 *
	 * @return  選択肢のラベル
	 * @see     #getValueLabel( String,boolean )
	 */
	@Override
	public String getValueLabel( final String selectValue ) {
		return getValueLabel( selectValue,false );
	}

	/**
	 * 選択肢(value)に対するラベルを返します。
	 * 無条件で、初期メッセージを返します。
	 *
	 * @og.rev 5.7.3.0 (2014/02/07) 新規追加
	 *
	 * @param	selectValue	選択肢の値
	 * @param	isSLbl	短縮ラベルを [true:使用する/false:しない](常に false)
	 *
	 * @return  選択肢のラベル
	 * @og.rtnNotNull
	 * @see     #getValueLabel( String )
	 */
	@Override
	public String getValueLabel( final String selectValue,final boolean isSLbl ) {
		return initMsg + " value=[" + selectValue + "]";
	}

	/**
	 * オブジェクトのキャッシュが時間切れかどうかを返します。
	 * キャッシュが時間切れ(無効)であれば、true を、有効であれば、
	 * false を返します。
	 *
	 * ※ ここでは、常に false を返します。
	 *
	 * @return  キャッシュが時間切れなら true
	 */
	@Override
	public boolean isTimeOver() {
		return false;
	}

	/**
	 * コードリソースのパラメータを適切な文字列に変換して返します。
	 * 通常であれば、そのままセットすればよいのですが、既存の処理が、
	 * class属性にセットするという仕様だったので、互換性を考慮した変換を行います。
	 * ここでは、
	 *   ① "=" を含む場合は、そのままセット
	 *   ② disabled 単体の場合は、disabled="disabled" をセット
	 *   ③ それ以外は、class= の後に、引数をセット
	 * します。
	 * 今後は、パラメータに、class="AAA" とセットすることで、徐々に
	 * この変換処理を無くしていきます。
	 *
	 * @og.rev 6.2.0.0 (2015/02/27) コードリソースのパラメータの指定方法を変更します。
	 *
	 * @param	buf			変換結果を格納するStringBuilder
	 * @param	paramKey	変換処理の対象となるパラメータ
	 * @return  引数と同じ、StringBuilder
	 */
	protected StringBuilder setCodeParam( final StringBuilder buf , final String paramKey ) {
		if( paramKey != null && !paramKey.isEmpty() ) {
			if( paramKey.indexOf( '=' ) > 0 ) {							// ①
				buf.append( ' ' ).append( paramKey );
			}
			else if( "disabled".equalsIgnoreCase( paramKey ) ) {		// ②
				buf.append( " disabled=\"" ).append( paramKey ).append( '"' );
			}
			else {														// ③
				buf.append( " class=\"" ).append( paramKey ).append( '"' );
			}
		}
		return buf ;
	}
}
