/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.view;

import java.util.List;
import java.util.regex.Pattern;									// 6.4.3.3 (2016/03/04)

import org.opengion.fukurou.system.OgBuilder ;					// 6.4.4.2 (2016/04/01)
import org.opengion.fukurou.util.StringUtil;
import org.opengion.hayabusa.common.HybsSystemException;
// import org.opengion.hayabusa.db.DBTableModel;
import org.opengion.hayabusa.html.TableFormatter;

/**
 * ヘッダー部分のフォーマットに応じたテーブルを自動作成する、フォーマットテーブル作成クラスです。
 *
 * AbstractViewForm により、setter/getterメソッドのデフォルト実装を提供しています。
 * 各HTMLのタグに必要な setter/getterメソッドのみ，追加定義しています。
 * [XXXX]は、カラムを指定します。ラベル＋入力フィールドをそれぞれtdで囲います。
 * [#XXXX]は、対応するカラムのラベルを出力します。
 * [$XXXX]は、対応するカラムのレンデラーを出力します。
 * [!XXXX]は、対応するカラムの値を出力します。
 * 特殊記号の解釈は、HTMLFormatTextField系とHTMLFormatTable系で異なりますので
 * ご注意ください。
 *
 * AbstractViewForm を継承している為,ロケールに応じたラベルを出力させる事が出来ます。
 *
 * @og.group 画面表示
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class ViewForm_HTMLFormatTable extends ViewForm_HTMLTable  {
	/** このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "6.4.4.2 (2016/04/01)" ;

	// 6.4.3.3 (2016/03/04) class属性(ColumnのDBType)置換で、td属性に[カラム]があると、誤ってそちらがセットされてしまう対応。
	// ※ 正規表現の「.」は、改行を除く任意の文字なので、改行を含みあい場合は、「[\\s\\S]」とします。
//	private static final Pattern TD_BODY = Pattern.compile( "[\\s\\S]*<td[^>]*>" );				// tdタグが完了して、BODY部分の解析に入る。
	private static final Pattern TD_BODY = Pattern.compile( "[\\s\\S]*<td[^>]*>[\\s]*" );		// 6.4.4.2 (2016/04/01) tdタグが完了して、BODY部分の解析に入る。
	private static final Pattern TD_END  = Pattern.compile( "[^<]*>[^<>]*" );					// 何らかのタグが完了して、BODY部分の解析に入る。

	// 3.5.4.0 (2003/11/25) TableFormatter クラス追加
	private TableFormatter format	;

	/**
	 * デフォルトコンストラクター
	 *
	 * @og.rev 6.4.2.0 (2016/01/29) PMD refactoring. Each class should declare at least one constructor.
	 */
	public ViewForm_HTMLFormatTable() {
		super();		// これも、自動的に呼ばれるが、空のメソッドを作成すると警告されるので、明示的にしておきます。
	}

	/**
	 * DBTableModel から HTML文字列を作成して返します。
	 * startNo(表示開始位置)から、pageSize(表示件数)までのView文字列を作成します。
	 * 表示残りデータが pageSize 以下の場合は,残りのデータをすべて出力します。
	 *
	 * @og.rev 3.5.0.0 (2003/09/17) BODY要素の noClass 属性を追加。
	 * @og.rev 3.5.0.0 (2003/09/17) &lt;tr&gt;属性は、元のフォーマットのまま使用します。
	 * @og.rev 3.5.2.0 (2003/10/20) ヘッダー繰り返し属性( headerSkipCount )を採用
	 * @og.rev 3.5.3.1 (2003/10/31) skip属性を採用。headerLine のキャッシュクリア
	 * @og.rev 3.5.4.0 (2003/11/25) TableFormatter クラスを使用するように変更。
	 * @og.rev 3.5.5.0 (2004/03/12) systemFormat(例：[KEY.カラム名]形式等)の対応
	 * @og.rev 3.5.5.0 (2004/03/12) No 欄そのものの作成判断ロジックを追加
	 * @og.rev 3.5.5.7 (2004/05/10) [#カラム名] , [$カラム名] に対応
	 * @og.rev 3.5.6.0 (2004/06/18) '!' 値のみ 追加 既存の '$' は、レンデラー
	 * @og.rev 3.5.6.4 (2004/07/16) ヘッダーとボディー部をJavaScriptで分離
	 * @og.rev 3.7.0.3 (2005/03/01) getBgColorCycleClass に、選択行マーカーを採用
	 * @og.rev 4.0.0.0 (2005/01/31) 新規作成(getColumnClassName ⇒ getColumnDbType)
	 * @og.rev 4.3.1.0 (2008/09/08) フォーマットが設定されていない場合のエラー追加・編集行のみを表示する属性(isSkipNoEdit)追加
	 * @og.rev 4.3.3.0 (2008/10/01) noTransition属性対応
	 * @og.rev 4.3.7.4 (2009/07/01) tbodyタグの入れ子を解消(FireFox対応)
	 * @og.rev 6.2.0.0 (2015/02/27) フォーマット系の noDisplay 対応
	 * @og.rev 6.4.3.3 (2016/03/04) class属性(ColumnのDBType)置換で、td属性に[カラム]があると、誤ってそちらがセットされてしまう対応。
	 * @og.rev 6.4.3.4 (2016/03/11) tdに、[カラム]が無いケースで、次の[カラム]のクラス属性が、前方すべてのtdにセットされてしまう対応。
	 * @og.rev 6.4.4.2 (2016/04/01) TableFormatterのタイプ別値取得処理の共通部をまとめる。
	 *
	 * @param  startNo    表示開始位置
	 * @param  pageSize   表示件数
	 *
	 * @return  DBTableModelから作成された HTML文字列
	 * @og.rtnNotNull
	 */
	@Override
	public String create( final int startNo, final int pageSize )  {
		if( getRowCount() == 0 ) { return ""; }	// 暫定処置

		// 4.3.1.0 (2008/09/08)
		if( format == null ) {
			final String errMsg = "ViewTagで canUseFormat() = true の場合、Formatter は必須です。";
			throw new HybsSystemException( errMsg );
		}
		// 6.2.0.0 (2015/02/27) フォーマット系の noDisplay 対応
		// ※ makeFormat の直後に setFormatNoDisplay を実行する順番を他のクラスと合わせます。
		// 元の場所では、setColumnDisplay のタイミングが後なので、うまく動きません
		format.makeFormat( getDBTableModel() );		// 6.2.0.0 (2015/02/27) #setFormatterList( List<TableFormatter> ) から移動
		setFormatNoDisplay( format );

		headerLine	 = null;		// 3.5.3.1 (2003/10/31) キャッシュクリア
		final int lastNo = getLastNo( startNo, pageSize );
		final int blc = getBackLinkCount();
		final int hsc = getHeaderSkipCount();		// 3.5.2.0 (2003/10/20)
		int hscCnt = 1;						// 3.5.2.0 (2003/10/20)

		final StringBuilder out = new StringBuilder( BUFFER_LARGE )
			.append( getCountForm( startNo,pageSize ) )
			.append( getHeader() );

		final String ckboxTD = "<td " + format.getRowspan() + ">";
		int bgClrCnt = 0;
		for( int row=startNo; row<lastNo; row++ ) {
			if( isSkip( row ) || isSkipNoEdit( row ) ) { continue; } // 4.3.1.0 (2008/09/08)
			if( ! format.isUse( row,getDBTableModel() ) ) { continue; }		// 3.5.4.0 (2003/11/25)
			out.append("<tbody").append( getBgColorCycleClass( bgClrCnt++,row ) );
			if( isNoTransition() ) {	// 4.3.3.0 (2008/10/01)
				out.append( getHiddenRowValue( row ) );
			}
			out.append('>')			// 6.0.2.5 (2014/10/31) char を append する。
				.append( format.getTrTag() );

			// 3.5.5.0 (2004/03/12) No 欄そのものの作成判断追加
			if( isNumberDisplay() ) {
				out.append( makeCheckbox( ckboxTD,row,blc ) );
			}

			int cl = 0;
			boolean isTdBody = false;										// 6.4.3.3 (2016/03/04) tdBody処理を行ってよいかどうかのフラグ
			for( ; cl<format.getLocationSize(); cl++ ) {
				String fmt = format.getFormat(cl);				// 3.5.0.0 (2003/09/17)
				final int loc = format.getLocation(cl);			// 3.5.5.0 (2004/03/12)
				if( ! format.isNoClass() && loc >= 0 ) {		// 3.5.5.7 (2004/05/10)
					if( isTdBody && TD_END.matcher( fmt ).matches() ) {		// 6.4.3.3 (2016/03/04) tdBody許可があり、TDが閉じた場合。
						isTdBody = false;
						final int idx = out.lastIndexOf( "<td" );
						if( idx >= 0 ) { 
							final String tdclass = " class=\"" + getColumnDbType(loc) + "\" ";
							out.insert( idx+3, tdclass );					// +3 は、"<td" の後ろという意味。fmtでなくそれ以前に戻る必要がある。
						}
					}
					if( TD_BODY.matcher( fmt ).matches() ) {				// 6.4.3.3 (2016/03/04) TDが閉じた場合。
						// 6.4.3.4 (2016/03/11) tdに、[カラム]が無いケースで、次の[カラム]のクラス属性が、前方すべてのtdにセットされてしまう対応。
						final int idx = fmt.lastIndexOf( "<td" );
						if( idx >= 0 ) {	// matchしてるので、あるはず
							final String tdclass = " class=\"" + getColumnDbType(loc) + "\" ";
							fmt = fmt.substring( 0,idx+3 ) + tdclass + fmt.substring( idx+3 ) ;
						}
//						final String tdclass = "<td class=\"" + getColumnDbType(loc) + "\" ";
//						fmt = StringUtil.replace( fmt ,"<td", tdclass );	// 6.2.0.1 (2015/03/06)
						isTdBody = false;									// 6.4.3.3 (2016/03/04) これは、要らない･･･はず。
					}
					else {
						isTdBody = true;									// TDが閉じていない。
					}
//					final String tdclass = "<td class=\"" + getColumnDbType(loc) + "\" " ;
////					fmt = StringUtil.replace( format.getFormat(cl) ,"<td", tdclass );
//					fmt = StringUtil.replace( fmt ,"<td", tdclass );							// 6.3.9.1 (2015/11/27)
				}
				out.append( fmt );			// 3.5.0.0

				// 3.5.5.7 (2004/05/10) #,$ 対応
				if( loc >= 0 ) {
					// 6.4.4.2 (2016/04/01) 処理の共通部をまとめる。
					out.append( getTypeCaseValue( format.getType(cl),row,loc ) );
//					switch( format.getType(cl) ) {
//						case '#' : out.append( getColumnLabel(loc) );		break;
//						case '$' : out.append( getRendererValue(row,loc) );	break;
//						case '!' : out.append( getValue(row,loc) );			break;
//						default  : out.append( getValueLabel(row,loc) );	break;
//					}
				}
				else {
					out.append( format.getSystemFormat(row,loc) );
				}
			}
			out.append( format.getFormat(cl) )
				.append("</tbody>").append( CR );

		// 3.5.2.0 (2003/10/20) ヘッダー繰り返し属性( headerSkipCount )を採用
			if( hsc > 0 && hscCnt % hsc == 0 ) {
				out.append("<tbody class=\"row_h\" >")
					.append( getHeadLine() )
					.append("</tbody>");
				hscCnt = 1;
			}
			else {
				hscCnt ++ ;
			}
		}
		out.append("</table>").append( CR )
			.append( getScrollBarEndDiv() );	// 3.8.0.3 (2005/07/15)

		return out.toString();
	}

	/**
	 * 内容をクリア(初期化)します。
	 *
	 * @og.rev 3.1.1.0 (2003/03/28) 同期メソッド(synchronized付き)を非同期に変更する。
	 * @og.rev 3.5.0.0 (2003/09/17) Noカラムに、表示を全て消せるように、class 属性を追加。
	 * @og.rev 3.5.4.0 (2003/11/25) TableFormatter クラスを使用するように変更。
	 *
	 */
	@Override
	public void clear() {
		super.clear();
		format          = null;
	}

	/**
	 * DBTableModel から テーブルのタグ文字列を作成して返します。
	 *
	 * @og.rev 3.2.4.0 (2003/06/12) makeFormat() する位置を移動。
	 * @og.rev 3.5.0.0 (2003/09/17) &lt;tr&gt;属性は、元のフォーマットのまま使用します。
	 * @og.rev 3.5.1.0 (2003/10/03) Noカラムに、numberType 属性を追加
	 * @og.rev 3.5.2.0 (2003/10/20) ヘッダー繰り返し部をgetHeadLine()へ移動
	 * @og.rev 3.5.3.1 (2003/10/31) VERCHAR2 を VARCHAR2 に修正。
	 * @og.rev 3.5.5.0 (2004/03/12) No 欄そのものの作成判断ロジックを追加
	 * @og.rev 3.5.6.5 (2004/08/09) thead に、id="header" を追加
	 * @og.rev 4.0.0.0 (2005/01/31) DBColumn の 属性(CLS_NM)から、DBTYPEに変更
	 * @og.rev 5.9.1.2 (2015/10/23) 自己終了警告対応
	 * @og.rev 6.4.4.1 (2016/03/18) NUMBER_DISPLAYを、static final 定数化します。
	 * @og.rev 6.4.4.2 (2016/04/01) StringBuilderの代わりに、OgBuilderを使用する。
	 *
	 * @return  テーブルのタグ文字列
	 * @og.rtnNotNull
	 */
	@Override
	protected String getTableHead() {
		return new OgBuilder()
					.appendIf( isNumberDisplay() , NUMBER_DISPLAY )
					.appendCR( "<thead id=\"header\">" )				// 3.5.6.5 (2004/08/09)
					.append(   getHeadLine() )
					.appendCR( "</thead>" )
					.toString();

//		final StringBuilder buf = new StringBuilder( BUFFER_MIDDLE );
//
//		// 3.5.5.0 (2004/03/12) No 欄そのものの作成判断追加
//		if( isNumberDisplay() ) {
//			// 5.9.1.2 (2015/10/23) 自己終了警告対応
////			buf.append("<colgroup class=\"X\" /><colgroup class=\"BIT\" /><colgroup class=\"S9\" />")
//			// 6.4.4.1 (2016/03/18) NUMBER_DISPLAYを、static final 定数化します。
////			buf.append("<colgroup class=\"X\" ><!-- --></colgroup><colgroup class=\"BIT\" ><!-- --></colgroup><colgroup class=\"S9\" ><!-- --></colgroup>")
////				.append(CR);
//			buf.append( NUMBER_DISPLAY );
//		}
//
//	// 3.5.2.0 (2003/10/20) ヘッダー繰り返し部をgetHeadLine()へ移動
//		buf.append("<thead id=\"header\">").append( CR )	// 3.5.6.5 (2004/08/09)
//			.append( getHeadLine() )
//			.append("</thead>").append( CR );
//
//		return buf.toString();
	}

	/**
	 * ヘッダー繰り返し部を、getTableHead()メソッドから分離。
	 *
	 * @og.rev 6.1.2.0 (2015/01/24) キャッシュを返すのを、#getHeadLine() に移動。
	 *
	 * @return	テーブルのタグ文字列
	 * @og.rtnNotNull
	 */
	@Override
	protected String getHeadLine() {
		if( headerLine == null ) {					// キャッシュになければ、設定する。
			headerLine = getHeadLine( "<th" + format.getRowspan() ) ;
		}

		return headerLine ;
	}

	/**
	 * ヘッダー繰り返し部を、getTableHead()メソッドから分離。
	 *
	 * @og.rev 3.5.2.0 (2003/10/20) 新規作成
	 * @og.rev 3.5.4.0 (2003/11/25) TableFormatter クラスを使用するように変更。
	 * @og.rev 3.5.4.3 (2004/01/05) useCheckControl 属性の機能を追加
	 * @og.rev 3.5.4.6 (2004/01/30) numberType="none" 時の処理を追加(Noラベルを出さない)
	 * @og.rev 3.5.4.7 (2004/02/06) ヘッダーにソート機能用のリンクを追加します。
	 * @og.rev 3.5.5.0 (2004/03/12) systemFormat(例：[KEY.カラム名]形式等)の対応
	 * @og.rev 3.5.5.0 (2004/03/12) No 欄そのものの作成判断ロジックを追加
	 * @og.rev 3.7.0.1 (2005/01/31) 全件チェックコントロール処理変更
	 * @og.rev 6.1.2.0 (2015/01/24) キャッシュを返すのを、#getHeadLine() に移動。
	 * @og.rev 6.4.3.4 (2016/03/11) ヘッダーでもTableFormatterのType(#,$,!)に対応した値を出すようにする。
	 * @og.rev 6.4.4.2 (2016/04/01) TableFormatterのタイプ別値取得処理の共通部をまとめる。
	 *
	 * @param	thTag タグの文字列
	 *
	 * @return	テーブルのタグ文字列
	 * @og.rtnNotNull
	 */
	@Override
	protected String getHeadLine( final String thTag ) {

		final StringBuilder buf = new StringBuilder( BUFFER_MIDDLE )
			.append( format.getTrTag() ).append( CR );

		// 3.5.5.0 (2004/03/12) No 欄そのものの作成判断追加
		if( isNumberDisplay() ) {
			// 6.1.2.0 (2015/01/24) thTag に、headerFormat.getRowspan() を含ませて受け取る。
			// 3.5.4.3 (2004/01/05) 追加分
			if( isUseCheckControl() && "checkbox".equals( getSelectedType() ) ) {
				buf.append( thTag ).append( "></th>" )
					.append( thTag ).append( '>' ).append( getAllCheckControl() ).append( "</th>" )
					.append( thTag ).append( '>' ).append( getNumberHeader()    ).append( "</th>" );
			}
			else {
				buf.append( thTag ).append(" colspan=\"3\">").append( getNumberHeader() ).append( "</th>" );	// 6.0.2.5 (2014/10/31) char を append する。
			}
		}

		int cl = 0;
		for( ; cl<format.getLocationSize(); cl++ ) {
			buf.append( StringUtil.replace( format.getFormat(cl) ,"td","th" ));
			final int loc = format.getLocation(cl);
			// 6.4.3.4 (2016/03/11) ヘッダーでもTableFormatterのType(#,$,!)に対応した値を出すようにする。
	//		if( loc >= 0 ) { buf.append( getSortedColumnLabel(loc) ); }
			if( loc >= 0 ) {
				// 6.4.4.2 (2016/04/01) 処理の共通部をまとめる。
				buf.append( getTypeCaseValue( format.getType(cl),-1,loc ) );
//				switch( format.getType(cl) ) {
//					case '#' : buf.append( getColumnLabel(loc) );		break;
//					case '$' : buf.append( getRendererValue(0,loc) );	break;
//					case '!' : buf.append( getValue(0,loc) );			break;
//					default  : buf.append( getSortedColumnLabel(loc) );	break;
//				}
			}
		}
		buf.append( StringUtil.replace( format.getFormat(cl) ,"td","th" ) ).append( CR );

		return buf.toString();				// 6.1.2.0 (2015/01/24)
	}

	/**
	 * フォーマットを設定します。
	 *
	 * @og.rev 3.5.4.0 (2003/11/25) 新規作成
	 * @og.rev 6.2.0.0 (2015/02/27) フォーマット系の noDisplay 対応。ロジックを共通にするため、処理を移動
	 *
	 * @param	list	TableFormatterのリスト
	 */
	@Override
	public void setFormatterList( final List<TableFormatter> list ) {		// 4.3.3.6 (2008/11/15) Generics警告対応
		format = list.get(0);		// 4.3.3.6 (2008/11/15) Generics警告対応
		// 6.2.0.0 (2015/02/27) フォーマット系の noDisplay 対応。ロジックを共通にするため、処理を移動
	}

	/**
	 * フォーマットメソッドを使用できるかどうかを問い合わせます。
	 *
	 * @return  使用可能(true)/ 使用不可能 (false)
	 */
	@Override
	public boolean canUseFormat() {
		return true;
	}

	/**
	 * ビューで表示したカラムの一覧をCSV形式で返します。
	 *
	 * @og.rev 5.1.6.0 (2010/05/01) 新規追加
	 * @og.rev 6.2.0.1 (2015/03/06) TableFormatter#getLocation(int)の有効判定
	 * @og.rev 6.4.3.4 (2016/03/11) getViewClms(TableFormatter) を使用して表示されたカラム一覧を求めます。
	 *
	 * @return	ビューで表示したカラムの一覧
	 * @og.rtnNotNull
	 */
	@Override
	public String getViewClms() {
		return getViewClms( format );

//		final DBTableModel table = getDBTableModel();
//		final StringBuilder buf = new StringBuilder( BUFFER_MIDDLE );
//		for( int i=0; i<format.getLocationSize(); i++ ) {
//			if( buf.length() > 0 ) { buf.append( ',' ); }
//			// 6.2.0.1 (2015/03/06) TableFormatter#getLocation(int)の有効判定
//			final int loc = format.getLocation(i);
//			if( loc >= 0 ) { buf.append( table.getColumnName( loc ) ); }
//		}
//		return buf.toString();
	}
}
