/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import org.opengion.hayabusa.html.ViewGanttTableParam;

import static org.opengion.fukurou.util.StringUtil.nval ;

// import java.io.ObjectOutputStream;
// import java.io.ObjectInputStream;
// import java.io.IOException;

/**
 * viewタグの viewFormType が HTMLGanttTable の場合にパラメータを設定します。
 *
 * ガントチャートを表示する、ViewForm_HTMLGanttTable クラスに対して、各種パラメータを
 * 設定します。
 * パラメータが設定されていない場合は、ViewGanttTableParam の初期値が
 * 使用されます。
 * (パラメータを使用するには、viewタグのuseParam 属性をtrueに設定する必要があります。)
 *
 * @og.formSample
 * ●形式：&lt;og:ganttParam groupColumns="PRJCODE,RESOURCE_ID" minDuration="0.5" /&gt;
 * ●body：なし
 *
 * ●Tag定義：
 *   &lt;og:ganttParam
 *       groupColumns       【TAG】各タスクを１行にまとめる時のカラム名(CSV形式)をセットします (初期値:ViewGanttTableParam#GROUP_COLUMNS_VALUE[=PRJCODE,RESOURCE_ID])
 *       durationColumn     【TAG】期間欄のカラム名をセットします
 *       dystartColumn      【TAG】開始日付けのカラム名をセットします (初期値:ViewGanttTableParam#DYSTART_COLUMN_VALUE[=DYSTART])
 *       dystartFormat      【TAG】開始日付けのフォーマットを指定します (初期値:ViewGanttTableParam#DYSTART_FORMAT_VALUE[=yyyyMMdd])
 *       minDuration        【TAG】最小期間(小数可)をセットします(初期値:1)
 *       headerDuration     【TAG】ヘッダーの表示期間(小数可)をセットします(初期値:最小期間(minDuration) と同じ)
 *       headerLocale       【TAG】ヘッダーを表示する時の、Locale(language のみ) を指定します (初期値:ViewGanttTableParam#HEADER_LOCALE_VALUE[=ja])
 *       useSeqDay          【TAG】ヘッダー日付けの連続表示を行うかどうか[true/false]を指定します(初期値:false)
 *       startDay           【TAG】ヘッダー日付けの開始日を指定します
 *       endDay             【TAG】ヘッダー日付けの終了日を指定します
 *       debug              【TAG】デバッグ情報を出力するかどうか[true/false]を指定します(初期値:false)
 *   /&gt;
 *
 * ●使用例
 *     ViewFormTag の viewFormType が、ViewForm_HTMLGanttTable の場合に使用します。
 *     &lt;og:view
 *         viewFormType = &quot;HTMLGanttTable&quot;
 *         command      = &quot;{&#064;command}&quot;
 *         startNo      = &quot;0&quot;
 *         pageSize     = &quot;20&quot;
 *         <b>useParam     = &quot;true&quot;</b>
 *     &gt;
 *         &lt;og:ganttParam
 *              groupColumns   = "PRJCODE,RESOURCE_ID"
 *              durationColumn = "DURATION"
 *              dystartColumn  = "DYSTART"
 *              dystartFormat  = "yyyyMMdd"
 *              minDuration    = "0.5"
 *              headerDuration = "1"
 *              headerLocale   = "en"
 *              useSeqDay      = "true"
 *              startDay       = "20041020"
 *              endDay         = "20041028"
 *         /&gt;
 *     &lt;/og:view &gt;
 *
 * @og.rev 3.5.5.8 (2004/05/20) 新規作成
 * @og.group 画面表示
 *
 * @version  4.0
 * @author	 Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
// public class ViewGanttParamTag extends ViewParamTag {
public class ViewGanttParamTag extends ViewParamImpl {
	/** このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "6.4.2.0 (2016/01/29)" ;
	private static final long serialVersionUID = 642020160129L ;

	/**
	 * デフォルトコンストラクター
	 *
	 * @og.rev 6.4.2.0 (2016/01/29) PMD refactoring. Each class should declare at least one constructor.
	 */
	public ViewGanttParamTag() {
		super();		// これも、自動的に呼ばれるが、空のメソッドを作成すると警告されるので、明示的にしておきます。
	}

	/**
	 * 【TAG】各タスクを１行にまとめる時のカラム名(CSV形式)をセットします
	 *		(初期値:ViewGanttTableParam#GROUP_COLUMNS_VALUE[={@og.value ViewGanttTableParam#GROUP_COLUMNS_VALUE}])。
	 *
	 * @og.tag
	 * ガントチャートで、同一要件のタスクを１行にまとめて表示します。
	 * その場合のグルーピングのキーとなるカラム名を指定します。
	 * 複数ある場合は、CSV形式で指定できます。
	 * 日付けでの重複は考慮されていませんので、データ側でチェックしておいて下さい。
	 * (初期値:ViewGanttTableParam#GROUP_COLUMNS_VALUE[={@og.value ViewGanttTableParam#GROUP_COLUMNS_VALUE}])。
	 *
	 * @param	groupColumns グループカラム名 (CSV形式)
	 * @see		org.opengion.hayabusa.html.ViewGanttTableParam#GROUP_COLUMNS_VALUE
	 */
	public void setGroupColumns( final String groupColumns ) {
		putParam( ViewGanttTableParam.GROUP_COLUMNS_KEY ,
				  nval( getRequestParameter( groupColumns ),null ) );
	}

	/**
	 * 【TAG】期間欄のカラム名をセットします。
	 *
	 * @og.tag
	 * ガントを表示する場合の、各タスクの幅(期間)のカラム名を指定します。
	 * 各タスクは、開始日と期間より、ガントの指定のロケーションに割り当てられます。
	 * 初期値は null です。
	 *
	 * @param	durationColumn 期間欄のカラム名
	 */
	public void setDurationColumn( final String durationColumn ) {
		putParam( ViewGanttTableParam.DURATION_COLUMN_KEY ,
				  nval( getRequestParameter( durationColumn ),null ) );
	}

	/**
	 * 【TAG】開始日付けのカラム名をセットします
	 *		(初期値:ViewGanttTableParam#DYSTART_COLUMN_VALUE[={@og.value ViewGanttTableParam#DYSTART_COLUMN_VALUE}])。
	 *
	 * @og.tag
	 * ガントを表示する場合の、各タスクの開始日付けのカラム名を指定します。
	 * 各タスクは、開始日と期間より、ガントの指定のロケーションに割り当てられます。
	 * (初期値:ViewGanttTableParam#DYSTART_COLUMN_VALUE[={@og.value ViewGanttTableParam#DYSTART_COLUMN_VALUE}])。
	 *
	 * @param	dystartColumn 開始日カラム名 (初期値:DYSTART)
	 */
	public void setDystartColumn( final String dystartColumn ) {
		putParam( ViewGanttTableParam.DYSTART_COLUMN_KEY ,
				   nval( getRequestParameter( dystartColumn ),null ) );
	}

	/**
	 * 【TAG】開始日付けのフォーマットを指定します
	 *		(初期値:ViewGanttTableParam#DYSTART_FORMAT_VALUE[={@og.value ViewGanttTableParam#DYSTART_FORMAT_VALUE}])。
	 *
	 * @og.tag
	 * ガントの開始日付 データを取り込むときの日付けフォーマットを指定します。
	 * この形式の日付け文字列を、内部で、日付けデータとして管理します。
	 * java.text.SimpleDateFormat で指定できる形式を指定します。
	 * (初期値:ViewGanttTableParam#DYSTART_FORMAT_VALUE[={@og.value ViewGanttTableParam#DYSTART_FORMAT_VALUE}])。
	 *
	 * @param	dystartFormat 開始日フォーマット
	 * @see java.text.SimpleDateFormat
	 */
	public void setDystartFormat( final String dystartFormat ) {
		putParam( ViewGanttTableParam.DYSTART_FORMAT_KEY ,
				   nval( getRequestParameter( dystartFormat ),null ) );
	}

	/**
	 * 【TAG】最小期間(小数可)をセットします
	 *		(初期値:ViewGanttTableParam#MIN_DURATION_VALUE[={@og.value ViewGanttTableParam#MIN_DURATION_VALUE}])。
	 *
	 * @og.tag
	 * Duration で指定する期間の最小単位を指定します。
	 * 例えば、１日単位の精度でタスクを割り当てる場合、期間の最小単位は、１ に
	 * なります。表示する場合も、この数字が最小単位となり、HTMLのテーブルの
	 * １カラムに相当します。例えば、最小単位を、０．５ にすると、
	 * 半日単位でタスクを割り当てることができます。１０ にすると、１０日単位
	 * になります。
	 * (初期値:ViewGanttTableParam#MIN_DURATION_VALUE[={@og.value ViewGanttTableParam#MIN_DURATION_VALUE}])。
	 *
	 * @param	minDuration 最小期間
	 */
	public void setMinDuration( final String minDuration ) {
		putParam( ViewGanttTableParam.MIN_DURATION_KEY ,
				   nval( getRequestParameter( minDuration ),null ) );
	}

	/**
	 * 【TAG】ヘッダーの表示期間(小数可)をセットします(初期値:最小期間(minDuration) と同じ)。
	 *
	 * @og.tag
	 * ヘッダー部の日付け欄で指定する表示間隔を指定します。
	 *
	 * 例えば、最小期間(minDuration)を、０．５日単位の精度で指定した場合、
	 * そのままでは、ヘッダーも最小期間単位で表示されますが、１日単位で表示
	 * させたい場合は、１ を設定します。
	 * また、１週間単位では、７ を指定します。
	 * １ヶ月単位などの、不定長での指定はできません。
	 * 初期値は最小期間(minDuration) と同じです。
	 *
	 * @param	headerDuration 表示期間 (小数可)
	 */
	public void setHeaderDuration( final String headerDuration ) {
		putParam( ViewGanttTableParam.HEADER_DURATION_KEY ,
				   nval( getRequestParameter( headerDuration ),null ) );
	}

	/**
	 * 【TAG】ヘッダーを表示する時の、Locale(language のみ) を指定します
	 *		(初期値:ViewGanttTableParam#HEADER_LOCALE_VALUE[={@og.value ViewGanttTableParam#HEADER_LOCALE_VALUE}])。
	 *
	 * @og.tag
	 * ヘッダー部の日付け欄で SimpleDateFormat を用いて解析していますが、
	 * この Locale (language のみ)を指定できます。
	 *
	 * language は、有効な ISO 言語コードです。
	 * これらのコードは、ISO-639 で定義される 2 桁の小文字です。
	 * コードの完全なリストは、
	 * <a href="http://www.ics.uci.edu/pub/ietf/http/related/iso639.txt">
	 * http://www.ics.uci.edu/pub/ietf/http/related/iso639.txt </a>
	 * を始めいくつかの場所で入手できます。
	 * SimpleDateFormat の Locale を指定するコンストラクタでは、全ての
	 * ロケールをサポートするわけではありませんのでご注意ください。
	 * (初期値:ViewGanttTableParam#HEADER_LOCALE_VALUE[={@og.value ViewGanttTableParam#HEADER_LOCALE_VALUE}])。
	 *
	 * @og.rev 3.5.5.9 (2004/06/07) 新規追加
	 *
	 * @param	headerLocale ロケール (languageのみ)
	 * @see		java.text.SimpleDateFormat#SimpleDateFormat(String , Locale)
	 */
	public void setHeaderLocale( final String headerLocale ) {
		putParam( ViewGanttTableParam.HEADER_LOCALE_KEY ,
				   nval( getRequestParameter( headerLocale ),null ) );
	}

	/**
	 * 【TAG】ヘッダー日付けの連続表示を行うかどうか[true/false]を指定します
	 *		(初期値:ViewGanttTableParam#USE_SEQ_DAY_VALUE[={@og.value ViewGanttTableParam#USE_SEQ_DAY_VALUE}])。
	 *
	 * @og.tag
	 * ヘッダー部の日付け欄で開始日と終了日の範囲の連続日付け表示を行う場合に、
	 * "true" に、設定します。開始日と終了日 の設定方法は、各属性を参照してください。
	 * true:連続表示を行う / false:dystartColumn カラムで使用した分のみ
	 * (初期値:ViewGanttTableParam#USE_SEQ_DAY_VALUE[={@og.value ViewGanttTableParam#USE_SEQ_DAY_VALUE}])。
	 *
	 * @og.rev 3.6.1.0 (2005/01/05) 新規追加
	 *
	 * @param	useSeqDay 連続表示可否 [true:行う/false:行わない]
	 * @see		#setStartDay( String )
	 * @see		#setEndDay( String )
	 */
	public void setUseSeqDay( final String useSeqDay ) {
		putParam( ViewGanttTableParam.USE_SEQ_DAY_KEY ,
				   nval( getRequestParameter( useSeqDay ),null ) );
	}

	/**
	 * 【TAG】ヘッダー日付けの開始日を指定します。
	 *
	 * @og.tag
	 * ヘッダー部の日付け欄で連続日付け表示使用時(seqDay="true")に、
	 * この開始日より、終了日までの日付けヘッダーを連続表示します。
	 * seqDay="true" で startDayを指定しない場合は、dystartColumn で
	 * 指定された、最小日付けを使用します。逆に、この最小日付けよりも
	 * 大きな日付けは、セットできません。データとヘッダーの整合性が
	 * 取れなくなります。
	 *
	 * この時の日付けフォーマットは、dystartFormat の指定と同じです。
	 *
	 * @og.rev 3.6.1.0 (2005/01/05) 新規追加
	 *
	 * @see		#setUseSeqDay( String )
	 * @param	startDay 開始日
	 * @see		#setEndDay( String )
	 */
	public void setStartDay( final String startDay ) {
		putParam( ViewGanttTableParam.START_DAY_KEY ,
				   nval( getRequestParameter( startDay ),null ) );
	}

	/**
	 * 【TAG】ヘッダー日付けの終了日を指定します。
	 *
	 * @og.tag
	 * ヘッダー部の日付け欄で連続日付け表示使用時(seqDay="true")に、
	 * 開始日より、この終了日までの日付けヘッダーを連続表示します。
	 * seqDay="true" で endDayを指定しない場合は、dystartColumn で
	 * 指定された、最大日付けを使用します。逆に、この最大日付けよりも
	 * 小さな日付けは、セットできません。データとヘッダーの整合性が
	 * 取れなくなります。
	 *
	 * この時の日付けフォーマットは、dystartFormat の指定と同じです。
	 *
	 * @og.rev 3.6.1.0 (2005/01/05) 新規追加
	 *
	 * @param	endDay 終了日
	 * @see		#setUseSeqDay( String )
	 * @see		#setStartDay( String )
	 */
	public void setEndDay( final String endDay ) {
		putParam( ViewGanttTableParam.END_DAY_KEY ,
				   nval( getRequestParameter( endDay ),null ) );
	}

	/**
	 * タグの名称を、返します。
	 * 自分自身のクラス名より、自動的に取り出せないため、このメソッドをオーバーライドします。
	 *
	 * @og.rev 4.0.0.0 (2005/01/31) 新規追加
	 *
	 * @return  タグの名称
	 * @og.rtnNotNull
	 */
	@Override
	protected String getTagName() {
		return "ganttParam" ;
	}

//	/**
//	 * シリアライズ用のカスタムシリアライズ書き込みメソッド。
//	 *
//	 * @og.rev 4.0.0.0 (2006/09/31) 新規追加
//	 * @serialData 一部のオブジェクトは、シリアライズされません。
//	 *
//	 * @param	strm	ObjectOutputStreamオブジェクト
//	 * @throws IOException	入出力エラーが発生した場合
//	 */
//	private void writeObject( final ObjectOutputStream strm ) throws IOException {
//		strm.defaultWriteObject();
//	}
//
//	/**
//	 * シリアライズ用のカスタムシリアライズ読み込みメソッド
//	 *
//	 * ここでは、transient 宣言された内部変数の内、初期化が必要なフィールドのみ設定します。
//	 *
//	 * @og.rev 4.0.0.0 (2006/09/31) 新規追加
//	 * @serialData 一部のオブジェクトは、シリアライズされません。
//	 *
//	 * @param	strm	ObjectInputStreamオブジェクト
//	 * @see #release2()
//	 * @throws IOException	シリアライズに関する入出力エラーが発生した場合
//	 * @throws ClassNotFoundException	クラスを見つけることができなかった場合
//	 */
//	private void readObject( final ObjectInputStream strm ) throws IOException , ClassNotFoundException {
//		strm.defaultReadObject();
//	}
}
