/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.hayabusa.resource.URLXfer;
import org.opengion.fukurou.util.StringUtil ;
import static org.opengion.fukurou.system.HybsConst.CR ;		// 6.1.0.0 (2014/12/26)

import javax.servlet.http.HttpSession ;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.jsp.tagext.TagSupport ;
import javax.servlet.jsp.JspWriter ;

import java.io.IOException;
import java.util.concurrent.atomic.AtomicInteger;			// 6.3.9.0 (2015/11/06)

/**
 * マルチセッション起動チェックを行います。
 *
 * このタグは、特殊で、一番最上位のJSP(通常は、jsp/index.jsp)に仕込むことで、
 * マルチセッション起動チェックを行います。
 * とくに、TopMenuTag と関連しており、このタグが存在しないと、メニューが
 * 動作しません。
 * このタグでは、URLXfer による、リンク変換転送をサポートします。
 * URLのXFER変数をキーに、GE17 テーブルを検索し、指定のURLへ sendRedirect します。
 *
 * @og.formSample
 * ●形式：&lt;og:jspInit /&gt;
 * ●body：なし
 *
 * ●使用例
 *    &lt;og:jspInit /&gt;
 *
 * @og.rev 4.0.0.0 (2005/08/31) 新規作成
 * @og.group メニュー制御
 *
 * @version  4.0
 * @author   Kohei Naruse
 * @since    JDK5.0,
 */
public class JspInitTag extends TagSupport {
	/** このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "6.3.9.0 (2015/11/06)" ;
	private static final long serialVersionUID = 639020151106L ;

//	// 3.8.0.0 (2005/06/07)
//	private static int count ;
	private static final AtomicInteger MSC_CNT = new AtomicInteger();		// 6.3.9.0 (2015/11/06)

	/**
	 * 同一セッションでのマルチ起動対策用カウンタを、同期処理します。
	 *
	 * この count は、適当でよかったのですが、findBugs で、警告されるため、きちんとしておきます。
	 *
	 * @og.rev 6.0.2.5 (2014/10/31) findBugs対応。
	 * @og.rev 6.3.9.0 (2015/11/06) Use block level rather than method level synchronization.(PMD)
	 *
	 * @return	マルチ起動対策用カウンタの値
	 */
//	private static synchronized String getCount() {
	private static String getCount() {
//		return String.valueOf( count++ ) ;
		return String.valueOf( MSC_CNT.getAndIncrement() );		// 互換性のため、今の値を取得してから、＋１する。
	}

	/**
	 * Taglibの終了タグが見つかったときに処理する doEndTag() を オーバーライドします。
	 *
	 * @og.rev 4.1.1.0 (2008/02/07) UserInfo の再作成する機能を追加
	 * @og.rev 4.2.2.0 (2008/05/28) Guestユーザ対応
	 * @og.rev 4.3.4.1 (2008/12/08) UserInfo の再作成する機能を削除
	 * @og.rev 5.7.4.3 (2014/03/28) 出力する HTML は、フィルターします。
	 * @og.rev 5.7.6.2 (2014/05/16) IEのHTML5機能が有効か無効かの判定キーを削除(初期化)します。
	 * @og.rev 6.0.2.5 (2014/10/31) マルチ起動対策用カウンタのfindBugs対応。
	 *
	 * @return	後続処理の指示
	 */
	@Override
	public int doEndTag() {
		// 5.7.4.3 (2014/03/28) エラー時でも、debug=true があれば、継続する。
		final HttpServletRequest request = ( HttpServletRequest )pageContext.getRequest();
		final boolean debug = "true".equalsIgnoreCase( request.getParameter( "debug" ) );

		// Tomcat 初期起動時の common/SystemParameter.java でエラーが発生した場合。
		// 6.4.2.1 (2016/02/05) PMD refactoring. Prefer StringBuffer over += for concatenating strings
//		String errMsg = HybsSystem.sys( HybsSystem.LOCAL_CONTX_ERR_KEY );
//		if( errMsg != null && !debug ) {
		final String cntxErrMsg = HybsSystem.sys( HybsSystem.LOCAL_CONTX_ERR_KEY );
		if( cntxErrMsg != null && !debug ) {
			try {
				final JspWriter out = pageContext.getOut();
				out.println( "<html><body><pre>" );
				// 5.7.4.3 (2014/03/28) 出力する HTML は、フィルターします。
//				out.println( StringUtil.htmlFilter( errMsg ) );
				out.println( StringUtil.htmlFilter( cntxErrMsg ) );
				out.println( "</pre></body></html>" );
			}
			catch(IOException ex) {
				// 6.4.2.1 (2016/02/05) PMD refactoring. Prefer StringBuffer over += for concatenating strings
				final String errMsg = "画面出力時の PageContext の取得時にエラーが発生しました。"
										+ CR
										+ cntxErrMsg ;
//				errMsg = "画面出力時の PageContext の取得時にエラーが発生しました。"
//						+ CR
//						+ errMsg ;
				throw new HybsSystemException( errMsg,ex );		// 3.5.5.4 (2004/04/15) 引数の並び順変更
			}
			return SKIP_PAGE ;		// ページの残りの処理を行わない。
		}

		final HttpSession session = pageContext.getSession();
		synchronized( JspInitTag.class ) {
			// 3.8.0.0 (2005/06/07) 同一セッションでのマルチ起動対策を行います。
			session.setAttribute( HybsSystem.MULTI_SESSION_CHECK, getCount() );		// 6.0.2.5 (2014/10/31)
		}

		// 5.7.6.2 (2014/05/16) IEのHTML5機能が有効か無効かの判定キーを削除(初期化)します。
		session.removeAttribute( HybsSystem.IE_HTML5_KEY );

		// URLXfer による、リンク変換転送機能
		final String key = request.getParameter( "XFER" );
		if( key != null ) {
			final URLXfer xfer = new URLXfer();
			final String url = xfer.getRedirectURL( key );
			if( url != null ) {
				try {
					final HttpServletResponse response = (HttpServletResponse)pageContext.getResponse();
					response.sendRedirect( url );
					return SKIP_PAGE ;		// ページの残りの処理を行わない。
				}
				catch ( IOException ex ) {
					// 6.4.2.1 (2016/02/05) PMD refactoring. Prefer StringBuffer over += for concatenating strings
//					final String errMsg2 = "URLの振り分け処理時に IOException が発生しました。 XFER=" + key;
//					throw new HybsSystemException( errMsg2,ex );
					final String errMsg = "URLの振り分け処理時に IOException が発生しました。 XFER=" + key;
					throw new HybsSystemException( errMsg,ex );
				}
			}
		}

		return EVAL_PAGE ;		// ページの残りを評価する。
	}
}
