/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.column;

import org.opengion.fukurou.model.NativeType;
import org.opengion.fukurou.util.ErrorMessage;
import org.opengion.fukurou.util.HybsDateUtil;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.hayabusa.db.AbstractDBType;
import org.opengion.hayabusa.db.DBTypeCheckUtil;

/**
 * 文字列の終了日付属性を規定する半角文字列を扱う為の、カラム属性を定義します。
 * 登録時に0000XXXXは00000101、9999XXXXは99991231、それ以外はYYYYMM31と変換します
 *
 * YYYYMM31 または、0000XXXX , 9999XXXX を許可します。
 * また、入力データとして, YYYYMM 形式 は、許可しません。あらかじめ,
 * valueSet( String ) でデータを変換しておく必要があります。
 * ただし、日付としての整合性チェックは行いません。
 *
 * タイプチェックとして、以下の条件を判定します。
 * ・文字列長は、直接計算で文字数との比較
 * ・日付使用文字チェック「('0' &gt; c || '9' &lt; c)以外」エラー
 * ・文字パラメータの 正規表現チェック
 *
 * @og.rev 5.4.3.6 (2012/01/19) コメント修正
 * @og.group データ属性
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class DBType_YM31 extends AbstractDBType {
	/** このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "5.6.6.0 (2013/07/05)" ;

	private static final String DEF_VALUE = "99991231" ;	// データのデフォルト値

	/**
	 * デフォルトコンストラクター
	 *
	 * @og.rev 4.0.0.0 (2005/01/31) type 廃止
	 */
	public DBType_YM31() {
		super( DEF_VALUE );
	}

	/**
	 * NATIVEの型の識別コードを返します。
	 *
	 * @og.rev 3.8.5.3 (2006/08/07) 新規作成
	 * @og.rev 4.1.1.2 (2008/02/28) Enum型(fukurou.model.NativeType)に変更
	 *
	 * @return  NATIVEの型の識別コード(DBType で規定)
	 * @og.rtnNotNull
	 * @see org.opengion.fukurou.model.NativeType
	 */
	@Override
	public NativeType getNativeType() {
		return NativeType.CALENDAR;
	}

	/**
	 * 半角スペースで固定長(半角換算の数)に変換した文字列を返します。
	 * なお、エラーチェックは行われません。
	 * 実行前に、必ず valueCheck( String value ,int len ) が行われる必要があります。
	 *
	 * @og.rev 3.5.4.5 (2004/01/23) エンコード指定に変更します。
	 *
	 * @param   value    ＦＩＬＬ埋めする文字列
	 * @param   sizeX    整数部分の文字列の長さ
	 * @param   sizeY    小数部分の文字列の長さ
	 * @param   encode   固定長で変換する文字エンコード
	 *
	 * @return  ＦＩＬＬ埋めした新しい文字列
	 */
	@Override
	public String valueFill( final String value ,final int sizeX ,final int sizeY,final String encode ) {
		if( value != null && value.length() >= sizeX ) {
			return value.substring( 0, sizeX );
		}

		final String errMsg = "指定文字が不正です。value=[" + value + "] length=[" + sizeX + "]";
		throw new HybsSystemException( errMsg );
	}

	/**
	 * String引数の文字列を＋１した文字列を返します。
	 * ここでは、YYYYMM 形式のデータに、＋１ するので、月を＋１します。
	 * 
	 * 0000XX と、9999XX は、引数の値そのままを返します。
	 * 引数が null の場合と、ゼロ文字列("")の場合は,物理的初期設定値(String getDefault())
	 * の値を返します。
	 * それ以外の場合は、日付計算を行い、月を加算します。よって、年月フォーマットに
	 * 合わないデータを指定した場合は、エラーになりますので、ご注意ください。
	 * valueSet( String ) する以前の文字については,整合性は保証していません。
	 *
	 * @og.rev 3.6.0.0 (2004/09/17) 月の加算ロジックの誤り修正。(roll ⇒ add)
	 * @og.rev 5.5.7.2 (2012/10/09) HybsDateUtil を利用するように修正します。
	 *
	 * @param	value	String引数の文字列
	 *
	 * @return  String引数の文字列を＋１した文字列
	 */
	@Override
	public String valueAdd( final String value ) {
		if( value == null || value.isEmpty() ) { return getDefault(); }
		if( value.startsWith( "0000" ) || value.startsWith( "9999" ) ) { return value; }

		return HybsDateUtil.getMonthPlus( value , 1 ) + "31";		// 5.5.7.2 (2012/10/09) HybsDateUtil を利用
	}

	/**
	 * 入力データの整合性を合わせた結果を返します。
	 * ここでは、日付データに含まれるフォーマット文字列を削除し、数字だけの
	 * 文字列を作成し、YYYYMM 形式の先頭６文字 ＋ "31" のデータを作成します。
	 * このメソッドでは、日付欄は、必ず "31" になります。
	 * 実登録データの作成は、DBType オブジェクトを利用しますので,
	 * これと Editor とがアンマッチの場合は、うまくデータ変換
	 * されない可能性がありますので、注意願います。
	 *
	 * @og.rev 3.3.3.0 (2003/07/09) 前後のスペースを取り除いておく。
	 * @og.rev 5.6.6.0 (2013/07/05) parseDate を改造し、日付の桁数指定を厳密に判定します。
	 *
	 * @param	value	(一般に編集データとして登録されたデータ)
	 *
	 * @return  修正後の文字列(一般にデータベースに登録するデータ)
	 */
	@Override
	public String valueSet( final String value ) {
		if( value == null || value.isEmpty() ) { return getDefault(); }

		return HybsDateUtil.parseDate( value , 6 ) + "31";		// 不要な記号等を削除し、桁数をそろえる。
	}

	/**
	 * データが登録可能かどうかをチェックします。
	 * データがエラーの場合は、そのエラー内容を返します。
	 * valueSet( String ) する以前の文字については,整合性は保証していません。
	 *
	 * @og.rev 3.6.0.0 (2004/09/22) dbType パラメータを引数に追加
	 * @og.rev 5.2.2.0 (2010/11/01) 厳密にチェック(isStrict=true)するフラグを追加
	 * @og.rev 5.6.0.3 (2012/01/24) 日付チェックの整合性まで含めたチェックを入れる。
	 *
	 * @param   key		タグのキー(カラムID)
	 * @param   value	チェックする値
	 * @param	sizeX	整数部分の文字列の長さ
	 * @param	sizeY	小数部分の文字列の長さ
	 * @param	typeParam 	dbTypeパラメータ
	 * @param	isStrict    	厳密にチェックするかどうか [true:する/false:標準的]
	 *
	 * @return  エラー内容
	 * @og.rtnNotNull
	 */
	@Override
	public ErrorMessage valueCheck( final String key ,final String value ,
									final int sizeX ,final int sizeY ,final String typeParam ,final boolean isStrict) {

		final ErrorMessage msg = new ErrorMessage();
		if( value == null || value.isEmpty() ) { return msg; }

		final int len = (sizeY == 0) ? sizeX : sizeX + sizeY + 1;
		// 5.2.2.0 (2010/11/01) 厳密にチェック(isStrict=true)するフラグを追加
		if( isStrict ) {
			if( len != value.length() ) {
				// 文字列の長さが指定の長さと異なります。
				msg.addMessage( 0,ErrorMessage.NG,"ERR0011", key,value, String.valueOf( value.length() ), String.valueOf( len ) );
			}
		}
		else {
			if( len < value.length() ) {
				// 文字列の長さが指定の長さよりも長いです。
				msg.addMessage( 0,ErrorMessage.NG,"ERR0006",key,value,
										String.valueOf( value.length() ),String.valueOf( len ) );
			}
		}

		// 5.6.0.3 (2012/01/24) 文字の範囲チェック
		String check = DBTypeCheckUtil.rangeCheck( value, '0', '9' );
		if( check != null ) {
			// 指定の文字以外の文字が使われています。
			msg.addMessage( 0,ErrorMessage.NG,"ERR0009", key,check );
		}

		// 5.6.0.3 (2012/01/24) 日付文字の整合性チェック
		// "0000","9999" で始まる日付は未条件許可。
		check = DBTypeCheckUtil.ymdFormatCheck( value );
		if( check != null ) {
			// 指定の文字以外の文字が使われています。
			msg.addMessage( 0,ErrorMessage.NG,"ERR0009", key,check );
		}

		// 3.6.0.0 (2004/09/22) dbType パラメータを使用したマッチチェック
		check = DBTypeCheckUtil.matcheCheck( value,typeParam );
		if( check != null ) {
			// 指定の文字以外の文字が使われています。
			msg.addMessage( 0,ErrorMessage.NG,"ERR0009", key,check );
		}

		return msg;
	}
}
