/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.report;

import org.opengion.fukurou.util.Shell;
import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.db.DBTableModel;

/**
 * CSVPrintPoint インターフェース のデフォルト実装クラスです。
 * execute() をオーバーライドして、各種CSV取込み方式のシステムに対応して下さい。
 *
 * @og.group 帳票システム
 *
 * @version  5.0
 * @author	 Masakazu Takahashi
 * @since    JDK6.0,
 */
public abstract class AbstractCSVPrintPointService implements CSVPrintPointService {

	protected String		ykno		= null;
	protected String 		systemId	= null;
	protected String		fgrun		= null;
	protected String		hostName	= null;
	protected String		prtName		= null;
	protected DBTableModel	table		= null;
	protected DBTableModel	tableH		= null;
	protected DBTableModel	tableF		= null;
	protected String		prgdir		= null;
	protected String		prgfile		= null;
	protected String		outdir		= null;
	protected String		prtid		= null;	
	protected String		portnm		= null;	
	protected String		listid		= null;	
	protected String		modelname		= null;	
	protected String		csvOutdir	= null;

	protected final StringBuilder	errMsg	= new StringBuilder();	// エラーメッセージ
	protected 		String			fgkan	= GE50Access.FG_ERR2;	// 初期値はアプリエラー

	protected int TIMEOUT = HybsSystem.sysInt( "REPORT_DAEMON_TIMEOUT" ); //Shellタイムアウト
	protected String		shellCmd	= null;
	
	protected final static String FGRUN_EXCEL = "H";
	protected final static String FGRUN_PDF = "I";
	
	private static final String CR		= System.getProperty("line.separator");

	/**
	 * 発行処理
	 * 対象のシステムに応じてこのメソッドをオーバーライドします
	 * 実行後はfgkanの値を正しい値でセットしなおして下さい。
	 *
	 * @return 結果 [true:正常/false:異常]
	 */
	public abstract boolean execute();

	/**
	 * 帳票起動された要求番号をセットします。
	 *
	 * @param   no 要求NO
	 */
	public void setYkno( final String no ) {
		ykno = no;
	}

	/**
	 * システムIDをセットします。
	 *
	 * @param   id システムID
	 */
	public void setSystemId( final String id ) {
		systemId = id;
	}

	/**
	 * 実行方法をセットします。
	 *
	 * @param   flag 実行方法
	 */
	public void setFgrun( final String flag ) {
		fgrun = flag;
	}

	/**
	 * 帳票デーモンが実行されているホスト名をセットします。
	 *
	 * @param   host ホスト名
	 */
	public void setHostName( final String host ) {
		hostName = host;
	}

	/**
	 * プリンター名をセットします。
	 *
	 * @param   printerName	プリンタ名
	 */
	public void setPrinterName( final String printerName ) {
		prtName = printerName;
	}

	/**
	 * DBTableModel をセットします。
	 *
	 * @param   tbl DBTableModelオブジェクト
	 */
	public void setTable( final DBTableModel tbl ) {
		table = tbl;
	}
	
	/**
	 * DBTableModel をセットします。
	 *
	 * @param   tbl DBTableModelオブジェクト
	 */
	public void setTableH( final DBTableModel tbl ) {
		tableH = tbl;
	}
	
	/**
	 * DBTableModel をセットします。
	 *
	 * @param   tbl DBTableModelオブジェクト
	 */
	public void setTableF( final DBTableModel tbl ) {
		tableF = tbl;
	}

	/**
	 * 起動するバッチ等のプログラム（ディレクトリ）をセットします。
	 *
	 * @param dir バッチプログラムディレクトリ
	 */
	public void setPrgDir( final String dir ){
		prgdir = dir;
	}

	/**
	 * 起動するバッチ等のプログラムをセットします。
	 * 空の場合は起動しません。
	 *
	 * @param file バッチプログラム名
	 */
	public void setPrgFile( final String file ){
		prgfile = file;
	}

	/**
	 * ファイル出力時のディレクトリを指定します
	 *
	 * @param dir ファイル出力ディレクトリ
	 */
	public void setOutDir( final String dir ){
		outdir = dir;
	}

	/**
	 * プリンタIDを指定します
	 *
	 * @param id プリンタID
	 */
	public void setPrtId( final String id ){
		prtid = id;
	}

	/**
	 * プリンタのポート名
	 *
	 * @param port ポート名
	 */
	public void setPortnm( final String port ){
		portnm = port;
	}

	/**
	 * 帳票IDをセットします
	 *
	 * @param   id 帳票ID
	 */
	public void setListId( final String id ) {
		listid = id;
	}
	
	/**
	 * 雛形ファイル名をセットします
	 *
	 * @param   name 雛形ファイル名
	 */
	public void setModelname( final String name ) {
		modelname = name;
	}

	/**
	 * 完成フラグを返します。
	 *
	 * @return 完成フラグ String
	 */
	public String getFgkan(){
		return fgkan;
	}

	/**
	 * エラーメッセージを返します。
	 *
	 * @return エラーメッセージ String
	 */
	public String getErrMsg(){
		return errMsg.toString();
	}

	/**
	 * シェルの実行を行います
	 *
	 * @og.rev 5.4.3.0 (2011/12/26)
	 *
	 * @return 結果 [true:正常/false:異常]
	 */
	protected boolean programRun(){
		Shell shell = new Shell();
		shell.setCommand( shellCmd,true );		// BATCHプロセスで実行する
		shell.setWait( true );					// プロセスの終了を待つ
		shell.setTimeout( TIMEOUT );

		int rtnCode = shell.exec();				// 0 は正常終了を示す

		if( rtnCode != 0 ) {
			errMsg.append( "Shell Command exequte Error." ).append( CR );
			errMsg.append( "==============================" ).append( CR );
			errMsg.append( shellCmd ).append( CR );
			errMsg.append( shell.getStdoutData() ).append( CR );
			errMsg.append( shell.getStderrData() ).append( CR );
			errMsg.append( CR );
			return false;
		}

		return true;
	}

}
