/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.html;

import org.opengion.hayabusa.common.HybsSystem;

/**
 * VViewForm オブジェクトを取得する為に使用する，ファクトリクラスです。
 *
 *  ViewForm オブジェクト の識別ID を元に、ViewFormFactory.newInstance( String id )
 * メソッドで，ViewForm オブジェクトを取得します。
 * ViewFormFactory.close( ViewForm viewForm ) メソッドで，内部的に ViewFormFactory に
 * オブジェクトを戻す事によって,ViewForm オブジェクトのプーリングを行なっています。
 *
 * 実装とマッピングの関係から,識別ID は、ViewFormFactory で static 定義します
 * 大前提として、ユーザー共通で使用することを考えており,ユーザー個別にプール
 * する必要があるならば, HttpSession オブジェクトに登録すべきです。
 *
 * @og.group 画面表示
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public final class ViewFormFactory {
	/** newInstance() 時のデフォルトクラス {@value} */
	public static final String DEFAULT  = "HTMLTable" ;

	/**
	 *  デフォルトコンストラクターをprivateにして、
	 *  オブジェクトの生成をさせないようにする。
	 *
	 */
	private ViewFormFactory() {
	}

	/**
	 * ViewForm オブジェクトを取得します。
	 * 遅い初期化を行なう事により,実際に必要となるまで ViewForm オブジェクトは
	 * 作成しません。
	 *
	 * @og.rev 3.5.4.2 (2003/12/15) ViewForm のサブクラス名変更。
	 * @og.rev 3.5.6.0 (2004/06/18) 各種プラグイン関連付け設定を、システムパラメータ に記述します。
	 * @og.rev 3.5.6.2 (2004/07/05) setID メソッド名がまぎらわしい為、変更します。
	 * @og.rev 4.0.0.0 (2005/01/31) キーの指定を、ViewForm. から、ViewForm_ に変更します。
	 *
	 * @param   id 接続先ID
	 *
	 * @return  ViewFormオブジェクト
	 */
	public static ViewForm newInstance( final String id ) {
		String type = ( id == null ) ? DEFAULT : id ;
		String cls = HybsSystem.sys( "ViewForm_" + type ) ;		// 4.0.0 (2005/01/31)
		ViewForm vf = (ViewForm)HybsSystem.newInstance( cls );	// 3.5.5.3 (2004/04/09)
		vf.setId( type );	// 3.5.6.2 (2004/07/05)

		return vf;
	}
}
