/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.view;

import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.hayabusa.html.TableFormatter;

import java.util.List;

/**
 * フォーマットを外部から指定して作成する自由レイアウトの、テキストフィールド表示クラスです。
 *
 * AbstractViewForm により、setter/getterメソッドのデフォルト実装を提供しています。
 * 各HTMLのタグに必要な setter/getterメソッドのみ，追加定義しています。
 * [XXXX]は、カラムを指定します。ラベル＋入力フィールドをそれぞれtdで囲います。
 * [#XXXX]は、テーブルタグのtdを使用せず、ラベルと入力フィールドを出力します。
 * [$XXXX]は、ラベルもtdも出さずに、入力フィールドのみ出力します。
 * [!XXXX]は、値のみ出力します。
 * 特殊記号の解釈は、HTMLFormatTextField系とHTMLFormatTable系で異なりますので
 * ご注意ください。
 *
 * AbstractViewForm を継承している為,ロケールに応じたラベルを出力させる事が出来ます。
 *
 * @og.group 画面表示
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class ViewForm_HTMLFormatTextField extends ViewForm_HTMLTextField {
	/** このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "6.2.0.0 (2015/02/27)" ;

	// 4.0.0 (2005/01/31) HTML_LABEL_SEPARATOR を boolean 変数として取得します。
	private final String CLM = HybsSystem.sysBool( "HTML_LABEL_SEPARATOR" ) ? ":" : "" ;

	// 3.5.4.0 (2003/11/25) TableFormatter クラス追加
	private TableFormatter format	;

	/**
	 * フォーマットを設定します。
	 *
	 * @og.rev 3.5.4.0 (2003/11/25) 新規作成
	 * @og.rev 6.2.0.0 (2015/02/27) フォーマット系の noDisplay 対応。ロジックを共通にするため、処理を移動
	 *
	 * @param	list	TableFormatterのリスト
	 */
	@Override
	public void setFormatterList( final List<TableFormatter> list ) {		// 4.3.3.6 (2008/11/15) Generics警告対応
		format = list.get(0);		// 4.3.3.6 (2008/11/15) Generics警告対応
	}

	/**
	 * DBTableModel から HTML文字列を作成して返します。
	 * startNo(表示開始位置)から、pageSize(表示件数)までのView文字列を作成します。
	 * 表示残りデータが pageSize 以下の場合は,残りのデータをすべて出力します。
	 *
	 * @og.rev 2.0.1.0 (2002/10/10) ラベルだけ、フィールドだけを取り出すフォーマットを追加
	 * @og.rev 2.0.1.0 (2002/10/10) ラベルとフィールドのセパレーターとして、コロン(：)を使用するかどうかを指定できる
	 * @og.rev 3.2.4.0 (2003/06/12) makeFormat() する位置を移動。
	 * @og.rev 3.5.4.0 (2003/11/25) TableFormatter クラスを使用するように変更。
	 * @og.rev 3.5.5.0 (2004/03/12) systemFormat(例：[KEY.カラム名]形式等)の対応
	 * @og.rev 5.0.0.2 (2009/09/15) フォーマットが設定されていない場合のエラー追加
	 * @og.rev 5.6.2.3 (2013/03/22) DBColumn に、useSLabel="false" の値をセットします。
	 *
	 * @param  startNo    表示開始位置
	 * @param  pageSize   表示件数
	 *
	 * @return  DBTableModelから作成された HTML文字列
	 * @og.rtnNotNull
	 */
	@Override
	public String create( final int startNo, final int pageSize )  {
		if( getRowCount() == 0 ) { return ""; }	// 暫定処置

		noSLabelSetting();		// 5.6.2.3 (2013/03/22) DBColumn に、useSLabel="false" の値をセットします。

		// 4.3.1.0 (2008/09/08)
		if( format == null ) {
			final String errMsg = "ViewTagで canUseFormat() = true の場合、Formatter は必須です。";
			throw new HybsSystemException( errMsg );
		}
		// 6.2.0.0 (2015/02/27) フォーマット系の noDisplay 対応
		// ※ makeFormat の直後に setFormatNoDisplay を実行する順番を他のクラスと合わせます。
		// 元の場所では、setColumnDisplay のタイミングが後なので、うまく動きません
		format.makeFormat( getDBTableModel() );		// 6.2.0.0 (2015/02/27) #setFormatterList( List<TableFormatter> ) から移動
		setFormatNoDisplay( format );

		final StringBuilder out = new StringBuilder( BUFFER_LARGE )
			.append( getCountForm( startNo,pageSize ) )
			.append( makeSelectNo( startNo ) ).append( CR )
			.append( format.getTrTag() );

		int cl = 0;
		for( ; cl < format.getLocationSize(); cl++ ) {
			out.append( format.getFormat(cl) );
			final int loc = format.getLocation(cl);
			if( loc < 0 ) {
				out.append(  format.getSystemFormat(startNo,loc) ); // 5.0.0.2 (2009/09/15)
				continue ;
			}

			final char type = format.getType(cl);
			if( type == '#' ) {
				out.append("<span id=\"label\">")
					.append( getColumnLabel(loc) )
					.append( CLM )
					.append("</span>");
			}
			else if( type == '$' ) {
				out.append( getValueLabel(startNo,loc) );
			}
			else if( type == '!' ) {
				out.append( getValue(startNo,loc) );
			}
			else {
				out.append("<td id=\"label\">")
					.append( getColumnLabel(loc) )
					.append( CLM )
					.append("</td><td>")
					.append( getValueLabel(startNo,loc) )
					.append("</td>");
			}
		}
		out.append( format.getFormat(cl) );

		return out.toString();
	}

	/**
	 * 内容をクリア(初期化)します。
	 *
	 * @og.rev 2.0.1.0 (2002/10/10) ラベルだけ、フィールドだけを取り出すフォーマットを追加
	 * @og.rev 3.1.1.0 (2003/03/28) 同期メソッド(synchronized付き)を非同期に変更する。
	 * @og.rev 3.5.4.0 (2003/11/25) TableFormatter クラスを使用するように変更。
	 *
	 */
	@Override
	public void clear() {
		super.clear();
		format		= null;
	}

	/**
	 * フォーマットメソッドを使用できるかどうかを問い合わせます。
	 *
	 * @return  使用可能(true)/ 使用不可能 (false)
	 */
	@Override
	public boolean canUseFormat() {
		return true;
	}

	/**
	 * 表示項目の編集(並び替え)が可能かどうかを返します。
	 *
	 * @og.rev 5.1.6.0 (2010/05/01) 新規追加
	 *
	 * @return	表示項目の編集(並び替え)が可能かどうか(false:不可能)
	 */
	@Override
	public boolean isEditable() {
		return false;
	}
}
