/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.view;

import org.opengion.hayabusa.db.DBTableModel;
import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.fukurou.util.XHTMLTag;

/**
 * エントリ形式フォーム作成クラスです。
 *
 * フォーマットを外部から指定することにより､自由にレイアウトを作成できます。
 *
 * AbstractViewForm により、setter/getterメソッドのデフォルト実装を提供しています。
 * 各HTMLのタグに必要な setter/getterメソッドのみ，追加定義しています。
 *
 * AbstractViewForm を継承している為,ロケールに応じたラベルを出力させる事が出来ます。
 *
 * @og.rev 3.1.8.0 (2003/05/16) ViewForm_HTMLEntry クラスの新規作成
 * @og.group 画面表示
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class ViewForm_HTMLEntry extends ViewForm_HTMLFormatTextField {
	/** このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "5.2.1.0 (2010/10/01)" ;

	private String mustHidden = "";

	/**
	 * DBTableModel から HTML文字列を作成して返します。
	 * startNo(表示開始位置)から、pageSize(表示件数)までのView文字列を作成します。
	 * 表示残りデータが pageSize 以下の場合は,残りのデータをすべて出力します。
	 *
	 * @og.rev 5.2.1.0 (2010/10/01) must 属性の処理を追加します。
	 *
	 * @param  startNo    表示開始位置
	 * @param  pageSize   表示件数
	 *
	 * @return  DBTableModelから作成された HTML文字列
	 * @og.rtnNotNull
	 */
	@Override
	public String create( final int startNo, final int pageSize )  {
		return super.create( startNo,pageSize ) + mustHidden;
	}

	/**
	 * row行，colum列 のデータの値をHTML文字列に変換して返します。
	 * Entry 系のため、通常の行番号付の Editor ではなく、行番号無しの
	 * Editorを使用して、HTML文字列を作成します。
	 *
	 * @og.rev 3.8.0.9 (2005/10/17) writableControl 追加による引数変更
	 *
	 * @param	row		行番号
	 * @param	column	カラム番号
	 * @param	inVal	設定値
	 *
	 * @return	row行，colum列 のデータの値
	 */
	@Override
	protected String getEditorValue( final int row, final int column , final String inVal ) {
		return getDBColumn(column).getEditorValue( inVal );
	}

	/**
	 * 画面に選択された番号を表示します。
	 * Entry 系のため、選択番号を作成しません。
	 *
	 * @param  row	 行番号
	 *
	 * @return	空文字列 ""(固定)
	 * @og.rtnNotNull
	 */
	@Override
	protected String makeSelectNo( final int row ) {
		return "" ;
	}

	/**
	 * 初期化します。
	 * このクラスでは、データが０件の場合は、初期データを１件作成します。
	 * 初期化時に、初期データ作成処理を行います。
	 *
	 * @og.rev 3.2.3.0 (2003/06/06) 新規追加
	 * @og.rev 3.5.6.0 (2004/06/18) null 比較でバグを修正
	 * @og.rev 3.5.6.1 (2004/06/25) lang 言語コード 属性を削除します。
	 * @og.rev 4.0.1.0 (2007/12/12) initの場所を変更
	 * @og.rev 5.2.1.0 (2010/10/01) must 属性の処理を追加します。
	 *
	 * @param	table	DBTableModelオブジェクト
	 */
	@Override
	public void init( final DBTableModel table ) {
	//	super.init( table );
		if( table != null && table.getRowCount() == 0 ) {
			String[] data = new String[table.getColumnCount()];
			for( int i=0; i<data.length; i++ ) {
				data[i] = table.getDBColumn(i).getDefault();
				if( data[i] == null ) { data[i] = ""; }
			}
			table.addValues( data,0 );

	 		// 5.2.1.0 (2010/10/01) must 属性の処理を追加します。
			final String[] clms = table.getMustArray();
			if( clms != null ) {
				final StringBuilder buf = new StringBuilder( BUFFER_MIDDLE );	// 6.1.0.0 (2014/12/26) refactoring
				for( int i=0; i<clms.length; i++ ) {
					buf.append( XHTMLTag.hidden( HybsSystem.MUST_KEY +"must", clms[i] ) );
				}
				mustHidden = buf.toString();
			}
		}
		super.init( table ); // 4.0.1.0 (2007/12/12)　0件時不具合対応につき場所変更
	}

	/**
	 * 表示項目の編集(並び替え)が可能かどうかを返します。
	 *
	 * @og.rev 5.1.6.0 (2010/05/01) 新規追加
	 *
	 * @return	表示項目の編集(並び替え)が可能かどうか(false:不可能)
	 */
	@Override
	public boolean isEditable() {
		return false;
	}
}
