/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.develop;

import java.util.List;
import java.util.ArrayList;
import java.util.Map;

import org.opengion.hayabusa.develop.AbstractJspCreate;
import org.opengion.hayabusa.develop.JspConvertEntity;
import org.opengion.fukurou.xml.OGElement;
import org.opengion.fukurou.xml.OGAttributes;

/**
 * entry.jspの&lt;og:tableUpdateParam&gt;タグを作成します。
 * tableUpdateParam は、tableUpdate タグのBODY部に記述されます。
 * tableUpdateParam で書き換えが発生するのは、対象テーブルと、omitNames属性です。
 * where条件の書き換えは行いません。(雛形読み込み時のまま使用します。)
 * テーブルの書き間違いで、異なるUNIQ番号の更新を避ける意味合いで、
 * UNIQ=[UNIQ]以外のキーを条件に入れておくと、より安全です。
 *
 * ●使用例
 *  &lt;og:tableUpdate command="{&#064;command}" queryType="JDBCTableUpdate" debug="false"&gt;
 *      &lt;og:tableUpdateParam
 *          sqlType   = "{&#064;sqlType}"
 *          table     = "GF02"
 *          where     = "UNIQ=[UNIQ] and DYSET=[DYSET]"
 *          omitNames = "SYSTEM_ID,TBLSYU,TABLESPACE_NAME"
 *      /&gt;
 *  &lt;/og:tableUpdate&gt;
 *
 * @og.rev 5.6.1.2 (2013/02/22) 文字列連結から、XML処理するように変更します。
 * @author Takeshi.Takada
 *
 */
public class JspCreate_TABLE_UPDATE extends AbstractJspCreate {
	/** このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "5.6.1.2 (2013/02/22)" ;

	private List<JspConvertEntity> RESULT_ROWS ;
	private boolean IS_NULL ;

	/**
	 * 初期化メソッド
	 *
	 * 内部で使用する JspConvertEntityのリストのマップを受け取り、初期化を行います。
	 *
	 * @og.rev 5.2.1.0 (2010/10/01) 名前空間を、og 決め打ちから、名前空間指定無しに変更します。
	 * @og.rev 5.6.1.2 (2013/02/22) 対象ファイルを、result だけから、update も含めるように変更。
	 *
	 * @param	master	JspConvertEntityのリストのマップ
	 */
	@Override
	protected void init( final Map<String,List<JspConvertEntity>> master ) {
		RESULT_ROWS	= master.get( "RESULT" );
		IS_NULL = !isNotEmpty( RESULT_ROWS );
		KEY  = ":tableUpdateParam";
		NAME = "entry";
	}

	/**
	 * JSPに出力するタグの内容を作成します。
	 * 引数より作成前のタグの属性内容を確認するする事が出来ます。
	 *
	 * @og.rev 5.2.1.0 (2010/10/01) メソッドの引数を、OGAttributes から OGElement に変更します。
	 * @og.rev 5.2.1.0 (2010/10/01) 名前空間を、og 決め打ちから、引数を使用するように変更します。
	 *
	 * @param ele OGElementエレメントオブジェクト
	 * @param	nameSpace	このドキュメントのnameSpace( og とか mis とか )
	 *
	 * @return	変換された文字列
	 * @og.rtnNotNull
	 * @throws Throwable 変換時のエラー
	 */
	@Override
	protected String execute( final OGElement ele , final String nameSpace )  throws Throwable {
		if( IS_NULL ) { return ""; }

		String table = null;
		final List<String> omitNames = new ArrayList<>();
		for( final JspConvertEntity column : RESULT_ROWS){
			// 非表示は、GF92の属性(Remarks)に、何もセットされていないカラムの事
			final String remks = column.getRemarks();
			final String astbl = column.getAsTableName();

			// DISP で、別名がA1以外の場合、データ登録しないので、omit カラムになる。
			if ( "DISP".equalsIgnoreCase( remks ) && !"A1".equalsIgnoreCase( astbl ) ) {
				omitNames.add( column.getColumnName() );
			}

			// 最初の１回だけ取り込む
			if( table == null && "A1".equalsIgnoreCase( astbl ) ) {
				table = column.getTableName();
			}
		}

		final OGAttributes attri = ele.getOGAttributes();		// OGElementの内部オブジェクトなので、副作用あり
		attri.setUseCR( true );
		attri.setVal( "table" , table );

		if ( ! omitNames.isEmpty() ) {
			attri.setVal( "omitNames" , chainChar( omitNames ,",") );		// あれば更新、なければ追加
		}

		return ele.getText( 1 );
	}
}
