/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.column;

import org.opengion.fukurou.util.ErrorMessage;
import org.opengion.fukurou.util.HybsDateUtil;
import org.opengion.hayabusa.db.DBTypeCheckUtil;

/**
 * 文字列の日付属性(年/月)の半角の日付を扱う為の、カラム属性を定義します。
 *
 * yyyyMM に対応している必要があります。
 * "000000" と、"999999" は、常に許可になります。
 * ただし、日付の整合性チェックは行いませんが、valueAdd( String value )による
 * 日付(月)の加算時には、0000XX と 9999XX は、加算されず、それ以外は、
 * 正式な日付で加算されます。
 *
 * タイプチェックとして、以下の条件を判定します。
 * ・文字列長は、直接計算で文字数との比較
 * ・日付使用文字チェック「('0' &gt; c || '9' &lt; c)以外」エラー
 * ・文字パラメータの 正規表現チェック
 *
 * @og.rev 3.6.0.0 (2004/09/17) 新規作成
 * @og.group データ属性
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class DBType_YM extends DBType_YMD {
	/** このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "5.6.6.0 (2013/07/05)" ;

	/**
	 * String引数の文字列を＋１した文字列を返します。
	 * ここでは、yyyyMM 形式のデータに、＋１ するので、月を＋１します。
	 * 
	 * 0000XX と、9999XX は、引数の値そのままを返します。
	 * 引数が null の場合と、ゼロ文字列("")の場合は,物理的初期設定値(String getDefault())
	 * の値を返します。
	 * それ以外の場合は、日付計算を行い、月を加算します。よって、年月フォーマットに
	 * 合わないデータを指定した場合は、エラーになりますので、ご注意ください。
	 *
	 * @og.rev 5.5.7.2 (2012/10/09) HybsDateUtil を利用するように修正します。
	 *
	 * @param	value	String引数の文字列
	 *
	 * @return  String引数の文字列を＋１した文字列
	 */
	@Override
	public String valueAdd( final String value ) {
		if( value == null || value.isEmpty() ) { return getDefault(); }
		if( value.startsWith( "0000" ) || value.startsWith( "9999" ) ) { return value; }	// 特別な値

		return HybsDateUtil.getMonthPlus( value , 1 );		// 5.5.7.2 (2012/10/09) HybsDateUtil を利用
	}

	/**
	 * String引数の文字列に、第２引数に指定の文字列(数字、日付等)を加算して返します。
	 *
	 * ここでは、yyyyMM 形式のデータに、月を加算します。
	 * 0000XX と、9999XX は、引数の値そのままを返します。
	 * 引数が null の場合と、ゼロ文字列("")の場合は,物理的初期設定値(String getDefault())
	 * の値を返します。
	 * それ以外の場合は、日付計算を行い、月を加算します。よって、年月フォーマットに
	 * 合わないデータを指定した場合は、エラーになりますので、ご注意ください。
	 *
	 * @og.rev 5.6.0.3 (2012/01/24) ADD に、引数の値を加算する機能を追加します。
	 *
	 * @param   value  String引数
	 * @param   add    加算する月文字列
	 *
	 * @return  引数の文字列に月を加算します。
	 */
	@Override
	public String valueAdd( final String value,final String add ) {
		if( value == null || value.isEmpty() ) { return getDefault(); }
		if( value.startsWith( "0000" ) || value.startsWith( "9999" ) ) { return value; }	// 特別な値

		int addSu = 1;
		if( add != null && !add.isEmpty() ) {
			addSu = Integer.parseInt( add );
		}

		return HybsDateUtil.getMonthPlus( value , addSu );
	}

	/**
	 * エディターで編集されたデータを登録する場合に、データそのものを
	 * 変換して、実登録データを作成します。
	 * 例えば,大文字のみのフィールドなら、大文字化します。
	 * 実登録データの作成は、DBType オブジェクトを利用しますので,
	 * これと Editor とがアンマッチの場合は、うまくデータ変換
	 * されない可能性がありますので、注意願います。
	 *
	 * @og.rev 5.5.7.2 (2012/10/09) HybsDateUtil を利用するように修正します。
	 * @og.rev 5.6.6.0 (2013/07/05) parseDate を改造し、日付の桁数指定を厳密に判定します。
	 *
	 * @param	value	(一般に編集データとして登録されたデータ)
	 * @return  修正後の文字列(一般にデータベースに登録するデータ)
	 */
	@Override
	public String valueSet( final String value ) {
		if( value == null || value.isEmpty() ) { return getDefault(); }

		return HybsDateUtil.parseDate( value , 6 );		// 不要な記号等を削除し、桁数をそろえる。
	}

	/**
	 * データが登録可能かどうかをチェックします。
	 * データがエラーの場合は、そのエラー内容を返します。
	 * "000000" と、"999999" は、常に許可になります。
	 *
	 * @og.rev 5.6.0.3 (2012/01/24) 日付チェックの整合性まで含めたチェックを入れる。
	 *
	 * @param   key		キー
	 * @param   value	値
	 * @param   sizeX	整数部分の文字列の長さ
	 * @param   sizeY	小数部分の文字列の長さ
	 * @param   typeParam	dbType パラメータ
	 * @param   isStrict	厳密にチェックするかどうか[true:する/false:標準的]
	 *
	 * @return  エラー内容
	 * @og.rtnNotNull
	 */
	@Override
	public ErrorMessage valueCheck( final String key ,final String value ,
									final int sizeX ,final int sizeY ,final String typeParam ,final boolean isStrict) {

		final ErrorMessage msg = new ErrorMessage();
		if( value == null || value.isEmpty() ) { return msg; }

		final int len = (sizeY == 0) ? sizeX : sizeX + sizeY + 1;
		// 厳密に一致しないとエラーとする。
		if( len != value.length() ) {
			// 文字列の長さが指定の長さと異なります。
			msg.addMessage( 0,ErrorMessage.NG,"ERR0011", key,value, 
									String.valueOf( value.length() ), String.valueOf( len ) );
		}

		// 5.6.0.3 (2012/01/24) 文字の範囲チェック
		String check = DBTypeCheckUtil.rangeCheck( value, '0', '9' );
		if( check != null ) {
			// 指定の文字以外の文字が使われています。
			msg.addMessage( 0,ErrorMessage.NG,"ERR0009", key,check );
		}

		// 5.6.0.3 (2012/01/24) 日付文字の整合性チェック
		// "0000","9999" で始まる日付は未条件許可。
		check = DBTypeCheckUtil.ymdFormatCheck( value );
		if( check != null ) {
			// 指定の文字以外の文字が使われています。
			msg.addMessage( 0,ErrorMessage.NG,"ERR0009", key,check );
		}

		// 3.6.0.0 (2004/09/22) dbType パラメータを使用したマッチチェック
		check = DBTypeCheckUtil.matcheCheck( value,typeParam );
		if( check != null ) {
			// 指定の文字以外の文字が使われています。
			msg.addMessage( 0,ErrorMessage.NG,"ERR0009", key,check );
		}

		return msg;
	}
}
