/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.SystemManager ;					// 6.3.8.4 (2015/10/09)
import org.opengion.hayabusa.resource.GUIInfo;
import org.opengion.fukurou.util.Attributes;
import org.opengion.fukurou.util.XHTMLTag;
import org.opengion.fukurou.util.ToString;							// 6.1.1.0 (2015/01/17)
import org.opengion.fukurou.util.StringUtil ;						// 6.2.2.0 (2015/03/27)
import org.opengion.fukurou.util.Cleanable;							// 6.3.8.4 (2015/10/09)
// import org.opengion.fukurou.util.FileInfo;						// 6.3.8.4 (2015/10/09)
import org.opengion.fukurou.util.FileMap;							// 6.3.8.4 (2015/10/09)
import org.opengion.fukurou.db.DBUtil;								// 6.3.8.4 (2015/10/09)

import static org.opengion.fukurou.util.StringUtil.nval ;

// import java.io.File;
import java.util.Set;												// 6.3.8.4 (2015/10/09)
import java.util.HashSet;											// 6.3.8.4 (2015/10/09)
// import java.util.Map;											// 6.3.8.4 (2015/10/09)
// import java.util.HashMap;										// 6.3.8.4 (2015/10/09)

/**
 * 画面ＩＤと同じヘルプファイルがあればリンクを作成するタグです(通常は query.jsp に組込み)。
 *
 * ヘルプファイルは、システムパラメータ の HELP_URL で定義されているhelpフォルダに配置します。
 * このフォルダに、画面IDと同じファイル(例えば、GE0001.html など)があれば、リンクを作成します。
 * ファイルがなければ、リンクは表示されません。
 * メッセージの表示の制御は、viewMsg 属性で指定します。(false でファイルが存在した場合のみ表示)
 * ファイルの拡張子も指定できますが、一般に、html でヘルプファイルを作成するほうが
 * すばやく表示できます。
// * ※ 6.3.8.4 (2015/10/09) topMenuタグ内のhelpタグ機能 廃止
// * また、og:topMenuタグ内にこのタグを記述することで、各画面分類に対するヘルプを表示することが
// * できるようになります。
 * (この場合も、画面分類のキーがヘルプファイルのキーになります)
 *
 * ※ 6.3.8.4 (2015/10/09)
 *    ヘルプファイルと、GE80(FAQﾃｰﾌﾞﾙ)の関連画面IDについて、先に検索してMapにセットする方法に変更します。
 *    従来は、ヘルプファイルは都度、GE80(FAQﾃｰﾌﾞﾙ)は、org.opengion.hayabusa.resource.GUIDataLoader
 *    で、読込処理していました。
 *
 * @og.formSample
 * ●形式：一般ユーザーが直接組み込むことはありません。
 * ●body：なし
 *
 * ●Tag定義：
 *   &lt;og:help
 *       guiInfoKey         【TAG】GUIInfo のキーを指定します
 *       extension          【廃止】拡張子を指定します(初期値:html)
 *       lbl                【TAG】ラベルリソースのラベルIDを指定します
 *       target             【TAG】TARGET 属性を指定します(初期値:_blank)
 *       viewMsg            【TAG】メッセージを常時表示させるかどうか[true/false]を指定します(初期値:false)
 *       iconURL            【TAG】ヘルプリンクをアイコンで指定する場合のアイコンURLを指定します (初期値:DEFAULT_HELP_ICON[=/image/help.png])
 *       faqIconURL         【TAG】FAQリンクをアイコンで指定する場合のアイコンURLを指定します (初期値:DEFAULT_FAQ_ICON[=/image/qaicon.png])
 *       useFaq             【TAG】FAQ表示の機能を利用するかどうか[true/false]を指定します (初期値:USE_GUI_FAQ[=false])
 *       debug              【TAG】デバッグ情報を出力するかどうか[true/false]を指定します(初期値:false)
 *   /&gt;
 *
 * ●使用例
 *     &lt;og:help guiInfoKey="{&#064;GUI.KEY}" lbl="HELP" /&gt;
 *
 *     &lt;og:help
 *        guiInfoKey    = "GUIInfo のキーを指定します(必須)。"
 *        extension     = "6.3.8.4 (2015/10/09) 廃止。拡張子を指定します(初期値:html)。"
 *        lbl           = "ラベルリソースのメッセージIDを指定します。"
 *        target        = "TARGET 属性を指定します(初期値:_blank)。"
 *        viewMsg       = "メッセージを常時表示させるかどうか[true/false]を指定します(初期値:false)。"
 *        iconURL       = "ヘルプアイコンのURL(初期値:/image/help.png)"; // 5.3.8.0 (2011/08/01)
 *     /&gt;
 *
 * @og.group メニュー制御
 *
 * @version  4.0
 * @author	 Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class HelpTag extends CommonTagSupport {
	/** このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "6.3.8.4 (2015/10/09)" ;
	private static final long serialVersionUID = 638420151009L ;

	private static final String	JSP = HybsSystem.sys( "JSP" );

	// 6.3.8.4 (2015/10/09) GE80(FAQﾃｰﾌﾞﾙ)の関連画面ID のキャッシュ
	// GE80(FAQﾃｰﾌﾞﾙ)の関連画面IDの初期一括読み込みのクエリー
	// 関連画面ID は、後で分割します。
	private final static String QUERY = "select KNRNGUI from GE80 where SYSTEM_ID = ? and FGJ='1' and KNRNGUI is not null" ;

	// リソースの接続先を、取得します。
	private final static String DBID = HybsSystem.sys( "RESOURCE_DBID" );

	// 6.3.8.4 (2015/10/09) ヘルプファイルと、GE80(FAQﾃｰﾌﾞﾙ)の関連画面ID のキャッシュ
	private static 		 FileMap		helpMap	;
	private static final Set<String>	faqSet	= new HashSet<>();

	// 6.3.8.4 (2015/10/09) ヘルプファイルと、GE80(FAQﾃｰﾌﾞﾙ)の関連画面ID のキャッシュ
	static {
		final Cleanable clr = new Cleanable() {
			/**
			 * 初期化(クリア)します。
			 * 主に、キャッシュクリアで利用します。
			 */
			public void clear() {
				helpMap = null;
				faqSet.clear();
			}
		};
		SystemManager.addCleanable( clr );
	}

	private String	guiInfoKey	;
//	private String	extension	= "html";		// 6.3.8.4 (2015/10/09) 廃止
//	private String  baseURL		= HybsSystem.sys( "HELP_URL" );
	private String	target		= "_blank";		// 3.6.0.7 (2004/11/06)
	private boolean	viewMsg		;
	private String  iconURL		= HybsSystem.sys( "DEFAULT_HELP_ICON" );	// 5.4.3.6 (2012/01/19)
	private String  faqIconURL	= HybsSystem.sys( "DEFAULT_FAQ_ICON" );		// 5.5.0.4 (2012/03/16)
	private String  faqGUI		= HybsSystem.sys( "DEFAULT_FAQ_GUI" );		// 5.5.0.4 (2012/03/16)

	private boolean  useFaq		= HybsSystem.sysBool( "USE_GUI_FAQ" );		// 5.6.7.3 (2013/08/23)
	private boolean  useFaqCtrl	= HybsSystem.sysBool( "USE_GUI_FAQ_CTRL" ); // 6.3.8.4 (2015/10/09) 廃止

	/**
	 * Taglibの終了タグが見つかったときに処理する doEndTag() を オーバーライドします。
	 *
	 * @og.rev 3.1.1.2 (2003/04/04) Tomcat4.1 対応。release2() を doEndTag()で呼ぶ。
	 * @og.rev 5.3.9.0 (2011/09/01) メニューでのヘルプアイコン対応
	 * @og.rev 5.5.0.4 (2012/03/16) FAQ対応
	 * @og.rev 5.6.4.3 (2013/05/26) FAQの画面別対応
	 * @og.rev 6.3.8.4 (2015/10/09) topMenu 内でのHelp機能を廃止します。
	 *
	 * @return	後続処理の指示
	 */
	@Override
	public int doEndTag() {
		debugPrint();		// 4.0.0 (2005/02/28)

		// 6.3.8.4 (2015/10/09) 初期化されているかどうかの判定を、helpMap で行う。
		if( helpMap == null ) {
			final String baseURL = HybsSystem.sys( "HELP_URL" );
			helpMap = new FileMap( HybsSystem.url2dir( baseURL ) , baseURL );
			loadGE80();
		}

		// 6.3.8.4 (2015/10/09) topMenu 内でのHelp機能を廃止します。
//		final TopMenuTag topMenu = (TopMenuTag)findAncestorWithClass( this,TopMenuTag.class );
//		if( topMenu == null ) {
//			jspPrint( makeTag() );
			jspPrint( makeHelpTag() );			// メソッド名変更
			if( useFaq ) {
				jspPrint( makeTagFaq() );
			}
//		}
//		else {
//			// 5.3.9.0 (2011/09/01) メニューでのヘルプアイコン対応
//			final String linkFormat = getLink( baseURL + "{FILENAME}" );
//			final String baseDir = HybsSystem.url2dir( baseURL );
//			topMenu.add( "helpLinkFormat",linkFormat );
//			topMenu.add( "helpBaseDir",baseDir );
//			if(useFaq){ // 5.5.0.4 (2012/03/16) FAQ対応
//				final GUIInfo guiInfo = getGUIInfo( faqGUI );
//				if( guiInfo != null ) { 
//					final String address = guiInfo.getRealAddress( get( "href" ) );
//					final String faqFormat = getFAQLink(getRequestParameter( address+"?command=NEW&GAMENID="+faqGUI+"&KNRNGUI={GUIKEY}" ));
//					topMenu.add( "faqLinkFormat",faqFormat );
//				}
//			}
//		}

		return EVAL_PAGE ;
	}

	/**
	 * タグリブオブジェクトをリリースします。
	 * キャッシュされて再利用されるので、フィールドの初期設定を行います。
	 *
	 * @og.rev 2.0.0.4 (2002/09/27) カスタムタグの release() メソッドを、追加
	 * @og.rev 3.0.0.3 (2003/02/21) ターゲット属性の新規追加他
	 * @og.rev 3.1.1.2 (2003/04/04) Tomcat4.1 対応。release2() を doEndTag()で呼ぶ。
	 * @og.rev 3.6.0.7 (2004/11/06) target 属性の初期値を _new から _blank に変更
	 * @og.rev 5.3.8.0 (2011/08/01) iconURL追加
	 * @og.rev 5.5.0.4 (2012/03/16) faq
	 * @og.rev 5.6.4.3 (2013/05/24) faqCtrl
	 * @og.rev 5.6.7.3 (2013/08/23) useFaq と useFaqCtrl のキーの後ろにスペースが入っていた。
	 * @og.rev 6.3.8.4 (2015/10/09) extension 廃止。
	 */
	@Override
	protected void release2() {
		super.release2();
		guiInfoKey	= null;
//		extension	= "html";										// 6.3.8.4 (2015/10/09) 廃止
//		baseURL		= HybsSystem.sys( "HELP_URL" );					// 6.3.8.4 (2015/10/09) ローカル変数
		target		= "_blank";										// 3.6.0.7 (2004/11/06)
		viewMsg		= false;
		iconURL		= HybsSystem.sys( "DEFAULT_HELP_ICON" );		// 5.4.3.6 (2012/01/19)
		faqIconURL	= HybsSystem.sys( "DEFAULT_FAQ_ICON" );			// 5.5.0.4 (2012/03/16)
		faqGUI		= HybsSystem.sys( "DEFAULT_FAQ_GUI" );			// 5.5.0.4 (2012/03/16)

		useFaq		= HybsSystem.sysBool( "USE_GUI_FAQ" );			// 5.6.7.3 (2013/08/23)
		useFaqCtrl	= HybsSystem.sysBool( "USE_GUI_FAQ_CTRL" );		// 5.6.7.3 (2013/08/23)
	}

	/**
	 * HELPリンクを作成します。
	 *
	 * @og.rev 3.0.0.3 (2003/02/21) ターゲット属性の新規追加
	 * @og.rev 3.0.1.0 (2003/03/03) viewMsg フラグの制御のバグ修正
	 * @og.rev 5.3.8.0 (2011/08/01) iconURL対応
	 * @og.rev 5.3.9.0 (2011/09/01) メニューでのヘルプアイコン対応
	 * @og.rev 5.5.0.4 (2012/03/16) faq
	 * @og.rev 6.3.8.4 (2015/10/09) ヘルプファイルと、GE80(FAQﾃｰﾌﾞﾙ)の関連画面ID のキャッシュ
	 * @og.rev 6.3.8.4 (2015/10/09) メソッド名変更と、private 化。
	 *
	 * @return	リンクタグ文字列
	 */
//	protected String makeTag() {
	private String makeHelpTag() {
		String rtn = "";

		if( guiInfoKey == null ) {
			guiInfoKey = getGUIInfoAttri( "KEY" );
		}

		// 6.3.8.4 (2015/10/09) ヘルプファイルと、GE80(FAQﾃｰﾌﾞﾙ)の関連画面ID のキャッシュ
		final String url = helpMap.getFilename( guiInfoKey );
		if( url != null ) {
			rtn = getLink( url );
		}

//		final String url = baseURL + guiInfoKey + "." + extension;
//		final File  file = new File( HybsSystem.url2dir( url ) );

//		// ファイルの存在チェック
//		if( file.exists() ) {						// 3.5.6.0 (2004/06/18)
//			// 5.3.9.0 (2011/09/01) メニューでのヘルプアイコン対応
//			rtn = getLink( url );
//		}
		else if( viewMsg ) {
			rtn = getLinkBody( null,getMsglbl() );	// 5.5.0.4
		}

		return rtn;
	}

	/**
	 * FAQリンクを作成します。
	 *
	 * @og.rev 5.3.9.0 (2011/09/01) メニューでのヘルプアイコン対応
	 * @og.rev 5.6.4.3 (2013/05/24) FAQ存在チェック対応
	 * @og.rev 6.3.8.4 (2015/10/09) 判定ロジックを、画面リソースではなく、内部Setで行う。
	 *
	 * @return	リンクタグ文字列
	 */
//	protected String makeTagFaq() {
	private String makeTagFaq() {
		String rtn = "";

//		// 6.3.8.4 (2015/10/09) 判定ロジック変更。
//		if( !useFaqCtrl && !"true".equals(getGUIInfoAttri( "FAQ" ) ) ) { return rtn; } // 5.6.4.3 (2013/05/24) 若干やっつけ

		if( guiInfoKey == null ) {
			guiInfoKey = getGUIInfoAttri( "KEY" );
		}

		// 6.3.8.4 (2015/10/09) 判定ロジック変更。前の判定方法は、間違っていたため、修正。
		if( !useFaqCtrl || faqSet.contains( guiInfoKey ) ) {
			final GUIInfo guiInfo = getGUIInfo( faqGUI );
			if( guiInfo == null ) { return rtn; }	// 見つからない場合は、アクセス不可

			final String address = guiInfo.getRealAddress( get( "href" ) );
			// KNRNGUI は、前後に %(のURLencode文字)を付けます。RequestParameter 処理は不要 
//			final String url = getRequestParameter( address+"?command=NEW&GAMENID="+faqGUI+"&KNRNGUI="+guiInfoKey );
			final String url = address+"index.jsp?command=NEW&GAMENID="+faqGUI+"&KNRNGUI=%25"+guiInfoKey+"%25" ;

			rtn = getFAQLink( url );
		}

		return rtn;
	}

	/**
	 * リンク文字列を作成します。
	 *
	 * @og.rev 5.3.9.0 (2011/09/01) 新規作成
	 * @og.rev 5.5.0.4 (2012/03/16) faq
	 *
	 * @param	url	リンクのURL
	 *
	 * @return	リンク文字列
	 * @og.rtnNotNull
	 */
	private String getLink( final String url ) {
		// 6.1.1.0 (2015/01/17) Attributesの連結記述
		return XHTMLTag.link(
				new Attributes()
					.set( "href"	, getContextPath() + "/" + url )
					.set( "body"	, getLinkBody( iconURL,getMsglbl() ) )		//5.5.0.4 (2012/03/16)
					.set( "target"	, target )
					.set( "class"	, "helplink" )
			);

	}

	/**
	 * FAQリンク文字列を作成します。
	 *
	 * @og.rev 5.5.0.4 (2012/03/16) 新規作成
	 *
	 * @param	url	リンクのURL
	 *
	 * @return	リンク文字列
	 * @og.rtnNotNull
	 */
	private String getFAQLink( final String url ) {
		// 6.1.1.0 (2015/01/17) Attributesの連結記述
		return XHTMLTag.link(
				new Attributes()
					.set( "href"	, url )
					.set( "body"	, getLinkBody( faqIconURL,"FAQ" ) )
					.set( "target"	, target )
					.set( "class"	, "faqlink" )
			) ;

	}

	/**
	 * リンクのボディー部分を作成します。
	 *
	 * @og.rev 5.3.8.0 (2011/08/01) 新規作成
	 * @og.rev 5.3.9.0 (2011/09/01) 画像表示時にtitle属性を付加
	 * @og.rev 5.5.0.4 (2012/03/16) 引数対応
	 * @og.rev 6.2.2.0 (2015/03/27) BRと\nを相互に変換する処理を追加
	 * @og.rev 6.2.2.3 (2015/04/10) htmlフィルターに、BR→改行処理機能を追加。
	 *
	 * @param	icon	アイコン
	 * @param	title	タイトル
	 *
	 * @return	リンクボディー文字列
	 * @og.rtnNotNull
	 */
	private String getLinkBody(final String icon, final String title) {
		// 6.1.1.0 (2015/01/17) refactoring. ロジックの見直し

		return icon == null || icon.isEmpty()
							? getMsglbl()
							: "<img src=\"" + JSP + icon + "\" title=\"" + StringUtil.htmlFilter( title,true ) + "\"/>";
	}

	/**
	 * 【TAG】GUIInfo のキーを指定します。
	 *
	 * @og.tag GUIInfo のキーを指定します。
	 *
	 * @param	key	GUIInfoのキー
	 */
	public void setGuiInfoKey( final String key ) {
		guiInfoKey = getRequestParameter( key );
	}

	/**
	 * 【廃止】拡張子を指定します(初期値:html)。
	 *
	 * @og.tag
	 * なにも設定されていない場合は、"html" が初期値となります。
	 * ここでは、ピリオドは、含める必要はありません。
	 *
	 * @og.rev 6.3.8.4 (2015/10/09) 廃止。
	 *     フォルダをスキャンして、必要なHelpファイルを見つけるため、
	 *     拡張子を指示する必要がなくなりました。
	 *
	 * @param	ext 拡張子
	 */
	public void setExtension( final String ext ) {
//		extension = nval( getRequestParameter( ext ),extension );
	}

	/**
	 * 【TAG】TARGET 属性を指定します(初期値:_blank)。
	 *
	 * @og.tag
	 * 初期値は、 "_blank" として、新規に画面を立ち上げます。
	 * CONTENTS 等を指定すれば、コンテンツフレーム(メニューの右側全面)に、
	 * RESULT を指定すれば、リザルトフレーム(メニュー右下側)に表示します。
	 *
	 * @og.rev 3.0.0.3 (2003/02/21) ターゲット属性の新規追加
	 *
	 * @param	val TARGET属性(初期値:"_blank")
	 */
	public void setTarget( final String val ) {
		target = nval( getRequestParameter( val ),target );
	}

	/**
	 * 【TAG】メッセージを常時表示させるかどうか[true/false]を指定します(初期値:false)。
	 *
	 * @og.tag
	 * "true"の場合は、常時表示させます。
	 * ファイルが、存在した場合は、リンクが張られ、存在しない場合は、リンクが
	 * 張られません。
	 * "false" の場合は、ファイルが、存在した場合は、リンクが張られ、存在しない場合は、
	 * なにも表示されません。
	 * 初期値は、 "false"(メッセージを常時表示しない)です。
	 *
	 * @og.rev 3.0.0.3 (2003/02/21) メッセージ表示属性の新規追加
	 *
	 * @param	flag メッセージ常時表示 [true:常時表示/false:非表示]
	 */
	public void setViewMsg( final String flag ) {
		viewMsg = nval( getRequestParameter( flag ),viewMsg );
	}

	/**
	 * 【TAG】ヘルプリンクをアイコンで指定する場合のアイコンURLを指定します
	 *		(初期値:DEFAULT_HELP_ICON[={@og.value SystemData#DEFAULT_HELP_ICON}])。
	 *
	 * @og.tag
	 * ヘルプリンクをアイコンで指定する場合、そのアイコン画像のURLを指定します。
	 * URLは、/[CONTEXT_PATH]/jspを基準として指定します。
	 * 例) /ge/jsp/image/help.pngに存在する画像を指定する場合、iconURL=/image/help.pngを指定します。
	 * このURLが指定されない場合、ヘルプリンクは、msgLbl属性で指定されたテキストで表示されます。
	 * (初期値:システム定数のDEFAULT_HELP_ICON[={@og.value SystemData#DEFAULT_HELP_ICON}])。
	 *
	 * @og.rev 5.3.8.0 (2011/08/01) 新規追加
	 *
	 * @param url アイコンURL
	 * @see		org.opengion.hayabusa.common.SystemData#DEFAULT_HELP_ICON
	 */
	public void setIconURL( final String url ) {
		iconURL = nval( getRequestParameter( url ),iconURL );
	}

	/**
	 * 【TAG】FAQリンクをアイコンで指定する場合のアイコンURLを指定します
	 *		(初期値:DEFAULT_FAQ_ICON[={@og.value SystemData#DEFAULT_FAQ_ICON}])。
	 *
	 * @og.tag
	 * FAQリンクをアイコンで指定する場合、そのアイコン画像のURLを指定します。
	 * URLは、/[CONTEXT_PATH]/jspを基準として指定します。
	 * 例) /ge/jsp/image/help.pngに存在する画像を指定する場合、iconURL=/image/help.pngを指定します。
	 * (初期値:システム定数のDEFAULT_FAQ_ICON[={@og.value SystemData#DEFAULT_FAQ_ICON}])。
	 *
	 * @og.rev 5.3.8.0 (2011/08/01) 新規追加
	 *
	 * @param url アイコンURL
	 * @see		org.opengion.hayabusa.common.SystemData#DEFAULT_FAQ_ICON
	 */
	public void setFaqIconURL( final String url ) {
		faqIconURL = nval( getRequestParameter( url ),faqIconURL );
	}

	/**
	 * 【TAG】FAQ表示の機能を利用するかどうか[true/false]を指定します
	 *		(初期値:USE_GUI_FAQ[={@og.value SystemData#USE_GUI_FAQ}])。
	 *
	 * @og.tag
	 * trueを指定すると、FAQ画面へのリンクが表示されます。(GE80にデータが存在するかは無関係)
	 * リンク先はfaqGUIでセットした画面に対して画面IDを引数としてわたします。
	 * (初期値:システム定数のUSE_GUI_FAQ[={@og.value SystemData#USE_GUI_FAQ}])。
	 *
	 * @og.rev 5.5.0.4 (2012/03/167) 新規追加
	 *
	 * @param	flag FAQ表示利用 [true:利用する/false:利用しない]
	 * @see		org.opengion.hayabusa.common.SystemData#USE_GUI_FAQ
	 * @see		#setUseFaqCtrl( String )
	 */
	public void setUseFaq( final String flag ) {
		useFaq = nval( getRequestParameter( flag ),useFaq );
	}

	/**
	 * 【TAG】FAQに関連画面機能を利用するかどうか[true/false]を指定します
	 *		(初期値:USE_GUI_FAQ_CTRL[={@og.value SystemData#USE_GUI_FAQ_CTRL}])。
	 *
	 * @og.tag
	 * trueを指定すると、GE80にデータが関連画面IDとして存在する場合のみアイコンを
	 * リンク先はfaqGUIでセットした画面に対して画面IDを引数として渡します。
	 * falseの場合は、瀬音ｚ内チェックを行わないため、常に表示されます。
	 * (ただし、useFaq=true の場合のみ)
	 * (初期値:システム定数のUSE_GUI_FAQ[={@og.value SystemData#USE_GUI_FAQ}])。
	 *
	 * @og.rev 5.6.4.3 (2013/05/24) 新規追加
	 *
	 * @param	flag FAQの存在チェック機能 [true:利用する/false:利用しない]
	 * @see		org.opengion.hayabusa.common.SystemData#USE_GUI_FAQ_CTRL
	 * @see		#setUseFaq( String )
	 */
	public void setUseFaqCtrl( final String flag ) {
		useFaqCtrl = nval( getRequestParameter( flag ),useFaqCtrl );
	}

	/**
	 * GE80(FAQﾃｰﾌﾞﾙ)より 関連画面IDを取得、分割して、FAQセットに設定します。
	 *
	 * 関連画面IDは、スペース、またはカンマ区切りとして、分割します。
	 * 分割後、trim() されたデータを、FAQセット に追加していきます。
	 * 画面IDは、存在チェックに使うため、重複は関係ありません。
	 *
	 * @og.rev 6.3.8.4 (2015/10/09) GE80(FAQﾃｰﾌﾞﾙ)の関連画面ID のキャッシュ
	 */
	private void loadGE80() {
		final String[] args = new String[] { HybsSystem.sys( "SYSTEM_ID" ) };

		final String[][] vals = DBUtil.dbExecute( QUERY,args,getApplicationInfo(),DBID );
		final int len = vals.length;

		for( int i=0; i<len; i++ ) {
			final String[] guis = vals[i][0].split( "[ ,]" );		// 最初のカラムを、スペースかカンマで分解する。
			for( int j=0; j<guis.length; j++ ) {
				faqSet.add( guis[j] );
			}
		}
	}

	/**
	 * このオブジェクトの文字列表現を返します。
	 * 基本的にデバッグ目的に使用します。
	 *
	 * @return このクラスの文字列表現
	 * @og.rtnNotNull
	 */
	@Override
	public String toString() {
		return ToString.title( this.getClass().getName() )
				.println( "VERSION"		,VERSION	)
				.println( "guiInfoKey"	,guiInfoKey	)
//				.println( "extension"	,extension	)		// 6.3.8.4 (2015/10/09) 廃止。
//				.println( "baseURL"		,baseURL	)		// 6.3.8.4 (2015/10/09) ローカル化。
				.println( "target"		,target		)
				.println( "viewMsg"		,viewMsg	)
				.println( "iconURL"		,iconURL	)
				.println( "Other..."	,getAttributes().getAttribute() )
				.fixForm().toString() ;
	}
}
