/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import org.opengion.fukurou.db.Transaction;
// import org.opengion.fukurou.db.TransactionReal;
import org.opengion.fukurou.db.ResultSetValue;			// 6.0.4.0 (2014/11/28)
import org.opengion.fukurou.util.ErrorMessage;
import org.opengion.fukurou.util.FileUtil;
import org.opengion.fukurou.util.StringUtil;
import org.opengion.fukurou.util.Closer ;
import org.opengion.fukurou.util.ToString;						// 6.1.1.0 (2015/01/17)
import static org.opengion.fukurou.util.StringUtil.nval ;
import static org.opengion.fukurou.util.HybsConst.BR;		// 6.1.0.0 (2014/12/26) refactoring

import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.hayabusa.resource.GUIInfo;
import org.opengion.hayabusa.resource.ResourceManager;
import org.opengion.hayabusa.db.DBErrMsg;

import java.sql.Connection;
import java.sql.Statement;
import java.sql.CallableStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Types;
import java.sql.Array;								// 6.0.0.0 (2014/04/11) Oracle11g(11.2.0.3のドライバ) 対応。oracle.sql.ARRAY の置き換え
import oracle.jdbc.OracleConnection;				// 6.0.0.0 (2014/04/11) Oracle11g(11.2.0.3のドライバ) 対応
import oracle.jdbc.OracleTypes;						// CURSOR が残る
import oracle.jdbc.OracleCallableStatement;			// CURSOR が残る

import java.io.File;
import java.io.PrintWriter;
import java.io.FileOutputStream;
import java.io.BufferedOutputStream;				// 6.0.4.0 (2014/11/28)
// import java.io.BufferedWriter;
// import java.io.OutputStreamWriter;

import java.io.IOException;
// import java.io.ObjectOutputStream;
// import java.io.ObjectInputStream;
import org.apache.commons.compress.archivers.zip.ZipArchiveEntry;			// 6.0.4.0 (2014/11/28)
import org.apache.commons.compress.archivers.zip.ZipArchiveOutputStream;	// 6.0.4.0 (2014/11/28)

import java.util.Map;

/**
 * SELECT文を直接実行して、指定のファイルに出力するタグです。
 *
 * 中間の、データ(DBTableModel)を作成しないため、余計なメモリを取らず、
 * 高速にデータを抜き出すことが可能です。
 * 一方、抜き出すデータは生データのため、データの再利用等、システム的な
 * 使用を想定しています。
 * JDBCErrMsg 形式のPL/SQL をコールして、その検索結果(カーソル)を抜く事もできます。
 *
 * ※ このタグは、Transaction タグの対象です。
 *
 * @og.formSample
 * ●形式：&lt;og:directWriteTable filename="[･･･]" ･･･ &gt;SELECT * FROM ZYXX &lt;/og:directWriteTable &gt;
 * ●body：あり(EVAL_BODY_BUFFERED:BODYを評価し、{&#064;XXXX} を解析します)
 *
 * ●Tag定義：
 *   &lt;og:directWriteTable
 *       fileURL            【TAG】保存先ディレクトリ名を指定します (初期値:FILE_URL[=filetemp/])
 *       filename           【TAG】ファイルを作成するときのファイル名をセットします(初期値:システムパラメータのFILE_FILENAME)
 *       encode             【TAG】ファイルを作成するときのファイルエンコーディング名をセットします (初期値:FILE_ENCODE[=UnicodeLittle])
 *       fileAppend         【TAG】追加モードで書き込むかどうか[true/false]を指定します(初期値:false[通常モード])
 *       zip                【TAG】結果をファイルに出力するときに、ZIPで圧縮するかどうか[true/false]を指定します(初期値:false)
 *       zipFilename        【TAG】ZIPファイルを作成するときのZIPファイル名をセットします(初期値:filename + ".zip")
 *       separator          【TAG】可変長ファイルを作成するときの項目区切り文字をセットします (初期値:TAB_SEPARATOR[={@og.value #TAB_SEPARATOR}])
 *       useHeader          【TAG】ヘッダーを書き込むかどうか[true/false]を指定します(初期値:true)
 *       useQuote           【TAG】データをダブルクオートで囲うかどうか指定します(初期値:false)
 *       useQuoteEscape     【TAG】データ中にダブルクオート文字が含まれる場合、エスケープするかどうか指定します(初期値:true)
 *       useReturnQuote     【TAG】データ中に改行コードが含まれる場合、ダブルクオートで囲うかどうか指定します(初期値:true)
 *       replaceFrom        【TAG】置換元文字を指定。一文字単位で置換します(初期値:null 置換なし)。
 *       replaceTo          【TAG】置換先文字を指定。一文字単位で置換します。
 *       displayMsg         【TAG】検索結果を画面上に表示するメッセージリソースIDを指定します(初期値:VIEW_DISPLAY_MSG[=])
 *       notfoundMsg        【TAG】検索結果がゼロ件の場合に表示するメッセージリソースIDを指定します(初期値:MSG0077[対象データはありませんでした])
 *       fetchSize          【TAG】(通常は使いません)データのフェッチサイズを指定します(初期値:100)
 *       names              【TAG】PL/SQLを利用する場合の引数にセットすべき データの名称をCSV形式で複数指定します
 *       queryType          【TAG】Query を発行する為のクラスID(JDBC,JDBCErrMsg)を指定します({@og.doc03Link queryType 初期値:JDBC})
 *       dbid               【TAG】(通常は使いません)検索時のDB接続IDを指定します(初期値:DEFAULT)
 *       useNumber          【TAG】行番号を出力するかどうか(初期値:true)
 *       quotCheck          【TAG】リクエスト情報の シングルクォート(') 存在チェックを実施するかどうか[true/false]を設定します(初期値:USE_SQL_INJECTION_CHECK)
 *       xssCheck           【TAG】リクエスト情報の HTMLTag開始/終了文字(&gt;&lt;) 存在チェックを実施するかどうか[true/false]を設定します (初期値:USE_XSS_CHECK[=true])
 *       useTimeView        【TAG】処理時間を表示する TimeView を表示するかどうかを指定します
 *										(初期値:VIEW_USE_TIMEBAR[={@og.value SystemData#VIEW_USE_TIMEBAR}])。
 *       caseKey            【TAG】このタグ自体を利用するかどうかの条件キーを指定します(初期値:null)
 *       caseVal            【TAG】このタグ自体を利用するかどうかの条件値を指定します(初期値:null)
 *       caseNN             【TAG】指定の値が、null/ゼロ文字列 でない場合(Not Null=NN)は、このタグは使用されます(初期値:判定しない)
 *       caseNull           【TAG】指定の値が、null/ゼロ文字列 の場合は、このタグは使用されます(初期値:判定しない)
 *       caseIf             【TAG】指定の値が、true/TRUE文字列の場合は、このタグは使用されます(初期値:判定しない)
 *       debug              【TAG】デバッグ情報を出力するかどうか[true/false]を指定します(初期値:false)
 *   &gt;   ... Body ...
 *   &lt;/og:directWriteTable&gt;
 *
 * ●使用例
 *     &lt;og:directWriteTable
 *         dbid        = "ORCL"               接続データベースID(初期値:DEFAULT)
 *         separator   = ","                  ファイルの区切り文字(初期値:タブ)
 *         fileURL     = "{&#064;USER.ID}"    保存先ディレクトリ名
 *         filename    = "{&#064;filename}"   保存ファイル名
 *         encode      = "UnicodeLittle"      保存ファイルエンコード名
 *         useHeader   = "true"               保存ファイルにヘッダーを出力するかどうか
 *         useQuote    = "false"              データをダブルクオートで囲うかどうか
 *         useQuoteEscape = "true"            ダブルクオート文字が含まれる場合、エスケープするかどうか
 *         useReturnQuote = "true"            改行コードが含まれる場合、ダブルクオートで囲うかどうか
 *         replaceFrom = "&apos;,&quot;*%|"   置換元文字を指定。一文字単位で置換します。
 *         replaceTo   = "’，”＊％｜"       置換先文字を指定。一文字単位で置換します。
 *         zip         = "true"               ZIPファイルに圧縮するかどうか
 *         zipFilename = "Sample.zip"         ZIPファイルのファイル名
 *         fileAppend  = "true"               ファイルを追加モードで登録するかどうか
 *         displayMsg  = "MSG0033"            実行後の表示メッセージ
 *         fetchSize   = "200"                DB検索する場合のフェッチするサイズ
 *     &gt;
 *         SELECT * FROM ZYXX 
 *     &lt;/og:directWriteTable &gt;
 *
 *     &lt;og:directWriteTable
 *         fileURL     = "{&#064;USER.ID}"    保存先ディレクトリ名
 *         filename    = "{&#064;filename}"   保存ファイル名
 *         names       = "AAA,BBB,CCC,･･･"    指定のキーに対応するリクエスト値を ARG_ARRAY にセットします。
 *         queryType   = "JDBCErrMsg"         JDBCErrMsg 形式のPL/SQL をコールします。
 *     &gt;
 *        { call PL/SQL(?,?,?,? ) } 
 *     &lt;/og:directWriteTable &gt;
 *
 * @og.rev 3.5.6.0 (2004/06/18) 新規作成
 * @og.rev 3.6.1.0 (2005/01/05) PL/SQLコール(JDBCErrMsg 形式)の実行を追加
 * @og.group ファイル出力
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class DirectWriteTableTag extends CommonTagSupport {
	/** このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "6.3.8.0 (2015/09/11)" ;
	private static final long serialVersionUID = 638020150911L ;

	private static final String TAB_SEPARATOR	= "\t" ;
	private static final String errMsgId		= HybsSystem.ERR_MSG_KEY;

	private final int DB_MAX_QUERY_TIMEOUT		= HybsSystem.sysInt( "DB_MAX_QUERY_TIMEOUT" ) ;
	private static final String ARG_ARRAY		= "ARG_ARRAY" ;
	private static final String ERR_MSG			= "ERR_MSG" ;
	private static final String ERR_MSG_ARRAY	= "ERR_MSG_ARRAY" ;

	// 4.0.0.0 (2007/10/10) dbid の初期値を、"DEFAULT" から null に変更
	private String	dbid		;
	private String  separator	= TAB_SEPARATOR;	// 項目区切り文字
	private boolean useHeader	= true;				// ヘッダーの使用可否
	private boolean useQuote	;					// 6.0.3.0 (2014/11/13) ダブルクオートで囲うかどうか
	private boolean useQuoteEscape	= true;			// 6.0.3.0 (2014/11/13) データ中にダブルクオート文字が含まれる場合、エスケープするかどうか
	private boolean useReturnQuote	= true;			// 6.0.3.0 (2014/11/13) データ中に改行コードが含まれる場合、ダブルクオートで囲うかどうか
	private String  fileURL		= HybsSystem.sys( "FILE_URL" );
	private String  filename	= HybsSystem.sys( "FILE_FILENAME" );	// ファイル名
	private String	sql			;
	private String  encode		= HybsSystem.sys( "FILE_ENCODE"   );	// ファイルエンコーディング  "DEFAULT","JISAutoDetect" ,"JIS", "EUC_JP", "MS932", "SJIS" , "Windows-31J" , "Shift_JIS"
	private boolean fileAppend	;					// ファイルをAPPENDモードで出力するか
	private boolean zip			;					// ファイルをZIPするか
	private String  zipFilename	;					// ZIPファイル名
	private String	displayMsg	= HybsSystem.sys( "VIEW_DISPLAY_MSG" );
	private String	notfoundMsg	= "MSG0077";		// 対象データはありませんでした。
	private long	dyStart		;					// 実行時間測定用のDIV要素を出力します。
	private boolean	useTimeView	= HybsSystem.sysBool( "VIEW_USE_TIMEBAR" );		// 6.3.6.0 (2015/08/16)
	private int		fetchSize	= 100 ;				// フェッチする行数(初期値:100)
	private boolean	useNumber	= true;				// 5.5.7.1(2012/10/05) 行番号出力

	private String  replaceFrom	;					// 6.0.3.0 (2014/11/13) 置換元文字を指定
	private String  replaceTo	;					// 6.0.3.0 (2014/11/13) 置換先文字を指定

	private boolean	quotCheck	= HybsSystem.sysBool( "USE_SQL_INJECTION_CHECK" );	// 6.2.2.0 (2015/03/27)
	private boolean	xssCheck	= HybsSystem.sysBool( "USE_XSS_CHECK" );			// 6.2.2.0 (2015/03/27)

	// 3.6.1.0 (2005/01/05) PL/SQLコール(JDBCErrMsg 形式)への対応
	private boolean	queryType	= true;				// ノーマルは、true/ JDBCErrMsg の時は、false
	private String	names		;					// 指定のリクエスト変数を、ARG_ARRAY にセットします。
	private int		errCode		= ErrorMessage.OK;
	private transient ErrorMessage errMessage	;

	/**
	 * Taglibの開始タグが見つかったときに処理する doStartTag() を オーバーライドします。
	 *
	 * @og.rev 6.0.3.0 (2014/11/13) 置換元文字,置換先文字のチェック
	 * @og.rev 6.3.4.0 (2015/08/01) caseKey,caseVal,caseNN,caseNull,caseIf 属性対応
	 *
	 * @return	後続処理の指示( EVAL_BODY_BUFFERED )
	 */
	@Override
	public int doStartTag() {
		if( !useTag() ) { return SKIP_BODY ; }	// 6.3.4.0 (2015/08/01)
		dyStart = System.currentTimeMillis();		// 時間測定用

		// 6.0.3.0 (2014/11/13) 置換元文字,置換先文字を指定
		if( replaceFrom != null || replaceTo != null ) {
			if( replaceFrom == null || replaceTo == null || replaceFrom.length() != replaceTo.length() ) {
				final String errMsg = "置換元文字と置換先文字の文字数が異なります。" + CR
							+ " replaceFrom=[" + replaceFrom + "] , replaceTo=[" + replaceTo + "]"
							+ CR ;
				throw new HybsSystemException( errMsg );
			}
		}

		return EVAL_BODY_BUFFERED ;	// Body を評価する。( extends BodyTagSupport 時)
	}

	/**
	 * Taglibのタグ本体を処理する doAfterBody() を オーバーライドします。
	 *
	 * @og.rev 3.8.6.3 (2006/11/30) SQL 文の前後のスペースを取り除きます。
	 * @og.rev 6.2.2.0 (2015/03/27) XSSチェック,クォートチェック をサポートします。
	 *
	 * @return	後続処理の指示(SKIP_BODY)
	 */
	@Override
	public int doAfterBody() {
		// 6.2.2.0 (2015/03/27) XSSチェック,クォートチェック をサポートします。
		useQuotCheck( quotCheck );
		useXssCheck( xssCheck );

		sql = getBodyString();
		if( sql == null || sql.isEmpty() ) {
			final String errMsg = "BODY 部の検索用 Select文は、必須です。";
			throw new HybsSystemException( errMsg );
		}
		sql = sql.trim();
		return SKIP_BODY ;				// Body を評価しない
	}

	/**
	 * Taglibの終了タグが見つかったときに処理する doEndTag() を オーバーライドします。
	 *
	 * @og.rev 3.6.1.0 (2005/01/05) PL/SQLコール(JDBCErrMsg 形式)への対応
	 * @og.rev 4.0.0.0 (2007/10/18) メッセージリソース統合( getResource().getMessage ⇒ getResource().getLabel )
	 * @og.rev 6.0.4.0 (2014/11/28) Zip処理を、ZipOutputStream → ZipArchiveOutputStream に変更
	 * @og.rev 6.3.4.0 (2015/08/01) caseKey,caseVal,caseNN,caseNull,caseIf 属性対応
	 * @og.rev 6.3.8.0 (2015/09/11) FileUtil#getPrintWriter( OutputStream,String ) を使用。
	 *
	 * @return	後続処理の指示
	 */
	@Override
	public int doEndTag() {
		debugPrint();		// 4.0.0 (2005/02/28)
		if( !useTag() ) { return EVAL_PAGE ; }	// 6.3.4.0 (2015/08/01)

		PrintWriter pw = null;
		final int executeCount;
		try {
			if( zip ) {
				final String directory = HybsSystem.url2dir( fileURL );

				if( zipFilename == null ) { zipFilename = filename + ".zip"; }
				ZipArchiveOutputStream gzip = null;			// 6.0.4.0 (2014/11/28)
				try {
					// 6.0.4.0 (2014/11/28) Zip処理を、ZipOutputStream → ZipArchiveOutputStream に変更
					gzip = new ZipArchiveOutputStream(
								new BufferedOutputStream (
									new FileOutputStream (
										new File( directory,zipFilename ))));	// 6.0.4.0 (2014/11/28)
					gzip.setEncoding( "Windows-31J" );
					gzip.putArchiveEntry( new ZipArchiveEntry( filename ) );
					// 6.0.4.0 (2014/11/28) ファイルのencode を指定できるようにする。
					// 6.3.8.0 (2015/09/11) FileUtil#getPrintWriter( OutputStream,String ) を使用。
//					pw = new PrintWriter(
//							new BufferedWriter(
//								new OutputStreamWriter( gzip,encode )));
					pw = FileUtil.getPrintWriter( gzip,encode );		// 6.3.8.0 (2015/09/11)
					executeCount = create( pw ) ;

					pw.flush();
					gzip.closeArchiveEntry();				// 6.0.4.0 (2014/11/28)
					gzip.finish() ;
				}
				finally {
					Closer.ioClose( gzip );		// 4.0.0 (2006/01/31) close 処理時の IOException を無視
				}
			}
			else {
				pw = getPrintWriter();
				executeCount = create( pw );
			}
		} catch( IOException ex ) {
			final String errMsg = "Error in DirectWriteTableTag: " + toString();
			throw new HybsSystemException( errMsg,ex );		// 3.5.5.4 (2004/04/15) 引数の並び順変更
		} finally {
			Closer.ioClose( pw );		// 4.0.0 (2006/01/31) close 処理時の IOException を無視
		}

		// 3.6.1.0 (2005/01/05) 検索結果の件数を、"DB.COUNT" キーでリクエストにセットする。
		setRequestAttribute( "DB.COUNT"   , String.valueOf( executeCount ) );
		setRequestAttribute( "DB.ERR_CODE", String.valueOf( errCode ) );

		final StringBuilder buf = new StringBuilder( BUFFER_MIDDLE );

		// 実行件数の表示
		if( executeCount > 0 && displayMsg != null && displayMsg.length() > 0 ) {
			buf.append( executeCount );
			buf.append( getResource().getLabel( displayMsg ) );
			buf.append( BR );
		}
		else if( executeCount == 0 && notfoundMsg != null && notfoundMsg.length() > 0 ) {
			buf.append( getResource().getLabel( notfoundMsg ) );
			buf.append( BR );
		}

		// 3.6.1.0 (2005/01/05) TaglibUtil.makeHTMLErrorTable メソッドを利用
		final String err = TaglibUtil.makeHTMLErrorTable( errMessage,getResource() );
		if( err != null && err.length() > 0 ) {
			buf.append( err );
			setSessionAttribute( errMsgId,errMessage );
		}
		else {
			removeSessionAttribute( errMsgId );
		}

		jspPrint( buf.toString() );

		// 3.6.1.0 (2005/01/05) 警告時に停止していましたが、継続処理させます。
		int rtnCode = EVAL_PAGE;
		if( errCode >= ErrorMessage.NG )  { 	// 異常
			rtnCode = SKIP_PAGE;
		}

		// 4.0.0 (2005/01/31) セキュリティチェック(データアクセス件数登録)
		final long dyTime = System.currentTimeMillis()-dyStart;
		final GUIInfo guiInfo = (GUIInfo)getSessionAttribute( HybsSystem.GUIINFO_KEY );
		if( guiInfo != null ) { guiInfo.addReadCount( executeCount,dyTime,sql ); }

		if( useTimeView ) {		// 6.3.6.0 (2015/08/16)
			// 時間測定用の DIV 要素を出力
			jspPrint( "<div id=\"queryTime\" value=\"" + (dyTime) + "\"></div>" );	// 3.5.6.3 (2004/07/12)
		}
		return rtnCode ;
	}

	/**
	 * タグリブオブジェクトをリリースします。
	 * キャッシュされて再利用されるので、フィールドの初期設定を行います。
	 *
	 * @og.rev 3.6.1.0 (2005/01/05) PL/SQLコール(JDBCErrMsg 形式)への対応
	 * @og.rev 4.0.0.0 (2007/10/10) dbid の初期値を、"DEFAULT" から null に変更
	 * @og.rev 5.5.7.1 (2012/10/05) useNumber追加
	 * @og.rev 6.0.3.0 (2014/11/13) useHeader,useQuote,useQuoteEscape,useReturnQuote,replaceFrom,replaceTo追加
	 * @og.rev 6.2.2.0 (2015/03/27) XSSチェック,クォートチェック をサポートします。
	 */
	@Override
	protected void release2() {
		super.release2();
		separator	= TAB_SEPARATOR;	// 項目区切り文字
		fileURL		= HybsSystem.sys( "FILE_URL" );
		filename	= HybsSystem.sys( "FILE_FILENAME" );	// ファイル名
		sql			= null;
		encode		= HybsSystem.sys( "FILE_ENCODE" );		// ファイルエンコーディング  "DEFAULT","JISAutoDetect" ,"JIS", "EUC_JP", "MS932", "SJIS" , "Windows-31J" , "Shift_JIS"
		fileAppend	= false;			// ファイルをAPPENDモードで出力するか
		zip			= false;			// ファイルをZIPするか
		zipFilename	= null;				// ZIPファイル名
		displayMsg	= HybsSystem.sys( "VIEW_DISPLAY_MSG" );
		notfoundMsg	= "MSG0077";		// 対象データはありませんでした。
		dbid		= null;
		fetchSize	= 100 ;				// フェッチする行数(初期値:0 参考にしない)
//		dyStart		= 0;
		useTimeView	= HybsSystem.sysBool( "VIEW_USE_TIMEBAR" );	// 6.3.6.0 (2015/08/16)
		queryType	= true;				// ノーマルは、true/ JDBCErrMsg の時は、false
		names		= null;				// 指定のリクエスト変数を、ARG_ARRAY にセットします。
		errCode		= ErrorMessage.OK;
		errMessage	= null;
		useNumber	= true;				// 5.5.7.1 (2012/10/05)
		useHeader	= true;				// ヘッダーの使用可否  … 6.0.3.0 (2014/11/13) 追加
		useQuote	= false;			// 6.0.3.0 (2014/11/13) ダブルクオートで囲うかどうか
		useQuoteEscape	= true;			// 6.0.3.0 (2014/11/13) データ中にダブルクオート文字が含まれる場合、エスケープするかどうか
		useReturnQuote	= true;			// 6.0.3.0 (2014/11/13) データ中に改行コードが含まれる場合、ダブルクオートで囲うかどうか
		replaceFrom	= null;				// 6.0.3.0 (2014/11/13) 置換元文字を指定
		replaceTo	= null;				// 6.0.3.0 (2014/11/13) 置換先文字を指定
		quotCheck	= HybsSystem.sysBool( "USE_SQL_INJECTION_CHECK" );	// 6.2.2.0 (2015/03/27)
		xssCheck	= HybsSystem.sysBool( "USE_XSS_CHECK" );			// 6.2.2.0 (2015/03/27)
	}

	/**
	 * 実オブジェクトを生成して，OutputStream に書き込みます。
	 *
	 * @og.rev 3.6.1.0 (2005/01/05) PL/SQLコール(JDBCErrMsg 形式)への対応
	 * @og.rev 3.8.6.0 (2006/09/29) ヘッダーにラベルを出力するように修正
	 * @og.rev 3.8.7.0 (2006/12/15) アクセスログ取得の為,ApplicationInfoオブジェクトを設定
	 * @og.rev 4.3.4.3 (2008/12/22) (Oracle11gDriver対応)PL/SQLコールの場合に、"クローズされた文です。"のエラーが発生する問題に対応
	 * @og.rev 5.1.9.0 (2010/08/01) Transaction 対応
	 * @og.rev 5.2.2.0 (2010/11/01) 改行を含む場合は、ダブルクオートを強制的に前後に追加する。
	 * @og.rev 5.2.2.0 (2010/11/01) ダブルクオートを含む場合は、その直前にダブルクオートを強制的に追加する。
	 * @og.rev 5.3.0.0 (2010/12/01) executeCall メソッドの引数見直し
	 * @og.rev 5.3.7.0 (2011/07/01) TransactionReal の引数変更
	 * @og.rev 5.5.7.1 (2012/10/05) useNumberの追加
	 * @og.rev 6.0.3.0 (2014/11/13) useQuote,useQuoteEscape,useReturnQuote,replaceFrom,replaceToの追加
	 * @og.rev 6.0.3.0 (2014/11/13) ヘッダーとラベルを、指定の separator で出力するように変更します。
	 * @og.rev 6.0.4.0 (2014/11/28) ResultSetValue を使用するように変更。
	 * @og.rev 6.2.0.0 (2015/02/27) データ出力の先頭カンマの判定処理変更
	 * @og.rev 6.3.6.1 (2015/08/28) Transaction でAutoCloseableを使用したtry-with-resources構築に対応。
	 *
	 * @param   out PrintWriterオブジェクト
	 *
	 * @return	検索件数
	 */
	private int create( final PrintWriter out )  {
		final int executeCount;
		Statement stmt = null;
		CallableStatement callStmt = null; // 4.3.4.3 (2008/12/22)
		ResultSet resultSet = null ;
//		boolean errFlag = true;
//		Transaction tran = null;	// 5.1.9.0 (2010/08/01) Transaction 対応
//		try {
//			// 5.1.9.0 (2010/08/01) Transaction 対応
//			final TransactionTag tranTag = (TransactionTag)findAncestorWithClass( this,TransactionTag.class );
//			if( tranTag == null ) {
//				tran = new TransactionReal( getApplicationInfo() );		// 5.3.7.0 (2011/07/01) 引数変更
//			}
//			else {
//				tran = tranTag.getTransaction();
//			}

		// 6.3.6.1 (2015/08/28) Transaction でAutoCloseableを使用したtry-with-resources構築に対応。
		try( final Transaction tran = getTransaction() ) {
			final Connection conn = tran.getConnection( dbid );				// 5.1.9.0 (2010/08/01) Transaction 対応
			// 3.6.1.0 (2005/01/05)
			if( queryType ) {		// JDBC 通常の SELECT 文
				stmt = conn.createStatement();
				if( fetchSize > 0 ) { stmt.setFetchSize( fetchSize ); }
				resultSet = stmt.executeQuery( sql );
			}
			else {					// PL/SQL Call 文
				String[] values = null;
				if( names != null ) {
					final String[] nameArray = StringUtil.csv2Array( names );
					values = getRequest( nameArray );
				}
				callStmt  = conn.prepareCall( sql );
				resultSet = executeCall( conn,callStmt,values );		// 5.3.0.0 (2010/12/01)
			}
			if( resultSet == null ) { return 0; }

			// 6.0.4.0 (2014/11/28) ResultSetValue を使用するように変更。
			final ResultSetValue rsv = new ResultSetValue( resultSet );
			final int numberOfColumns =  rsv.getColumnCount();

			// ヘッダー部の出力
			if( useHeader && numberOfColumns > 0 ) {
				final StringBuilder headName  = new StringBuilder( BUFFER_MIDDLE );
				final StringBuilder headLabel = new StringBuilder( BUFFER_MIDDLE );
				if( useNumber ){					// 6.0.3.0 (2014/11/13) ヘッダー部の useNumber 対応漏れ
					headName.append(  "#Name" );
					headLabel.append( "#Label" );
				}
				final ResourceManager resource = getResource();
				// 6.0.4.0 (2014/11/28) ResultSetValue を使用するように変更。
				for(int clmNo=0; clmNo<numberOfColumns; clmNo++) {
					final String clm = rsv.getColumnName(clmNo);
					if( clmNo > 0 || useNumber ){						// 5.5.7.1 (2012/10/05)
						//この場合だけセパレータ出力する。
						headName.append( separator );					// 6.0.3.0 (2014/11/13)
						headLabel.append( separator );					// 6.0.3.0 (2014/11/13)
					}
					headName.append( clm );								// 6.0.3.0 (2014/11/13)
					headLabel.append( resource.getLabel( clm ) );		// 6.0.3.0 (2014/11/13)
				}
				out.println( headName.toString() );
				out.println( headLabel.toString() );
			}

			int rowNo = 0;
			// 6.0.4.0 (2014/11/28) ResultSetValue を使用するように変更。
			while( rsv.next() ) {
				// 6.0.4.0 (2014/11/28) ResultSetValue を使用するように変更。
				if( useNumber ){					// 5.5.7.1 (2012/10/05)
					if( useQuote ) { out.print( "\"" + rowNo + "\"" ); }	// 行番号
					else {			 out.print( rowNo ); }
				}
				for(int clmNo=0; clmNo<numberOfColumns; clmNo++) {
					// 6.0.2.5 (2014/10/31) refactoring:Avoid empty if statements 警告の対応
					if( clmNo > 0 || useNumber ){					// 6.2.0.0 (2015/02/27)
						//この場合だけセパレータ出力する。
						out.print( separator );
					}
					String sval = replace( rsv.getValue(clmNo) );				// 禁則文字の置換処理
					if( sval != null && sval.length() > 0 ) {
						// 6.0.3.0 (2014/11/13) データ中にダブルクオート文字が含まれる場合、エスケープするかどうか
						if( useQuoteEscape && sval.indexOf( '"' ) >= 0 ) { sval = sval.replaceAll( "\"" ,"\"\"" ) ; }
						// 6.0.3.0 (2014/11/13) データ中に改行コードが含まれる場合、ダブルクオートで囲うかどうか
						if( useQuote || useReturnQuote && sval.indexOf( CR ) >= 0 ) {
							sval = "\"" + sval + "\"" ;
						}
					}
					else {
						sval = useQuote ? "\"\"" : "" ;
					}
					out.print( sval );
				}
				out.println();
				rowNo++ ;
			}
			executeCount = rowNo ;
//			errFlag = false;			// エラーではない
			tran.commit();				// 6.3.6.1 (2015/08/28)
		}
		catch ( SQLException ex ) {		// 3.6.1.0 (2005/01/05)
			final String errMsg = "データベース処理を実行できませんでした。"
						 + CR + stmt + CR
						 + "err=[" + ex.getSQLState() + "]"
						 + ex.getMessage();
			throw new HybsSystemException( errMsg,ex );
		}
		finally {
			Closer.resultClose( resultSet );
			Closer.stmtClose( stmt );
			Closer.stmtClose( callStmt );	// 4.3.4.3 (2008/12/22)
//			if( tran != null ) {				// 5.5.2.6 (2012/05/25) findbugs対応
//				tran.close( errFlag );			// 5.1.9.0 (2010/08/01) Transaction 対応
//			}
		}

		return executeCount ;
	}

	/**
	 * replaceFrom,replaceTo に基づく禁則文字の置換処理を行います。
	 *
	 * replaceFrom の１文字づつを、対応するreplaceToの１文字づつに変換します。
	 * replaceFrom と replaceTo の文字数は同じでなければなりません。
	 *
	 * @og.rev 6.0.3.0 (2014/11/13) 新規追加
	 *
	 * @param   str 置換する文字列
	 *
	 * @return	置換後の文字列
	 */
	private String replace( final String str ) {
		String rtn = str;
		if( rtn != null && replaceFrom != null && replaceTo != null ) {
			for( int i=0; i<replaceTo.length(); i++ ) {
				rtn = rtn.replace( replaceFrom.charAt(i) , replaceTo.charAt(i) );		// charの置き換えは、全件
			}
		}
		return rtn ;
	}

	/**
	 * 引数配列付のクエリーを実行します。
	 * 処理自体は, #execute() と同様に、各サブクラスの実装に依存します。
	 * これは、CallableStatement を用いて、データベース検索処理を行います。
	 * {call TYPE3B01.TYPE3B01(?,?,?,?)} で、４番目の引数には、
	 * names で指定したリクエスト情報が、ARG_ARRAY 配列に順次セットされます。
	 * 使用する場合は、一旦わかり易い変数に受けて利用してください。
	 * 呼び出す PL/SQL では、検索系PL/SQL です。
	 *
	 * @og.rev 3.6.1.0 (2005/01/05) PL/SQLコール(JDBCErrMsg 形式)への対応
	 * @og.rev 4.3.4.3 (2008/12/22) (Oracle11gDriver対応)PL/SQLコールの場合に、"クローズされた文です。"のエラーが発生する問題に対応
	 * @og.rev 5.3.0.0 (2010/12/01) executeCall メソッドの引数見直し
	 * @og.rev 6.0.0.0 (2014/04/11) Oracle11g(11.2.0.3のドライバ)対応
	 *
	 * @param	conn		コネクション
	 * @param   callStmt	コーラブルステートメント
	 * @param   args		オブジェクトの引数配列(可変長引数)
	 *
	 * @return	結果オブジェクト
	 */
	private ResultSet executeCall( final Connection conn,final CallableStatement callStmt,final String... args ) throws SQLException {
		ResultSet resultSet = null;
		callStmt.setQueryTimeout( DB_MAX_QUERY_TIMEOUT );
		if( fetchSize > 0 ) { callStmt.setFetchSize( fetchSize ); }
		final Map<String,Class<?>> map = conn.getTypeMap();
		try {
			map.put( ERR_MSG,Class.forName( "org.opengion.hayabusa.db.DBErrMsg" ) );
		}
		catch( ClassNotFoundException ex ) {
			final String errMsg = "org.opengion.hayabusa.db.DBErrMsg クラスが見つかりません。" + CR
					+ ex.getMessage();			// // 5.1.8.0 (2010/07/01) errMsg 修正
			throw new HybsSystemException( errMsg,ex );		// 3.5.5.4 (2004/04/15) 引数の並び順変更
		}

		// 6.0.0.0 (2014/04/11) Oracle11g(11.2.0.3のドライバ)対応 http://docs.oracle.com/cd/E28389_01/web.1111/b60995/thirdparty.htm
		final Array newArray = ((OracleConnection)conn).createOracleArray( ARG_ARRAY, StringUtil.rTrims( args ));		// 6.0.0.0 (2014/04/11) Oracle11g(11.2.0.3のドライバ)対応

		callStmt.registerOutParameter(1, Types.INTEGER);
		callStmt.registerOutParameter(2, Types.ARRAY,ERR_MSG_ARRAY);		// 6.0.0.0 (2014/04/11) Oracle11g(11.2.0.3のドライバ)対応
		callStmt.registerOutParameter(3, OracleTypes.CURSOR);
		callStmt.setArray( 4,newArray );									// 6.0.0.0 (2014/04/11) Oracle11g(11.2.0.3のドライバ)対応

		callStmt.execute();

		errCode = callStmt.getInt(1);

		if( errCode < ErrorMessage.NG ) {		// 異常以外の場合
			resultSet = ((OracleCallableStatement)callStmt).getCursor(3);
		}
		if( errCode > ErrorMessage.OK ) {		// 正常以外の場合
			final Array rtn3 = callStmt.getArray(2);								// 6.0.0.0 (2014/04/11) Oracle11g(11.2.0.3のドライバ)対応
			final Object[] rtnval3 = (Object[])rtn3.getArray();
			errMessage = new ErrorMessage( "Query_JDBCErrMsg Error!!" );
			for( int i=0; i<rtnval3.length; i++ ) {
				final DBErrMsg er = (DBErrMsg)rtnval3[i];
				if( er == null ) { break; }
				errMessage.addMessage( er.getErrMsg() );
			}
		}
		return resultSet;
	}

	/**
	 * PrintWriter を取得します。
	 *
	 * ここでは、一般的なファイル出力を考慮した PrintWriter を作成します。
	 *
	 * @og.rev 3.7.1.1 (2005/05/23) フォルダがない場合は、複数階層分のフォルダを自動で作成します。
	 * @og.rev 3.8.0.0 (2005/06/07) FileUtil#getPrintWriter を利用。
	 * @og.rev 5.6.1.0 (2013/02/01) 3.7.1.1のコメントに入っているが対応されていないのでフォルダ作成追加
	 *
	 * @return	 出力用PrintWriterオブジェクト
	 * @og.rtnNotNull
	 */
	private PrintWriter getPrintWriter() {
		if( filename == null ) {
			final String errMsg = "ファイル名がセットされていません。";
			throw new HybsSystemException( errMsg );
		}
		final String directory = HybsSystem.url2dir( fileURL );

		// 5.6.1.0 (2013/02/01) 
		final File dir = new File(directory);
		if( ! dir.exists() && ! dir.mkdirs() ) {
			final String errMsg = "ディレクトリの作成に失敗しました。[" + directory + "]";
			throw new HybsSystemException( errMsg );
		}

		// ※ 注意 StringUtil.urlAppend を組み込んでいる意図が不明。一旦削除していますが、注意
	 	// 3.8.0.0 (2005/06/07) FileUtil#getPrintWriter を利用。
	//	out = FileUtil.getPrintWriter( StringUtil.urlAppend( directory,filename ),fileAppend,encode);

		// 処理を簡素化します。
		return FileUtil.getPrintWriter( new File( directory,filename ),encode,fileAppend );
	}

	/**
	 * 名称配列を元に、リクエスト情報のデータを取得します。
	 *
	 * @og.rev 3.6.1.0 (2005/01/05) PL/SQLコール(JDBCErrMsg 形式)への対応
	 *
	 * @param	nameArray	キーとなる名称の配列(可変長引数)
	 *
	 * @return	そのリクエスト情報
	 * @og.rtnNotNull
	 */
	private String[] getRequest( final String... nameArray ) {
		String[] rtn = new String[nameArray.length];

		for( int i=0; i<rtn.length; i++ ) {
			rtn[i] = getRequestValue( nameArray[i] );
		}

		return rtn;
	}

	/**
	 * 【TAG】(通常は使いません)検索時のDB接続IDを指定します(初期値:DEFAULT)。
	 *
	 * @og.tag
	 *   検索時のDB接続IDを指定します。初期値は、DEFAULT です。
	 *
	 * @param	id DB接続ID
	 */
	public void setDbid( final String id ) {
		dbid = nval( getRequestParameter( id ),dbid );
	}

	/**
	 * 【TAG】可変長ファイルを作成するときの項目区切り文字(セパレータ)をセットします
	 *		(初期値:TAB_SEPARATOR[={@og.value #TAB_SEPARATOR}])。
	 *
	 * @og.tag 可変長ファイルを作成するときの項目区切り文字をセットします。
	 * (初期値:ローカル定義のTAB_SEPARATOR[={@og.value #TAB_SEPARATOR}])。
	 *
	 * @param   sep セパレータ
	 * @see		#TAB_SEPARATOR
	 */
	public void setSeparator( final String sep ) {
		separator = nval( getRequestParameter( sep ),TAB_SEPARATOR );
	}

	/**
	 * 【TAG】保存先ディレクトリ名を指定します
	 *		(初期値:FILE_URL[={@og.value SystemData#FILE_URL}])。
	 *
	 * @og.tag
	 * この属性で指定されるディレクトリに、ファイルをセーブします。
	 * 指定方法は、通常の fileURL 属性と同様に、先頭が、'/' (UNIX) または、２文字目が、
	 * ":" (Windows)の場合は、指定のURLそのままのディレクトリに、そうでない場合は、
	 * fileURL = "{&#064;USER.ID}" と指定すると、FILE_URL 属性で指定のフォルダの下に、
	 * さらに、各個人ID別のフォルダを作成して、そこにセーブします。
	 * (初期値:システム定数のFILE_URL[={@og.value SystemData#FILE_URL}])。
	 *
	 * @og.rev 3.5.4.3 (2004/01/05) 内部処理を、makeFileURL に移動。
	 * @og.rev 4.0.0.0 (2005/01/31) StringUtil.urlAppend メソッドの利用
	 * @og.rev 4.0.0.0 (2007/11/20) 指定されたディレクトリ名の最後が"\"or"/"で終わっていない場合に、"/"を付加する。
	 *
	 * @param	url 保存先ディレクトリ名
	 * @see		org.opengion.hayabusa.common.SystemData#FILE_URL
	 */
	public void setFileURL( final String url ) {
		String furl = nval( getRequestParameter( url ),null );
		if( furl != null ) {
			final char ch = furl.charAt( furl.length()-1 );
			if( ch != '/' && ch != '\\' ) { furl = furl + "/"; }
			fileURL = StringUtil.urlAppend( fileURL,furl );
		}
	}

	/**
	 * 【TAG】ファイルを作成するときのファイル名をセットします(初期値:システムパラメータのFILE_FILENAME)。
	 *
	 * @og.tag ファイルを作成するときのファイル名をセットします。
	 *
	 * @param   fname ファイル名
	 */
	public void setFilename( final String fname ) {
		filename = nval( getRequestParameter( fname ),filename );
	}

	/**
	 * 【TAG】ファイルを作成するときのファイルエンコーディング名をセットします
	 *		(初期値:FILE_ENCODE[={@og.value SystemData#FILE_ENCODE}])。
	 *
	 * @og.tag
	 * "DEFAULT","JISAutoDetect" ,"JIS", "EUC_JP", "MS932", "SJIS" , "Windows-31J" , "Shift_JIS"
	 * (初期値:システム定数のFILE_ENCODE[={@og.value SystemData#FILE_ENCODE}])。
	 *
	 * @og.rev 2.2.0.0 (2002/12/17) 中国語(国際化)対応 エンコードの取得方法変更
	 * @og.rev 3.1.3.0 (2003/04/10) FILE_ENCODE から、エンコード情報を取得する。
	 *
	 * @param   enc ファイルエンコーディング名
	 * @see     <a href="http://www.iana.org/assignments/character-sets">IANA Charset Registry</a>
	 * @see		org.opengion.hayabusa.common.SystemData#FILE_ENCODE
	 */
	public void setEncode( final String enc ) {
		encode = nval( getRequestParameter( enc ),encode );
	}

	/**
	 * 【TAG】ヘッダーを書き込むかどうか[true/false]を指定します(初期値:true)。
	 *
	 * @og.tag
	 *  #Name ････ ヘッダーの書き込みを指定します。
	 * 通常は、書き込み(true)にしておき、使用側でコメントと解釈するように
	 * 処理を行うべきです。コメントのため、append モードで途中に現れても
	 * 無視できます。また、エンジン標準でデータを取り込む場合に、データの配置が
	 * 変更されても取り込みプログラムはそのまま使用できます。
	 * 初期値は、true(書き込む)です。
	 *
	 * @param   flag ヘッダー有無 [true:書き込む/false:書き込まない]
	 */
	public void setUseHeader( final String flag ) {
		useHeader = nval( getRequestParameter( flag ),useHeader );
	}

	/**
	 * 【TAG】データをダブルクオートで囲うかどうか指定します(初期値:false)。
	 *
	 * @og.tag
	 * データを出力する場合、ダブルクオートで囲うかどうか指定します。
	 * 主に、区切り文字(separator)を、タブではなく、カンマを使う場合に、使用します。
	 * なお、ヘッダー部は、この指定に関わらず、ダブルクオートで囲いません。
	 * 初期値は、false(囲わない)です。
	 *
	 * @og.rev 6.0.3.0 (2014/11/13) 新規追加
	 *
	 * @param   flag ダブルクオート使用 [true:書き込む/false:書き込まない]
	 */
	public void setUseQuote( final String flag ) {
		useQuote = nval( getRequestParameter( flag ),useQuote );
	}

	/**
	 * 【TAG】データ中にダブルクオート文字が含まれる場合、エスケープするかどうか指定します(初期値:true)。
	 *
	 * @og.tag
	 * データ中にダブルクオート文字が含まれる場合、エスケープするかどうか指定します。
	 * ここでいうエスケープとは、ダブルクオート文字を重ねる処理を指します。
	 * 初期値は、互換性の関係で、true(処理する)です。
	 *
	 * @og.rev 6.0.3.0 (2014/11/13) 新規追加
	 *
	 * @param   flag ダブルクオートエスケープ有無 [true:する/false:しない]
	 */
	public void setUseQuoteEscape( final String flag ) {
		useQuoteEscape = nval( getRequestParameter( flag ),useQuoteEscape );
	}

	/**
	 * 【TAG】データ中に改行コードが含まれる場合、ダブルクオートで囲うかどうか指定します(初期値:true)。
	 *
	 * @og.tag
	 * データ中に改行コードが含まれたテキストの場合、EXCELで開くと、改行されてしまう。
	 * その場合、ダブルクオートで囲うと、セルに設定してくれます。
	 * この処理は、useQuote="true" にすると、無条件に、データは囲われます。
	 * 初期値は、互換性の関係で、true(処理する)です。
	 *
	 * @og.rev 6.0.3.0 (2014/11/13) 新規追加
	 *
	 * @param   flag 改行コード処理 [true:する/false:しない]
	 * @see		#setUseQuote( String )
	 */
	public void setUseReturnQuote( final String flag ) {
		useReturnQuote = nval( getRequestParameter( flag ),useReturnQuote );
	}

	/**
	 * 【TAG】一文字単位で置換する置換元文字を指定します(初期値:null 置換なし)。
	 *
	 * @og.tag
	 * データ出力時に、禁則文字を、置き換える元の文字を指定します。
	 * ここでは、一文字単位で、置換しますので、禁則文字は、連続の文字列の
	 * 形で、指定します。
	 * なお、ヘッダー部は、この指定に関わらず、ダブルクオートで囲いません。
	 * 初期値は、null の場合は、何も変換しません。
	 * 文字数は、replaceTo と同じでなければなりません。
	 *
	 * @og.rev 6.0.3.0 (2014/11/13) 新規追加
	 * @og.rev 6.2.2.0 (2015/03/27) \n,\r,\t をサポートします。
	 *
	 * @param   str 置換元文字
	 */
	public void setReplaceFrom( final String str ) {
		replaceFrom = changeRNT( nval( getRequestParameter( str ),replaceFrom ) );
	}

	/**
	 * 【TAG】一文字単位で置換する置換先文字を指定します。
	 *
	 * @og.tag
	 * データ出力時に、禁則文字を、置き換える先の文字を指定します。
	 * ここでは、一文字単位で、置換しますので、禁則文字は、連続の文字列の
	 * 形で、指定します。(例えば、全角文字にするとか)
	 * 初期値は、null の場合は、何も変換しません。
	 * 文字数は、replaceFrom と同じでなければなりません。
	 *
	 * @og.rev 6.0.3.0 (2014/11/13) 新規追加
	 * @og.rev 6.2.2.0 (2015/03/27) \n,\r,\t をサポートします。
	 *
	 * @param   str 置換先文字
	 */
	public void setReplaceTo( final String str ) {
		replaceTo = changeRNT( nval( getRequestParameter( str ),replaceTo ) );
	}

	/**
	 * replaceFrom,replaceTo で、\n,\r,\t をサポートします。
	 *
	 * データ置換で、改行、復帰、タブを、指定する場合、２文字必要です。
	 * ここでは、\n,\r,\t が指定された場合、キャラクタコードに置き換えます。
	 *
	 * @og.rev 6.2.2.0 (2015/03/27) \n,\r,\t をサポートします。
	 *
	 * @param	str 置換先文字
	 * @return	置換先文字
	 */
	private String changeRNT( final String str ) {
		String rtn = str ;
		if( rtn != null && !rtn.isEmpty() ) {
			final StringBuilder buf = new StringBuilder();
			for( int i=0; i<rtn.length(); i++ ) {
				char ch = rtn.charAt(i) ;
				if( ch == '\\' ) {
					final char ch2 = rtn.charAt(++i) ;		// ￥ の次の文字(n,r,tのみサポート)
					switch( ch2 ) {
						case 'n' : ch = '\n'; break;
						case 'r' : ch = '\r'; break;
						case 't' : ch = '\t'; break;
						default  : 
							final String errMsg = getClass().getName() + "の置換文字列で、「\\" + ch2 + "」は、サポートされていません。";
							System.err.println( errMsg );
							break;
					}
				}
				buf.append( ch );
			}
			rtn = buf.toString();
		}
		return rtn ;
	}

	/**
	 * 【TAG】追加モードで書き込むかどうか[true/false]を指定します(初期値:false[新規モード])。
	 *
	 * @og.tag
	 * ファイルを書き込む場合、追加モードで書き込むかどうかをセットします。
	 * 新規モード(true)の場合、既存のファイルが存在し、かつ書き込み許可があれば、
	 * 上書きで新規に作成します。
	 * 初期値は、false(新規モード)です。
	 *
	 * @param   flag 追加モード [true:追加モード/false:新規モード]
	 */
	public void setFileAppend( final String flag ) {
		fileAppend = nval( getRequestParameter( flag ),fileAppend );
	}

	/**
	 * 【TAG】結果をファイルに出力するときに、ZIPで圧縮するかどうか[true/false]を指定します(初期値:false)。
	 *
	 * @og.tag
	 * 大量に抜き出す場合、そのまま、サーバーから取り出すだけでも大変です。
	 * zip 属性を、true にすると、GZIP で圧縮したファイルを作成します。
	 * 初期値は、false(圧縮しない)です。
	 *
	 * @param  flag ZIP圧縮 [true:する/それ以外:しない]
	 * @see    #setZipFilename( String )
	 */
	public void setZip( final String flag ) {
		zip = nval( getRequestParameter( flag ),zip );
	}

	/**
	 * 【TAG】ZIPファイルを作成するときのZIPファイル名をセットします(初期値:filename + ".zip")。
	 *
	 * @og.tag
	 * zip 属性に、true を指定した場合に、ZIPファイル化します。その場合のファイル名を指定します。
	 * なにも指定しない場合は、filename + ".zip" になります。
	 *
	 * @param   zipFile ZIPファイル名
	 * @see #setZip( String )
	 */
	public void setZipFilename( final String zipFile ) {
		zipFilename = nval( getRequestParameter( zipFile ),zipFilename );
	}

	/**
	 * 【TAG】検索結果を画面上に表示するメッセージリソースIDを指定します
	 *		(初期値:VIEW_DISPLAY_MSG[={@og.value SystemData#VIEW_DISPLAY_MSG}])。
	 *
	 * @og.tag
	 * ここでは、検索結果の件数や登録された件数をまず出力し、
	 * その次に、ここで指定したメッセージをリソースから取得して
	 * 表示します。
	 * 件数を表示させる場合は、displayMsg = "MSG0033"[　件検索しました] をセットしてください。
	 * 表示させたくない場合は, displayMsg = "" をセットしてください。
	 * (初期値:システム定数のVIEW_DISPLAY_MSG[={@og.value SystemData#VIEW_DISPLAY_MSG}])。
	 *
	 * @param	id 結果表示メッセージID
	 */
	public void setDisplayMsg( final String id ) {
		final String ids = getRequestParameter( id );
		if( ids != null ) { displayMsg = ids; }
	}

	/**
	 * 【TAG】検索結果がゼロ件の場合に表示するメッセージリソースIDを指定します(初期値:MSG0077[対象データはありませんでした])。
	 *
	 * @og.tag
	 * ここでは、検索結果がゼロ件の場合のみ、特別なメッセージを表示させます。
	 * 従来は、displayMsg と兼用で、『0　件検索しました』という表示でしたが、
	 * displayMsg の初期表示は、OFF になりましたので、ゼロ件の場合のみ別に表示させます。
	 * 表示させたくない場合は, notfoundMsg = "" をセットしてください。
	 * 初期値は、MSG0077[対象データはありませんでした]です。
	 *
	 * @param	id ゼロ件時表示メッセージID
	 */
	public void setNotfoundMsg( final String id ) {
		final String ids = getRequestParameter( id );
		if( ids != null ) { notfoundMsg = ids; }
	}

	/**
	 * 【TAG】(通常は使いません)データのフェッチサイズを指定します(初期値:100)。
	 *
	 * @og.tag
	 * より多くの行が必要なときに、データベースから取り出す必要がある行数に
	 * ついてのヒントを JDBC ドライバに提供します。
	 * 指定された行数は、この Statement を使って作成された結果セットにだけ影響します。
	 * 指定された値が 0 の場合、ヒントは無視されます。
	 * 初期値は、100 です。
	 *
	 * @param	size フェッチ行数 (初期値:100)
	 */
	public void setFetchSize( final String size ) {
		fetchSize = nval( getRequestParameter( size ),fetchSize );
	}

	/**
	 * 【TAG】PL/SQLを利用する場合の引数にセットすべき データの名称をCSV形式で複数指定します。
	 *
	 * @og.tag
	 * 複数ある場合は、CSV形式で渡します。
	 * PL/SQL を使用しない場合は、無視されます。
	 *
	 * @og.rev 3.6.1.0 (2005/01/05) PL/SQLコール(JDBCErrMsg 形式)への対応
	 *
	 * @param	nm 引数の名称 (CSV形式)
	 */
	public void setNames( final String nm ) {
		names = nval( getRequestParameter( nm ),names );
	}

	/**
	 * 【TAG】Query を発行する為のクラスID(JDBC,JDBCErrMsg)を指定します({@og.doc03Link queryType 初期値:JDBC})。
	 *
	 * @og.tag
	 * ストアドプロシージャ等を実行する場合に、queryType="JDBCErrMsg" を
	 * 指定する必要があります。(それ以外の指定は、初期値の JDBC になります。)
	 * 初期値は、"JDBC" です。
	 * {@og.doc03Link queryType Query_**** クラス}
	 *
	 * @og.rev 3.6.1.0 (2005/01/05) PL/SQLコール(JDBCErrMsg 形式)への対応
	 *
	 * @param	id Query発行クラスID
	 */
	public void setQueryType( final String id ) {
		// 内部的には、JDBCErrMsg:false / それ以外:true で管理しています。
		queryType = ! "JDBCErrMsg".equalsIgnoreCase( getRequestParameter( id ) );
	}

	/**
	 * 【TAG】ファイル出力時に、行番号情報を、出力する/しない[true/false]を指定します(初期値:true)。
	 *
	 * @og.tag
	 * 通常のフォーマットでは、各行の先頭に行番号を出力します。
	 * これは、#NAME 属性を使用する場合には、必ず出力する必要があります。
	 * (#NAME 属性は、読み取り時にあれば、自動的にカラムに割り当てられます。)
	 * この、先頭の行番号が不要な場合(つまり、他のシステムへのデータ出力等)
	 * の為に出力する場合に、false を設定することで、行番号列を出力しない
	 * ようにできます。
	 * 初期値は、true(出力する) です。
	 * 
	 * @og.rev 5.5.7.1 (2012/10/05) 新規追加
	 * @param  flag 行番号出力 [true:する/それ以外:しない]
	 */
	public void setUseNumber( final String flag ) {
		useNumber = nval( getRequestParameter( flag ),useNumber );
	}

	/**
	 * 【TAG】リクエスト情報の シングルクォート(') 存在チェックを実施するかどうか[true/false]を設定します
	 *		(初期値:USE_SQL_INJECTION_CHECK[={@og.value SystemData#USE_SQL_INJECTION_CHECK}])。
	 *
	 * @og.tag
	 * ＳＱＬインジェクション対策の一つとして、暫定的ではありますが、SQLのパラメータに
	 * 渡す文字列にシングルクォート(') を許さない設定にすれば、ある程度は防止できます。
	 * 数字タイプの引数には、 or 5=5 などのシングルクォートを使用しないコードを埋めても、
	 * 数字チェックで検出可能です。文字タイプの場合は、必ず (')をはずして、
	 * ' or 'A' like 'A のような形式になる為、(')チェックだけでも有効です。
	 * (') が含まれていたエラーにする(true)／かノーチェックか(false)を指定します。
	 * 初期値は、SystemData#USE_SQL_INJECTION_CHECK です。
	 *
	 * @og.rev 6.2.2.0 (2015/03/27) XSSチェック,クォートチェック をサポートします。
	 *
	 * @param   flag クォートチェック [true:する/それ以外:しない]
	 */
	public void setQuotCheck( final String flag ) {
		quotCheck = nval( getRequestParameter( flag ),quotCheck );
	}

	/**
	 * 【TAG】リクエスト情報の HTMLTag開始/終了文字(&gt;&lt;) 存在チェックを実施するかどうか[true/false]を設定します
	 *		(初期値:USE_XSS_CHECK[={@og.value SystemData#USE_XSS_CHECK}])。
	 *
	 * @og.tag
	 * クロスサイトスクリプティング(XSS)対策の一環としてless/greater than signについてのチェックを行います。
	 * (&gt;&lt;) が含まれていたエラーにする(true)／かノーチェックか(false)を指定します。
	 * (初期値:システム定数のUSE_XSS_CHECK[={@og.value SystemData#USE_XSS_CHECK}])
	 *
	 * @og.rev 6.2.2.0 (2015/03/27) XSSチェック,クォートチェック をサポートします。
	 *
	 * @param	flag	XSSチェック [true:する/false:しない]
	 * @see		org.opengion.hayabusa.common.SystemData#USE_XSS_CHECK
	 */
	public void setXssCheck( final String flag ) {
		xssCheck = nval( getRequestParameter( flag ),xssCheck );
	}

	/**
	 * 【TAG】処理時間を表示する TimeView を表示するかどうか[true:する/false:しない]を指定します
	 *		(初期値:VIEW_USE_TIMEBAR[={@og.value SystemData#VIEW_USE_TIMEBAR}])。
	 *
	 * @og.tag
	 * true に設定すると、処理時間を表示するバーイメージが表示されます。
	 * これは、DB検索、APサーバー処理、画面表示の各処理時間をバーイメージで
	 * 表示させる機能です。処理時間の目安になります。
	 * (初期値:VIEW_USE_TIMEBAR[={@og.value SystemData#VIEW_USE_TIMEBAR}])。
	 *
	 * @og.rev 6.3.6.0 (2015/08/16) useTimeView の初期値を、VIEW_USE_TIMEBAR にする。
	 *
	 * @param	flag	処理時間を表示 [true:する/false:しない]
	 */
	public void setUseTimeView( final String flag ) {
		useTimeView = nval( getRequestParameter( flag ),useTimeView );
	}

//	/**
//	 * シリアライズ用のカスタムシリアライズ書き込みメソッド。
//	 *
//	 * @og.rev 4.0.0.0 (2006/09/31) 新規追加
//	 * @serialData 一部のオブジェクトは、シリアライズされません。
//	 *
//	 * @param strm ObjectOutputStreamオブジェクト
//	 * @throws IOException	シリアライズに関する入出力エラーが発生した場合
//	 */
//	private void writeObject( final ObjectOutputStream strm ) throws IOException {
//		strm.defaultWriteObject();
//	}

//	/**
//	 * シリアライズ用のカスタムシリアライズ読み込みメソッド
//	 *
//	 * ここでは、transient 宣言された内部変数の内、初期化が必要なフィールドのみ設定します。
//	 *
//	 * @og.rev 4.0.0.0 (2006/09/31) 新規追加
//	 * @serialData 一部のオブジェクトは、シリアライズされません。
//	 *
//	 * @param strm ObjectInputStreamオブジェクト
//	 * @see #release2()
//	 * @throws IOException	シリアライズに関する入出力エラーが発生した場合
//	 * @throws ClassNotFoundException	クラスを見つけることができなかった場合
//	 */
//	private void readObject( final ObjectInputStream strm ) throws IOException , ClassNotFoundException {
//		strm.defaultReadObject();
//	}

	/**
	 * このオブジェクトの文字列表現を返します。
	 * 基本的にデバッグ目的に使用します。
	 *
	 * @return このクラスの文字列表現
	 * @og.rtnNotNull
	 */
	@Override
	public String toString() {
		return ToString.title( this.getClass().getName() )
				.println( "VERSION"		,VERSION	)
				.println( "dbid"		,dbid		)
				.println( "separator"	,separator	)
				.println( "useHeader"	,useHeader	)
				.println( "fileURL"		,fileURL	)
				.println( "filename"	,filename	)
				.println( "sql"			,sql		)
				.println( "encode"		,encode		)
				.println( "fileAppend"	,fileAppend	)
				.println( "zip"			,zip		)
				.println( "zipFilename"	,zipFilename)
				.println( "displayMsg"	,displayMsg	)
//				.println( "dyStart"		,dyStart	)
				.println( "fetchSize"	,fetchSize	)
				.println( "queryType"	,queryType	)
				.println( "names"		,names		)
				.println( "errCode"		,errCode	)
				.println( "Other..."	,getAttributes().getAttribute() )
				.fixForm().toString() ;
	}
}
