/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.fukurou.util;

import java.io.FileInputStream;
import java.io.FileOutputStream;

import static org.opengion.fukurou.util.HybsConst.BUFFER_MIDDLE;	// 6.1.0.0 (2014/12/26) refactoring

/**
 * DateSet.java は、入力ファイルの日付，時刻キーワードを実行時の日時で変換して,出力します。
 *
 * 変換には,$(yyyy)の形式で指定し,カッコの文字列は,java.text.SimpleDateFormat で使用する，
 * 時刻フォーマット構文を用います。
 * また、引数に keys,vals を渡すことで、$(KEY1) 文字列を VAL1 文字列と置き換えます。
 *
 *  サンプルファイル
 *  $(yyyy/MM/dd)        年／月／日を表します。
 *  $(yy)                年だけを２桁で表します。
 *  $(MM)                月を２桁 (02,03など)で表します。
 *  $(dd)                日を２桁 (02,03など)で表します。
 *  $(HH:mm:ss)          時：分：秒を表します。
 *  $(MMMMMMMM)          月をフルスペルで表します。
 *  $(MMM)               月を３桁固定(Mar，Aplなど)で表します。
 *  $(EEEEEEEE)          曜日をフルスペルで表します。
 *  $(EEE)               曜日を３桁固定(Sun,Monなど)で表します。
 *
 *   時刻フォーマット構文
 *
 *   記号     意味                    表示                例
 *   ------   -------                 ------------        -------
 *   G        年号                    (テキスト)          AD
 *   y        年                      (数値)              1996
 *   M        月                      (テキスト &amp; 数値)  July &amp; 07
 *   d        日                      (数値)              10
 *   h        午前/午後の時 (1~12)    (数値)              12
 *   H        一日における時 (0~23)   (数値)              0
 *   m        分                      (数値)              30
 *   s        秒                      (数値)              55
 *   S        ミリ秒                  (数値)              978
 *   E        曜日                    (テキスト)          火曜日
 *   D        年における日            (数値)              189
 *   F        月における曜日          (数値)              2 (7月の第2水曜日)
 *   w        年における週            (数値)              27
 *   W        月における週            (数値)              2
 *   a        午前/午後               (テキスト)          PM
 *   k        一日における時 (1~24)   (数値)              24
 *   K        午前/午後の時 (0~11)    (数値)              0
 *   z        時間帯                  (テキスト)          PDT
 *   '        テキスト用エスケープ
 *   ''       単一引用符                                  '
 *
 *  パターン文字のカウントによって、そのフォーマットが決まります。
 *  (テキスト): 4以上: フル形式を使用します。4以下: 短いまたは省力された形式があれば、それを使用します。
 *
 *  (数値): 最小桁数。これより短い数値は、この桁数までゼロが追加されます。年には特別な処理があります。
 *  つまり、'y'のカウントが2なら、年は2桁に短縮されます。
 *
 *  (テキスト ＆ 数値): 3以上ならテキストを、それ以外なら数値を使用します。
 *
 *  パターンの文字が['a'..'z']と['A'..'Z']の範囲になければ、その文字は引用テキストとして扱われます。
 *  たとえば、':'、'.'、' '、'#'、'@'などの文字は、単一引用符に囲まれていなくても、
 *  結果の時刻テキストに使用されます。
 *
 *  無効なパターン文字がパターンに入っていると、フォーマットや解析で例外がスローされます。
 *
 *  USロケールを使った例:
 *
 *   フォーマットパターン                   結果
 *   --------------------                   ----
 *   "yyyy.MM.dd G 'at' hh:mm:ss z"    ⇒  1996.07.10 AD at 15:08:56 PDT
 *   "EEE, MMM d, ''yy"                ⇒  Wed, July 10, '96
 *   "h:mm a"                          ⇒  12:08 PM
 *   "hh 'o''''clock' a, zzzz"         ⇒  12 o'clock PM, Pacific Daylight Time
 *   "K:mm a, z"                       ⇒  0:00 PM, PST
 *   "yyyyy.MMMMM.dd GGG hh:mm aaa"    ⇒  1996.July.10 AD 12:08 PM
 *
 * @version  0.9.0  1999/03/09
 * @author   Kazuhiko Hasegawa
 * @since    JDK1.1,
 */
public class DateSet {
	private String[] keys ;
	private String[] vals ;

	/**
	 * フォーマット解析時に置き換える キーと値の配列を設定します。
	 *
	 * $(KEY1) 文字列を VAL1 文字列と置き換える処理を行います。これにより日付以外の
	 * 文字列を置き換える処理を実行できます。
	 *
	 * @param	inkeys	置き換え元キー配列
	 * @param	invals	置き換え元値配列
	 */
	public void setKeysVals( final String[] inkeys, final String[] invals ) {
		if( inkeys != null && invals != null && inkeys.length == invals.length ) {
			final int size = inkeys.length ;
			keys = new String[size];
			vals = new String[size];
			System.arraycopy( inkeys,0,keys,0,size );
			System.arraycopy( invals,0,vals,0,size );
		}
	}

	/**
	 * 現在日付、時刻をフォーマット指定個所に埋め込みます。
	 * フォーマットの指定方法は、java.text.SimpleDateFormat の指定方法と同一です。
	 *
	 * @og.rev 6.3.6.0 (2015/08/16) System.arraycopy が使える箇所は、置き換えます。
	 *
	 * @param	inByte	変換元バイト配列(可変長引数)
	 * @return	変換後のバイト配列
	 */
	public byte[] change( final byte... inByte ) {
		byte[] outByte = new byte[inByte.length+100];
		int add = 0;
		for( int i=0; i<inByte.length; i++) {
			if( inByte[i] == '$' && i<inByte.length-1 && inByte[i+1] == '(' ) {
				int j = 0;
				while( inByte[i+j+2] != ')') { j++; }
				final String str = changeForm( new String( inByte,i+2,j,StringUtil.DEFAULT_CHARSET ) );		// 5.5.2.6 (2012/05/25) findbugs対応
				final byte[] byteDate = str.getBytes( StringUtil.DEFAULT_CHARSET ) ;						// 5.5.2.6 (2012/05/25) findbugs対応
				// 6.3.6.0 (2015/08/16) System.arraycopy が使える箇所は、置き換えます。
//				for( int k = 0; k<byteDate.length; k++) {
//					outByte[add] = byteDate[k];
//					add++;
//				}
				System.arraycopy( byteDate,0,outByte,add,byteDate.length );		// 6.3.6.0 (2015/08/16)
				add += byteDate.length ;										// 6.3.6.0 (2015/08/16)
				i   += j+2;
			}
			else {
				outByte[add] = inByte[i];
				add++;
			}
		}
		final byte[] rtnByte = new byte[add];
		System.arraycopy( outByte,0,rtnByte,0,add );
		return rtnByte;
	}

	/**
	 * keys,vals の変換、および、現在日付、時刻のフォーマット変換を行います。
	 *
	 * 先に、keys,vals の変換を行います。form が、keys にマッチすれば、vals を
	 * 返します。最後までマッチしなければ、時刻のフォーマット変換を行います。
	 * フォーマットの指定方法は、java.text.SimpleDateFormat の指定方法と同一です。
	 *
	 * @og.rev 5.5.7.2 (2012/10/09) HybsDateUtil を利用するように修正します。
	 *
	 * @param	form フォーム文字列 ( 例 "yyyy/MM/dd HH:mm:ss" )
	 *
	 * @return	フォーマット変換結果
	 */
	public String changeForm( final String form ) {
		if( keys != null ) {
			for( int i=0; i<keys.length; i++ ) {
				if( form.equals( keys[i] ) ) {
					return vals[i];
				}
			}
		}

		return HybsDateUtil.getDate( form );
	}

	/**
	 * keys,vals の変換、および、現在日付、時刻のフォーマット変換を行います。
	 *
	 * 先に、keys,vals の変換を行います。form が、keys にマッチすれば、vals を
	 * 返します。最後までマッチしなければ、時刻のフォーマット変換を行います。
	 * フォーマットの指定方法は、java.text.SimpleDateFormat の指定方法と同一です。
	 *
	 * @param	form フォーム文字列 ( 例 "yyyy/MM/dd HH:mm:ss" )
	 *
	 * @return	フォーマット変換結果
	 * @og.rtnNotNull
	 */
	public String changeString( final String form ) {
		final StringBuilder buf = new StringBuilder( BUFFER_MIDDLE );
		int bkst = 0;
		int st = form.indexOf( "$(" );
		while( st >= 0 ) {
			buf.append( form.substring( bkst,st ) );
			final int ed = form.indexOf( ')',st+2 );				// 6.0.2.5 (2014/10/31) refactoring
			buf.append( changeForm( form.substring( st+2,ed ) ) );
			bkst = ed + 1;
			st = form.indexOf( "$(",bkst );
		}
		buf.append( form.substring( bkst ) );

		return buf.toString();
	}

	/**
	 * 入力ファイルの時刻フォーマットを変換して出力ファイルに書き込みます。
	 *
	 * 引数に &lt;key1&gt; &lt;val1&gt; のペア情報を渡すことが可能です。
	 * 先に、keys,vals の変換を行います。form が、keys にマッチすれば、vals を
	 * 返します。最後までマッチしなければ、時刻のフォーマット変換を行います。
	 * フォーマットの指定方法は、java.text.SimpleDateFormat の指定方法と同一です。
	 * フォーム文字列例 (  "yyyy/MM/dd HH:mm:ss" )
	 *
	 * @param	args 引数配列( 入力ファイル 出力ファイル キー１ 値１ ･･･
	 * @throws Throwable なんらかのエラーが発生した場合。
	 */
	public static void main( final String[] args ) throws Throwable {
		if( args.length > 2 && ( args.length % 2 != 0 ) ) {
			System.err.println( "Usage: java org.opengion.fukurou.util.DateSet <inputFile> <outputFile> [<key1> <val1> ･･･]" );
			return ;
		}

		String[] keys = new String[ (args.length-2)/2 ];
		String[] vals = new String[ (args.length-2)/2 ];
		for( int i=1; i<=keys.length; i++ ) {
			keys[i-1] = args[i*2];
			vals[i-1] = args[i*2+1];
		}

		final FileInputStream filein = new FileInputStream( args[0] );
		final byte[] byteIn = new byte[ filein.available() ];
		final int len = filein.read( byteIn );
		if( len != byteIn.length ) {
			final String errMsg = "読み取りファイルのデータが切り捨てられました。" +
							"File=" + args[0] + " Length=" + len  + " Input=" + byteIn.length ;
			System.err.println( errMsg );
		}
		filein.close();

		final DateSet dateSet = new DateSet();
		dateSet.setKeysVals( keys,vals );
		final byte[] byteout = dateSet.change( byteIn );

		final FileOutputStream fileout = new FileOutputStream( args[1] );
		fileout.write( byteout );
		fileout.close();
	}
}
