/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.fukurou.util.TagBuffer;
import org.opengion.fukurou.util.ToString;						// 6.1.1.0 (2015/01/17)

import static org.opengion.fukurou.util.StringUtil.nval ;		// 6.1.0.0 (2014/12/26)

// import java.util.Arrays ;		// 6.1.0.0 (2014/12/26)

/**
 * 配置図／座席表を表す特殊系のビューです。
 *
 * itd タグは、ガントヘッダー部の TDタグの繰返しに使用されます。
 * この繰返しは、ganttParam タグの minDuration で指定された間隔で行われます。
 * (例えば、0.5 を指定すれば、半日単位で処理されます。)
 * itd タグの colspan 属性を指定した場合は、itd 自身が、td タグに colspan を
 * 追加すると共に、繰返し自身を、その指定数だけに抑制します。
 * 具体的には、colspan="2" とすると、２回に一回しか、itd タグが呼び出されなく
 * なります。
 *
 * @og.formSample
 * ●形式：&lt;og:iMatrix  ... /&gt;
 * ●body：なし
 * ●前提：headタグで、adjustEvent="Matrix" を指定してください。
 *
 * ●Tag定義：
 *   &lt;og:iMatrix
 *       mode               【TAG】[Layout:レイアウト編集/Value:設定値の編集](初期値:Layout)
 *       countPro           【TAG】データノードの属性数( 行, 列, 名称, 値, 色, 編集可否, ステータス )(初期値:7)
 *                                 (SQL文のカラム数(非表示項目を除く))
 *       inputWidth         【TAG】編集時の入力欄幅(px)(初期値:50px)
 *       cellWidth          【TAG】セル幅(px) (初期値:75px)
 *       cellHeight         【TAG】セル高さ(px) (初期値:30px)
 *       showStatus         【TAG】ステータスの出力を行うかどうか[[true:表示する/false:表示しない]](初期値:false)
 *       countStatus        【TAG】ステータスの種類数（1:会議、2:来客、3:出張、4:外出、5:休暇、6:他）(初期値:6)
 *       writable           【TAG】画面編集フラグを[true:可/false:不可](初期値:false)
 *       editColor          【TAG】セル背景色の編集可否[true:編集可/false:編集不可](初期値:false)
 *       separator          【TAG】セル名称と設定値の間の区切り文字(初期値:":")
 *       paramVal           【TAG】セルの設定値はパラメータより渡す(初期値:"")
 *       paramColor         【TAG】セルの背景色はパラメータより渡す(初期値:"")
 *       notEditBgColor     【TAG】編集不可のセルの背景色(初期値:gray)
 *       debug              【TAG】デバッグ情報を出力するかどうか[true/false]を指定します(初期値:false)
 *   /&gt;
 *
 * ●使用例
 *  Select 文は、行, 列, 名称, 値, 色, 編集可否, ステータス の順で検索します。
 *  &lt;og:query &gt;
 *    select 行番号,列番号,セル名称,セル設定値,セル背景色,セルの編集可否,セルのステータス
 *    from   レイアウトテーブル
 *    where  ロケーションID = 'XXXX'
 *    order by 行番号,列番号
 *  &lt;/og:query&gt;
 *
 *  viewタグによるHTML出力 は、この通りにしてください。
 *  &lt;og:view
 *      viewFormType    = "HTMLDynamic"
 *      useScrollBar    = "false"
 *      useHilightRow   = "false"
 *      noMessage       = "true"
 *      pageSize        = "0"
 *  /&gt;
 *
 *  &lt;og:iMatrix
 *     mode             =   "Layout/Value" 
 *     countPro         =   "7"            
 *     inputWidth       =   "80px"         
 *     cellWidth        =   "100px"        
 *     cellHeight       =   "30px"         
 *     showStatus       =   "true/false"   
 *     countStatus      =   "6"            
 *     writable         =   "true/false"   
 *     editColor        =   "true/false"   
 *     separator        =   ":"            
 *     paramVal         =   "{&#064;XXX}"       
 *     paramColor       =   "{&#064;XXX}"       
 *     notEditBgColor   =   "#XXXXXX"      
 *  /&gt;
 *
 * @og.rev 5.6.3.2 (2013/04/12) 新規作成
 * @og.group 画面部品
 *
 * @version  5.0
 * @author	 Kazuhiko Hasegawa
 * @since    JDK6.0,
 */
public class ViewIMatrixTag extends CommonTagSupport {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "6.3.4.0 (2015/08/01)" ;
	private static final long serialVersionUID = 634020150801L ;

	/** mode 引数に渡す事の出来る モードリスト  */
	private static final String[] MODE_LIST = { "Layout" , "Value" };

	private transient TagBuffer tag = new TagBuffer( "iMatrix" ).addBody( "<!-- -->" );		// 6.1.1.0 (2015/01/17) TagBufferの連結記述

	/**
	 * Taglibの終了タグが見つかったときに処理する doEndTag() を オーバーライドします。
	 *
	 * @og.rev 6.0.2.5 (2014/10/31) HTML5対応。javaScriptで、BODYがないと入れ子になってしまう。
	 *
	 * @return	後続処理の指示
	 */
	@Override
	public int doEndTag() {
		debugPrint();		// 4.0.0 (2005/02/28)

		// 6.1.1.0 (2015/01/17) TagBufferの連結記述
		jspPrint( tag.makeTag() );

		return EVAL_PAGE ;		// ページの残りを評価する。
	}

	/**
	 * タグリブオブジェクトをリリースします。
	 * キャッシュされて再利用されるので、フィールドの初期設定を行います。
	 *
	 */
	@Override
	protected void release2() {
		super.release2();
		tag = new TagBuffer( "iMatrix" ).addBody( "<!-- -->" );		// 6.1.1.0 (2015/01/17) TagBufferの連結記述
	}

	/**
	 * 【TAG】配置図／座席表の編集モード(Layout:レイアウト編集/Value:設定値の編集)を指定します(初期値:Layout)。
	 *
	 * @og.tag
	 * mode は、Layout:レイアウト編集を行うのか、Value:設定値の編集を行うかを指定します。
	 * Layout:レイアウト編集(CELLNAMEとFGCELLEDITの値を編集します。)
	 * Value :設定値の編集(VALUEとCOLORの値を編集します。)
	 * (adjustMatrix.jsの)初期値は、Layout:レイアウト編集 です。
	 *
	 * @og.rev 6.3.4.0 (2015/08/01) Arrays.toString から String.join に置き換え。
	 *
	 * @param   mode 編集モード [Layout:レイアウト編集/Value:設定値の編集]
	 */
	public void setMode( final String mode ) {
		final String tmpMode = nval( getRequestParameter( mode ),null );

		if( !check( tmpMode, MODE_LIST ) ) {
			final String errMsg = "指定のモード(mode)は指定できません。モード指定エラー"	+ CR
							+ "mode=[" + tmpMode + "] "										+ CR
//							+ Arrays.toString( MODE_LIST ) ;			// 6.1.0.0 (2014/12/26)
							+ "modeList=" + String.join( ", " , MODE_LIST ) ;
			throw new HybsSystemException( errMsg );
		}

		tag.add( "mode",tmpMode );
	}

	/**
	 * 【TAG】データノードの属性数( 行, 列, 名称, 値, 色, 編集可否, ステータス )を指定します(初期値:7)。
	 *
	 * @og.tag
	 * SQL文のカラム数(非表示項目を除く)を指定します。
	 * SQL文は、先頭から、「行, 列, 名称, 値, 色, 編集可否, ステータス」の順に検索する必要があります。
	 * これ以外に、表示項目になるカラムを指定する場合に、表示カラム数を設定します。
	 * 必要分だけ（行, 列, 名称, 値, 色, 編集可否, ステータス）の場合は、"7" を指定します。
	 * (adjustMatrix.jsの)初期値は、"7" です。
	 *
	 * @param   countPro データノードの属性数
	 */
	public void setCountPro( final String countPro ) {
		tag.add( "countPro",nval( getRequestParameter( countPro ),null ) );
	}

	/**
	 * 【TAG】編集時の入力欄幅(px)を指定します(初期値:50px)。
	 *
	 * @og.tag
	 * 編集時の入力欄幅(px)を指定します。
	 * (adjustMatrix.jsの)初期値は、"50px" です。
	 *
	 * @param   inputWidth 入力欄幅(px)
	 */
	public void setInputWidth( final String inputWidth ) {
		tag.add( "inputWidth",nval( getRequestParameter( inputWidth ),null ) );
	}

	/**
	 * 【TAG】セル幅(px)を指定します(初期値:75px)。
	 *
	 * @og.tag
	 * セル幅(px)を指定します。
	 * (adjustMatrix.jsの)初期値は、"75px" です。
	 *
	 * @param   cellWidth セル幅(px)
	 */
	public void setCellWidth( final String cellWidth ) {
		tag.add( "cellWidth",nval( getRequestParameter( cellWidth ),null ) );
	}

	/**
	 * 【TAG】セル高さ(px)を指定します(初期値:30px)。
	 *
	 * @og.tag
	 * セル高さ(px)を指定します。
	 * (adjustMatrix.jsの)初期値は、"30px" です。
	 *
	 * @param   cellHeight セル高さ(px)
	 */
	public void setCellHeight( final String cellHeight ) {
		tag.add( "cellHeight",nval( getRequestParameter( cellHeight ),null ) );
	}

	/**
	 * 【TAG】ステータスの出力を行うかどうか[true:表示する/false:表示しない]を指定します(初期値:false)。
	 *
	 * @og.tag
	 * ステータスの出力を行う場合は、../image/status_番号.png イメージを画面に表示します。
	 * 番号が、ステータス番号になります。
	 * (adjustMatrix.jsの)初期値は、false:表示しない です。
	 *
	 * @param   showStatus ステータスの出力可否 [true:する/false:しない]
	 */
	public void setShowStatus( final String showStatus ) {
		tag.add( "showStatus",nval( getRequestParameter( showStatus ),null ) );
	}

	/**
	 * 【TAG】ステータスの種類数（1:会議、2:来客、3:出張、4:外出、5:休暇、6:他）を指定します(初期値:6)。
	 *
	 * @og.tag
	 * ステータスの種類数（1:会議、2:来客、3:出張、4:外出、5:休暇、6:他）は、編集ポップアップの作成に使います。
	 * 上記の場合、６種類になるため、countStatus="6" とします。
	 * 内部的には、status_0.png があります。
	 * (adjustMatrix.jsの)初期値は、"6" です。
	 *
	 * @param   countStatus ステータスの種類数
	 */
	public void setCountStatus( final String countStatus ) {
		tag.add( "countStatus",nval( getRequestParameter( countStatus ),null ) );
	}

	/**
	 * 【TAG】画面編集を許可するかどうか[true:可/false:不可]を指定します(初期値:false)。
	 *
	 * @og.tag
	 * true:画面編集可/false:画面編集不可を指定します。
	 * ここでは、mode=[Layout/Value] に関係なく、false にすると編集不可になります。
	 * (adjustMatrix.jsの)初期値は、false:編集不可 です。
	 *
	 * @param	writable 画面編集可否 [true:許可/false:不可]
	 */
	public void setWritable( final String writable ) {
		tag.add( "writable",nval( getRequestParameter( writable ),null ) );
	}

	/**
	 * 【TAG】セル背景色の編集可否[true:編集可/false:編集不可]を指定します(初期値:false)。
	 *
	 * @og.tag
	 * セル背景色の編集可否[true:編集可/false:編集不可]を指定します。
	 * これは、mode="Value" で、writable="true" の場合のみ有効です。
	 * false にすると、COLOR属性の編集ができなくなります。
	 * (adjustMatrix.jsの)初期値は、false:編集不可 です。
	 *
	 * @param   editColor セル背景色の編集可否 [true:する/false:しない]
	 */
	public void setEditColor( final String editColor ) {
		tag.add( "editColor",nval( getRequestParameter( editColor ),null ) );
	}

	/**
	 * 【TAG】セル名称と設定値の間の区切り文字を指定します(初期値:":")。
	 *
	 * @og.tag
	 * 画面上には、セル名称(CELLNAME)と設定値(VALUE)が表示されます。その時の、区切り文字を指定します。
	 * (adjustMatrix.jsの)初期値は、":" です。
	 *
	 * @param   separator 区切り文字
	 */
	public void setSeparator( final String separator ) {
		tag.add( "separator",nval( getRequestParameter( separator ),null ) );
	}

	/**
	 * 【TAG】セルの設定値(VALUE)を指定します(初期値:"")。
	 *
	 * @og.tag
	 * セルの設定値(VALUE)を指定します。
	 *
	 * @param   paramVal セルの設定値
	 */
	public void setParamVal( final String paramVal ) {
		tag.add( "paramVal",nval( getRequestParameter( paramVal ),null ) );
	}

	/**
	 * 【TAG】セルの背景色(COLOR)を指定します(初期値:"")。
	 *
	 * @og.tag
	 * セルの背景色を指定します。
	 *
	 * @param   paramColor セルの背景色
	 */
	public void setParamColor( final String paramColor ) {
		tag.add( "paramColor",nval( getRequestParameter( paramColor ),null ) );
	}

	/**
	 * 【TAG】編集不可のセルの背景色を指定します(初期値:gray)。
	 *
	 * @og.tag
	 * 編集不可のセルの背景色を指定します。
	 * (adjustMatrix.jsの)初期値は、"gray" です。
	 *
	 * @param   notEditBgColor 編集不可背景色
	 */
	public void setNotEditBgColor( final String notEditBgColor ) {
		tag.add( "notEditBgColor",nval( getRequestParameter( notEditBgColor ),null ) );
	}

	/**
	 * タグの名称を、返します。
	 * 自分自身のクラス名より、自動的に取り出せないため、このメソッドをオーバーライドします。
	 *
	 * @return  タグの名称
	 * @og.rtnNotNull
	 */
	@Override
	protected String getTagName() {
		return "iMatrix" ;
	}

	/**
	 * このオブジェクトの文字列表現を返します。
	 * 基本的にデバッグ目的に使用します。
	 *
	 * @return このクラスの文字列表現
	 * @og.rtnNotNull
	 */
	@Override
	public String toString() {
		return ToString.title( this.getClass().getName() )
				.println( "VERSION"		,VERSION	)
				.println( "tag"			,tag.makeTag()	)
				.println( "Other..."	,getAttributes().getAttribute() )
				.fixForm().toString() ;
	}
}
