/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.fukurou.util.StringUtil;
import org.opengion.fukurou.util.TagBuffer;

/**
 * 明細棒グラフで、バー部のヘッダーに、進捗率を示す線を表示します。
 *
 * iHead タグは、ガントの ヘッダー部に使用されます。
 * 進捗率を示す線は、２本あり、デフォルトでは、中間点と最大値になります。
 * グラフの右に、何かを表示する場合は、幅(width)を指定する必要があります。
 *
 * @og.formSample
 * ●形式：&lt;og:iHead  ... /&gt;
 * ●body：なし
 * ●前提：headタグで、adjustEvent="Bar" を指定してください。
 *
 * ●Tag定義：
 *   &lt;og:iHead
 *       width              【TAG】グラフ部分の横幅を[px]で指定します。
 *       leftVal            【TAG】左線のヘッダ部分の文字を指定します（初期値：最大桁の半分）
 *       leftLoc            【TAG】左線のヘッダ部文字（及び線）の位置を左からの割合で指定します。
 *       leftBorderStyle    【TAG】左線のスタイルを指定します(初期値=dotted gray 1px)。
 *       adjustLeft         【TAG】左線のヘッダ部分の文字の上下位置をピクセル単位で指定します(負の数で下方向)。
 *       rightVal           【TAG】右線のヘッダ部分の文字を指定します（初期値：最大桁）
 *       rightLoc           【TAG】右線のヘッダ部文字（及び線）の位置を左からの割合で指定します。
 *       rightBorderStyle   【TAG】右線のスタイルを指定します(初期値=dotted gray 1px)。
 *       adjustRight        【TAG】右線のヘッダ部分の文字の上下位置をピクセル単位で指定します(負の数で下方向)。
 *       debug              【TAG】デバッグ情報を出力するかどうか[true/false]を指定します(初期値:false)
 *   /&gt;
 *
 * ●使用例
 *    &lt;og:view
 *        viewFormType = "HTMLCustomTable"
 *        command      = "{&#064;command}"
 *        writable     = "false"
 *        useScrollBar = "false"
 *        headerSkipCount="10"
 *    &gt;
 *      &lt;og:thead rowspan="2"&gt;
 *        &lt;tr&gt;
 *          &lt;td rowspan="2"&gt;[NOORDER]&lt;/td&gt;
 *          &lt;td rowspan="2"&gt;
 *              &lt;og:iHead width="150px"
 *                      leftVal ="now"      leftLoc ="0.8"  leftBorderStyle ="solid red 1px"    adjustLeft ="15px" 
 *                      rightVal="Target"   rightLoc="1.0"  rightBorderStyle="dashed red 2px"   adjustRight="-15px"
 *              /&gt;
 *          &lt;td rowspan="2"&gt;[SINTYOKU]&lt;/td&gt;
 *        &lt;/tr&gt;
 *      &lt;/og:thead&gt;
 *      &lt;og:tbody rowspan="2"&gt;
 *          &lt;td rowspan="2"&gt;[NOORDER]&lt;/td&gt;
 *          &lt;td rowspan="2"&gt;
 *              &lt;iBar id="B0_[I]" type="0" value="[SINTYOKU]" baseVal="[KNORDER]" /&gt;
 *              &lt;iBar id="B1_[I]" type="1" value="[KNORDER_G]" color="[COLOR]" text="[KNORDER]" /&gt;
 *              &lt;br/&gt;
 *              &lt;iBar id="B2_[I]" type="1" value="[KNJISK_G],[KNMIKM_G]" color="black,green" /&gt;
 *              &lt;iBar id="B3_[I]" type="2" value="[KNJISK_G],[KNMIKM_G]" baseVal="[KNORDER_G]" color="green" /&gt;
 *          &lt;td rowspan="2"&gt;[SINTYOKU]&lt;/td&gt;
 *      &lt;/og:tbody&gt;
 *    &lt;/og:view&gt;
 *
 * @og.rev 5.6.3.2 (2013/04/12) 新規作成
 * @og.group 画面部品
 *
 * @version  5.0
 * @author	 Kazuhiko Hasegawa
 * @since    JDK6.0,
 */
public class ViewIHeadTag extends CommonTagSupport {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "5.6.3.2 (2013/04/12)" ;

	private static final long serialVersionUID = 563220130412L ;

	private TagBuffer tag = new TagBuffer( "iHead" ) ;

	/**
	 * Taglibの終了タグが見つかったときに処理する doEndTag() を オーバーライドします。
	 *
	 * @return	後続処理の指示
	 */
	@Override
	public int doEndTag() {
		debugPrint();		// 4.0.0 (2005/02/28)

		jspPrint( tag.makeTag() );

		return(EVAL_PAGE);		// ページの残りを評価する。
	}

	/**
	 * タグリブオブジェクトをリリースします。
	 * キャッシュされて再利用されるので、フィールドの初期設定を行います。
	 *
	 */
	@Override
	protected void release2() {
		super.release2();
		tag = new TagBuffer( "iHead" );
	}

	/**
	 * 【TAG】グラフ部分の横幅を[px]で指定します。
	 *
	 * @og.tag
	 * 指定しない場合は残りの横幅全てとなります。
	 * グラフの右側に何かを表示したい場合は指定する必要があります。
	 * 単位(px)は設定不要です。それ以外の単位は設定できません。
	 *
	 * @param   width グラフ部分の横幅
	 */
	public void setWidth( final String width ) {
		// JavaScript 側の都合で、px 単位は、削除します。
		String tmpW = StringUtil.nval( getRequestParameter( width ),null ) ;
		if( tmpW != null && tmpW.endsWith("px") ) {
			tmpW = tmpW.substring( 0,tmpW.length()-2 );
		}

		tag.add( "width",tmpW );
	}

	/**
	 * 【TAG】左線のヘッダ部分の文字を指定します（初期値：最大桁の半分）。
	 *
	 * @og.tag
	 * 左線のヘッダ部分に表示する文字を指定します。
	 * 左線は、中間データ、または、途中経過を意味しますので、その様な単語にします。
	 * なにも指定しない場合は、右側が最大桁以下を切り捨てした値となり、左側はその半分の値です。
	 *
	 * @param   leftVal 左線のヘッダ部分の文字
	 */
	public void setLeftVal( final String leftVal ) {
		tag.add( "leftVal",StringUtil.nval( getRequestParameter( leftVal ),null ) );
	}

	/**
	 * 【TAG】左線のヘッダ部文字（及び線）の位置を左からの割合(少数)で指定します。
	 *
	 * @og.tag
	 * グラフ幅全体を1として、左からの割合で指定します。(左が0.3で右が0.9など）
	 *
	 * @param   leftLoc 左線のヘッダ部文字（及び線）の位置の割合(少数)
	 */
	public void setLeftLoc( final String leftLoc ) {
		tag.add( "leftLoc",StringUtil.nval( getRequestParameter( leftLoc ),null ) );
	}

	/**
	 * 【TAG】左線のスタイルを指定します(初期値=dotted gray 1px)。
	 *
	 * @og.tag
	 * グラフに引かれる左線のスタイルを指定します。
	 * 初期値は、dotted gray 1px です。
	 *
	 * @param   leftBorderStyle 左線のヘッダ部文字（及び線）の位置の割合(少数)
	 */
	public void setLeftBorderStyle( final String leftBorderStyle ) {
		tag.add( "leftBorderStyle",StringUtil.nval( getRequestParameter( leftBorderStyle ),null ) );
	}

	/**
	 * 【TAG】左線のヘッダ部分の文字の上下位置をピクセル単位で指定します(負の数で下方向)。
	 *
	 * @og.tag
	 * 左右の文字が重なる可能性がある場合に利用します。
	 * 10と指定すると、10マイナスされて文字が上方向にずれます。
	 * 負の数を指定すると、文字が下方向にずれます。
	 * 単位(px)は設定不要です。それ以外の単位は設定できません。
	 *
	 * @param   adjustLeft 文字の上下位置指定
	 */
	public void setAdjustLeft( final String adjustLeft ) {
		// JavaScript 側の都合で、px 単位は、削除します。
		String tmpA = StringUtil.nval( getRequestParameter( adjustLeft ),null ) ;
		if( tmpA != null && tmpA.endsWith("px") ) {
			tmpA = tmpA.substring( 0,tmpA.length()-2 );
		}

		tag.add( "adjustLeft",tmpA );
	}

	/**
	 * 【TAG】右線のヘッダ部分の文字を指定します（初期値：最大桁）。
	 *
	 * @og.tag
	 * 右線のヘッダ部分に表示する文字を指定します。
	 * 右線は、最終データ、または、最大データを意味しますので、その様な単語にします。
	 * なにも指定しない場合は、右側が最大桁以下を切り捨てした値となり、左側はその半分の値です。
	 *
	 * @param   rightVal 左線のヘッダ部分の文字
	 */
	public void setRightVal( final String rightVal ) {
		tag.add( "rightVal",StringUtil.nval( getRequestParameter( rightVal ),null ) );
	}

	/**
	 * 【TAG】右線のヘッダ部文字（及び線）の位置を左からの割合(少数)で指定します。
	 *
	 * @og.tag
	 * グラフ幅全体を1として、左からの割合で指定します。(左が0.3で右が0.9など）
	 *
	 * @param   rightLoc 右線のヘッダ部文字（及び線）の位置の割合(少数)
	 */
	public void setRightLoc( final String rightLoc ) {
		tag.add( "rightLoc",StringUtil.nval( getRequestParameter( rightLoc ),null ) );
	}

	/**
	 * 【TAG】右線のスタイルを指定します(初期値=dotted gray 1px)。
	 *
	 * @og.tag
	 * グラフに引かれる右線のスタイルを指定します。
	 * 初期値は、dotted gray 1px です。
	 *
	 * @param   rightBorderStyle 左線のヘッダ部文字（及び線）の位置の割合(少数)
	 */
	public void setRightBorderStyle( final String rightBorderStyle ) {
		tag.add( "rightBorderStyle",StringUtil.nval( getRequestParameter( rightBorderStyle ),null ) );
	}

	/**
	 * 【TAG】右線のヘッダ部分の文字の上下位置をピクセル単位で指定します(負の数で下方向)。
	 *
	 * @og.tag
	 * 左右の文字が重なる可能性がある場合に利用します。
	 * 10と指定すると、10マイナスされて文字が上方向にずれます。
	 * 負の数を指定すると、文字が下方向にずれます。
	 * 単位(px)は設定不要です。それ以外の単位は設定できません。
	 *
	 * @param   adjustRight 文字の上下位置指定
	 */
	public void setAdjustRight( final String adjustRight ) {
		// JavaScript 側の都合で、px 単位は、削除します。
		String tmpA = StringUtil.nval( getRequestParameter( adjustRight ),null ) ;
		if( tmpA != null && tmpA.endsWith("px") ) {
			tmpA = tmpA.substring( 0,tmpA.length()-2 );
		}

		tag.add( "adjustRight",tmpA );
	}

	/**
	 * タグの名称を、返します。
	 * 自分自身のクラス名より、自動的に取り出せないため、このメソッドをオーバーライドします。
	 *
	 * @return  タグの名称
	 */
	@Override
	protected String getTagName() {
		return "iHead" ;
	}

	/**
	 * このオブジェクトの文字列表現を返します。
	 * 基本的にデバッグ目的に使用します。
	 *
	 * @return このクラスの文字列表現
	 */
	@Override
	public String toString() {
		return org.opengion.fukurou.util.ToString.title( this.getClass().getName() )
				.println( "VERSION"		,VERSION	)
				.println( "tag"			,tag.makeTag()	)
				.println( "Other..."	,getAttributes().getAttribute() )
				.fixForm().toString() ;
	}
}
