/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import static org.opengion.fukurou.util.StringUtil.nval;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.opengion.fukurou.util.StringUtil;
import org.opengion.fukurou.util.TagBuffer;
import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.hayabusa.db.DBColumn;
import org.opengion.hayabusa.db.DBEditConfig;
import org.opengion.hayabusa.db.DBLastSql;
import org.opengion.hayabusa.db.DBTableModel;

/**
 * 画面表示、集計に関する設定情報の表示、登録を行うためのタグです。
 * (このタグは標準の設定編集画面に組み込んで使用され、各画面JSPから呼び出すことはありません)
 *
 * このタグは、ユーザー単位に管理されるエディット設定オブジェクトに対するI/Fの機能を
 * 提供しています。このエディット設定オブジェクトについては、画面毎に設定を行うため、
 * タグの呼び出しには、画面IDが必須となっています。
 *
 * 具体的な機能としては、3つの機能を提供します。
 * (1)設定画面表示(command="GET")
 *    ユーザー単位に管理されるエディット設定オブジェクトをHTMLに変換して表示
 *    また、表示カラムの一覧(カンマ区切り)については、画面側のJavaScriptで再設定を行うため、
 *    その値を"viewClms"という名前のhiddenタグで出力します。
 * (2)エディット名一覧(command="LIST")
 *    指定の画面IDに対して、設定されているエディット名の一覧をプルダウン(selectタグ)に
 *    変換して表示します。(name="editName")
 * (3)設定情報登録/削除(command="SET"/"DELETE")
 *    (1)で設定された内容に対して、エディット名を指定してその内容を保存/削除します。
 *    情報の保存は、command="GET"で表示される項目名と連動していますので、単独での使用は
 *    できません。
 *
 * @og.formSample
 * ●形式：一般ユーザーが直接組み込むことはありません。
 * ●body：なし
 *
 * ●Tag定義：
 *   &lt;og:editConfig
 *       command          ○【TAG】command を指定します。(必須)
 *       gamenId          ○【TAG】画面ID を指定します。(必須)
 *       editName           【TAG】エディット名 を指定します
 *       debug              【TAG】デバッグ情報を出力するかどうか[true/false]を指定します(初期値:false)
 *   /&gt;
 *
 * ●使用例
 *     &lt;og:editConfig command="{&#064;command}" gamenId="{&#064;gamenId}" editName="{&#064;editName}" /&gt;
 *
 *     &lt;og:editConfig
 *         command        = command設定 (GET/LIST/SET/REMOVE)
 *         gamenId        = "GE0000"    画面ID
 *       [ editName ]     = "EDITNAME"  エディット名
 *     /&gt;
 *
 * @og.group エディット設定
 *
 * @og.rev 5.3.6.0 (2011/06/01)
 *
 * @version  5.0
 * @author	 Hiroki Nakamura
 * @since    JDK6.0,
 */
public class EditConfigTag extends CommonTagSupport {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "5.5.5.2 (2012/08/10)" ;

	private static final long serialVersionUID = 555220120810L ;

	private static final String VIEW_PREFIX			= "EDIT_VIEW_";
	private static final String SUM_PREFIX			= "EDIT_SUM_";
	private static final String GROUP_PREFIX		= "EDIT_GROUP_";
	private static final String SUBTOTAL_PREFIX		= "EDIT_SUBTOTAL_";
	private static final String TOTAL_PREFIX		= "EDIT_TOTAL_";
	private static final String ORDERBY_PREFIX		= "EDIT_ORDERBY_";
	private static final String DESC_PREFIX			= "EDIT_DESC_";
	private static final String GRANDTOTAL_PREFIX	= "EDIT_GRANDTOTAL_";
	private static final String COMMON_PREFIX		= "EDIT_COMMON_";

	private String	command			= null;		// EW" 、アップロード="COPY|INSERT"
	private String	gamenId			= null;
	private String	editName		= null;

	private transient DBTableModel table		= null;		// 5.5.2.4 (2012/05/16) transient 定義追加
	private transient DBEditConfig config		= null;		// 5.5.2.4 (2012/05/16) transient 定義追加
	
	private boolean orderOnly		= false; // 5.5.5.2 (2012/08/10)

	/**
	 * Taglibの開始タグが見つかったときに処理する doStartTag() を オーバーライドします。
	 *
	 * @return	後続処理の指示(SKIP_BODY)
	 */
	@Override
	public int doStartTag() {
		return(SKIP_BODY);				// Body を評価しない
	}

	/**
	 * Taglibの終了タグが見つかったときに処理する doEndTag() を オーバーライドします。
	 * @return	後続処理の指示
	 */
	@Override
	public int doEndTag() {
		debugPrint();

		StringBuilder buf = new StringBuilder( HybsSystem.BUFFER_MIDDLE );

		// エディット情報をHTMLに変換して表示します。
		// 表示に当たって、最後に発行されたQUERYのtableId、scopeをチェックした上で
		// 表示するかを判断します。
		if( "GET".equals( command ) ) {
			DBLastSql lastSql = (DBLastSql)getSessionAttribute( HybsSystem.DB_LAST_SQL_KEY );
			if( lastSql != null ) {
				if( !lastSql.isViewEditable() ) {
					// この画面は、項目の並び替えはできません。
					String rtn = "<b style=\"font-color:red;\">" + getResource().getLabel( "GEE0003" ) + "</b>";
					jspPrint( rtn );
				}
				else if( lastSql.isGuiMatch( gamenId ) ) {
					setScope( lastSql.getScope() );
					table = (DBTableModel)getObject( lastSql.getTableId() );
					if( table != null ) {
						config = getUser().getEditConfig( gamenId, editName );
						String viewClms = null;
						if( config == null ) {
							viewClms = lastSql.getViewClmNames();
							config = new DBEditConfig();
						}
						else {
							viewClms = config.getViewClms();
						}
						buf.append( makeEditTable( viewClms ) );
					}
				}
			}
		}
		// エディット情報を保存します。
		else if( "SET".equals( command ) ) {
			if( editName == null || editName.length() == 0 ) {
				String msg = "エディット名が指定されていません。";
				throw new HybsSystemException( msg );
			}
			saveEditConfig();
		}
		// エディット情報を削除します。
		else if( "DELETE".equals( command ) ) {
			if( editName == null || editName.length() == 0 ) {
				String msg = "エディット名が指定されていません。";
				throw new HybsSystemException( msg );
			}
			deleteEditConfig();
		}
		// 指定された画面IDに対するエディット情報の一覧(プルダウン)を表示します。
		else if( "LIST".equals( command ) ) {
			DBEditConfig[] configs = getUser().getEditConfigs( gamenId );
			if( configs != null && configs.length > 0 ) {
				buf.append( getEditSelect( configs ) ).append( HybsSystem.CR );
			}
		}

		jspPrint( buf.toString() );

		return(EVAL_PAGE);
	}

	/**
	 * タグリブオブジェクトをリリースします。
	 * キャッシュされて再利用されるので、フィールドの初期設定を行います。
	 *
	 * @og.rev 5.5.5.2 (2012/08/10) orderOnly対応
	 */
	@Override
	protected void release2() {
		super.release2();
		command = "GET";
		gamenId = null;
		editName = null;
		table = null;
		config = null;
		orderOnly		= false; //5.5.5.2 (2012/08/10)
	}

	/**
	 * エディット情報をHTMLに変換して表示します。
	 *
	 * @og.rev 5.4.2.0 (2011/12/01) 入替え対象のカラム列でのみスクロールが表示されるように対応します。
	 * @og.rev 5.5.5.2 (2012/08/10) orderOnly対応
	 *
	 * @param viewClms 表示カラム(カンマ区切り)
	 *
	 * @return エディット情報のHTML
	 */
	private String makeEditTable( final String viewClms ) {
		StringBuilder buf = new StringBuilder( HybsSystem.BUFFER_MIDDLE );

		boolean useSum = getUseSum( viewClms );
		String[] viewGroups = StringUtil.csv2Array( viewClms, '|' );
		buf.append( "<input type=\"hidden\" name=\"viewClms\" id=\"viewClms\" value=\"" + viewClms + "\"/>" );
		buf.append( "<div />" );
		buf.append( "<div style=\"float:left;\">" );
		buf.append( makeLabelRow( useSum ) );
		buf.append( "</div>" );
		buf.append( "<div id=\"clmLayer\" style=\" float:left; width: 670px;overflow-x:scroll;\">" );
		for( int i=0; i<viewGroups.length; i++ ) {
			if( i > 0 ) {
				buf.append( makeSeparateRow( useSum ) );
			}
			buf.append( "<table class=\"clmGroup\" style=\"float:left;\"><tr>" );
			String[] clms = StringUtil.csv2Array( viewGroups[i] );
			for( int j=0; j<clms.length; j++ ) {
				String clm = ( !clms[j].startsWith( "!" ) ? clms[j] : clms[j].substring( 1 ) );
				if( "rowCount".equals( clm ) ) { continue; }
				boolean isView = ( !clms[j].startsWith( "!" ) ? true : false );
				buf.append( makeColumnRow( clm, isView, useSum, config ) );
			}
			buf.append( "</tr></table>" );
		}
		buf.append( "</div>" );

		String grandTotalLabel = "<b>" + getDBColumn( GRANDTOTAL_PREFIX + "LABEL" ).getLongLabel() + ":</b>";
		buf.append( "<div style=\"clear:both;\">" );
		buf.append( "<table>" );
//		buf.append( makeCheckbox( GRANDTOTAL_PREFIX, config.useGrandTotal(), "h", grandTotalLabel ) );
		buf.append( makeCheckbox( GRANDTOTAL_PREFIX, config.useGrandTotal(), "h", grandTotalLabel, orderOnly ) ); // 5.5.5.2 (2012/08/10)
		buf.append( "</table>" );
		buf.append( "</div>" );

		return buf.toString();
	}

	/**
	 * エディット情報のヘッダー(ラベル行)のHTMLを生成します。
	 *
	 * @og.rev 5.4.2.0 (2011/12/01) 表示項目の全チェック機能を追加
	 * @og.rev 5.5.5.2 (2012/08/10) orderOnly対応
	 *
	 * @param useSum 集計対象のカラム(=NUMBER型)が存在しているか
	 *
	 * @return エディット情報のヘッダー(ラベル行)のHTML
	 */
	private String makeLabelRow( final boolean useSum ) {
		StringBuilder buf = new StringBuilder( HybsSystem.BUFFER_SMALL );
		String commonLabel = "<b>" + getDBColumn( COMMON_PREFIX + "LABEL" ).getLongLabel() + ":</b>";
		String canEditCommon = HybsSystem.sys( "EDIT_COMMON_ROLES" );

		String groupLabel = "<b>" + getDBColumn( GROUP_PREFIX + "LABEL" ).getLongLabel() + "</b>";
		groupLabel += "<img id=\"groupBtn\" src=\"" + HybsSystem.sys( "JSP" ) + "/image/ball-green.gif\" />";

		buf.append( "<table><tr>" );
		buf.append( "<td style=\"margin:0px; padding:0px;\"><table>" );
		if( getUser().isAccess( canEditCommon ) ) {
//			buf.append( makeCheckbox( COMMON_PREFIX, config.isCommon(), "h", commonLabel ) );
			buf.append( makeCheckbox( COMMON_PREFIX, config.isCommon(), "h", commonLabel, orderOnly ) ); // 5.5.5.2 (2012/08/10)
		}
		else {
			buf.append( makeLabel	( commonLabel ) );
		}
//		buf.append( makeLabel	( VIEW_PREFIX		+ "LABEL" ) );
		String viewLabel = "<b>" + getDBColumn( VIEW_PREFIX + "LABEL" ).getLongLabel() + ":</b>";
//		buf.append( makeCheckbox( "VIEW_ALL_CHECK", true, "h", viewLabel ) );
		buf.append( makeCheckbox( "VIEW_ALL_CHECK", true, "h", viewLabel, orderOnly ) ); // 5.5.5.2 (2012/08/10)
		if( useSum ) {
			buf.append( makeLabel	( SUM_PREFIX		+ "LABEL" ) );
		}
		buf.append( makeCell	( groupLabel, "h" ) );
		buf.append( makeLabel	( SUBTOTAL_PREFIX	+ "LABEL" ) );
		buf.append( makeLabel	( TOTAL_PREFIX		+ "LABEL" ) );
		buf.append( makeLabel	( ORDERBY_PREFIX	+ "LABEL" ) );
		buf.append( makeLabel	( DESC_PREFIX		+ "LABEL" ) );
		buf.append( "</table></td>" );
		buf.append( "</tr></table>" );
		return buf.toString();
	}

	/**
	 * エディット情報のカラム列のHTMLを生成します。
	 * 
	 * @og.rev 5.5.5.2 (2012/08/10) orderOnly対応
	 *
	 * @param clm カラム
	 * @param isView 表示対象かどうか
	 * @param useSum 集計対象のカラム(=NUMBER型)が存在しているか
	 * @param config エディット設定オブジェクト
	 *
	 * @return エディット情報のカラム列のHTMLを生成します。
	 */
	private String makeColumnRow( final String clm, final boolean isView, final boolean useSum, final DBEditConfig config ) {
		StringBuilder buf = new StringBuilder( HybsSystem.BUFFER_SMALL );

		int clmNo = table.getColumnNo( clm, false  );
		DBColumn column = ( clmNo < 0 ? getDBColumn( clm ) : table.getDBColumn( clmNo ) );
		buf.append( "<td name=\"" ).append( clm ).append( "\" class=\"sortItem\" style=\"margin:0px; padding:0px;\">" );
		buf.append( "<table>" );
		buf.append( makeLabel	( column.getLongLabel() ) );
//		buf.append( makeCheckbox( VIEW_PREFIX			+ clm, isView						, "0", null ) );
		buf.append( makeCheckbox( VIEW_PREFIX			+ clm, isView						, "0", null, orderOnly ) ); // 5.5.5.2 (2012/08/10)
		if( useSum ) {
			boolean isSumClm = isNumberClm( clm );
//			buf.append( makeCheckbox( SUM_PREFIX		+ clm, config.isSumClm( clm )		, "1", isSumClm , null ) );
			buf.append( makeCheckbox( SUM_PREFIX		+ clm, config.isSumClm( clm )		, "1", isSumClm , null, orderOnly ) ); // 5.5.5.2 (2012/08/10)
		}
//		buf.append( makeCheckbox( GROUP_PREFIX		+ clm, config.isGroupClm( clm )		, "0", null ) );
//		buf.append( makeCheckbox( SUBTOTAL_PREFIX	+ clm, config.isSubTotalClm( clm )	, "1", null ) );
//		buf.append( makeCheckbox( TOTAL_PREFIX		+ clm, config.isTotalClm( clm )		, "0", null ) );
//		buf.append( makeInput	( ORDERBY_PREFIX		+ clm, config.getOrder( clm )		, "1", null ) );
//		buf.append( makeCheckbox( DESC_PREFIX			+ clm, config.isOrderByDesc( clm )	, "0", null ) );
		buf.append( makeCheckbox( GROUP_PREFIX		+ clm, config.isGroupClm( clm )		, "0", null, orderOnly ) ); // 5.5.5.2 (2012/08/10)
		buf.append( makeCheckbox( SUBTOTAL_PREFIX	+ clm, config.isSubTotalClm( clm )	, "1", null, orderOnly ) ); // 5.5.5.2 (2012/08/10)
		buf.append( makeCheckbox( TOTAL_PREFIX		+ clm, config.isTotalClm( clm )		, "0", null, orderOnly ) ); // 5.5.5.2 (2012/08/10)
		buf.append( makeInput	( ORDERBY_PREFIX		+ clm, config.getOrder( clm )		, "1", null ) );
		buf.append( makeCheckbox( DESC_PREFIX			+ clm, config.isOrderByDesc( clm )	, "0", null, orderOnly ) ); // 5.5.5.2 (2012/08/10)
		buf.append( "</table>" );
		buf.append( "</td>" );

		return buf.toString();
	}

	/**
	 * チェックボックスのHTML文字列を生成します。
	 * 生成したHTMLは以下のようになります。
	 * 例)&lt;tr&gt;&lt;td class="row_[bgCnt]" ...&gt;[prefix]&lt;input type="checkbox" name="[clm]" ... /&gt;&lt;/td&gt;&lt;/tr&gt;
	 *
	 * @param clm カラム
	 * @param checked 初期チェックするかどうか
	 * @param bgCnt 背景色ゼブラカラーの指定("0"or"1"or"h")
	 * @param prefix チェックボックスのタグの前に挿入するHTML文字列
	 * @param readonly リードオンリー
	 * 
	 * @og.rev 5.5.5.2 (2012/08/10) readOnly追加
	 *
	 * @return チェックボックスのHMTL文字列
	 */
//	private String makeCheckbox( final String clm, final boolean checked, final String bgCnt, final String prefix ) {
	private String makeCheckbox( final String clm, final boolean checked, final String bgCnt, final String prefix, final boolean readonly ) {
		return makeCheckbox( clm, checked, bgCnt, true, prefix, readonly );
	}

	/**
	 * チェックボックスのHTML文字列を生成します。
	 * 生成したHTMLは以下のようになります。
	 * 例)&lt;tr&gt;&lt;td class="row_[bgCnt]" ...&gt;[prefix]&lt;input type="checkbox" name="[clm]" ... /&gt;&lt;/td&gt;&lt;/tr&gt;
	 *
	 * @param clm カラム
	 * @param checked 初期チェックするかどうか
	 * @param bgCnt 背景色ゼブラカラーの指定("0"or"1"or"h")
	 * @param isChbox チェックボックスを生成するかどうか(falseの場合、チェックボックスのinputタグは生成されません)
	 * @param prefix チェックボックスのタグの前に挿入するHTML文字列
	 * @param readonly リードオンリー
	 * 
	 * @og.rev 5.5.5.2 (2012/08/10) readOnly追加
	 *
	 * @return チェックボックスのHMTL文字列
	 */
//	private String makeCheckbox( final String clm, final boolean checked, final String bgCnt, final boolean isChbox, final String prefix ) {
	private String makeCheckbox( final String clm, final boolean checked, final String bgCnt, final boolean isChbox, final String prefix, final boolean readonly ) {
		if( isChbox ) {
			String suffix = "";
			TagBuffer tag = new TagBuffer( "input" );
			tag.add( "type", "checkbox" );
			tag.add( "name", clm );
			tag.add( "value", "1" );
			if( checked ) {
				tag.add( "checked", "checked" );
			}
			if( readonly ){ // 5.5.5.2 (2012/08/10)
				tag.add( "disabled", "disabled" );
				if( checked ){
					TagBuffer tag2 = new TagBuffer( "input" );
					tag2.add( "type", "hidden" );
					tag2.add( "name", clm );
					tag2.add( "value", "1" );
					suffix += tag2.makeTag();
				}
				
			}
//			return makeCell( ( prefix == null ? "" : prefix ) + tag.makeTag(), bgCnt );
			return makeCell( ( prefix == null ? "" : prefix ) + tag.makeTag() + suffix, bgCnt ); // 5.5.5.2 (2012/08/10)
		}
		else {
			return makeCell( ( prefix == null ? "" : prefix ) + "&nbsp;", bgCnt );
		}
	}

	/**
	 * テキスト入力HTML文字列を生成します。
	 * 生成したHTMLは以下のようになります。
	 * 例)&lt;tr&gt;&lt;td class="row_[bgCnt]" ...&gt;[prefix]&lt;input type="text" name="[clm]" ... /&gt;&lt;/td&gt;&lt;/tr&gt;
	 *
	 * @param clm カラム
	 * @param value 初期チェックするかどうか
	 * @param bgCnt 背景色ゼブラカラーの指定("0"or"1"or"h")
	 * @param prefix チェックボックスのタグの前に挿入するHTML文字列
	 *
	 * @return チェックボックスのHMTL文字列
	 */
	private String makeInput( final String clm, final String value, final String bgCnt, final String prefix ) {
		TagBuffer tag = new TagBuffer( "input" );
		tag.add( "type", "text" );
		tag.add( "name", clm );
		tag.add( "value", value );
		tag.add( "style", "width: 10px; font-size:10px;" );
		tag.add( "maxlength", "2" );
		tag.add( "class", "S9" );

		return makeCell( ( prefix == null ? "" : prefix ) + tag.makeTag(), bgCnt );
	}

	/**
	 * 左右分割されている際の分割列のHTML文字列を生成します。
	 *
	 * @param useSum 集計対象のカラム(=NUMBER型)が存在しているか
	 *
	 * @return チェックボックスのHMTL文字列
	 */
	private String makeSeparateRow( final boolean useSum ) {
		StringBuilder buf = new StringBuilder( HybsSystem.BUFFER_SMALL );

		buf.append( "<table style=\"float:left;\"><tr>" );
		buf.append( "<td style=\"margin:0px; padding:0px;\"><table>" );
		buf.append( makeCell( "&nbsp", "h" ) );		// ラベル
		buf.append( makeCell( "&nbsp", "h" ) );		// 表示
		if( useSum ) {
			buf.append( makeCell( "&nbsp", "h" ) );		// 集計項目
		}
		buf.append( makeCell( "&nbsp", "h" ) ); // 集計キー
		buf.append( makeCell( "&nbsp", "h" ) );	// 小計キー
		buf.append( makeCell( "&nbsp", "h" ) ); // 合計キー
		buf.append( makeCell( "&nbsp", "h" ) ); // 表示順
		buf.append( makeCell( "&nbsp", "h" ) ); // 昇順・降順
		buf.append( "</table></td>");
		buf.append( "</tr></table>" );

		return buf.toString();
	}

	/**
	 * ラベルのHTML文字列を生成します。
	 *
	 * @param clm カラム
	 *
	 * @return ラベルのHTML文字列
	 */
	private String makeLabel( final String clm ) {
		return makeCell( getDBColumn( clm ).getLongLabel(), "h" );
	}

	/**
	 * セルのHTML文字列を生成します。
	 *
	 * @param body tdタグ内のHTML文字列
	 * @param bgCnt 背景色ゼブラカラーの指定("0"or"1"or"h")
	 *
	 * @return セルのHTML文字列
	 */
	private String makeCell( final String body, final String bgCnt ) {
		return "<tr><td align=\"center\" style=\"height:22px;\" class=\"row_" + bgCnt + "\">" + body + "</td></tr>";
	}

	/**
	 * このエディット設定で集計対象のカラム(=NUMBER型)が存在しているかを返します。
	 *
	 * @param viewClms カラム
	 *
	 * @return 集計対象のカラム(=NUMBER型)が存在しているか
	 */
	private boolean getUseSum( final String viewClms ) {
		if( viewClms == null ) { return false; }

		boolean rtn = false;
		String[] clms = StringUtil.csv2Array( viewClms.replace( '|', ',' ) );
		for( int j=0; j<clms.length; j++ ) {
			String clm = ( !clms[j].startsWith( "!" ) ? clms[j] : clms[j].substring( 1 ) );
			rtn = isNumberClm( clm );
			if( rtn ) { break; }
		}
		return rtn;
	}

	/**
	 * 引数のカラムがNUMBER型かどうかをチェックします。
	 *
	 * @param clm カラム
	 *
	 * @return NUMBER型かどうか
	 */
	private boolean isNumberClm( final String clm ) {
		if( clm == null ) { return false; }

		int no = table.getColumnNo( clm, false );
		if( no >= 0 ) {
			DBColumn dbClm = table.getDBColumn( table.getColumnNo( clm ) );
			if( dbClm != null ) {
				if( "NUMBER".equals( dbClm.getClassName()) ) {
					return true;
				}
			}
		}
		return false;
	}

	/**
	 * エディット設定情報を保存します。
	 */
	private void saveEditConfig() {
		String viewClms		= getRequest().getParameter( "viewClms" );
		String sumClms		= getColumns( SUM_PREFIX );
		String groupClms	= getColumns( GROUP_PREFIX );
		String subTotalClms = getColumns( SUBTOTAL_PREFIX );
		String totalClms	= getColumns( TOTAL_PREFIX );
		String useGrandTotal= getRequest().getParameter( GRANDTOTAL_PREFIX );
		String orderByClms	= getOrderByColumns();
		String isCommon		= getRequest().getParameter( COMMON_PREFIX );

		DBEditConfig config
			= new DBEditConfig( editName, viewClms, sumClms, groupClms
								, subTotalClms, totalClms, useGrandTotal, orderByClms, isCommon );

		getUser().addEditConfig( gamenId, editName, config );
	}

	/**
	 * エディット設定情報を削除します。
	 */
	private void deleteEditConfig() {
		getUser().deleteEditConfig( gamenId, editName );
	}

	/**
	 * パラメーターから引数のプレフィックスをキーに、チェックされたカラム一覧(カンマ区切り)を返します。
	 *
	 * @param prefixKey 各キーの取得するためのプレフィックス
	 *
	 * @return カラム一覧(カンマ区切り)
	 */
	private String getColumns( final String prefixKey ) {
		StringBuilder buf = new StringBuilder();

		Enumeration<?> enume = getParameterNames();
		while( enume.hasMoreElements() ) {
			String key = (String)(enume.nextElement());
			if( key.startsWith( prefixKey ) ) {
				String val = getRequest().getParameter( key );
				if( "1".equals( val ) ) {
					String clm = key.substring( prefixKey.length() );
					if( buf.length() > 0 ) { buf.append( "," ); }
					buf.append( clm );
				}
			}
		}

		return buf.toString();
	}

	/**
	 * 表示順のカラム一覧(カンマ区切り)を返します。
	 *
	 * @return 表示順のカラム一覧(カンマ区切り)
	 */
	private String getOrderByColumns() {
		Enumeration<?> enume = getParameterNames();
		List<Integer> orderNo = new ArrayList<Integer>();
		Map<Integer,String> orderClm = new HashMap<Integer,String>();
		while( enume.hasMoreElements() ) {
			String key = (String)(enume.nextElement());
			if( key.startsWith( ORDERBY_PREFIX ) ) {
				String val = getRequest().getParameter( key );
				if( val != null && val.length() > 0 ) {
					String clm = key.substring( ORDERBY_PREFIX.length() );
					String desc = getRequest().getParameter( DESC_PREFIX + clm );
					if( "1".equals( desc ) ) {
						clm = "!"  + clm;
					}
					// 数字項目以外が入力された場合は無視
					Integer odno = null;
					try {
						odno = Integer.valueOf( val );
					}
					catch ( NumberFormatException ex ) {
						continue;
					}
					String str = orderClm.get( odno );
					// 同じ番号の場合でも重ならないように振り直しする。
					while( str != null ) {
						odno = Integer.valueOf( odno.intValue() + 1 );
						str = orderClm.get( odno );
					}
					orderClm.put( odno, clm );
					orderNo.add( odno );
				}
			}
		}

		Collections.sort( orderNo );

		StringBuilder buf = new StringBuilder();
		for( Integer i : orderNo ) {
			if( buf.length() > 0 ) { buf.append( "," ); }
			String clm = orderClm.get( i );
			buf.append( clm );
		}

		return buf.toString();
	}

	/**
	 * エディット設定一覧のプルダウンメニューを作成します。
	 *
	 * @param	configs	DBEditConfig配列
	 *
	 * @return	エディット一覧のプルダウン
	 */
	private String getEditSelect( final DBEditConfig[] configs ) {
		DBColumn column = getDBColumn( "editName" );

		StringBuilder buf = new StringBuilder();
		buf.append( "<span class=\"label editName\">" )
			.append( column.getLongLabel() )
			.append( ":</span><span class=\"editName\">" )
			.append( "<select name=\"editName\">" )
			.append( "<option />" );
		for( DBEditConfig config : configs ) {
			String name = config.getEditName();
			buf.append( "<option value=\"" ).append( name ).append( "\"" );
			if( config.isCommon() ) {
				buf.append( " class=\"commonEdit\"" );
			}
			buf.append( "\">" );
			buf.append( name ).append( "</option>" );
		}
		buf.append( "</select></span>" );
		return buf.toString();
	}

	/**
	 * 【TAG】command を指定します。
	 *
	 * @og.tag
	 * command を指定します。
	 * [GET/LIST/SET/DELETE]のみが設定可能です。それ以外の場合、何も処理されません。
	 *
	 * @param	cmd コマンド[GET/LIST/SET/DELETE]
	 */
	public void setCommand( final String cmd ) {
		command = nval( getRequestParameter( cmd ),command );
	}

	/**
	 * 【TAG】画面ID を指定します。
	 *
	 * @og.tag
	 * 画面ID を指定します。
	 *
	 * @param	key 画面ID
	 */
	public void setGamenId( final String key ) {
		gamenId = nval( getRequestParameter( key ),gamenId );
	}

	/**
	 * 【TAG】エディット名 を指定します。
	 *
	 * @og.tag
	 * エディット名 を指定します。
	 * commandがSETまたはDELETEの場合は必須です。
	 * commandがGETまたはLISTの場合は無効です。
	 *
	 * @param	name エディット名
	 */
	public void setEditName( final String name ) {
		editName = nval( getRequestParameter( name ),editName );
	}
	
	/**
	 * 【TAG】チェックボックスのリードオンリー化を行います
	 *
	 * @og.tag
	 * 順番の入れ替えと、表示順の設定のみを行う場合にtrueにします。
	 * 表示/非表示切替や、集計機能は利用できなくなります。
	 * （チェックボックスのリードオンリーはできないため、実際にはdisable+hiddenで出力しています）
	 *
	 * @og.rev 5.5.5.2 (2012/08/10) 新規追加
	 *
	 * @param   flag  [true:リードオンリー/それ以外:編集可]
	 */
	public void setOrderOnly( final String flag ) {
		orderOnly = nval( getRequestParameter( flag ),orderOnly );
	}

	/**
	 * このオブジェクトの文字列表現を返します。
	 * 基本的にデバッグ目的に使用します。
	 *
	 * @return このクラスの文字列表現
	 */
	@Override
	public String toString() {
		return org.opengion.fukurou.util.ToString.title( this.getClass().getName() )
				.println( "VERSION"		,VERSION		)
				.println( "command"		,command		)
				.println( "gamenId"		,gamenId		)
				.println( "editName"	,editName		)
				.println( "Other..."	,getAttributes().getAttribute() )
				.fixForm().toString() ;
	}
}
