/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

/**
 * JSPアプリケーションに対してバージョン管理できるようにコメントを
 * 記述するためのタグです。
 *
 * バージョン、ビルド番号、日付、変更者、タイトルなどを記述できます。
 * 変更内容は、text属性か、BODY 部に記述します。
 * パラメーター変数({&#064;XXXX})は、使用できません。
 *
 * @og.formSample
 * ●形式：&lt;og:comment title="･･･" version="･･･" date="･･･" author="･･･" ･･･ /&gt;
 * ●body：あり(EVAL_BODY_BUFFERED:BODYを評価し、{&#064;XXXX} を解析します)
 *
 * ●Tag定義：
 *   &lt;og:comment
 *       rdcCode            【TAG】コメントの設計変更要求コードを設定します
 *       title            ○【TAG】コメントのタイトルを設定します。(必須)
 *       version          ○【TAG】コメントのバージョンを設定します。(必須)
 *       build              【TAG】コメントのビルド番号を設定します
 *       date             ○【TAG】コメントの変更日付を設定します。(必須)
 *       author           ○【TAG】コメントの変更者を設定します。(必須)
 *       text               【TAG】コメントの内容を設定します
 *       debug              【TAG】デバッグ情報を出力するかどうか[true/false]を指定します(初期値:false)
 *       pgid               【TAG】コメントのプログラムIDを設定します
 *       system             【TAG】コメントのシステム名称を設定します
 *   &gt;   ... Body ...
 *   &lt;/og:comment&gt;
 *
 * ●使用例
 *     &lt;og:comment
 *         title   = "変更概要"
 *         version = "001"
 *         date    = "YYYY/MM/DD"
 *         author  = "Y.MATSUI(Hybs) "
 *         rdcCode = "RDC00001"
 *         text    = "当処理中の変更内容を記述します。"
 *     /&gt;
 *
 *     &lt;og:comment
 *         title   = "変更概要"
 *         version = "001"
 *         date    = "YYYY/MM/DD"
 *         author  = "Y.MATSUI(Hybs) "
 *         rdcCode = "RDC00001"
 *         build   = "ビルド番号"           (必須属性では有りません。)
 *     &gt;
 *       当処理中の変更内容を記述します。
 *     &lt;/og:comment&gt;
 *
 * @og.rev 2.1.3.0 (2002/12/12) JSP画面の管理用に、画面に変更履歴を持たせる為のタグを新規作成
 * @og.rev 3.3.1.0 (2003/06/24) text 属性追加、パラメータ変数を使えないように変更。
 * @og.group 画面部品
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class CommentTag extends CommonTagSupport {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "4.0.0.0 (2005/08/31)" ;

	private static final long serialVersionUID = 400020050831L ;

	private String rdcCode	= null;
	private String title	= null;
	private String version	= null;
	private String build	= null;
	private String date		= null;
	private String author	= null;
	private String text		= null;
	private String pgid		= null; // 3.8.9.3
	private String system		= null; // 3.8.9.3

	/**
	 * Taglibの開始タグが見つかったときに処理する doStartTag() を オーバーライドします。
	 *
	 * @og.rev 3.3.1.0 (2003/06/24) text 属性追加、パラメータ変数を使えないように変更。
	 *
	 * @return	後続処理の指示
	 */
	@Override
	public int doStartTag() {
		if( text == null ) {
			return( EVAL_BODY_BUFFERED );	// Body を評価する。( extends BodyTagSupport 時)
		}
		else {
			return(SKIP_BODY);				// Body を評価しない
		}
	}

	/**
	 * Taglibのタグ本体を処理する doAfterBody() を オーバーライドします。
	 *
	 * @og.rev 2.2.0.0 (2002/12/17) 中国語(国際化)対応 エンコードの取得方法変更
	 * @og.rev 3.0.0.0 (2002/12/25) StringUtil#changeString 廃止
	 * @og.rev 3.1.1.0 (2003/03/28) ボディの内容を取得する処理を、CommonTagSupport で行う。
	 *
	 * @return	後続処理の指示(SKIP_BODY)
	 */
	@Override
	public int doAfterBody() {
		text = getBodyString();

		return(SKIP_BODY);
	}

	/**
	 * タグリブオブジェクトをリリースします。
	 * キャッシュされて再利用されるので、フィールドの初期設定を行います。
	 *
	 * @og.rev 2.0.0.4 (2002/09/27) カスタムタグの release() メソッドを、追加
	 * @og.rev 3.1.1.2 (2003/04/04) Tomcat4.1 対応。release2() を doEndTag()で呼ぶ。
	 * @og.rev 3.3.2.0 (2003/07/07) 設計変更要求コード(rdcCode)属性を追加。
 	 * @og.rev 3.8.9.3 (2007/08/31) PGID(pgid)、システム名称(system)属性を追加。
	 *
	 */
	@Override
	protected void release2() {
		super.release2();
		rdcCode		= null;
		title		= null;
		version		= null;
		build		= null;
		date		= null;
		author		= null;
		text		= null;
		pgid		= null; // 3.8.9.3
		system		= null; // 3.8.9.3
	}

	/**
	 * 【TAG】コメントの設計変更要求コードを設定します。
	 *
	 * @og.tag コメントの設計変更要求コードを設定します。
	 *
	 * @og.rev 3.3.1.0 (2003/06/24) text 属性追加、パラメータ変数を使えないように変更。
	 *
	 * @param   val 設計変更要求コード
	 */
	public void setRdcCode( final String val ) {
		rdcCode = val;
	}

	/**
	 * 【TAG】コメントのタイトルを設定します。
	 *
	 * @og.tag コメントのタイトルを設定します。
	 *
	 * @og.rev 2.2.0.0 (2002/12/17) 中国語(国際化)対応 エンコードの取得方法変更
	 * @og.rev 3.0.0.0 (2002/12/25) StringUtil#changeString 廃止
	 * @og.rev 3.3.1.0 (2003/06/24) text 属性追加、パラメータ変数を使えないように変更。
	 *
	 * @param   val タイトル
	 */
	public void setTitle( final String val ) {
		title = val;
	}

	/**
	 * 【TAG】コメントのバージョンを設定します。
	 *
	 * @og.tag コメントのバージョンを設定します
	 *
	 * @og.rev 3.3.1.0 (2003/06/24) text 属性追加、パラメータ変数を使えないように変更。
	 *
	 * @param   val バージョン
	 */
	public void setVersion( final String val ) {
		version = val;
	}

	/**
	 * 【TAG】コメントのビルド番号を設定します。
	 *
	 * @og.tag コメントのビルド番号を設定します
	 *
	 * @og.rev 3.3.1.0 (2003/06/24) text 属性追加、パラメータ変数を使えないように変更。
	 *
	 * @param   val ビルド番号
	 */
	public void setBuild( final String val ) {
		build = val;
	}

	/**
	 * 【TAG】コメントの変更日付を設定します。
	 *
	 * @og.tag コメントの変更日付を設定します
	 *
	 * @og.rev 3.3.1.0 (2003/06/24) text 属性追加、パラメータ変数を使えないように変更。
	 *
	 * @param   val 変更日付
	 */
	public void setDate( final String val ) {
		date = val;
	}

	/**
	 * 【TAG】コメントの変更者を設定します。
	 *
	 * @og.tag コメントの変更者を設定します
	 *
	 * @og.rev 3.3.1.0 (2003/06/24) text 属性追加、パラメータ変数を使えないように変更。
	 *
	 * @param   val 変更者
	 */
	public void setAuthor( final String val ) {
		author = val;
	}

	/**
	 * 【TAG】コメントの内容を設定します。
	 *
	 * @og.tag コメントの内容を設定します。
	 * ここでの内容は、BODY部に記述することも可能です。
	 * 両方に記述した場合は、text属性を優先します。
	 *
	 * @og.rev 3.3.1.0 (2003/06/24) text 属性追加、パラメータ変数を使えないように変更。
	 *
	 * @param   val コメントの内容
	 */
	public void setText( final String val ) {
		text = val;
	}

	/**
	 * 【TAG】コメントのプログラムIDを設定します。
	 *
	 * @og.tag コメントのプログラムIDを設定します。
	 *
	 * @param   val プログラムID
	 */
	public void setPgid( final String val ) {
		pgid = val;
	}

	/**
	 * 【TAG】コメントのシステム名称を設定します。
	 *
	 * @og.tag コメントのシステム名称を設定します。
	 *
	 * @param   val システム名称
	 */
	public void setSystem( final String val ) {
		system = val;
	}

	/**
	 * このオブジェクトの文字列表現を返します。
	 * 基本的にデバッグ目的に使用します。
	 *
	 * @return このクラスの文字列表現
	 */
	@Override
	public String toString() {
		return org.opengion.fukurou.util.ToString.title( this.getClass().getName() )
				.println( "VERSION"		,VERSION	)
				.println( "rdcCode"		,rdcCode	)
				.println( "title"		,title		)
				.println( "version"		,version	)
				.println( "build"		,build		)
				.println( "date"		,date		)
				.println( "author"		,author		)
				.println( "text"		,text		)
				.println( "pgid"		,pgid		)
				.println( "system"		,system		)
				.println( "Other..."	,getAttributes().getAttribute() )
				.fixForm().toString() ;
	}
}
