/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.io;

import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.hayabusa.db.DBTableModelUtil;
import org.opengion.hayabusa.io.AbstractTableReader;
import org.opengion.fukurou.util.StringUtil;

import jxl.Workbook;
import jxl.WorkbookSettings;
import jxl.Sheet;
import jxl.Cell;
import jxl.read.biff.BiffException;

import java.io.File;
import java.io.BufferedReader;
import java.io.IOException;

/**
 * JExcelによるEXCELバイナリファイルを読み取る実装クラスです。
 *
 * ファイル名、シート名を指定して、データを読み取ることが可能です。
 * 第一カラムが # で始まる行は、コメント行なので、読み飛ばします。
 * カラム名の指定行で、カラム名が null の場合は、その列は読み飛ばします。
 *
 * @og.rev 3.5.4.8 (2004/02/23) 新規作成
 * @og.group ファイル入力
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class TableReader_JExcel extends AbstractTableReader {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "4.0.0 (2005/08/31)" ;

	private String  sheetName		= null;		// 3.5.4.2 (2003/12/15)
	private String  filename		= null;		// 3.5.4.3 (2004/01/05)

	/**
	 * DBTableModel から 各形式のデータを作成して,BufferedReader より読み取ります。
	 * コメント/空行を除き、最初の行は、必ず項目名が必要です。
	 * それ以降は、コメント/空行を除き、データとして読み込んでいきます。
	 * このメソッドは、EXCEL 読み込み時に使用します。
	 *
	 * @og.rev 4.0.0 (2006/09/31) 新規追加
	 * @og.rev 5.1.6.0 (2010/05/01) columns 処理 追加
	 * @og.rev 5.1.6.0 (2010/05/01) skipRowCount , useNumber の追加
	 * @og.rev 5.2.1.0 (2010/10/01) setTableColumnValues メソッドを経由して、テーブルにデータをセットする。
	 *
	 * @see #isExcel()
	 */
	public void readDBTable() {
		Workbook wb = null;
		try {
			WorkbookSettings settings = new WorkbookSettings();
			// System.gc()「ガベージコレクション」の実行をOFFに設定
			settings.setGCDisabled(true);
			wb = Workbook.getWorkbook(new File(filename),settings);

			Sheet sheet ;

			if( sheetName == null || sheetName.length() == 0 ) {
				sheet = wb.getSheet(0);
			}
			else {
				sheet = wb.getSheet( sheetName );
				if( sheet == null ) {
					String errMsg = "対応するシートが存在しません。 Sheet=[" + sheetName + "]" ;
					throw new HybsSystemException( errMsg );
				}
			}

			boolean nameNoSet = true;
			table = DBTableModelUtil.newDBTable();

			int numberOfRows = 0;
			JxlHeaderData data = new JxlHeaderData();

			// 5.1.6.0 (2010/05/01) columns 処理
			data.setUseNumber( isUseNumber() );
			if( data.setColumns( columns ) ) {
				nameNoSet = false;
				table.init( data.getColumnSize() );
				setTableDBColumn( data.getNames() ) ;
			}

			int rowCnt = sheet.getRows();
			int skip = getSkipRowCount();		// 5.1.6.0 (2010/05/01)
//			for( int nIndexRow = 0; nIndexRow < rowCnt; nIndexRow++) {
			for( int nIndexRow = skip; nIndexRow < rowCnt; nIndexRow++) {
				Cell[] cells = sheet.getRow( nIndexRow );
				if( data.isSkip( cells ) ) { continue; }
				if( nameNoSet ) {
					nameNoSet = false;
					table.init( data.getColumnSize() );
					setTableDBColumn( data.getNames() ) ;
				}

				if( numberOfRows < getMaxRowCount() ) {
					setTableColumnValues( data.toArray( cells ) );		// 5.2.1.0 (2010/10/01)
//					table.addColumnValues( data.toArray( cells ) );
					numberOfRows ++ ;
				}
				else {
					table.setOverflow( true );
				}
			}

			// 最後まで、#NAME が見つから無かった場合
			if( nameNoSet ) {
				String errMsg = "最後まで、#NAME が見つかりませんでした。"
								+ HybsSystem.CR
								+ "ファイルが空か、もしくは損傷している可能性があります。"
								+ HybsSystem.CR ;
				throw new HybsSystemException( errMsg );
			}
		}
		catch (IOException ex) {
			String errMsg = "ファイル読込みエラー[" + filename + "]"  ;
			throw new HybsSystemException( errMsg,ex );
		}
		catch (BiffException ex) {
			String errMsg = "ファイル読込みエラー。データ形式が不正です[" + filename + "]"  ;
			throw new HybsSystemException( errMsg,ex );
		}
		finally {
			if( wb != null ) { wb.close(); }
		}
	}

	/**
	 * DBTableModel から 各形式のデータを作成して,BufferedReader より読み取ります。
	 * コメント/空行を除き、最初の行は、必ず項目名が必要です。
	 * それ以降は、コメント/空行を除き、データとして読み込んでいきます。
	 *
	 * @og.rev 3.5.4.3 (2004/01/05) 引数に、BufferedReader を受け取る要に変更します。
	 * @og.rev 4.0.0 (2006/09/31) UnsupportedOperationException を発行します。
	 *
	 * @param   reader BufferedReader (使用していません)
	 */
	public void readDBTable( final BufferedReader reader ) {
		String errMsg = "このクラスでは実装されていません。";
		throw new UnsupportedOperationException( errMsg );
	}

	/**
	 * DBTableModelのデータとして読み込むときのシート名を設定します。
	 * デフォルトは、第一シートです。
	 *
	 * @og.rev 3.5.4.2 (2003/12/15) 新規追加
	 *
	 * @param   sheetName String
	 */
	public void setSheetName( final String sheetName ) {
		this.sheetName = sheetName;
	}

	/**
	 * このクラスが、EXCEL対応機能を持っているかどうかを返します。
	 *
	 * EXCEL対応機能とは、シート名のセット、読み込み元ファイルの
	 * Fileオブジェクト取得などの、特殊機能です。
	 * 本来は、インターフェースを分けるべきと考えますが、taglib クラス等の
	 * 関係があり、問い合わせによる条件分岐で対応します。
	 *
	 * @og.rev 3.5.4.3 (2004/01/05) 新規追加
	 *
	 * @return boolean EXCEL対応機能を持っているかどうか(常にtrue)
	 */
	public boolean isExcel() {
		return true;
	}

	/**
	 * 読み取り元ファイル名をセットします。(DIR + Filename)
	 * これは、EXCEL追加機能として実装されています。
	 *
	 * @og.rev 3.5.4.3 (2004/01/05) 新規作成
	 *
	 * @param   filename 読み取り元ファイル名
	 */
	public void setFilename( final String filename ) {
		this.filename = filename;
		if( filename == null ) {
			String errMsg = "ファイル名が指定されていません。" ;
			throw new HybsSystemException( errMsg );
		}
	}
}

/**
 * EXCEL ネイティブのデータを処理する ローカルクラスです。
 * このクラスでは、コメント行のスキップ判定、ヘッダー部のカラム名取得、
 * 行情報（HSSFRow）から、カラムの配列の取得などを行います。
 *
 * @og.rev 3.5.4.8 (2004/02/23) 新規追加
 * @og.group ファイル入力
 *
 * @version  4.0
 * @author   儲
 * @since    JDK5.0,
 */
class JxlHeaderData {
	private String[] names ;
	private int[]    index;
	private int		 columnSize = 0;
	private boolean  nameNoSet = true;
	private boolean  useNumber = true;

	/**
	 * 行番号情報を、使用している(true)/していない(false)を指定します。
	 *
	 * デフォルトは、true（使用する） です。
	 *
	 * @og.rev 5.1.6.0 (2010/05/01) 新規作成
	 *
	 * @param useNumber boolean 行番号情報を、使用している(true)/していない(false)を指定
	 */
	void setUseNumber( final boolean useNumber ) {
		this.useNumber = useNumber ;
	}

	/**
	 * カラム名を外部から指定します。
	 * カラム名が、NULL でなければ、＃NAME より、こちらが優先されます。
	 * カラム名は、順番に、指定する必要があります。
	 *
	 * @og.rev 5.1.6.0 (2010/05/01) 新規作成
	 *
	 * @param columns String EXCELのカラム列(CSV形式)
	 * @return true:処理実施/false:無処理
	 */
	boolean setColumns( final String columns ) {
		if( columns != null && columns.length() > 0 ) {
			names = StringUtil.csv2Array( columns );
			columnSize = names.length ;
			index = new int[columnSize];
			int adrs = (useNumber) ? 1:0 ;	// useNumber =true の場合は、１件目(No)は読み飛ばす。
			for( int i=0; i<columnSize; i++ ) { index[i] = adrs++; }
			nameNoSet = false;

			return true;
		}
		return false;
	}

	/**
	 * EXCEL ネイティブのデータを処理する ローカルクラスです。
	 * このクラスでは、コメント行のスキップ判定、ヘッダー部のカラム名取得、
	 * 行情報（HSSFRow）から、カラムの配列の取得などを行います。
	 *
	 * @param cells Cell[] EXCELのセル配列(行)
	 * @return true:コメント行/false:通常行
	 */
	boolean isSkip( final Cell[] cells ) {
		int size = cells.length ;
		if( size == 0 ) { return true; }

		String strText =  cells[0].getContents();
		if( strText != null && strText.length() > 0 ) {
			if( nameNoSet ) {
				if( strText.equalsIgnoreCase( "#Name" ) ) {
					makeNames( cells );
					nameNoSet = false;
					return true;
				}
				else if( strText.charAt( 0 ) == '#' ) {
					return true;
				}
				else {
					String errMsg = "#NAME が見つかる前にデータが見つかりました。"
									+ HybsSystem.CR
									+ "可能性として、ファイルが、ネイティブExcelでない事が考えられます。"
									+ HybsSystem.CR ;
					throw new HybsSystemException( errMsg );
				}
			}
			else {
				if( strText.charAt( 0 ) == '#' ) {
					return true;
				}
			}
		}

		return nameNoSet ;
	}

	/**
	 * EXCEL ネイティブの行のセル配列からカラム名情報を取得します。
	 *
	 * @og.rev 5.1.6.0 (2010/05/01) useNumber(行番号情報を、使用している(true)/していない(false)を指定)
	 *
	 * @param cells Cell[] EXCELの行のセル配列
	 */
	private void makeNames( final Cell[] cells ) {
		int maxCnt = cells.length;
		String[] names2 = new String[maxCnt];
		int[]    index2 = new int[maxCnt];

		// 先頭カラムは、#NAME 属性行である。
//		for( int nIndexCell = 1; nIndexCell < maxCnt; nIndexCell++) {
		// 先頭カラムは、#NAME 属性行であるかどうかを、useNumber で判定しておく。
		int nFirstCell = (useNumber) ? 1:0 ;
		for( int nIndexCell = nFirstCell; nIndexCell < maxCnt; nIndexCell++) {
			String strText = cells[nIndexCell].getContents();

			if( strText != null && strText.length() > 0 ) {
				names2[columnSize] = strText;
				index2[columnSize] = nIndexCell;
				columnSize++;
			}
		}

		// #NAME を使用しない場合：no欄が存在しないケース
		if( maxCnt == columnSize ) {
			names = names2;
			index = index2;
		}
		else {
			names = new String[columnSize];
			index = new int[columnSize];
			System.arraycopy(names2, 0, names, 0, columnSize);
			System.arraycopy(index2, 0, index, 0, columnSize);
		}
	}

	/**
	 * カラム名情報を返します。
	 * ここでは、内部配列をそのまま返します。
	 *
	 * @return String[] カラム列配列情報
	 */
	String[] getNames() {
		return names;
	}

	/**
	 * カラムサイズを返します。
	 *
	 * @return int カラムサイズ
	 */
	int getColumnSize() {
		return columnSize;
	}

	/**
	 * カラム名情報を返します。
	 *
	 * @param cells Cell[] EXCELの行のセル配列
	 * @return String[] カラム列配列情報
	 */
	String[] toArray( final Cell[] cells ) {
		if( nameNoSet ) {
			String errMsg = "#NAME が見つかる前にデータが見つかりました。";
			throw new HybsSystemException( errMsg );
		}

		int cellSize = cells.length;
		String[] data = new String[columnSize];
		for( int i=0;i<columnSize; i++ ) {
			int indx = index[i];
			if( indx < cellSize ) {
				data[i] = cells[indx].getContents();
			}
			else {
				data[i] = null;
			}
		}

		return data;
	}
}
