/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.fukurou.util.XHTMLTag;

/**
 * 指定の位置に画像を配置するHTML拡張タグです。
 *
 * @og.formSample
 * ●形式：&lt;og:img src="･･･" alt="･･･"; /&gt;
 * ●body：なし
 *
 * ●使用例
 *    &lt;og:img src="{&#064;URL}" alt="sample" /&gt;
 *
 *        src  : 表示させる画像のURL（必須）
 *        alt  : 画像が表示されない場合、代わりに表示させるテキスト（必須）
 *
 * @og.group 画面部品
 *
 * @version  4.0
 * @author   Kohei Naruse
 * @since    JDK5.0,
 */
public class ImageTag extends HTMLTagSupport {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "4.0.0 (2005/08/31)" ;

	private static final long serialVersionUID = 4000 ;	// 4.0.0 (2005/01/31)

	/**
	 * 指定位置に画像を配置するためのイメージタグを作成します。
	 *
	 * @return  イメージタグ
	 */
	protected String makeTag() {
		StringBuilder rtn = new StringBuilder( HybsSystem.BUFFER_MIDDLE );

		rtn.append(XHTMLTag.img(getAttributes()));

		return rtn.toString();
	}

	/**
	 * 【HTML】名前を指定します。
	 *
	 * @og.tag 名前を指定します。
	 *
	 * @param   name String
	 */
	public void setName( final String name ) {
		set( "name",getRequestParameter( name ) );
	}

	/**
	 * 【HTML】表示させる画像のURLを指定します。
	 *
	 * @og.tag 表示させる画像のURLを指定します。
	 *
	 * @param   src String
	 */
	public void setSrc( final String src ) {
		set( "src",getRequestParameter( src ) );
	}

	/**
	 * 【HTML】画像が表示されない場合、代わりに表示させるテキストを指定します。
	 *
	 * @og.tag 画像が表示されない場合、代わりに表示させるテキストを指定します。
	 *
	 * @param   alt String
	 */
	public void setAlt( final String alt ) {
		set( "alt",getRequestParameter( alt ) );
	}

	/**
	 * 【HTML】画像部分についての詳しい情報があるURL(longdesc)を指定します。
	 *
	 * @og.tag longdescを指定します。
	 *
	 * @param   longdesc String
	 */
	public void setLongdesc( final String longdesc ) {
		set( "longdesc",getRequestParameter( longdesc ) );
	}

	/**
	 * 【HTML】横幅をピクセル数または％で指定します。
	 *
	 * @og.tag 横幅を指定します。
	 *
	 * @param   width String
	 */
	public void setWidth( final String width ) {
		set( "width",getRequestParameter( width ) );
	}

	/**
	 * 【HTML】縦幅をピクセル数または％で指定します。
	 *
	 * @og.tag 縦幅を指定します。
	 *
	 * @param   height String
	 */
	public void setHeight( final String height ) {
		set( "height",getRequestParameter( height ) );
	}

	/**
	 * 【HTML】クライアントサイドクリッカブルを指定します。
	 *
	 * @og.tag クライアントサイドクリッカブルを指定します。
	 *
	 * @param   usemap String
	 */
	public void setUsemap( final String usemap ) {
		set( "usemap",getRequestParameter( usemap ) );
	}

	/**
	 * 【HTML】サーバーサイドクリッカブルを指定します。
	 *
	 * @og.tag サーバーサイドクリッカブルを指定します。
	 *
	 * @param   ismap String
	 */
	public void setIsmap( final String ismap ) {
		set( "ismap",getRequestParameter( ismap ) );
	}

	/**
	 * タグの名称を、返します。
	 * 自分自身のクラス名より、自動的に取り出せないため、このメソッドをオーバーライドします。
	 *
	 * @og.rev 4.0.0 (2005/01/31) 新規追加
	 *
	 * @return  タグの名称
	 */
	protected String getTagName() {
		return "img" ;
	}
}
