/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.fukurou.process;

import org.opengion.fukurou.model.DataModel;
import org.opengion.fukurou.model.NativeType;
import org.opengion.fukurou.util.StringUtil;

import java.util.Arrays;

/**
 * LineModel は、データの１行分を管理する為の TableModel と類似の実装クラスです。
 *
 * データの１行分を LineModel に割り当てます。
 * カラム番号は、0 から始まります。カラム名よりカラム番号を求める場合に、
 * 存在しない場合は、-1 を返します。
 * カラム番号が -1 の場合は、処理を行いません。
 *
 * 注意：このクラスは、同期処理されていません。
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class LineModel implements DataModel<Object> {		// 4.3.3.6 (2008/11/15) Generics警告対応
	/** タブセパレータ  */
	private static final String TAB = "\t";   // タブ区切り文字

	/** リターンコード  System.getProperty("line.separator")  */
	private static final String CR = System.getProperty("line.separator");

	private	String[]	names	= null;
	private	Object[]	data	= null;
	private	int			dtSize	= 0;
	private	int			rowNo	= -1;

	/**
	 * このオブジェクトを初期化します。
	 * 指定の引数分の内部配列を作成します。
	 *
	 * @param   columnCount int
	 */
	public void init( final int columnCount ) {
		if( columnCount <= 0 ) {
			String errMsg = "内部配列の数量が 0か、負です。count=[" + columnCount + "]";
			throw new RuntimeException( errMsg );
		}
		dtSize	= columnCount;
		names	= new String[dtSize];
		data	= new Object[dtSize];
	}

	/**
	 * カラム名配列を指定して、このオブジェクトを初期化します。
	 *
	 * @param   clmNms String[]
	 */
	public void init( final String[] clmNms ) {
		if( clmNms == null ) {
			String errMsg = "カラム名配列が、 null です。";
			throw new RuntimeException( errMsg );
		}
		dtSize	= clmNms.length;
		names	= new String[dtSize];
		System.arraycopy( clmNms,0,names,0,dtSize );
		data	= new Object[dtSize];
	}

	/**
	 * 名前をセットします。
	 * 指定のカラム番号が、-1 の場合は、なにもしません。
	 *
	 * @param   clm	 値が変更されるカラム番号
	 * @param   key	 新しい名前
	 *
	 */
	public void setName( final int clm,final String key ) {
		if( clm >= 0 ) { names[clm] = key; }
	}

	/**
	 * カラム番号に対するカラム名を取得します。
	 * 指定のカラム番号が、-1 の場合は、null を返します。
	 *
	 * @param   clm  最初のカラム番号は 0、2番目のカラム番号は 1、などとする。
	 * @return  カラム名
	 *
	 */
	public String getName( final int clm ) {
		return ( clm >= 0 ) ? names[clm] : null ;
	}

	/**
	 * カラム名配列を返します。
	 * 配列オブジェクトは、clone されたコピーを返しますので、
	 * 内容を書き換えたい場合は、setName() メソッドを使用してください。
	 *
	 * @return   nm String[] カラム名配列
	 */
	public String[] getNames() {
		return names.clone();
	}

	/**
	 * column に対応した 値を登録します。
	 * 指定のカラム番号が、-1 の場合は、なにもしません。
	 *
	 * @param   clm	 値が変更されるカラム番号
	 * @param   value   新しい値。null も可
	 *
	 */
	public void setValue( final int clm,final Object value ) {
		if( clm >= 0 ) { data[clm] = value; }
	}

	/**
	 * column にあるセルの属性値をObjectに変換して返します。
	 * 指定のカラム番号が、-1 の場合は、null を返します。
	 *
	 * @param   clm  値が参照されるカラム番号
	 * @return  指定されたセルの値 Object
	 *
	 */
	public Object getValue( final int clm ) {
		return ( clm >= 0 ) ? data[clm] : null ;
	}

	/**
	 * row および clm にあるセルの属性値をStringに変換して返します。
	 *
	 * @param   row     値が参照される行
	 * @param   clm     値が参照される列
	 * @return  指定されたセルの値 String
	 *
	 */
	public String getValue( final int row, final int clm) {
		String errMsg = "このクラスでは実装されていません。";
		throw new UnsupportedOperationException( errMsg );
	}

	/**
	 * 属性値配列をセットします。
	 * このメソッドでは、カラム名配列は更新しません。配列数が異なる場合や、
	 * 属性値配列が null の場合は設定できません。
	 * 設定は、配列オブジェクトのコピーをセットしますので、登録元の配列オブジェクトを
	 * 書き換えた場合でも、このオブジェクトの内部値は変更されませんので、
	 * 副作用を期待したコーディングは、行わないで下さい。
	 * 注意：値オブジェクト配列自身はコピーしますが、個々のオブジェクトそのものは、
	 * arraycopy しているだけです。個々のオブジェクトの状態変化に対しては、
	 * 各クライアント側で対応が必要です。
	 *
	 * @param   values	String[]	セットする値配列
	 * @param   rno		int			処理中の行番号
	 */
	public void setValues( final Object[] values, final int rno ) {
		if( values == null ) {
			String errMsg = "値配列が、 null です。row=[" + rno + "]";
			throw new RuntimeException( errMsg );
		}
		else if( names.length != values.length ) {
			String errMsg = "カラム名配列と異なる要素の属性値配列は登録できません。" + CR
						+ " names.length=[" + names.length + "],"
						+ " values.length[" + values.length + "],"
						+ " row=[" + rno + "]" 	+ CR
						+ " names=" + StringUtil.array2csv( names )
						+ " values=" + Arrays.toString( values ) ;		// 5.1.8.0 (2010/07/01) errMsg 修正
			throw new RuntimeException( errMsg );
		}

		rowNo = rno;
		System.arraycopy( values,0,data,0,values.length );
	}

	/**
	 * 属性値を配列で返します。
	 * 配列オブジェクトは、clone されたコピーを返しますので、
	 * 内容を書き換えたい場合は、setValue() メソッドを使用してください。
	 *
	 * @return  指定されたセルの属性値 Object[]
	 */
	public Object[] getValues() {
		return data.clone();
	}

	/**
	 * row にあるセルの属性値を配列で返します。
	 *
	 * @param   row     値が参照される行
	 * @return  指定されたセルの属性値 String[]
	 */
	public String[] getValues( final int row ) {
		String errMsg = "このクラスでは実装されていません。";
		throw new UnsupportedOperationException( errMsg );
	}

	/**
	 * 行データモデル属性に、処理中の行番号情報を設定します。
	 * 一般に、setValue( int clm, Object value ) との併用時に使用します。
	 *
	 * @param   rno	int	処理中の行番号
	 */
	public void setRowNo( final int rno ) {
		rowNo = rno;
	}

	/**
	 * 行データモデル属性の、処理中の行番号情報を返します。
	 *
	 * @return  処理中の行番号
	 *
	 */
	public int getRowNo() {
		return rowNo;
	}

	/**
	 * columnName 名称に対応する カラム番号を返します。存在しない場合は、-1 を返します。
	 *
	 * @param   columnName  値が参照されるカラム番号
	 * @return  指定されたセルのカラム番号（存在しない場合は、-1）
	 *
	 */
	public int getColumnNo( final String columnName ) {
		int clmNo = -1;
		if( columnName != null ) {
			for( int i=0; i<dtSize; i++ ) {
				if( columnName.equalsIgnoreCase( names[i] ) ) {
					clmNo = i;
					break;
				}
			}
		}

		return clmNo;
	}

	/**
	 * データテーブル内の列の数を返します。
	 *
	 * @return  モデルの列数
	 *
	 */
	public int size() {
		return dtSize ;
	}

	/**
	 * この内部データの名前配列を連結した文字列で、出力します。
	 *
	 * @return  String 内部データの名前配列を連結した文字列
	 */
	public String nameLine() {
		String rtn = "No Data";

		if( dtSize > 0 ) {
			StringBuilder buf = new StringBuilder();
			buf.append( "#Name" );
			for( int clm=0; clm<dtSize; clm++ ) {
				buf.append( TAB );
				buf.append( names[clm] );
			}
			rtn = buf.toString() ;
		}
		return rtn ;
	}

	/**
	 * この内部データの名前配列を連結した文字列で、出力します。
	 *
	 * @return  String 名前配列を連結した文字列
	 */
	public String dataLine() {
		String rtn = "No Data";

		if( dtSize > 0 ) {
			StringBuilder buf = new StringBuilder();
			buf.append( rowNo );
			for( int clm=0; clm<dtSize; clm++ ) {
				buf.append( TAB );
				buf.append( data[clm] );
			}
			rtn = buf.toString() ;
		}
		return rtn ;
	}

	/**
	 * この内部データを見える形で出力します。
	 *
	 * @return  String 内部データを見える形
	 */
	public String toString() {
		return nameLine() + CR + dataLine() ;
	}
	
	/**
	 * clm のNativeタイプを返します。
	 * Nativeタイプはorg.opengion.fukurou.model.NativeTypeで定義されています。
	 * 
	 * @og.rev 4.1.1.2 (2008/02/28) 新規追加
	 * @og.rev 5.1.8.0 (2010/07/01) NativeType#getType(String) のメソッドを使用するように変更。
	 * 
	 * @param  clm      値が参照される列
	 * @return Nativeタイプ
	 * @see org.opengion.fukurou.model.NativeType
	 */
	public NativeType getNativeType( final int clm ) {
//		return StringUtil.getNativeType( data[clm].toString() );
		return NativeType.getType( data[clm].toString() );
	}
}
