/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.table;

import org.opengion.hayabusa.db.AbstractTableFilter;
import org.opengion.hayabusa.db.DBTableModel;
import org.opengion.hayabusa.db.DBColumn;

import org.opengion.hayabusa.resource.ResourceFactory;
import org.opengion.hayabusa.resource.ResourceManager;
import org.opengion.hayabusa.resource.CodeData;

import org.opengion.fukurou.util.ErrorMessage;
import org.opengion.fukurou.util.StringUtil;

/**
 * <p>TableFilter_CLMSET は、TableFilter インターフェースを継承した、DBTableModel 処理用の
 * 実装クラスです。<br />
 *
 * ここでは、CLM,SYSTEM_ID,LANG より、カラムリソースのRENDERER,EDITOR,DBTYPE,BIKOを設定します。
 * 検索した DBTableModel の属性として、RENDERER,EDITOR,DBTYPE,BIKO という名称の
 * カラムが必要です。
 * 引数として指定可能なのは、SYSTEM_ID,LANG のみです。
 *  CLM :カラムリソースのキーとなる値が設定されているカラム名を指定します。
 *  SYSTEM_ID:コードリソースの作成システムIDを指定します。無指定時は、ログイン時のリソースになります。
 *  LANG:ラベルリソースの言語を指定します。無指定時は、日本語になります。
 *
 * また、CLM,RENDERER,EDITOR,DBTYPE,BIKO で指定したカラムが DBTableModel に存在しない場合は、
 * 処理そのものを無視します。その場合は、警告も出力されませんので、ご注意ください。
 *
 * @og.rev 4.1.0.0(2008/01/18) 新規作成
 *
 * @version  0.9.0  2000/10/17
 * @author   Kazuhiko Hasegawa
 * @since    JDK1.1,
 */
public class TableFilter_CLMSET extends AbstractTableFilter {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "4.0.0 (2005/08/31)" ;

	/**
	 * DBTableModel処理を実行します。<br />
	 *
	 * @return DBTableModel
	 */
	public DBTableModel execute() {
		String systemId = getValue( "SYSTEM_ID" );
		String lang		= getValue( "LANG" );
		ResourceManager resource = ResourceFactory.newInstance( systemId,lang,false );

		int clmNo  = table.getColumnNo( "CLM",false );		// 存在しない場合は、-1 を返す。
		int renNo  = table.getColumnNo( "RENDERER",false );
		int ediNo  = table.getColumnNo( "EDITOR",false );
		int typNo  = table.getColumnNo( "DBTYPE",false );
		int bikoNo = table.getColumnNo( "BIKO",false );

		if( clmNo >= 0 && renNo >= 0 && ediNo >= 0 && typNo >= 0 && bikoNo >= 0 ) {
			String[] data  = null;
			int rowCnt = table.getRowCount();
			DBColumn column ;
			CodeData code ;
			for( int row=0; row<rowCnt; row++ ) {
				String clmVal = null;
				try {
					data   = table.getValues( row );
					clmVal = data[clmNo];		// 変換する元のカラム名
					column = resource.getDBColumn( clmVal );
					// 以下の設定は、副作用を及ぼしています。注意
					if( column != null ) {
						data[renNo] = column.getRenderer() ;
						data[ediNo] = column.getEditor() ;
						data[typNo] = column.getDbType() ;

						// 本当は、RENDERER='MENU' の場合のみ有効。
						code = resource.getCodeData( clmVal );
						if( code != null ) {
							data[bikoNo] = code.toCodeString() ;
						}
						// オブジェクトが存在しない場合は、クリアします。
						else {
							data[bikoNo] = "";
						}
					}
					// オブジェクトが存在しない場合は、クリアします。
					else {
						data[renNo]  = "";
						data[ediNo]  = "";
						data[typNo]  = "";
						data[bikoNo] = "";
					}
				}
				catch( RuntimeException ex ) {
					ErrorMessage errMessage = makeErrorMessage( "TableFilter_CLMSET Error",ErrorMessage.NG );
					errMessage.addMessage( row+1,ErrorMessage.NG,ex.getMessage() );
					errMessage.addMessage( row+1,ErrorMessage.NG,StringUtil.array2csv( data ) );
					errMessage.addMessage( row+1,ErrorMessage.NG,"CLM=[" + clmVal + "]" );
				}
			}
		}

		return table;
	}
}
