/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.column;

import org.opengion.fukurou.util.StringUtil;
import org.opengion.hayabusa.db.AbstractDBType;

/**
 * Unicode文字列の値を HTML のエスケープ記号(&amp;#xZZZZ;)に変換する、カラム属性を定義します。
 *
 * SJIS(JA16SJIS) で作成されたデータベースに、(NVARCHAR2)を使用して中国語等を登録するのは
 * 非常に複雑でかつ、リスクが大きい処理になります。
 * ORACLE殿でも、自信を持っては勧められない機能とのコメントを頂いています。
 * そこで、HTMLでのエスケープ文字を使用して、Unicodeを文字列化して登録する為の
 * DBType として、新規に作成します。
 * ここでは、入力文字を、キャラクタ(char)型に分解し、(&amp;#xZZZZ;)に変換していきます。
 * よって、通常に１文字(Shift-JISで２Byte,UTF-8で３Byte)が、８Byteになります。
 * この変換された文字列を、HTML上でそのまま取り出すと、元のUnicode文字に戻る為、
 * 通常のShift-JISでは、扱えない文字(中国語など)でも表示可能になります。
 *
 * (NVARCHAR2)を使用して中国語等を登録する場合の文字コード変換
 *
 * APレイヤ     ネットレイヤ                        DBレイヤ
 * 文字入力  → 全てDBのCHARセットに変換  →  NCHARの分だけUTFに変換
 * （UTF）      （S-jis）                         （UTF）
 *
 * タイプチェックは、HTMLのエスケープ記号(&amp;#xZZZZ;)に変換後に実施されます。
 * データベースへの登録には、半角英数字になります。
 * ・文字列長は、Byte換算での文字数との比較
 * ・半角文字列チェック「 c &lt; 0x20 || c &gt; 0x7e 以外」エラー
 * ・文字パラメータの 正規表現チェック
 * ・クロスサイトスクリプティングチェック
 *
 * @og.rev 3.8.0.0 (2005/06/07) 新規作成
 * @og.group データ属性
 *
 * @version  4.0
 * @author	 Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class DBType_NVAR extends AbstractDBType {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "4.0.0 (2005/08/31)" ;

	/**
	 * 半角０文字の固定長でFILL埋めされた文字列を返します。
	 * なお、エラーチェックは行われません。
	 * 実行前に、必ず valueCheck( String value ,int len ) が行われる必要があります。
	 *
	 * @param	value	 ＦＩＬＬ埋めする文字列
	 * @param	sizeX	 整数部分の文字列の長さ
	 * @param	sizeY	 少数部分の文字列の長さ
	 * @param	encode	 エンコード指定
	 * @return	ＦＩＬＬ埋めした新しい文字列
	 */
	public String valueFill( final String value ,final int sizeX ,final int sizeY,final String encode ) {
		return valueSet( super.valueFill( value,sizeX,sizeY,encode ) );
	}

	/**
	 * String引数の文字列を＋１した文字列を返します。
	 * これは、英字の場合（A,B,C など）は、B,C,D のように,最終桁の文字コードを
	 * ＋１ します。
	 * 文字列が数字タイプの場合は, 数字に変換して、+1 します。（桁上がりもあり）
	 * 混在タイプの場合は,最後の桁だけを確認して ＋１します。
	 * 引数が null の場合と、ゼロ文字列（""）の場合は,物理的初期設定値(String getDefault())
	 * の値を返します。
	 *
	 * @param	value  String
	 * @return	String引数の文字列を＋１した文字列
	 */
	public String valueAdd( final String value ) {
		return valueSet( super.valueAdd( value ) );
	}

	/**
	 * エディターで編集されたデータを登録する場合に、データそのものを
	 * 変換して、実登録データを作成します。
	 * 例えば,大文字のみのフィールドなら、大文字化します。
	 * 実登録データの作成は、DBType オブジェクトを利用しますので,
	 * これと Editor とがアンマッチの場合は、うまくデータ変換
	 * されない可能性がありますので、注意願います。
	 *
	 * @param	value String （一般に編集データとして登録されたデータ）
	 * @return	修正後の文字列（一般にデータベースに登録するデータ）
	 */
	public String valueSet( final String value ) {
		return StringUtil.getUnicodeEscape( value );
	}
}
