/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.mail;

import java.util.Map;
import java.util.TreeMap;

import org.opengion.hayabusa.db.DBColumn;
import org.opengion.hayabusa.db.DBTableModelUtil;
import org.opengion.hayabusa.resource.ResourceManager;
import org.opengion.hayabusa.db.DBTableModel;

public class MailManager_DIRECT extends AbstractMailManager {

/**
 * タグ mailSender2 による送信を行う際に利用するメール送信マネージャの処理クラスです。
 * タグ mailSender2 よりパラメータマップを受取って、メール文の合成、送信を行います。
 * バッチ送信する場合と共通する部分はスーパークラス AbstractMailManager に実装していますが、
 * タグ独自ロジックの部分は本クラスより実装を行っています。
 * 独自ロジックはセッションから取得した宛先テーブルにより宛先マップを作成、セッションから取得したメール
 * 文により送信を行うロジックとあります。
 * 
 * @og.group メールモジュール
 *
 * @version  4.0
 * @author   Sen.Li
 * @since    JDK1.6
 */
	private final static String[] names	= { "DST_ID", "GROUP_ID", "GROUP_NAME", "DST_NAME", "DST_ADDR", "DST_KBN", "FGJ_MAIL" };
	private final static int IDX_DST_ID 	= 0;
	private final static int IDX_FGJ_MAIL 	= 6;
	private DBColumn[]	dbColumn	= null;
	private ResourceManager resource = null;

	/**
	 * action="SEND"の時にこのメソッドが呼ばれます。
	 * セッションから取得した宛先テーブルにより宛先マップを作成します。
	 * まだ、action="CHECK"の時に、確認画面から添付ファイルを追加するケースがあるため、
	 * パラメータを再読込を行います。そして、action="SEND"の時に添付ファイルを送信します。
	 * 
	 * @param params Map<String, String>	パラメータのマップ
	 * @param table DBTableModel
	 * 
	 */
	public void create( final Map<String, String> params, final DBTableModel table ){
		setInitParams( params );
		setAttachFiles(   params.get( "ATTACH1" )
		        		, params.get( "ATTACH2" )
		                , params.get( "ATTACH3" )
		                , params.get( "ATTACH4" )
		                , params.get( "ATTACH5" ));
		Map <String, String[]> dstMap = makeMailDstMap( table );
		setMailDstMap( dstMap );
	}

	/**
	 * 画面に各宛先の送信状況を表示するために、送信の宛先マップに基づいてテーブルモデルを作成します。
	 * 作成されたテーブルモデルを指定されるスコープに入れます。
	 * 
	 * @og.rev 5.1.9.0 (2010/08/01) keySet() → entrySet() に変更
	 * 
	 * @return table DBTableModel
	 */
	public DBTableModel makeDstTable(){
		Map<String, String[]> mailDst = getMailDstMap();
		DBTableModel table;
		int numberOfColumns = names.length;
		
		table = DBTableModelUtil.newDBTable();
		table.init( numberOfColumns );
		setTableDBColumn( table, names );

//		for( String dstId : mailDst.keySet() ) {
//			String[] dstInfo = mailDst.get( dstId );
//			table.addColumnValues( dstInfo );
//		}

		// 5.1.9.0 (2010/08/01) keySet() → entrySet() に変更
		for( Map.Entry<String, String[]> en : mailDst.entrySet() ) {
			table.addColumnValues( en.getValue() );
		}

		return table;
	}

	/**
	 * リソースマネージャーをセットします。
	 * これは、言語（ロケール）に応じた DBColumn をあらかじめ設定しておく為に
	 * 必要です。
	 * リソースマネージャーが設定されていない、または、所定のキーの DBColumn が
	 * リソースに存在しない場合は、内部で DBColumn オブジェクトを作成します。
	 *
	 * @param  res リソースマネージャー
	 */
	// 注意：この resource は、実質利用されていません。
	public void setResourceManager( final ResourceManager res ) {
		resource = res;
	}

	/**
	 * DBColumn オブジェクトをテーブルモデルに設定します。
	 *
	 * @param table DBTableModel
	 * @param names String[]
	 */
	// 注意：この dbColumn は、実質利用されていません。
	protected void setTableDBColumn( final DBTableModel table, final String[] names ) {
		dbColumn = new DBColumn[names.length] ;
		for( int i=0; i<names.length; i++ ) {
			DBColumn clm = resource.makeDBColumn( names[i] );
			table.setDBColumn( i,clm );
			dbColumn[i] = clm;
		}
	}

	/**
	 * セッションから取得した宛先テーブルにより宛先マップを作成します。
	 * 宛先テーブルの各レコードに対して、"送信待ち"となっているレコードのみ宛先マップに入れるようにしています。
	 * 
	 * @param  table  DBTableModel
	 * @return dstMap Map<String, String[]>
	 */
	private Map<String, String[]> makeMailDstMap( final DBTableModel table ){
		Map<String, String[]> dstMap = new TreeMap<String, String[]>();
		int rowCount = table.getRowCount();
		int colCount = names.length;
		for( int i = 0; i < rowCount; i++ ) {
			String[] tmpDst = new String[colCount];
			if( AbstractMailManager.FGJ_SEND_WAIT.equals( table.getValue( i, IDX_FGJ_MAIL ) ) ) {
				for( int j = 0; j < colCount; j++ ) {
					tmpDst[j] = table.getValue( i, j );
				}
				dstMap.put( tmpDst[IDX_DST_ID], tmpDst );
			}
		}

		return dstMap;
	}
}
