/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.fukurou.taglet;

import com.sun.javadoc.Tag;

import org.opengion.fukurou.util.FileUtil;

import java.io.File;
import java.io.PrintWriter;
import java.io.IOException;

/**
 * Tag 情報を出力する PrintWriter 相当クラスです。
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public final class DocletTagWriter {
	private final PrintWriter outFile ;
	private final boolean rtn2br ;

	private static final String ENCODE = "UTF-8" ;

	/** リターンコード  System.getProperty("line.separator")  */
	public static final String CR = System.getProperty("line.separator");
	/** HTML上のブレーク  &lt;br&gt; + CR  */
	public static final String BR = "&lt;br&gt;" + CR ;

	/**
	 * Doclet のエントリポイントメソッドです。
	 *
	 * @param file String
	 */
	public DocletTagWriter( final String file ) throws IOException {
		this( file,ENCODE,false );
	}

	/**
	 * Doclet のエントリポイントメソッドです。
	 *
	 * @param file String
	 * @param encode String
	 */
	public DocletTagWriter( final String file,final String encode ) throws IOException {
		this( file,encode,false );
	}

	/**
	 * Doclet のエントリポイントメソッドです。
	 *
	 * @param file String
	 * @param encode String
	 * @param r2b boolean
	 */
	public DocletTagWriter( final String file,final String encode,final boolean r2b ) throws IOException {
		outFile = FileUtil.getPrintWriter( new File( file ),encode );
		rtn2br = r2b;
	}

	/**
	 * 出力ファイルをクロースします。
	 *
	 */
	public void close() {
		outFile.close();
	}

	/**
	 * 可変長の文字列引数を取り、文字列を出力します。
	 * 文字列の最後に改行が入ります。
	 *
	 * @param str String...
	 */
	public void printTag( final String... str ) {
		for( int i=0; i<str.length; i++ ) {
			if( rtn2br ) { outFile.print( str[i].replaceAll( CR,BR ) ); }
			else		 { outFile.print( str[i] ); }
		}
		outFile.println();
	}

	/**
	 * タグ配列を受け取り、タグ出力します。
	 *
	 * @param tag Tag[]
	 */
	public void printTag( final Tag[] tag ) {
		if( tag.length == 1 ) {
			String data = DocletUtil.htmlFilter( tag[0].text() );
			if( rtn2br ) { outFile.print( data.replaceAll( CR,BR ) ); }
			else		 { outFile.print( data ); }
		}
		else {
			for( int i=0; i<tag.length; i++ ) {
				String data = DocletUtil.htmlFilter( tag[i].text() );
				if( rtn2br ) {
					outFile.print( data.replaceAll( CR,BR ) );
					outFile.print( BR );
				}
				else {
					outFile.println( data );
				}
			}
		}
	}

	/**
	 * タグ配列を受け取り、タグ出力します。
	 * 複数のタグを出力する場合に、カンマ区切り文字で連結します。
	 *
	 * @param tag Tag[]
	 */
	public void printCSVTag( final Tag[] tag ) {
		for( int i=0; i<tag.length; i++ ) {
			String data = DocletUtil.htmlFilter( tag[i].text() );
			if( i > 0 ) { outFile.print( "," ); }
			outFile.print( data );
		}
	}

	/**
	 * タグ配列を受け取り、タグ出力します。
	 * ここでは、タグ毎にタグの名称と内容を出力し、改行を行います。
	 * 特殊処理：ここでは、og.rev タグは取り込みません。
	 *
	 * @param tag Tag[]
	 */
	public void printTagsInfo( final Tag[] tag ) {
		for( int i=0; i<tag.length; i++ ) {
			String tagName = tag[i].name();
			if( tagName.equalsIgnoreCase( "@og.rev" ) ) {
				continue;
			}
			outFile.print( tagName );
			outFile.print( " " );
			outFile.print( DocletUtil.htmlFilter( tag[i].text() ) );
			if( rtn2br ) { outFile.print( BR ); }
			else		 { outFile.println(); }
		}
	}

	/**
	 * 文字列引数を ２つと、タグ配列を受け取り、タグ出力します。
	 *
	 * @param str1 String
	 * @param tag Tag[]
	 * @param str3 String
	 */
	public void printTag( final String str1,final Tag[] tag, final String str3 ) {
		outFile.print( str1 );
		printTag( tag );
		outFile.println( str3 );
	}

	/**
	 * 文字列引数を ２つと、タグ配列を受け取り、先頭一文字のタグ出力します。
	 *
	 * @param str1 String
	 * @param tag Tag[]
	 * @param str3 String
	 */
	public void printChar( final String str1,final Tag[] tag, final String str3 ) {
		outFile.print( str1 );
		if( tag.length > 0 ) {
			String str = tag[0].text();
			if( str != null && str.length() > 0 ) {
				outFile.print( str.charAt(0) );
			}
		}
		outFile.println( str3 );
	}
}
