/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.fukurou.db;

import org.opengion.fukurou.util.Closer;
import org.opengion.fukurou.util.ApplicationInfo;

import java.sql.Connection;

import java.util.Locale;
import java.util.Map;
import java.util.HashMap;
// import java.util.Map.Entry;

/**
 * コネクションを共有して、トランザクションを実現します。
 *
 * 基本的には、TransactionTag で利用されますが、一部、このオブジェクトを
 * 渡して、直接、利用するケースもあります。
 *
 * トランザクションがすべて完了した後で、realClose() メソッドを呼び出します。
 * 一度でも、rollback が指定されていれば、ロールバックを行い、コネクションを
 * 破棄します。それ以外で、commit が指定されていれば、コミットを行い、
 * コネクションを、プールに戻します。どちらも指定されていなければ、
 * コネクションプールに戻すだけになります。
 *
 * 考え方として、下記のような流れになります。
 * <pre>
 *   TransactionImpl tran = new TransactionImpl( appInfo ) ;
 *   try {
 *      ・・・・・
 *      tran.commit();
 *      tran.finish();
 *   }
 *   catch( Exception ex ) {
 *      tran.rollback();
 *   }
 *   finally {
 *      tran.realClose()
 *   }
 * </pre>
 *
 * @og.rev 5.1.9.0 (2010/08/01) 新規作成
 *
 * @version  5.0
 * @author	 Kazuhiko Hasegawa
 * @since    JDK6.0,
 */
public class TransactionImpl implements Transaction {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "5.1.9.0 (2010/08/01)" ;
	private static final long serialVersionUID = 5190 ;	// 5.1.9.0 (2010/08/01)

	private static final String  DBID = "DEFAULT";
	private Connection defconn = null;		// 最も利用率の高いDEFAULTだけ、別に変数を用意。

	private final ApplicationInfo appInfo ;

	private Map<String,Connection> dbidMap = new HashMap<String,Connection>();
	private boolean isCommit   = false;
	private boolean isRollback = false;
	private boolean isError    = false;
	private boolean isFinish   = false;

	/**
	 * ApplicationInfo を指定して作成する、コンストラクター
	 *
	 * このクラスは、基本的には、TransactionTag クラスから作成されます。
	 *
	 * @param	appInfo  ApplicationInfo 内部統制用のアクセス情報
	 */
	public TransactionImpl( final ApplicationInfo appInfo ) {
		this.appInfo = appInfo ;
	}

	/**
	 * 指定のDBID に対応した、Connection オブジェクトを返します。
	 * 内部Mapに存在していれば、そのコネクションを、存在しなければ、
	 * 新しく作成します。
	 *
	 * @param	dbid  String DBID
	 * @return	Connection 指定のDBID に対応した、Connection オブジェクト
	 */
	public Connection getConnection( final String dbid ) {
		if( dbid == null || dbid.length() == 0 || DBID.equalsIgnoreCase( dbid ) ) {
			if( defconn == null ) {
				defconn = ConnectionFactory.connection( DBID,appInfo );
			}
			return defconn;
		}

		String udbid = dbid.toUpperCase( Locale.JAPAN );	// 大文字化

		Connection conn = dbidMap.get( udbid );
		if( conn == null ) {
			conn = ConnectionFactory.connection( udbid,appInfo );
			dbidMap.put( udbid,conn );
		}

		return conn;
	}

	/**
	 * コミット処理が行われた場合に、内部フラグ(isCommit)を true にセットします。
	 * １回でもコミットが行われており、ロールバックが行われていなければ、
	 * コミットされます。
	 *
	 * 検索処理のみで、エラーが発生していない場合は、コミットも行われないケースがあります。
	 *
	 * @return 正常:true/異常:false
	 */
	public boolean commit() {
		isCommit = true;
		return true;
	}

	/**
	 * ロールバック処理が行われた場合に、内部フラグ(isRollback)を true にセットします。
	 * １回でもロールバックが行われていれば、最終的にはロールバックされます。
	 *
	 * ロールバック指定の場合は、isError フラグを true(エラー有)にセットします。
	 *
	 * @return 正常:true/異常:false
	 */
	public boolean rollback() {
		isRollback = true;
		isError    = true;
		return true;
	}

	/**
	 * トランザクションの、終了時処理を行います。
	 *
	 * 実質的には、なにもしません。
	 *
	 * @see #close( boolean )
	 * @return 正常:true/異常:false
	 */
	public boolean close() {
		return close( false );
	}

	/**
	 * トランザクションの、終了時処理を行います。
	 *
	 * 引数は、正常かどうかを判定するフラグです。異常の場合は、true をセットします。
	 * ここでは、実際には何もしませんが、内部的にエラーフラグをセットします。
	 * （エラーの場合のみセット。リセットはされません）
	 * 一度でも、エラーが発生したコネクションは、破棄します。それ以外は、プールに戻します。
	 *
	 * @param	errFlag  boolean エラー状態(true)/通常(false)
	 * @return 正常:true/異常:false
	 */
	public boolean close( final boolean errFlag ) {
		if( errFlag ) { isError = true; }
		return true;
	}

	/**
	 * トランザクションとして、正常終了時に処理を行います。
	 *
	 * 実質的には、内部のfinishフラグをセットするだけです。
	 * ただし、このフラグがセットされていない場合は、処理が途中で止まった
	 * 可能性があるため、トランザクションとしては、正常終了させることができません。
	 *
	 * @see #realClose()
	 */
	public void finish() {
		isFinish = true;
	}

	/**
	 * トランザクションとして、終了時処理を行います。
	 *
	 * トランザクションがすべて完了した後で、呼び出します。
	 * 一度でも、Rollback が指定されていれば、ロールバックを行い、コネクションを
	 * 破棄します。それ以外で、Commit が指定されていれば、コミットを行い、
	 * コネクションを、プールに戻します。どちらも指定されていなければ、
	 * コネクションプールに戻すだけになります。
	 *
	 */
	public void realClose() {
		if( defconn != null ) {
			connClose( defconn,DBID );
		}

		for( Map.Entry<String,Connection> entry : dbidMap.entrySet() ) {
			String     dbid = entry.getKey();
			Connection conn = entry.getValue();

			connClose( conn,dbid );
		}

		// 内部変数を初期化します。
		defconn    = null;
		dbidMap    = null;
	}

	/**
	 * Connection オブジェクトをクローズします。
	 *
	 * 実際にはクローズせず、commit または、rollback を行った後、
	 * エラー状況に応じて、ConnectionFactory に返却するか、削除します。
	 * なお、commit または、rollback 時にエラーが発生した場合でも、無視して
	 * そのまま、処理を継続します。
	 *
	 * @param  conn Connection クローズ処理を行う、Connection オブジェクト
	 * @param  dbid String DBID
	 */
	private void connClose( final Connection conn, final String dbid ) {
		// まず、コミットかロールバックされた場合は、どちらかの処理が必要
		if( isCommit || isRollback ) {
			// commit できる条件：コミットされ、フィニッシュされ、エラーでなく、ロールバックでない
			if( isCommit && isFinish && (!isError) && (!isRollback) ) {
				Closer.commit( conn );
			}
			// それ以外は、ロールバックする。
			else {
				Closer.rollback( conn );
			}
		}

		// 残りは、コミットもロールバックもされていないため、単にキャッシュへの返し方のみ判定する。
		// isFinish されていないが、エラーにもなっていない場合は、接続要因以外の問題なので、返却でよい。
		if( isError ) { ConnectionFactory.remove( conn,dbid ); }	// 削除
		else {			ConnectionFactory.close( conn,dbid );  }	// 返却

	}
}
