/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.fukurou.db;

import java.sql.Connection;

/**
 * コネクションを共有して、トランザクションを実現するインターフェースです。
 *
 * 基本的には、TransactionTag で利用されますが、一部、このオブジェクトを
 * 渡して、直接、利用するケースもあります。
 *
 * トランザクション の実クラスには、リアルタイムで commit,rollback を行う、
 * TransactionReal クラスと、内部にキャッシュされた コネクションを、終了時点で
 * 一括処理を行う、TransactionImpl があります。
 * TransactionTag で利用するのが、TransactionImpl で、トランザクション処理を行わない
 * ケースで利用するのが、TransactionReal クラスになります。
 * TransactionReal クラス は、トランザクション処理を｢行わない｣場合に、共通ロジックを
 * 提供するために用意されたクラスです。
 *
 * @og.rev 5.1.9.0 (2010/08/01) 新規作成
 *
 * @version  5.0
 * @author	 Kazuhiko Hasegawa
 * @since    JDK6.0,
 */
public interface Transaction {

	/**
	 * 指定のDBID に対応した、Connection オブジェクトを返します。
	 * 内部Mapに存在していれば、そのコネクションを、存在しなければ、
	 * 新しく作成します。
	 *
	 * @param	dbid  String DBID
	 * @return	Connection 指定のDBID に対応した、Connection オブジェクト
	 */
	Connection getConnection( final String dbid ) ;

	/**
	 * コミット処理が行われた場合に、内部フラグ(isCommit)を true にセットします。
	 * １回でもコミットが行われており、ロールバックが行われていなければ、
	 * コミットされます。
	 *
	 * 検索処理のみで、エラーが発生していない場合は、コミットも行われないケースがあります。
	 *
	 * @return 正常:true/異常:false
	 */
	boolean commit() ;

	/**
	 * ロールバック処理が行われた場合に、内部フラグ(isRollback)を true にセットします。
	 * １回でもロールバックが行われていれば、最終的にはロールバックされます。
	 *
	 * @return 正常:true/異常:false
	 */
	boolean rollback() ;

	/**
	 * トランザクションの、終了時処理を行います。
	 *
	 * それまでの処理は、すべて正常に処理できた場合に、使用します。
	 * close( false ) と同じです。
	 *
	 * @see #close( boolean )
	 * @return 正常:true/異常:false
	 */
	boolean close() ;

	/**
	 * トランザクションの、終了時処理を行います。
	 *
	 * 引数は、正常かどうかを判定するフラグです。異常の場合は、true をセットします。
	 * ここでは、実際には何もしませんが、内部的にエラーフラグをセットします。
	 * （エラーの場合のみセット。リセットはされません）
	 * 一度でも、エラーが発生したコネクションは、破棄します。それ以外は、プールに戻します。
	 *
	 * @param	errFlag  boolean エラー状態(true)/通常(false)
	 * @return 正常:true/異常:false
	 */
	boolean close( final boolean errFlag ) ;
}
