/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import org.opengion.hayabusa.resource.UserInfo;
import org.opengion.hayabusa.db.DBTableModel;
import org.opengion.hayabusa.db.Query;
import org.opengion.hayabusa.db.QueryFactory;

import static org.opengion.fukurou.util.StringUtil.nval ;

import java.util.Locale ;
import java.io.ObjectOutputStream;
import java.io.ObjectInputStream;
import java.io.IOException;

/**
 * ユーザー情報を管理しているタグです。
 *
 * 設定した値は、{&#064;USER.XXXX} 形式で 取り出すことができます。
 * また、command 属性で 直接画面に値を書き出すことも可能です。
 *
 * 以下の値は UserInfo オブジェクトの項目から取得します。
 * ・USER.JNAME     ユーザー日本語名称
 * ・USER.ID        ユーザーＩＤ
 * ・USER.IDNO      USER.ID が５Byte以上の時のみ先頭１文字を除いたユーザーＩＤ
 * ・USER.INFO      ユーザー情報（ユーザーID：日本語名称）
 * ・USER.LANG      言語
 * ・USER.ROLES     ロール
 * ・USER.IPADDRESS IPアドレス
 * ・USER.LOGINTIME ログイン時刻
 *
 * 以下の値はあらかじめ、動的に作成されます。
 * ・USER.YMD       ８byte の今日のシステム日付
 * ・USER.YMDH    １４byte の今日のシステム日時
 *
 * それ以外は、外部より設定された値です。
 *
 * @og.formSample
 * ●形式：&lt;og:userInfo command="[…]" key="[…]" value="[…]" /&gt;
 * ●body：あり
 *
 * ●使用例
 *		&lt;og:userInfo
 *		   command		 = &quot;コマンド(SET,GET,NVLEGT,REMOVE,SQL)をセットします(初期値:SET)。&quot;
 *		   key			 = &quot;UserInfo に登録するキーをセットします(予約語:JNAME,ID,PASSWD,IDNO,INFO,LANG,ROLE,IPADDRESS,LOGINTIME)。&quot;
 *		   value		 = &quot;UserInfo に登録する値をセットします。&quot;
 *		   dbid			 = &quot;(通常は使いません)Queryオブジェクトを作成する時のDB接続IDを指定します。&quot;
 *		&gt;
 *
 * @og.group その他部品
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class UserInfoTag extends CommonTagSupport {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "4.0.0 (2005/08/31)" ;

	private static final long serialVersionUID = 4000 ;	// 4.0.0 (2005/01/31)

	/** command 引数に渡す事の出来る コマンド  セット {@value} */
	public static final String CMD_SET   = "SET" ;
	/** command 引数に渡す事の出来る コマンド  ゲット {@value} */
	public static final String CMD_GET   = "GET" ;
	/** command 引数に渡す事の出来る コマンド  ゲット {@value} */
	public static final String CMD_REMOVE   = "REMOVE" ;
	/** command 引数に渡す事の出来る コマンド  ゲット {@value} */
	public static final String CMD_SQL   = "SQL" ;
	// 3.5.5.3 (2004/04/09) 値が null の場合は、""(ゼロ文字列)を返すNVLGETを採用
	/** command 引数に渡す事の出来る コマンド  NVLゲット {@value} */
	public static final String CMD_NVLGET   = "NVLGET" ;

	// 3.5.5.3 (2004/04/09) 値が null の場合は、""(ゼロ文字列)を返すNVLGETを採用
	/** command 引数に渡す事の出来る コマンド リスト  */
	private static final String[] COMMAND_LIST = new String[] {
		CMD_SET , CMD_GET , CMD_NVLGET , CMD_REMOVE , CMD_SQL };

	// 3.5.6.0 (2004/06/18) すべてを protected から private に変更します。
	private String			command		= CMD_SET;
	private String			key			= null;
	private String			value		= null;
//	private boolean			isNullSet	= true;			// NULL のときにセットし直すかどうか。
	private transient DBTableModel	table		= null;
	// 4.0.0.0 (2007/10/10) dbid の初期値を、"DEFAULT" から null に変更
//	private String			dbid		= "DEFAULT";
	private String			dbid		= null;
	private String			sql			= null;
	// 4.3.4.0 (2008/12/01) save属性追加
	private boolean			save		= false;

	/**
	 * Taglibの開始タグが見つかったときに処理する doStartTag() を オーバーライドします。
	 *
	 * @return  int
	 */
	@Override
	public int doStartTag() {
		if( CMD_SQL.equals( command ) ) {
			return( EVAL_BODY_BUFFERED );		// Body を評価する
		}
		else {
			return( SKIP_BODY ); // Body を評価しない
		}
	}

	/**
	 * Taglibのタグ本体を処理する doAfterBody() を オーバーライドします。
	 *
	 * @og.rev 3.1.1.0 (2003/03/28) ボディの内容を取得する処理を、CommonTagSupport で行う。
	 * @og.rev 3.6.0.8 (2004/11/19) エラー発生時に確実にリリースされるように try finally 追加
	 * @og.rev 3.8.6.3 (2006/11/30) SQL 文の前後のスペースを取り除きます。
	 * @og.rev 3.8.7.0 (2006/12/15) アクセスログ取得の為,ApplicationInfo オブジェクトを設定
	 * @og.rev 4.0.0 (2005/01/31) lang ⇒ ResourceManager へ変更
	 *
	 * @return  int
	 */
	@Override
	public int doAfterBody() {
		sql = getBodyString().trim();

		Query query = QueryFactory.newInstance();		// 4.0.0 (2005/01/31)
		try {
			query.setConnectionID( dbid );
			query.setResourceManager( getResource() );	// 4.0.0 (2005/01/31)

			query.setStatement( sql );
			query.setApplicationInfo( getApplicationInfo() );	// 3.8.7.0 (2006/12/15)
			query.execute();

			table = query.getDBTableModel();
		}
		finally {
			if( query != null ) { query.close(); }
			QueryFactory.close( query );
		}
		return(SKIP_BODY);
	}

	/**
	 * Taglibの終了タグが見つかったときに処理する doEndTag() を オーバーライドします。
	 *
	 * @og.rev 3.1.1.2 (2003/04/04) Tomcat4.1 対応。release2() を doEndTag()で呼ぶ。
	 * @og.rev 3.5.5.3 (2004/04/09) 値が null の場合は、""(ゼロ文字列)を返すフラグを採用
	 *
	 * @return  int
	 */
	@Override
	public int doEndTag() {
		debugPrint();		// 4.0.0 (2005/02/28)
		if( check( command, COMMAND_LIST ) ) {
			commandExec( command );

			if( CMD_GET.equals( command ) ) {
				jspPrint( value );
			}
			else if( CMD_NVLGET.equals( command ) ) {	// 3.5.5.3 (2004/04/09)
				jspPrint( nval(value,"") );
			}
		}

		return(EVAL_PAGE);
	}

	/**
	 * タグリブオブジェクトをリリースします。
	 * キャッシュされて再利用されるので、フィールドの初期設定を行います。
	 *
	 * @og.rev 2.0.0.4 (2002/09/27) カスタムタグの release() メソッドを、追加
	 * @og.rev 3.1.1.2 (2003/04/04) Tomcat4.1 対応。release2() を doEndTag()で呼ぶ。
	 * @og.rev 4.0.0.0 (2007/10/10) dbid の初期値を、"DEFAULT" から null に変更
	 * @og.rev 4.3.4.0 (2008/12/01) save属性対応
	 *
	 */
	@Override
	protected void release2() {
		super.release2();
		command		= CMD_SET;
		key			= null;
		value		= null;
//		isNullSet	= true;			// NULL のときにセットし直すかどうか。
		table		= null;
//		dbid		= "DEFAULT";
		dbid		= null;
		sql			= null;
		save		= false;
	}

	/**
	 * コマンドを実行します。
	 *
	 * コマンドは,HTMLから（get/post)指定されますので,setCommand()メソッドで
	 * 登録します。
	 * コマンドを登録すると同時に,実行も行ないます。
	 *
	 * @og.rev 3.5.5.3 (2004/04/09) 値が null の場合は、""(ゼロ文字列)を返すフラグを採用
	 *
	 * @param   command コマンド（public static final 宣言されている文字列)
	 * @see		<a href="{@docRoot}/constant-values.html#org.opengion.hayabusa.taglib.UserInfoTag.CMD_SET">コマンド定数</a>
	 */
	protected void commandExec( final String command ) {
		if( CMD_SQL.equals( command ) ) { setSQLAttribute() ; }
		else if( CMD_SET.equals(    command ) ) { setAttribute() ;    }
		else if( CMD_GET.equals(    command ) ) { getAttribute() ;    }
		else if( CMD_NVLGET.equals( command ) ) { getAttribute() ;    }	// 3.5.5.3 (2004/04/09)
		else if( CMD_REMOVE.equals( command ) ) { removeAttribute() ; }
	}

	/**
	 * UserInfoの文字列を登録します。
	 *
	 * @og.rev 4.3.4.0 (2008/12/01) GE20(ユーザー定数)へ登録するかのフラグを追加
	 *
	 */
	private void setAttribute() {
//		if( isNullSet ) {
//			setUserInfo( key, value );
			setUserInfo( key, value, save );
//		}
	}

	/**
	 * UserInfoの文字列を作成します。
	 *
	 */
	private void getAttribute() {
		value = getUserInfo( key );
	}

	/**
	 * UserInfoの文字列を削除します。
	 *
	 */
	private void removeAttribute() {
		UserInfo userInfo = getUser();
		userInfo.removeAttribute( key );
	}

	/**
	 * UserInfoの文字列を指定のSQL文より作成します。
	 *
	 * @og.rev 4.3.4.0 (2008/12/01) GE20(ユーザー定数)へ登録するかのフラグを追加
	 *
	 */
	private void setSQLAttribute() {
		if( table == null || table.getRowCount() == 0 ) { return ; }

		int row    = 0;
		int clmCnt = table.getColumnCount();
		for( int clm=0; clm<clmCnt; clm++ ) {
			String clmkey = table.getColumnName( clm );
			String clmval = table.getValue( row,clm );
//			setUserInfo( clmkey,clmval );
			setUserInfo( clmkey, clmval, save );
		}
	}

	/**
	 * 【TAG】コマンド(SET,GET,NVLEGT,REMOVE,SQL)をセットします(初期値:SET)。
	 *
	 * @og.tag
	 * コマンドは,HTMLから（get/post)指定されますので,CMD_xxx で設定される
	 * フィールド定数値のいずれかを、指定できます。
	 * 何も設定されない、または、null の場合は、"SET" が初期値にセットされます。
	 * SQL の場合、検索結果の戻り値が複数存在する場合は、最初の１件目のみ使用します。
	 *
	 * @param	cmd コマンド（public static final 宣言されている文字列)
	 * @see		<a href="{@docRoot}/constant-values.html#org.opengion.hayabusa.taglib.UserInfoTag.CMD_SET">コマンド定数</a>
	 */
	public void setCommand( final String cmd ) {
		String cmd2 = getRequestParameter( cmd );
		if( cmd2 != null && cmd2.length() > 0 ) { command = cmd2.toUpperCase(Locale.JAPAN); }
	}

	/**
	 * 【TAG】UserInfo に登録するキーをセットします(予約語:JNAME,ID,PASSWD,IDNO,INFO,LANG,ROLE,IPADDRESS,LOGINTIME)。
	 *
	 * @og.tag UserInfo に登録するキーをセットします。
	 *
	 * @param	inkey UserInfo に登録するキー
	 */
	public void setKey( final String inkey ) {
		key = getRequestParameter( inkey ).toUpperCase(Locale.JAPAN);
		if( key.startsWith( "USER." ) ) {
			key = key.substring( 5 );
		}
	}

	/**
	 * 【TAG】UserInfo に登録する値をセットします。
	 *
	 * @og.tag UserInfo に登録する値をセットします。
	 *
	 * @param   val String UserInfoに登録する値
	 */
	public void setValue( final String val ) {
		value = getRequestParameter( val );
	}

	/**
	 * 【TAG】UserInfo に登録した値を永続化するかを指定します。
	 *
	 * @og.tag
	 * UserInfo に登録した値を永続化するかを指定します。
	 * trueが指定された場合、UserInfoに設定された値は、GE20(ユーザー定数)に保存され、
	 * UserInfoが再作成されるタイミングで自動的に復元されます。
	 * ここで、登録された値は、そのユーザーの全てのロールに対して有効となります。
	 * 初期値は、false(永続化しない)です。
	 *
	 * @param   sv String 登録値を永続化するか
	 */
	public void setSave( final String sv ) {
		save = nval( getRequestParameter( sv ),save );
	}

//	/**
//	 * 【廃止】Query を実行して、value に値をセットするかどうかを指定します。
//	 *
//	 * @og.tag
//	 * 初期値は、実行しない ("false")
//	 *
//	 * @og.rev 3.1.0.1 (2003/03/26) query 属性を非推奨属性とします。内部では使用しません。
//	 *
//	 * @param   flag Query を実行する ("true")／実行しない(それ以外)
//	 * @deprecated
//	 */
//	@Deprecated public void setQuery( final String flag ) {
//		// ここでは処理を行いません。
//	}

//	/**
//	 * 【廃止】value が NULL の時に、設定するかどうか(true/false)を指定します(初期値:true)。
//	 *
//	 * @og.tag
//	 * true の場合は, null のときでもセットします。
//	 * false の場合は, null のときは、既存の値を置き換えません。
//	 * 初期値は、null のときでもセットするです。 ("true")
//	 *
//	 * @param   flag NULL の時に、設定する ("true")／設定しない(それ以外)
//	 * @deprecated
//	 */
//	@Deprecated public void setNullSet( final String flag ) {
//		isNullSet = nval( getRequestParameter( flag ),isNullSet );
//	}

	/**
	 * 【TAG】(通常は使いません)Queryオブジェクトを作成する時のDB接続IDを指定します。
	 *
	 * @og.tag Queryオブジェクトを作成する時のDB接続IDを指定します。
	 *
	 * @param	id データベース接続ID
	 */
	public void setDbid( final String id ) {
		dbid = nval( getRequestParameter( id ),dbid );
	}

	/**
	 * シリアライズ用のカスタムシリアライズ書き込みメソッド
	 *
	 * @og.rev 4.0.0 (2006/09/31) 新規追加
	 * @serialData
	 *
	 * @param strm ObjectOutputStream
	 */
	private void writeObject( final ObjectOutputStream strm ) throws IOException {
		strm.defaultWriteObject();
	}

	/**
	 * シリアライズ用のカスタムシリアライズ読み込みメソッド
	 *
	 * ここでは、transient 宣言された内部変数の内、初期化が必要なフィールドのみ設定します。
	 *
	 * @og.rev 4.0.0 (2006/09/31) 新規追加
	 * @serialData
	 *
	 * @param strm ObjectInputStream
	 * @see #release2()
	 */
	private void readObject( final ObjectInputStream strm ) throws IOException , ClassNotFoundException {
		strm.defaultReadObject();
	}

	/**
	 * このオブジェクトの文字列表現を返します。
	 * 基本的にデバッグ目的に使用します。
	 *
	 * @return このクラスの文字列表現
	 */
	public String toString() {
		return org.opengion.fukurou.util.ToString.title( this.getClass().getName() )
				.println( "VERSION"			,VERSION	)
				.println( "command"			,command	)
				.println( "key"				,key		)
				.println( "value"			,value		)
//				.println( "isNullSet"		,isNullSet	)
				.println( "dbid"			,dbid		)
				.println( "sql"				,sql		)
				.println( "save"			,save		)
				.println( "COMMAND_LIST"	,COMMAND_LIST	)
				.println( "Other..."	,getAttributes().getAttribute() )
				.fixForm().toString() ;
	}
}
