/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.fukurou.xml;

import org.opengion.fukurou.util.Closer;
import org.opengion.fukurou.util.ApplicationInfo;
import org.opengion.fukurou.util.LogWriter;

import java.io.Reader;
import java.io.BufferedReader;
import java.io.InputStreamReader;
import java.io.FileInputStream;
import java.util.Map;
import java.util.List;
import java.util.ArrayList;
import java.util.regex.Pattern;
import java.util.regex.Matcher;

import java.sql.DriverManager;
import java.sql.Connection;
import java.sql.Statement;
import java.sql.PreparedStatement;
import java.sql.ParameterMetaData;
import java.sql.SQLException;

/**
 * このクラスは、オラクル XDKの oracle.xml.sql.dml.OracleXMLSave クラスと
 * ほぼ同様の目的で使用できるクラスです。
 * 拡張XDK形式のXMLファイルを読み込み、データベースに INSERT します。
 *
 * 拡張XDK形式の元となる オラクル XDK(Oracle XML Developer's Kit)については、以下の
 * リンクを参照願います。<br />
 * <a href="http://otn.oracle.co.jp/software/tech/xml/xdk/index.html" target="_blank" >
 * XDK(Oracle XML Developer's Kit)</a>
 *
 * このクラスでは、MAP を登録する[ setDefaultMap( Map ) ]ことにより、
 * XMLファイルに存在しないカラムを初期値として設定することが可能になります。
 * 例えば、登録日や、登録者、または、テンプレートより各システムID毎に
 * 登録するなどです。
 * 同様に、読み取った XMLファイルの情報を書き換える機能[ setAfterMap( Map ) ]メソッド
 * により、カラムの値の置き換えも可能です。
 *
 * 拡張XDK形式の元となる オラクル XDK(Oracle XML Developer's Kit)については、以下の
 * リンクを参照願います。<br />
 * <a href="http://otn.oracle.co.jp/software/tech/xml/xdk/index.html" target="_blank" >
 * XDK(Oracle XML Developer's Kit)</a>
 *
 * 拡張XDK形式とは、ROW 以外に、SQL処理用タグ(EXEC_SQL)を持つ XML ファイルです。
 * また、登録するテーブル(table)を ROWSETタグの属性情報として付与することができます。
 * (大文字小文字に注意)
 * これは、オラクルXDKで処理する場合、無視されますので、同様に扱うことが出来ます。
 * この、EXEC_SQL は、それそれの XMLデータをデータベースに登録する際に、
 * SQL処理を自動的に流す為の、SQL文を記載します。
 * この処理は、イベント毎に実行される為、その配置順は重要です。
 * このタグは、複数記述することも出来ますが、BODY部には、１つのSQL文のみ記述します。
 *
 *   &lt;ROWSET tableName="XX" &gt;
 *       &lt;EXEC_SQL&gt;                    最初に記載して、初期処理（データクリア等）を実行させる。
 *           delete from GEXX where YYYYY
 *       &lt;/EXEC_SQL&gt;
 *       &lt;MERGE_SQL&gt;                   このSQL文で UPDATEして、結果が０件ならINSERTを行います。
 *           update GEXX set AA=[AA] , BB=[BB] where CC=[CC]
 *       &lt;/MERGE_SQL&gt;
 *       &lt;ROW num="1"&gt;
 *           &lt;カラム1&gt;値1&lt;/カラム1&gt;
 *             ･･･
 *           &lt;カラムn&gt;値n&lt;/カラムn&gt;
 *       &lt;/ROW&gt;
 *        ･･･
 *       &lt;ROW num="n"&gt;
 *          ･･･
 *       &lt;/ROW&gt;
 *       &lt;EXEC_SQL&gt;                    最後に記載して、項目の設定(整合性登録)を行う。
 *           update GEXX set AA='XX' , BB='XX' where YYYYY
 *       &lt;/EXEC_SQL&gt;
 *   &lt;ROWSET&gt;
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class HybsXMLSave implements TagElementListener {
	/** システム依存の改行記号をセットします。	*/
	private static final String CR = System.getProperty("line.separator");

	private String tableName		= null;
	private String[] keyColumns		= null;
	private Connection connection	= null;
	private PreparedStatement insPstmt	= null;		// INSERT用の PreparedStatement
	private PreparedStatement updPstmt	= null;		// UPDATE用の PreparedStatement
	private ParameterMetaData insMeta	= null;
	private ParameterMetaData updMeta	= null;
	private int insCnt		= 0;
	private int updCnt		= 0;
	private int delCnt		= 0;
	private Map<String,String>	defaultMap	= null;
	private Map<String,String>	afterMap	= null;
	private List<String>		updClms		= null;
	private String[]			insClms		= null;

	private final boolean useParamMetaData ;	// 4.0.0.0 (2007/09/25)

	// UPDATE時の [XXX] を取り出します。\w は、単語構成文字: [a-zA-Z_0-9]と同じ
	private static final Pattern pattern = Pattern.compile( "\\[\\w*\\]" );

	/**
	 * コネクションを指定して、オブジェクトを構築します。
	 * テーブル名は、拡張XDK形式のROWSETタグのtableName属性に
	 * 記述しておく必要があります。
	 *
	 * @param conn Connection データベース接続
	 */
	public HybsXMLSave( final Connection conn ) {
		this( conn,null );
	}

	/**
	 * コネクションとテーブル名を指定して、オブジェクトを構築します。
	 * ここで指定するテーブル名は、デフォルトテーブルという扱いです。
	 * 拡張XDK形式のROWSETタグのtableName属性にテーブル名が記述されている場合は、
	 * そちらが優先されます。
	 *
	 * @og.rev 4.0.0.0 (2007/09/25) ParameterMetaData を使用したパラメータ設定追加。
	 *
	 * @param conn Connection データベース接続
	 * @param table String テーブル名(ROWSETタグのtable属性が未設定時に使用)
	 */
	public HybsXMLSave( final Connection conn,final String table ) {
		connection = conn;
		tableName  = table;
		useParamMetaData = ApplicationInfo.useParameterMetaData( conn );
	}

	/**
	 * &lt;ROWSET&gt; タグの一番最初に呼び出されます。
	 * ROWSET の属性である、table 属性と、dbid 属性 を、TagElement の
	 * get メソッドで取得できます。
	 * 取得時のキーは、それぞれ、"TABLE" と "DBID" です。
	 *
	 * @param tag TagElement
	 * @see org.opengion.fukurou.xml.TagElement
	 * @see HybsXMLHandler#setTagElementListener( TagElementListener )
	 */
	public void actionInit( final TagElement tag ) {
		String table = tag.get( "tableName" );
		if( table != null ) { tableName = table; }
	}

	/**
	 * &lt;ROW&gt; タグの endElement 処理毎に呼び出されます。
	 * この Listener をセットすることにより、行データを取得都度、
	 * TagElement オブジェクトを作成し、このメソッドが呼び出されます。
	 *
	 * @og.rev 4.0.0 (2007/05/09) ParameterMetaData を使用したパラメータ設定追加。
	 * @og.rev 4.0.0.0 (2007/09/25) isOracle から useParamMetaData に変更
	 * @og.rev 4.3.7.0 (2009/06/01) HSQLDB対応
	 *
	 * @param tag TagElement
	 * @see org.opengion.fukurou.xml.TagElement
	 * @see HybsXMLHandler#setTagElementListener( TagElementListener )
	 */
	public void actionRow( final TagElement tag ) {
		tag.setAfterMap( afterMap );

		try {
			// 更新SQL(MERGE_SQLタグ)が存在する場合の処理
			int tempCnt = 0;
			if( updPstmt != null ) {
				String[] vals = tag.getValues( updClms );
				for( int j=0; j<vals.length; j++ ) {
					// 4.3.7.0 (2009/06/01) HSQLDB対応。空文字の場合nullに置換え
					if( vals[j] != null && vals[j].length() == 0 ){
						vals[j] = null;
					}
					
					// 4.0.0.0 (2007/09/25) ParameterMetaData を使用したパラメータ設定追加
					if( useParamMetaData ) {
						int type = updMeta.getParameterType( j+1 );
						updPstmt.setObject( j+1,vals[j],type );
					}
					else {
						updPstmt.setObject( j+1,vals[j] );
					}

				}
				tempCnt = updPstmt.executeUpdate();
				if( tempCnt > 1 ) {
					String errMsg = "Update キーが重複しています。"
							+ "TABLE=[" + tableName + "] ROW=["
							+ tag.getRowNo() + "]" + CR
							+ tag.toString() + CR ;
					throw new RuntimeException( errMsg );
				}
				updCnt += tempCnt;
			}
			// 更新が 0件の場合は、INSERT処理を行います。
			if( tempCnt == 0 ) {
				// 初回INSERT時のタグより、DB登録SQL文を構築します。
				if( insPstmt == null ) {
					insClms = tag.getKeys();
					String sql = insertSQL( insClms,tableName );
					insPstmt = connection.prepareStatement( sql );
					// 4.0.0.0 (2007/09/25) ParameterMetaData を使用したパラメータ設定追加
					if( useParamMetaData ) { insMeta = insPstmt.getParameterMetaData(); }
				}
				String[] vals = tag.getValues( insClms );
				for( int j=0; j<vals.length; j++ ) {
					// 4.3.7.0 (2009/06/01) HSQLDB対応。空文字の場合nullに置換え
					if( vals[j] != null && vals[j].length() == 0 ){
						vals[j] = null;
					}
					
					// 4.0.0.0 (2007/09/25) ParameterMetaData を使用したパラメータ設定追加
					if( useParamMetaData ) {
						int type = insMeta.getParameterType( j+1 );
						insPstmt.setObject( j+1,vals[j],type );
					}
					else {
						insPstmt.setObject( j+1,vals[j] );
					}
				}
				insCnt += insPstmt.executeUpdate();
			}
		}
		catch( SQLException ex ) {
			String errMsg = "DB登録エラーが発生しました。"
						+ "TABLE=[" + tableName + "] ROW=["
						+ tag.getRowNo() + "]" + CR
						+ tag.toString() + CR
						+ ex.getMessage() + ":" + ex.getSQLState() + CR ;
			throw new RuntimeException( errMsg,ex );
		}
	}

	/**
	 * &lt;EXEC_SQL&gt; タグの endElement 処理毎に呼び出されます。
	 * getBody メソッドを使用して、このタグのBODY部の文字列を取得します。
	 * この Listener をセットすることにより、EXEC_SQL データを取得都度、
	 * TagElement オブジェクトを作成し、このメソッドが呼び出されます。
	 *
	 * @param tag TagElement
	 * @see org.opengion.fukurou.xml.TagElement
	 * @see HybsXMLHandler#setTagElementListener( TagElementListener )
	 */
	public void actionExecSQL( final TagElement tag ) {
		Statement execSQL = null ;
		try {
			String sql = tag.getBody();
			execSQL = connection.createStatement();
			delCnt += execSQL.executeUpdate( sql ) ;
		}
		catch( SQLException ex ) {
			String errMsg = "DB登録エラーが発生しました。"
						+ "TABLE=[" + tableName + "] ROW=["
						+ tag.getRowNo() + "]" + CR
						+ tag.toString() + CR
						+ ex.getMessage() + ":" + ex.getSQLState() + CR ;
			throw new RuntimeException( errMsg,ex );
		}
		finally {
			Closer.stmtClose( execSQL );
		}
	}

	/**
	 * &lt;MERGE_SQL&gt; タグの endElement 処理時に呼び出されます。
	 * getBody メソッドを使用して、このタグのBODY部の文字列を取得します。
	 * MERGE_SQLタグは、マージ処理したいデータ部よりも上位に記述しておく
	 * 必要がありますが、中間部に複数回記述しても構いません。
	 * このタグが現れるまでは、INSERT のみ実行されます。このタグ以降は、
	 * 一旦 UPDATE し、結果が ０件の場合は、INSERTする流れになります。
	 * 完全に INSERT のみであるデータを前半に、UPDATE/INSERTを行う
	 * データを後半に、その間に、MERGE_SQL タグを入れることで、無意味な
	 * UPDATE を避けることが可能です。
	 * この Listener をセットすることにより、MERGE_SQL データを取得都度、
	 * TagElement オブジェクトを作成し、このメソッドが呼び出されます。
	 *
	 * @og.rev 4.0.0 (2007/05/09) ParameterMetaData を使用したパラメータ設定追加。
	 * @og.rev 4.0.0.0 (2007/09/25) isOracle から useParamMetaData に変更
	 *
	 * @param tag TagElement
	 * @see org.opengion.fukurou.xml.TagElement
	 * @see HybsXMLHandler#setTagElementListener( TagElementListener )
	 */
	public void actionMergeSQL( final TagElement tag ) {
		if( updPstmt != null ) {
			String errMsg = "MERGE_SQLタグが、複数回記述されています。"
						+ "TABLE=[" + tableName + "] ROW=["
						+ tag.getRowNo() + "]" + CR
						+ tag.toString() + CR;
			throw new RuntimeException( errMsg );
		}

		String orgSql = tag.getBody();
		Matcher matcher = pattern.matcher( orgSql );
		updClms = new ArrayList<String>();
		while( matcher.find() ) {
			// ここでは、[XXX]にマッチする為、前後の[]を取り除きます。
			updClms.add( orgSql.substring( matcher.start()+1,matcher.end()-1 ) );
		}
		String sql = matcher.replaceAll( "?" );

		try {
			updPstmt = connection.prepareStatement( sql );
			// 4.0.0.0 (2007/09/25) ParameterMetaData を使用したパラメータ設定追加
			if( useParamMetaData ) { updMeta = updPstmt.getParameterMetaData(); }
		}
		catch( SQLException ex ) {
			String errMsg = "Statement作成時にエラーが発生しました。"
						+ "TABLE=[" + tableName + "] ROW=["
						+ tag.getRowNo() + "]" + CR
						+ tag.toString() + CR
						+ ex.getMessage() + ":" + ex.getSQLState() + CR ;
			throw new RuntimeException( errMsg,ex );
		}
	}

	/**
	 * UPDATE,DELETE を行う場合の WHERE 条件になるキー配列
	 * このキーの AND 条件でカラムを特定し、UPDATE,DELETE などの処理を
	 * 行います。
	 *
	 * @param keyCols String[] WHERE 条件になるキー配列
	 */
	public void setKeyColumns( final String[] keyCols ) {
		keyColumns = new String[keyCols.length];
		System.arraycopy( keyCols,0,keyColumns,0,keyColumns.length );
	}

	/**
	 * XMLファイルを読み取る前に指定するカラムと値のペア（Map)情報
	 *
	 * このカラムと値のペアのマップは、オブジェクト構築前に設定される為、
	 * XMLファイルにキーが存在している場合は、値が書き変わります。（XML優先）
	 * XMLファイルにキーが存在していない場合は、ここで指定するMapの値が
	 * 初期設定値として使用されます。
	 * ここで指定する Map に LinkedHashMap を使用する場合、カラム順も
	 * 指定することが出来ます。
	 *
	 * @param map Map<String,String> 初期設定するカラムデータ
	 * @see #setAfterMap( Map )
	 */
	public void setDefaultMap( final Map<String,String> map ) { defaultMap = map; }

	/**
	 * XMLファイルを読み取った後で指定するカラムと値のペア（Map)情報
	 *
	 * このカラムと値のペアのマップは、オブジェクト構築後に設定される為、
	 * XMLファイルのキーの存在に関係なく、Mapのキーと値が使用されます。（Map優先）
	 * null を設定した場合は、なにも処理されません。
	 *
	 * @param map Map<String,String> 後設定するカラムデータ
	 * @see #setDefaultMap( Map )
	 */
	public void setAfterMap( final Map<String,String> map ) { afterMap = map; }

	/**
	 * データベースに追加処理(INSERT)を行います。
	 *
	 * 先に指定されたコネクションを用いて、指定のテーブルに INSERT します。
	 * 引数には、XMLファイルを指定したリーダーをセットします。
	 * コネクションは、終了後、コミットされます。(close されません。)
	 * リーダーのクローズは、ここでは行っていません。
	 *
	 * @og.rev 5.1.1.0 (2009/11/11) insMeta , updMeta のクリア(気休め)
	 *
	 * @param reader Reader XMLファイルを指定するリーダー
	 */
	public void insertXML( final Reader reader ) {
		try {
			HybsXMLHandler handler = new HybsXMLHandler();
			handler.setTagElementListener( this );
			handler.setDefaultMap( defaultMap );

			handler.parse( reader );
		}
		finally {
			Closer.stmtClose( insPstmt );
			Closer.stmtClose( updPstmt );
			insPstmt = null;
			updPstmt = null;
			insMeta	= null;		// 5.1.1.0 (2009/11/11)
			updMeta	= null;		// 5.1.1.0 (2009/11/11)
		}
	}

	/**
	 * インサート用のSQL文を作成します。
	 *
	 * @param columns String[] インサートするカラム名
	 * @param tableName String インサートするテーブル名
	 * @return String インサート用のSQL文
	 */
	private String insertSQL( final String[] columns,final String tableName ) {
		if( tableName == null ) {
			String errMsg = "tableName がセットされていません。" + CR
						+ "tableName は、コンストラクタで指定するか、ROWSETのtableName属性で"
						+ "指定しておく必要があります" + CR ;
			throw new RuntimeException( errMsg );
		}

		StringBuilder sql = new StringBuilder();
		sql.append( "INSERT INTO " ).append( tableName );
		sql.append( " ( " );
		sql.append( columns[0] );
		for( int i=1; i<columns.length; i++ ) {
			sql.append( "," ).append( columns[i] );
		}
		sql.append( " ) VALUES ( " );
		sql.append( "?" );
		for( int i=1; i<columns.length; i++ ) {
			sql.append( "," ).append( "?" );
		}
		sql.append( " )" );

		return sql.toString();
	}

	/**
	 * データベースに追加した件数を返します。
	 *
	 * @return 登録件数
	 */
	public int getInsertCount() { return insCnt; }

	/**
	 * データベースを更新した件数を返します。
	 * これは、拡張XDK形式で、MERGE_SQL タグを使用した場合の更新処理件数を
	 * 合計した値を返します。
	 *
	 * @return 更新件数
	 */
	public int getUpdateCount() { return updCnt; }

	/**
	 * データベースに変更（更新、削除を含む）した件数を返します。
	 * これは、拡張XDK形式で、EXEC_SQL タグを使用した場合の実行件数を合計した
	 * 値を返します。
	 * よって、更新か、追加か、削除かは、判りませんが、通常 登録前に削除する
	 * ケースで使われることから、deleteCount としています。
	 *
	 * @return 変更件数(主に、削除件数)
	 */
	public int getDeleteCount() { return delCnt; }

	/**
	 * 実際に登録された テーブル名を返します。
	 *
	 * テーブル名は、拡張XDK形式のROWSETタグのtableName属性に
	 * 記述しておくか、コンストラクターで引数として渡します。
	 * 両方指定された場合は、ROWSETタグのtableName属性が優先されます。
	 * ここでの返り値は、実際に使用された テーブル名です。
	 *
	 * @return 変更件数(主に、削除件数)
	 */
	public String getTableName() { return tableName; }

	/**
	 * テスト用のメインメソッド
	 *
	 * java org.opengion.fukurou.xml.HybsXMLSave USER PASSWD URL TABLE FILE [ENCODE] [DRIVER]
	 *    USER  : DB接続ユーザー（GE)
	 *    PASSWD: DB接続パスワード(GE)
	 *    URL   : DB接続JDBCドライバURL(jdbc:oracle:thin:@localhost:1521:HYBS
	 *    TABLE : 登録するテーブルID(GE21)
	 *    FILE  : 登録するORACLE XDK 形式 XMLファイル(GE21.xml)
	 *    ENCODE: ファイルのエンコード 初期値:UTF-8
	 *    DRIVER: JDBCドライバー 初期値:oracle.jdbc.OracleDriver
	 *
	 * @og.rev 5.1.1.0 (2009/12/01) MySQL対応 明示的に、TRANSACTION_READ_COMMITTED を指定する。
	 *
	 * @param args String[]
	 * @throws Exception
	 */
	public static void main( final String[] args )
			throws ClassNotFoundException , SQLException {
		if( args.length < 5 ) {
			LogWriter.log( "Usage: java org.opengion.fukurou.xml.HybsXMLSave USER PASSWD URL TABLE FILE [ENCODE] [DRIVER]" );
			LogWriter.log( "   USER  : DB接続ユーザー（GE)" );
			LogWriter.log( "   PASSWD: DB接続パスワード(GE)" );
			LogWriter.log( "   URL   : DB接続JDBCドライバURL(jdbc:oracle:thin:@localhost:1521:HYBS)" );
			LogWriter.log( "   TABLE : 登録するテーブルID(GE21)" );
			LogWriter.log( "   FILE  : 登録するORACLE XDK 形式 XMLファイル(GE21.xml)" );
			LogWriter.log( " [ ENCODE: ファイルのエンコード 初期値:UTF-8 ]" );
			LogWriter.log( " [ DRIVER: JDBCドライバー 初期値:oracle.jdbc.OracleDriver ]" );
			return ;
		}

		String user   = args[0] ;
		String passwd = args[1] ;
		String url    = args[2] ;
		String table  = args[3] ;
		String file   = args[4] ;
		String encode = ( args.length == 6 ) ? args[5] : "UTF-8"  ;
		String driver = ( args.length == 7 ) ? args[6] : "oracle.jdbc.OracleDriver"  ;

		Class.forName(driver);

		Connection conn = DriverManager.getConnection( url,user,passwd );
		Reader reader = null;
		int insCnt;
		int updCnt;
		int delCnt;
		try {
			conn.setAutoCommit( false );
			conn.setTransactionIsolation(Connection.TRANSACTION_READ_COMMITTED);  // 5.1.1.0 (2009/12/01)
			HybsXMLSave save = new HybsXMLSave( conn,table );

			reader = new BufferedReader(new InputStreamReader(
								new FileInputStream( file ) ,encode ));
			save.insertXML( reader );
			insCnt = save.getInsertCount();
			updCnt = save.getUpdateCount();
			delCnt = save.getDeleteCount();

			Closer.commit( conn );
		}
		// FileNotFoundException , UnsupportedEncodingException
		catch( java.io.FileNotFoundException ex ) {
			String errMsg = "ファイルが存在しません。" + ex.getMessage()
							+ CR + "Table=[" + table + "] File =[" + file + "]" ;
			throw new RuntimeException( errMsg,ex );
		}
		catch( java.io.UnsupportedEncodingException ex ) {
			String errMsg = "指定のエンコードが存在しません。" + ex.getMessage()
							+ CR + "Table=[" + table + "] Encode =[" + encode + "]" ;
			throw new RuntimeException( errMsg,ex );
		}
		finally {
			Closer.ioClose( reader );
			Closer.connClose( conn );
		}

		System.out.println( "XML File[" + file + "] Into [" + table + "] Table" );
		System.out.println( "   Delete Count : [" + delCnt + "]" );
		System.out.println( "   Insert Count : [" + insCnt + "]" );
		System.out.println( "   Update Count : [" + updCnt + "]" );
	}
}
