/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.table;

import org.opengion.hayabusa.common.ErrorMessage;
import org.opengion.hayabusa.db.DBTableModel;

import org.opengion.fukurou.util.StringUtil;

/**
 * <p>TableFilter_DTYPE は、TableFilter インターフェースを継承した、DBTableModel 処理用の
 * 実装クラスです。<br />
 *
 * ここでは、キーの CLS_NAME,USE_LENGTH より、DTYPE の値を設定します。
 *  CLS_NAME は、VARCHAR2, NUMBER などのカラムの属性を表します。
 *  USE_LENGTH は、長さ(使用桁数)です。
 *  DTYPE は、X(10) や、 S9(8) などの簡易型カラム属性です。
 * エンジンを使用したシステムでは、この属性より、さらに詳細にカラムを定義する、
 * DBTYPE 属性が、あります。将来的には、この定義を使用するように切り替えていく予定です。
 *
 * CLS_NAME,USE_LENGTH,DTYPE の カラム名については、初期値はこのままですが、
 * keys , vals に指定すれば、別名についても使用することが可能です。
 *
 * @version  0.9.0  2000/10/17
 * @author   Kazuhiko Hasegawa
 * @since    JDK1.1,
 */
public class TableFilter_DTYPE extends AbstractTableFilter {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "4.0.0 (2005/08/31)" ;

	/**
	 * DBTableModel処理を実行します。<br />
	 *
	 * @return DBTableModel
	 */
	public DBTableModel execute() {

		String cls = StringUtil.nval( getValue( "CLS_NAME" )	,"CLS_NAME" );
		String len = StringUtil.nval( getValue( "USE_LENGTH" )	,"USE_LENGTH" );
		String typ = StringUtil.nval( getValue( "DTYPE" )		,"DTYPE" );

		int clsNo  = table.getColumnNo( cls,false );	// 存在しない場合は、-1 を返す。
		int lenNo  = table.getColumnNo( len,false );
		int typNo  = table.getColumnNo( typ,false );

		if( clsNo >= 0 && lenNo >= 0 && typNo >= 0 ) {
			String[] data  = null;
			int rowCnt = table.getRowCount();
			String clsVal = null;
			String lenVal = null;
			for( int row=0; row<rowCnt; row++ ) {
				try {
					data   = table.getValues( row );
					clsVal = data[clsNo];		// CLS_NAME(VARCHAR,NUMBER など)
					lenVal = data[lenNo];		// USE_LENGTH(使用桁数)

					StringBuilder typVal = new StringBuilder();
					if( "NUMBER".equalsIgnoreCase( clsVal ) ) {
						typVal.append( "S9(" );
					}
					else {
						typVal.append( "X(" );
					}

					int cm = lenVal.indexOf( ".0" );
					if( cm > 0 ) {
						typVal.append( lenVal.substring( 0,cm ) );
					}
					else {
						typVal.append( lenVal );
					}
					typVal.append( ')' );

					table.setValueAt( typVal.toString(),row,typNo );
				}
				catch( RuntimeException ex ) {
					ErrorMessage errMessage = makeErrorMessage( "TableFilter_DTYPE Error",ErrorMessage.NG );
					errMessage.addMessage( row+1,ErrorMessage.NG,ex.getMessage() );
					errMessage.addMessage( row+1,ErrorMessage.NG,StringUtil.array2csv( data ) );
					errMessage.addMessage( row+1,ErrorMessage.NG,"CLS=[" + clsVal + "],LEN=[" + lenVal + "]" );
				}
			}
		}

		return table;
	}
}
