/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.query;

import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.db.DBConstValue;

/**
 * データベース登録時に、固定値カラムの設定を行う情報を管理するクラスです。
 * このクラスは、MISシステム共通の管理クラスです。
 * 具体的には、
 *     更新フラグ  FGJ
 *     作成日時    DYSET
 *     作成ＰＧ    PGSET
 *     作成親ＰＧ  PGPSET
 *     作成社員    USRSET
 *     更新日時    DYUPD
 *     更新ＰＧ    PGUPD
 *     更新親ＰＧ  PGPUPD
 *     更新社員    USRUPD
 * などの情報を、INSERT、UPDATE の状況に応じて提供します。
 * INSERT には、すべての情報(XXUPD)にも値をセットします。
 * これは、UPDATE された日付や時刻で、一覧検索できるようにする為で、
 * null にしておくと、IS NULL で問い合わせる必要があります。
 * INSERT データだけ取出す場合は、WHERE DYSET = DYUPD とします。
 *
 * コンストラクターでは、テーブル名、ユーザーID,画面ID を受け取ります。
 * 登録時刻は、このオブジェクトが生成された時間を通常は使用します。
 * テーブル名を渡すのは、ワークなど、テーブル単位に処理するカラムが
 * 異なるので、それへの対応です。
 * 処理が簡素化できるように、ルールを規定すべきです。
 *
 * @og.rev 4.3.7.5 (2009/07/08) 新規追加
 * @og.group データ登録
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class DBConstValue_UA implements DBConstValue {
	private static final String[] MSTR_INSKEYS = 
		new String[] { "FGJ","DYSET","DYUPD","USRSET","USRUPD","PGSET","PGUPD","PGPSET","PGPUPD" };

	private static final String[] MSTR_UPDKEYS = 
		new String[] { "FGJ","DYUPD","USRUPD","PGUPD","PGPUPD" };

	private static final String[] MSTR_LOGDELKEYS = // 4.3.7.0 (2009/06/01)
		new String[] { "FGJ","DYUPD","USRUPD","PGUPD","PGPUPD" };

	private static final String[] ZERO = new String[0] ;

	private String[] insKeys = ZERO;
	private String[] insVals = ZERO;

	private String[] updKeys = ZERO;
	private String[] updVals = ZERO;
	
	private String[] logDelKeys = ZERO; // 4.3.7.0 (2009/06/01)
	private String[] logDelVals = ZERO; // 4.3.7.0 (2009/06/01)

	/**
	 * 初期化情報を設定します。
	 *
	 * @param  table String テーブル名
	 * @param  user  String ユーザーID
	 * @param  gamen String 画面ID
	 */
	public void init( final String table, final String user, final String gamen ) {
		if( isUseClm( table ) ) {
			String userID  = "'" + user + "'";
			String gamenID = "'" + gamen + "'";
			String timestamp = "'" + HybsSystem.getDate( "yyyyMMddHHmmss" ) + "'" ;

			insKeys = MSTR_INSKEYS;
			insVals = new String[] {
							 "'1'"			// FGJ
							,timestamp		// DYSET
							,timestamp		// DYUPD
							,userID			// USRSET
							,userID			// USRUPD
							,gamenID		// PGSET
							,gamenID		// PGUPD
							,gamenID		// PGPSET
							,gamenID		// PGPUPD
			};

			updKeys = MSTR_UPDKEYS;
			updVals = new String[] {
							 "'1'"			// FGJ
							,timestamp		// DYUPD
							,userID			// USRUPD
							,gamenID		// PGUPD
							,gamenID		// PGPUPD
			};

			// 4.3.7.0 (2009/06/01)
			logDelKeys = MSTR_LOGDELKEYS;
			logDelVals = new String[] {
							 "'9'"			// FGJ
							,timestamp		// DYUPD
							,userID			// USRUPD
							,gamenID		// PGUPD
							,gamenID		// PGPUPD
			};
		}
	}

	/**
	 * INSERT する場合のキー配列を返します。
	 *
	 * 指定の値がなければ、長さゼロ配列を返します。
	 *
	 * @og.rev 4.3.1.1 (2008/08/23) 変更可能なオブジェクトの参照を直接返さないようにします。
	 *
	 * @return  String[]  (非null)
	 */
	public String[] getInsertKeys() {
//		return insKeys ;
		return insKeys.clone();
	}

	/**
	 * INSERT する場合の値配列を返します。
	 *
	 * 指定の値がなければ、長さゼロ配列を返します。
	 *
	 * @og.rev 4.3.1.1 (2008/08/23) 変更可能なオブジェクトの参照を直接返さないようにします。
	 *
	 * @return  String[] (非null)
	 */
	public String[] getInsertVals() {
//		return insVals ;
		return insVals.clone();
	}

	/**
	 * UPDATE する場合のキー配列を返します。
	 *
	 * 指定の値がなければ、長さゼロ配列を返します。
	 *
	 * @og.rev 4.3.1.1 (2008/08/23) 変更可能なオブジェクトの参照を直接返さないようにします。
	 *
	 * @return  String[] (非null)
	 */
	public String[] getUpdateKeys() {
//		return updKeys ;
		return updKeys.clone();
	}

	/**
	 * UPDATE する場合の値配列を返します。
	 *
	 * 指定の値がなければ、長さゼロ配列を返します。
	 *
	 * @og.rev 4.3.1.1 (2008/08/23) 変更可能なオブジェクトの参照を直接返さないようにします。
	 *
	 * @return  String[] (非null)
	 */
	public String[] getUpdateVals() {
//		return updVals ;
		return updVals.clone();
	}


	/**
	 * DELETE(論理削除) する場合のキー配列を返します。
	 *
	 * 指定の値がなければ、長さゼロ配列を返します。
	 *
	 * @og.rev 4.3.7.0 (2009/06/01) 新規作成
	 *
	 * @return  String[] (非null)
	 */
	public String[] getLogicalDeleteKeys() {
		return logDelKeys.clone();
	}

	/**
	 * DELETE(論理削除) する場合の値配列を返します。
	 *
	 * 指定の値がなければ、長さゼロ配列を返します。
	 *
	 * @og.rev 4.3.7.0 (2009/06/01) 新規作成
	 *
	 * @return  String[] (非null)
	 */
	public String[] getLogicalDeleteVals() {
//		return updVals ;
		return logDelVals.clone();
	}

	/**
	 * カラムを返すかどうかを決めるロジック
	 *
	 * テーブル名より、ロジック的に導き出します。
	 *
	 * @return  boolean (true:カラムを生成/false:カラムは生成しない)
	 */
	private boolean isUseClm( final String table ) {
		boolean flag = ( table != null &&
						 table.startsWith( "UA" ) ) ;

		return flag ;
	}
}
