/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.hayabusa.common.ErrorMessage;
import org.opengion.hayabusa.db.DBErrMsg;
import org.opengion.hayabusa.resource.GUIInfo;
import org.opengion.hayabusa.resource.ResourceManager;
import org.opengion.fukurou.db.ConnectionFactory;
import org.opengion.fukurou.util.FileUtil;
import org.opengion.fukurou.util.StringUtil;
import org.opengion.fukurou.util.Closer ;
import static org.opengion.fukurou.util.StringUtil.nval ;

import java.util.Locale ;

import java.sql.Connection;
import java.sql.Statement;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.sql.Types;
import java.sql.CallableStatement;
import oracle.sql.ARRAY;
import oracle.sql.ArrayDescriptor;
import oracle.jdbc.OracleTypes;
import oracle.jdbc.OracleCallableStatement;

import java.io.File;
import java.io.PrintWriter;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.ObjectOutputStream;
import java.io.ObjectInputStream;
import java.util.zip.ZipOutputStream;
import java.util.zip.ZipEntry;

import java.util.Map;

/**
 * SELECT文を直接実行して、指定のファイルに出力するタグです。
 *
 * 中間の、データ（DBTableModel)を作成しないため、余計なメモリを取らず、
 * 高速にデータを抜き出すことが可能です。
 * 一方、抜き出すデータは生データのため、データの再利用等、システム的な
 * 使用を想定しています。
 * JDBCErrMsg 形式のPL/SQL をコールして、その検索結果(カーソル)を抜きこともできます。
 *
 * @og.formSample
 * ●形式：&lt;og:directWriteTable filename="[･･･]" ･･･ &gt;SELECT * FROM ZYXX &lt;/og:directWriteTable &gt;
 * ●body：あり
 *
 * ●使用例
 *     &lt;og:directWriteTable
 *         dbid        = "ORCL"               接続データベースID(初期値：DEFAULT)
 *         separator   = ","                  ファイルの区切り文字（初期値：タブ）
 *         fileURL     = "{&#064;USER.ID}"    保存先ディレクトリ名
 *         filename    = "{&#064;filename}"   保存ファイル名
 *         encode      = "UnicodeLittle"      保存ファイルエンコード名
 *         useHeader   = "true"               保存ファイルにヘッダーを出力するかどうか
 *         zip         = "true"               ZIPファイルに圧縮するかどうか
 *         zipFilename = "Sample.zip"         ZIPファイルのファイル名
 *         fileAppend  = "true"               ファイルを追加モードで登録するかどうか
 *         displayMsg  = "MSG0033"            実行後の表示メッセージ
 *         fetchSize   = "200"                DB検索する場合のフェッチするサイズ
 *     &gt;
 *        &lt;jsp:text&gt; SELECT * FROM ZYXX &lt;/jsp:text&gt;
 *     &lt;/og:directWriteTable &gt;
 *
 *     &lt;og:directWriteTable
 *         fileURL     = "{&#064;USER.ID}"    保存先ディレクトリ名
 *         filename    = "{&#064;filename}"   保存ファイル名
 *         names       = "AAA,BBB,CCC,･･･"    指定のキーに対応するリクエスト値を ARG_ARRAY にセットします。
 *         queryType   = "JDBCErrMsg"         JDBCErrMsg 形式のPL/SQL をコールします。
 *     &gt;
 *        &lt;jsp:text &gt; { call PL/SQL(?,?,?,? ) } &lt;/jsp:text&gt;
 *     &lt;/og:directWriteTable &gt;
 *
 * @og.rev 3.5.6.0 (2004/06/18) 新規作成
 * @og.rev 3.6.1.0 (2005/01/05) PL/SQLコール(JDBCErrMsg 形式)の実行を追加
 * @og.group ファイル出力
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class DirectWriteTableTag extends CommonTagSupport {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "4.0.0 (2005/08/31)" ;

	private static final long serialVersionUID = 4000 ;	// 4.0.0 (2005/01/31)

	private static final String TAB_SEPARATOR	= "\t" ;
	private static final String errMsgId		= HybsSystem.ERR_MSG_KEY;

	private final int DB_MAX_QUERY_TIMEOUT		= HybsSystem.sysInt( "DB_MAX_QUERY_TIMEOUT" ) ;
//	private static final String ARG_ARRAY		= HybsSystem.sys( "ARG_ARRAY" ) ;
//	private static final String ERR_MSG			= HybsSystem.sys( "ERR_MSG" ) ;
//	private static final String ERR_MSG_ARRAY	= HybsSystem.sys( "ERR_MSG_ARRAY" ) ;
	private static final String ARG_ARRAY		= "ARG_ARRAY" ;
	private static final String ERR_MSG			= "ERR_MSG" ;
	private static final String ERR_MSG_ARRAY	= "ERR_MSG_ARRAY" ;

	// 4.0.0.0 (2007/10/10) dbid の初期値を、"DEFAULT" から null に変更
//	private String	dbid		= "DEFAULT";
	private String	dbid		= null;
	private String  separator	= TAB_SEPARATOR;   // 項目区切り文字
	private boolean useHeader	= true;		// ヘッダーの使用可否
	private String  fileURL		= HybsSystem.sys( "FILE_URL" );
	private String  filename	= HybsSystem.sys( "FILE_FILENAME" );   // ファイル名
	private String  zipFilename	= null;		// ZIPファイル名
	private String	sql			= null;
	private String  encode		= HybsSystem.sys( "FILE_ENCODE"   );   // ファイルエンコーディング  "DEFAULT","JISAutoDetect" ,"JIS", "EUC_JP", "MS932", "SJIS" , "Windows-31J" , "Shift_JIS"
	private boolean fileAppend	= false;	// ファイルをAPPENDモードで出力するか
	private boolean zip			= false;	// ファイルをZIPするか
//	private String	displayMsg	= "MSG0033";	// 　件検索しました。
	private String	displayMsg	= HybsSystem.sys( "VIEW_DISPLAY_MSG" );
	private String	notfoundMsg	= "MSG0077";	// 対象データはありませんでした。
	private long	dyStart		= 0;	// 実行時間測定用のDIV要素を出力します。
	private int		fetchSize	= 100 ;	// フェッチする行数(初期値:100)

	// 3.6.1.0 (2005/01/05) PL/SQLコール(JDBCErrMsg 形式)への対応
	private boolean	queryType	= true;	// ノーマルは、true/ JDBCErrMsg の時は、false
	private String	names		= null;	// 指定のリクエスト変数を、ARG_ARRAY にセットします。
	private int		errCode		= ErrorMessage.OK;
	private transient ErrorMessage errMessage = null;

	/**
	 * Taglibの開始タグが見つかったときに処理する doStartTag() を オーバーライドします。
	 *
	 * @return	int
	 */
	@Override
	public int doStartTag() {
		dyStart = System.currentTimeMillis();		// 時間測定用
		return( EVAL_BODY_BUFFERED );	// Body を評価する。（ extends BodyTagSupport 時）
	}

	/**
	 * Taglibのタグ本体を処理する doAfterBody() を オーバーライドします。
	 *
	 * @og.rev 3.8.6.3 (2006/11/30) SQL 文の前後のスペースを取り除きます。
	 *
	 * @return	int
	 */
	@Override
	public int doAfterBody() {
		sql = getBodyString();
		if( sql == null || sql.length() == 0 ) {
			String errMsg = "BODY 部の検索用 Select文は、必須です。";
			throw new HybsSystemException( errMsg );
		}
		sql = sql.trim();
		return(SKIP_BODY);				// Body を評価しない
	}

	/**
	 * Taglibの終了タグが見つかったときに処理する doEndTag() を オーバーライドします。
	 *
	 * @og.rev 3.6.1.0 (2005/01/05) PL/SQLコール(JDBCErrMsg 形式)への対応
	 * @og.rev 4.0.0.0 (2007/10/18) メッセージリソース統合( getResource().getMessage > getResource().getLabel )
	 *
	 * @return  int
	 */
	@Override
	public int doEndTag() {
		debugPrint();		// 4.0.0 (2005/02/28)

		PrintWriter pw = null;
		final int executeCount;
		try {
			if( zip ) {
				String directory = HybsSystem.url2dir( fileURL );

				if( zipFilename == null ) { zipFilename = filename + ".zip"; }
				ZipOutputStream gzip = null;
				try {
					gzip = new ZipOutputStream(
								new FileOutputStream(
										StringUtil.urlAppend( directory,zipFilename )));
					gzip.putNextEntry( new ZipEntry( filename ) );
					pw = new PrintWriter( gzip );
					executeCount = create( pw ) ;

					pw.flush();
					gzip.closeEntry();
					gzip.finish() ;
				}
				finally {
					Closer.ioClose( gzip );		// 4.0.0 (2006/01/31) close 処理時の IOException を無視
				}
			}
			else {
				pw = getPrintWriter();
				executeCount = create( pw );
			}
		} catch( IOException ex ) {
			String errMsg = "Error in DirectWriteTableTag: " + toString();
			throw new HybsSystemException( errMsg,ex );		// 3.5.5.4 (2004/04/15) 引数の並び順変更
		} finally {
			Closer.ioClose( pw );		// 4.0.0 (2006/01/31) close 処理時の IOException を無視
		}

		// 3.6.1.0 (2005/01/05) 検索結果の件数を、"DB.COUNT" キーでリクエストにセットする。
		setRequestAttribute( "DB.COUNT"   , String.valueOf( executeCount ) );
		setRequestAttribute( "DB.ERR_CODE", String.valueOf( errCode ) );

		StringBuilder buf = new StringBuilder( HybsSystem.BUFFER_SMALL );

		// 実行件数の表示
//		boolean	useStatusBar = HybsSystem.sysBool( "VIEW_USE_DISPLAY_MSG" );
//		if( useStatusBar && executeCount > 0 && displayMsg != null && displayMsg.length() > 0 ) {
		if( executeCount > 0 && displayMsg != null && displayMsg.length() > 0 ) {
			buf.append( executeCount );
//			buf.append( getResource().getMessage( displayMsg ) );
			buf.append( getResource().getLabel( displayMsg ) );
			buf.append( HybsSystem.BR );
		}
		else if( executeCount == 0 && notfoundMsg != null && notfoundMsg.length() > 0 ) {
//			buf.append( getResource().getMessage( notfoundMsg ) );
			buf.append( getResource().getLabel( notfoundMsg ) );
			buf.append( HybsSystem.BR );
		}

		// 3.6.1.0 (2005/01/05) TaglibUtil.makeHTMLErrorTable メソッドを利用
		String err = TaglibUtil.makeHTMLErrorTable( errMessage,getResource() );
		if( err != null && err.length() > 0 ) {
			buf.append( err );
			setSessionAttribute( errMsgId,errMessage );
		}
		else {
			removeSessionAttribute( errMsgId );
		}

		jspPrint( buf.toString() );

		// 時間測定用の DIV 要素を出力
		long dyTime = System.currentTimeMillis()-dyStart;
		jspPrint( "<div id=\"queryTime\" value=\"" + (dyTime) + "\"></div>" );	// 3.5.6.3 (2004/07/12)

		// 3.6.1.0 (2005/01/05) 警告時に停止していましたが、継続処理させます。
		int rtnCode = EVAL_PAGE;
		if( errCode >= ErrorMessage.NG )  { 	// 異常
			rtnCode = SKIP_PAGE;
		}

		// 4.0.0 (2005/01/31) セキュリティチェック(データアクセス件数登録)
		GUIInfo guiInfo = (GUIInfo)getSessionAttribute( HybsSystem.GUIINFO_KEY );
		if( guiInfo != null ) { guiInfo.addReadCount( executeCount,dyTime,sql ); }

		return( rtnCode );
	}

	/**
	 * タグリブオブジェクトをリリースします。
	 * キャッシュされて再利用されるので、フィールドの初期設定を行います。
	 *
	 * @og.rev 3.6.1.0 (2005/01/05) PL/SQLコール(JDBCErrMsg 形式)への対応
	 * @og.rev 4.0.0.0 (2007/10/10) dbid の初期値を、"DEFAULT" から null に変更
	 */
	@Override
	protected void release2() {
		super.release2();
		separator	= TAB_SEPARATOR;   // 項目区切り文字
		fileURL		= HybsSystem.sys( "FILE_URL" );
		filename	= HybsSystem.sys( "FILE_FILENAME" );   // ファイル名
		zipFilename	= null;		// ZIPファイル名
		sql			= null;
		encode		= HybsSystem.sys( "FILE_ENCODE" );   // ファイルエンコーディング  "DEFAULT","JISAutoDetect" ,"JIS", "EUC_JP", "MS932", "SJIS" , "Windows-31J" , "Shift_JIS"
		fileAppend	= false;	// ファイルをAPPENDモードで出力するか
		zip			= false;	// ファイルをZIPするか
//		displayMsg	= "MSG0033";	// 　件検索しました。
		displayMsg	= HybsSystem.sys( "VIEW_DISPLAY_MSG" );
		notfoundMsg	= "MSG0077";	// 対象データはありませんでした。
//		dbid		= "DEFAULT";
		dbid		= null;
		fetchSize	= 100 ;	// フェッチする行数(初期値:0 参考にしない)
		dyStart		= 0;
		queryType	= true;	// ノーマルは、true/ JDBCErrMsg の時は、false
		names		= null;	// 指定のリクエスト変数を、ARG_ARRAY にセットします。
		errCode		= ErrorMessage.OK;
		errMessage	= null;
	}

	/**
	 * 実オブジェクトを生成して，<del>PrintWriter</del>OutputStream に書き込みます。
	 *
	 * @og.rev 3.6.1.0 (2005/01/05) PL/SQLコール(JDBCErrMsg 形式)への対応
	 * @og.rev 3.8.6.0 (2006/09/29) ヘッダーにラベルを出力するように修正
	 * @og.rev 3.8.7.0 (2006/12/15) アクセスログ取得の為,ApplicationInfo オブジェクトを設定
	 * @og.rev 4.3.4.3 (2008/12/22) (Oracle11gDriver対応)PL/SQLコールの場合に、"クローズされた文です。"のエラーが発生する問題に対応
	 *
	 * @param   out OutputStream
	 * @return  executeCount 検索件数
	 */
	private int create( final PrintWriter out )  {
		final int executeCount;
		Statement stmt = null;
		CallableStatement callStmt = null; // 4.3.4.3 (2008/12/22)
		ResultSet resultSet = null ;
		boolean errFlag = true;
		Connection conn = null;
		try {
			conn = ConnectionFactory.connection( dbid,getApplicationInfo() );	// 3.8.7.0 (2006/12/15)

			// 3.6.1.0 (2005/01/05)
			if( queryType ) {		// JDBC 通常の SELECT 文
				stmt = conn.createStatement();
				if( fetchSize > 0 ) { stmt.setFetchSize( fetchSize ); }
				resultSet = stmt.executeQuery( sql );
			}
			else {					// PL/SQL Call 文
				String[] values = null;
				if( names != null ) {
					String[] nameArray = StringUtil.csv2Array( names );
					values = getRequest( nameArray );
				}
				callStmt  = conn.prepareCall( sql );
				resultSet = executeCall( conn,callStmt,sql,values ); // 4.3.4.3 (2008/12/22)
			}
			if( resultSet == null ) { return 0; }

			ResultSetMetaData metaData  = resultSet.getMetaData();
			int numberOfColumns =  metaData.getColumnCount();

			// ヘッダー部の出力
			if( useHeader && numberOfColumns > 0 ) {
				StringBuilder headName  = new StringBuilder();
				StringBuilder headLabel = new StringBuilder();
				headName.append( "#Name" );
				headLabel.append( "#Label" );
				String clm ;
				ResourceManager resource = getResource();
				for(int column = 1; column <= numberOfColumns; column++) {
					clm = (metaData.getColumnLabel(column)).toUpperCase(Locale.JAPAN);
					headName.append( TAB_SEPARATOR ).append( clm );
					headLabel.append( TAB_SEPARATOR ).append( resource.getLabel( clm ) );
				}
				out.println( headName.toString() );
				out.println( headLabel.toString() );
			}

			int rowNo = 0;
			Object obj ;
			while( resultSet.next() ) {
				out.print( rowNo );				// 行番号
				for(int column = 1; column <= numberOfColumns; column++) {
					out.print( separator );
					obj = resultSet.getObject(column);
					if( obj != null ) {
						out.print( obj );
					}
				}
				out.println();
				rowNo++ ;
			}
			executeCount = rowNo ;
			errFlag = false;		// エラーではない
		}
		catch ( SQLException ex ) {		// 3.6.1.0 (2005/01/05)
			String errMsg = "データベース処理を実行できませんでした。"
						 + HybsSystem.CR + stmt + HybsSystem.CR
						 + "err=[" + ex.getSQLState() + "]"
						 + ex.getMessage();
			throw new HybsSystemException( errMsg,ex );
		}
		finally {
			Closer.resultClose( resultSet );
			Closer.stmtClose( stmt );
			Closer.stmtClose( callStmt ); // 4.3.4.3 (2008/12/22)
			if( errFlag ) { ConnectionFactory.remove( conn,dbid ); }	// 削除
			else {			ConnectionFactory.close( conn,dbid );  }	// 返却
//				conn = null;
		}

		return executeCount ;
	}

	/**
	 * 引数配列付のクエリーを実行します。
	 * 処理自体は, #execute() と同様に、各サブクラスの実装に依存します。
	 * これは、CallableStatement を用いて、データベース検索処理を行います。
	 * {call TYPE3B01.TYPE3B01(?,?,?,?)} で、４番目の引数には、
	 * names で指定したリクエスト情報が、ARG_ARRAY 配列に順次セットされます。
	 * 使用する場合は、一旦わかり易い変数に受けて利用してください。
	 * 呼び出す PL/SQL では、検索系PL/SQL です。
	 *
	 * @og.rev 3.6.1.0 (2005/01/05) PL/SQLコール(JDBCErrMsg 形式)への対応
	 * @og.rev 4.3.4.3 (2008/12/22) (Oracle11gDriver対応)PL/SQLコールの場合に、"クローズされた文です。"のエラーが発生する問題に対応
	 *
	 * @param   args オブジェクトの引数配列
	 */
	private ResultSet executeCall( final Connection conn,final CallableStatement callStmt,final String sql,final String[] args ) throws SQLException {
//		CallableStatement callStmt = null ; // 4.3.4.3 (2008/12/22)
		ResultSet resultSet = null;
//		try {
//			callStmt  = conn.prepareCall( sql );
			callStmt.setQueryTimeout( DB_MAX_QUERY_TIMEOUT );
			if( fetchSize > 0 ) { callStmt.setFetchSize( fetchSize ); }
			Map<String,Class<?>> map = conn.getTypeMap();
			try {
				map.put( ERR_MSG,Class.forName( "org.opengion.hayabusa.db.DBErrMsg" ) );
			}
			catch( ClassNotFoundException ex ) {
				String errMsg = "org.opengion.hayabusa.db.DBErrMsg クラスが見つかりません。";
				throw new HybsSystemException( errMsg,ex );		// 3.5.5.4 (2004/04/15) 引数の並び順変更
			}

			ArrayDescriptor sd = ArrayDescriptor.createDescriptor( ARG_ARRAY, conn );
			ARRAY newArray = new ARRAY( sd,conn,StringUtil.rTrims( args ) );

			callStmt.registerOutParameter(1, Types.INTEGER);
			callStmt.registerOutParameter(2, OracleTypes.ARRAY,ERR_MSG_ARRAY);
			callStmt.registerOutParameter(3, OracleTypes.CURSOR);
			((OracleCallableStatement)callStmt).setARRAY( 4,newArray );

			callStmt.execute();

			errCode = callStmt.getInt(1);

			if( errCode < ErrorMessage.NG ) {		// 異常以外の場合
				resultSet = ((OracleCallableStatement)callStmt).getCursor(3);
			}
			if( errCode > ErrorMessage.OK ) {		// 正常以外の場合
				ARRAY rtn3 = ((OracleCallableStatement)callStmt).getARRAY(2);
				Object[] rtnval3 = (Object[])rtn3.getArray();
				errMessage = new ErrorMessage( "Query_JDBCErrMsg Error!!" );
				for( int i=0; i<rtnval3.length; i++ ) {
					DBErrMsg er = (DBErrMsg)rtnval3[i];
					if( er == null ) { break; }
					errMessage.addMessage( er.getErrMsg() );
				}
			}
//		}
//		finally {
//			Closer.stmtClose( callStmt );
//			callStmt = null;
//		}
		return resultSet;
	}

	/**
	 * <del>PrintWriter</del> OutputStream を取得します。
	 *
	 * ここでは、一般的なファイル出力を考慮した <del>PrintWriter</del>OutputStream を作成します。
	 *
	 * @og.rev 3.7.1.1 (2005/05/23) フォルダがない場合は、複数階層分のフォルダを自動で作成します。
	 * @og.rev 3.8.0.0 (2005/06/07) FileUtil#getPrintWriter を利用。
	 *
	 * @return InputStream
	 */
	private PrintWriter getPrintWriter() {
		if( filename == null ) {
			String errMsg = "ファイル名がセットされていません。";
			throw new HybsSystemException( errMsg );
		}
		String directory = HybsSystem.url2dir( fileURL );

		// ※ 注意 StringUtil.urlAppend を組み込んでいる意図が不明。一旦削除していますが、注意
	 	// 3.8.0.0 (2005/06/07) FileUtil#getPrintWriter を利用。
	//	out = FileUtil.getPrintWriter( StringUtil.urlAppend( directory,filename ),fileAppend,encode);

//		処理を簡素化します。
//		PrintWriter out = FileUtil.getPrintWriter( new File( directory,filename ),encode,fileAppend );
//		return out ;
		return FileUtil.getPrintWriter( new File( directory,filename ),encode,fileAppend );
	}

	/**
	 * 名称配列を元に、リクエスト情報のデータを取得します。
	 *
	 * @og.rev 3.6.1.0 (2005/01/05) PL/SQLコール(JDBCErrMsg 形式)への対応
	 *
	 * @param	nameArray String[] キーとなる名称の配列
	 * @return	String[] そのリクエスト情報
	 */
	private String[] getRequest( final String[] nameArray ) {
		String[] rtn = new String[nameArray.length];

		for( int i=0; i<rtn.length; i++ ) {
			rtn[i] = getRequestValue( nameArray[i] );
		}

		return rtn;
	}

	/**
	 * 【TAG】(通常は使いません)検索時のDB接続IDを指定します(初期値:DEFAULT)。
	 *
	 * @og.tag
	 *   検索時のDB接続IDを指定します。初期値は、DEFAULT です。
	 *
	 * @param	id データベース接続ID
	 */
	public void setDbid( final String id ) {
		dbid = nval( getRequestParameter( id ),dbid );
	}

	/**
	 * 【TAG】可変長ファイルを作成するときの項目区切り文字をセットします(初期値:TAB_SEPARATOR)。
	 *
	 * @og.tag 可変長ファイルを作成するときの項目区切り文字をセットします。
	 *
	 * @param   sep 項目区切り文字
	 */
	public void setSeparator( final String sep ) {
		separator = nval( getRequestParameter( sep ),TAB_SEPARATOR );
	}

	/**
	 * 【TAG】保存先ディレクトリ名を指定します(初期値:システムパラメータのFILE_URL)。
	 *
	 * @og.tag
	 * この属性で指定されるディレクトリに、ファイルをセーブします。
	 * 指定方法は、通常の fileURL 属性と同様に、先頭が、'/' （UNIX) または、２文字目が、
	 * ":" （Windows）の場合は、指定のURLそのままのディレクトリに、そうでない場合は、
	 * システムパラメータ の FILE_URL 属性で指定のフォルダの下に、作成されます。
	 * fileURL = "{&#064;USER.ID}" と指定すると、FILE_URL 属性で指定のフォルダの下に、
	 * さらに、各個人ID別のフォルダを作成して、そこにセーブします。
	 *
	 * @og.rev 3.5.4.3 (2004/01/05) 内部処理を、makeFileURL に移動。
	 * @og.rev 4.0.0 (2005/01/31) StringUtil.urlAppend メソッドの利用
	 * @og.rev 4.0.0.0 (2007/11/20) 指定されたディレクトリ名の最後が"\"or"/"で終わっていない場合に、"/"を付加する。
	 *
	 * @param	url 保存先ディレクトリ名
	 */
	public void setFileURL( final String url ) {
		String furl = nval( getRequestParameter( url ),null );
		if( furl != null ) {
			char ch = furl.charAt( furl.length()-1 );
			if( ch != '/' && ch != '\\' ) { furl = furl + "/"; }
			fileURL = StringUtil.urlAppend( fileURL,furl );
		}
	}

	/**
	 * 【TAG】ファイルを作成するときのファイル名をセットします(初期値:システムパラメータのFILE_FILENAME)。
	 *
	 * @og.tag ファイルを作成するときのファイル名をセットします。
	 *
	 * @param   fname ファイル名
	 */
	public void setFilename( final String fname ) {
		filename = nval( getRequestParameter( fname ),filename );
	}

	/**
	 * 【TAG】ZIPファイルを作成するときのZIPファイル名をセットします(初期値:filename + ".zip")。
	 *
	 * @og.tag
	 * zip 属性に、true を指定した場合に、ZIPファイル化します。その場合のファイル名を指定します。
	 * なにも指定しない場合は、filename + ".zip" になります。
	 *
	 * @param   zipFile ZIPファイル名
	 * @see #setZip( String )
	 */
	public void setZipFilename( final String zipFile ) {
		zipFilename = nval( getRequestParameter( zipFile ),zipFilename );
	}

	/**
	 * 【TAG】ファイルを作成するときのファイルエンコーディング名をセットします(初期値:システムパラメータのFILE_ENCODE)。
	 *
	 * @og.tag
	 * "DEFAULT","JISAutoDetect" ,"JIS", "EUC_JP", "MS932", "SJIS" , "Windows-31J" , "Shift_JIS"
	 *
	 * @og.rev 2.2.0.0 (2002/12/17) 中国語（国際化）対応 エンコードの取得方法変更
	 * @og.rev 3.1.3.0 (2003/04/10) FILE_ENCODE から、エンコード情報を取得する。
	 *
	 * @param   enc ファイルエンコーディング名
	 * @see     <a href="http://www.iana.org/assignments/character-sets">IANA Charset Registry</a>
	 */
	public void setEncode( final String enc ) {
		encode = nval( getRequestParameter( enc ),encode );
	}

	/**
	 * 【TAG】ヘッダーを書き込むかどうかを指定します(初期値:true)。
	 *
	 * @og.tag
	 *  #Name ････ ヘッダーの書き込みを指定します。
	 * 通常は、書き込み（true)にしておき、使用側でコメントと解釈するように
	 * 処理を行うべきです。コメントのため、append モードで途中に現れても
	 * 無視できます。また、エンジン標準でデータを取り込む場合に、データの配置が
	 * 変更されても取り込みプログラムはそのまま使用できます。
	 * 初期値は、true(書き込む）です。
	 *
	 * @param   flag ヘッダーを書き込むかどうか（true:書き込む／false:書き込まない)
	 */
	public void setUseHeader( final String flag ) {
		useHeader = nval( getRequestParameter( flag ),useHeader );
	}

	/**
	 * 【TAG】追加モードで書き込むかどうか(true/false)を指定します(初期値:false[通常モード])。
	 *
	 * @og.tag
	 * ファイルを書き込む場合、追加モードで書き込むかどうかをセットします。
	 * 新規モード(true)の場合、既存のファイルが存在し、かつ書き込み許可があれば、
	 * 上書きで新規に作成します。
	 * 初期値は、false(新規モード)です。
	 *
	 * @param   flag 追加モード（true)／新規モード（false)
	 */
	public void setFileAppend( final String flag ) {
		fileAppend = nval( getRequestParameter( flag ),fileAppend );
	}

	/**
	 * 【TAG】結果をファイルに出力するときに、ZIPで圧縮するかどうか(true/false)を指定します(初期値:false)。
	 *
	 * @og.tag
	 * 大量に抜き出す場合、そのまま、サーバーから取り出すだけでも大変です。
	 * zip 属性を、true にすると、GZIP で圧縮したファイルを作成します。
	 * 初期値は、false(圧縮しない)です。
	 *
	 * @param  flag ZIPで圧縮する（true)／ しない（その他)
	 * @see    #setZipFilename( String )
	 */
	public void setZip( final String flag ) {
		zip = nval( getRequestParameter( flag ),zip );
	}

	/**
	 * 【TAG】検索結果を画面上に表示するメッセージリソースIDを指定します(初期値:MSG0033[　件検索しました])。
	 *
	 * @og.tag
	 * ここでは、検索結果の件数や登録された件数をまず出力し、
	 * その次に、ここで指定したメッセージをリソースから取得して
	 * 表示します。
	 * 表示させたくない場合は, displayMsg = "" をセットしてください。
	 * 初期値は、検索件数を表示します。
	 *
	 * @param	id ディスプレイに表示させるメッセージ ID
	 */
	public void setDisplayMsg( final String id ) {
		String ids = getRequestParameter( id );
		if( ids != null ) { displayMsg = ids; }
	}

	/**
	 * 【TAG】検索結果がゼロ件の場合に表示するメッセージリソースIDを指定します(初期値:MSG0077[対象データはありませんでした])。
	 *
	 * @og.tag
	 * ここでは、検索結果がゼロ件の場合のみ、特別なメッセージを表示させます。
	 * 従来は、displayMsg と兼用で、『0　件検索しました』という表示でしたが、
	 * displayMsg の初期表示は、OFF になりましたので、ゼロ件の場合のみ別に表示させます。
	 * 表示させたくない場合は, notfoundMsg = "" をセットしてください。
	 * 初期値は、MSG0077[対象データはありませんでした]です。
	 *
	 * @param	id ディスプレイに表示させるメッセージ ID
	 */
	public void setNotfoundMsg( final String id ) {
		String ids = getRequestParameter( id );
		if( ids != null ) { notfoundMsg = ids; }
	}

	/**
	 * 【TAG】(通常は使いません)データのフェッチサイズを指定します(初期値:100)。
	 *
	 * @og.tag
	 * より多くの行が必要なときに、データベースから取り出す必要がある行数に
	 * ついてのヒントを JDBC ドライバに提供します。
	 * 指定された行数は、この Statement を使って作成された結果セットにだけ影響します。
	 * 指定された値が 0 の場合、ヒントは無視されます。
	 * 初期値は、100 です。
	 *
	 * @param	size フェッチする行数(初期値:100)
	 */
	public void setFetchSize( final String size ) {
		fetchSize = nval( getRequestParameter( size ),fetchSize );
	}

	/**
	 * 【TAG】PL/SQLを利用する場合の引数にセットすべき データの名称をCSV形式で複数指定します。
	 *
	 * @og.tag
	 * 複数ある場合は、カンマ区切り文字で渡します。
	 * PL/SQL を使用しない場合は、無視されます。
	 *
	 * @og.rev 3.6.1.0 (2005/01/05) PL/SQLコール(JDBCErrMsg 形式)への対応
	 *
	 * @param	nm 引数の名称（複数ある場合は、カンマ区切り文字）
	 */
	public void setNames( final String nm ) {
		names = nval( getRequestParameter( nm ),names );
	}

	/**
	 * 【TAG】Query を発行する為のクラスID(JDBC,JDBCErrMsg)を指定します(初期値:JDBC)。
	 *
	 * @og.tag
	 * ストアドプロシージャ等を実行する場合に、queryType="JDBCErrMsg" を
	 * 指定する必要があります。(それ以外の指定は、初期値の JDBC になります。)
	 * 初期値は、"JDBC" です。
	 *
	 * @og.rev 3.6.1.0 (2005/01/05) PL/SQLコール(JDBCErrMsg 形式)への対応
	 *
	 * @param	id Query を発行する為の実クラス ID
	 */
	public void setQueryType( final String id ) {
		// 内部的には、JDBCErrMsg:false / その他:true で管理しています。
		queryType = ! "JDBCErrMsg".equalsIgnoreCase( getRequestParameter( id ) );
	}

	/**
	 * シリアライズ用のカスタムシリアライズ書き込みメソッド
	 *
	 * @og.rev 4.0.0 (2006/09/31) 新規追加
	 * @serialData
	 *
	 * @param strm ObjectOutputStream
	 */
	private void writeObject( final ObjectOutputStream strm ) throws IOException {
		strm.defaultWriteObject();
	}

	/**
	 * シリアライズ用のカスタムシリアライズ読み込みメソッド
	 *
	 * ここでは、transient 宣言された内部変数の内、初期化が必要なフィールドのみ設定します。
	 *
	 * @og.rev 4.0.0 (2006/09/31) 新規追加
	 * @serialData
	 *
	 * @param strm ObjectInputStream
	 * @see #release2()
	 */
	private void readObject( final ObjectInputStream strm ) throws IOException , ClassNotFoundException {
		strm.defaultReadObject();
	}

	/**
	 * このオブジェクトの文字列表現を返します。
	 * 基本的にデバッグ目的に使用します。
	 *
	 * @return このクラスの文字列表現
	 */
	public String toString() {
		return org.opengion.fukurou.util.ToString.title( this.getClass().getName() )
				.println( "VERSION"		,VERSION	)
				.println( "dbid"		,dbid		)
				.println( "separator"	,separator	)
				.println( "useHeader"	,useHeader	)
				.println( "fileURL"		,fileURL	)
				.println( "filename"	,filename	)
				.println( "zipFilename"	,zipFilename)
				.println( "sql"			,sql		)
				.println( "encode"		,encode		)
				.println( "fileAppend"	,fileAppend	)
				.println( "zip"			,zip		)
				.println( "displayMsg"	,displayMsg	)
				.println( "dyStart"		,dyStart	)
				.println( "fetchSize"	,fetchSize	)
				.println( "queryType"	,queryType	)
				.println( "names"		,names		)
				.println( "errCode"		,errCode	)
				.println( "Other..."	,getAttributes().getAttribute() )
				.fixForm().toString() ;
	}
}
