/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.hayabusa.common.ErrorMessage;
import org.opengion.hayabusa.db.DBColumn;
import org.opengion.hayabusa.db.DBTableModel;
import org.opengion.fukurou.util.StringUtil;

import static org.opengion.fukurou.util.StringUtil.nval ;

import java.util.Enumeration;
import java.util.Arrays;
import java.util.Locale ;

/**
 * 登録データの物理チェック(整合性チェック、nullチェック)を行うタグです。
 *
 * コマンドに対応したチェック方式で、指定のカラムをチェックします。
 * command="NEW" の場合は、columns 引数にCSV形式で指定したカラム名より
 * リクエスト情報を取得して、値をチェックします。
 * 引数チェックでは、"%" や "_" などのあいまい検索時に指定する記号を
 * 含むとエラーになるカラムはチェック対象からはずす必要がある為、
 * チェックすべきカラムを指定する必要があります。
 * command="ENTRY" の場合は、columns 引数に無関係に、全てのリクエストされたカラム
 * の値をチェックします。これは、先の検索時の場合と異なり、ENTRYでは
 * データベースに値を設定する為、無条件にチェックする必要があります。
 * nullCheck は、command に無関係に指定のカラムが null （ゼロ文字列）かどうかを
 * 判定します。
 * maxRowCount は、一覧検索時のチェックされた件数が、指定の値を超えた場合に
 * エラーになります。minRowCount は、同様に、最小選択件数（設定値を含む）の指定です。
 * このタグは、エラー時には、それ以降のJSP画面を評価しません。BODY 部に記述された
 * 値は、エラー時のみ表示され、正常時には、なにも表示されません。これは、エラー時の
 * メッセージや、ユーザーにエラー時のアクションを行ってもらう場合の処理（例えば、
 * 画面を戻る為のボタンなど）を表示させます。
 *
 * command="MODIFY" , "DELETE" 時には、強制的に、minRowCount="1" で処理します。
 *
 * @og.formSample
 * ●形式：&lt;og:columnCheck command = "…" /&gt;
 * ●形式：&lt;og:columnCheck command = "…" &gt;エラー時のみ処理 &lt;/og:columnCheck&gt;
 * ●body：なし/あり
 *
 * ●使用例
 * &lt;og:columnCheck
 *     command = "{&#064;command}"
 *     columns = "AAA,BBB,CCC"  DB定義(DBColumnリソース)で定義した項目(桁数,タイプ等)でチェックします。
 *     maxRowCount = "1"        チェックで選ばれた 最大選択件数（設定値を含む）を指定。
 *     minRowCount = "1"        チェックで選ばれた 最小選択件数（設定値を含む）を指定。
 *     nullCheck = "AAA,CCC"    NULLチェックを実行します。
 * /&gt;
 *
 * [エラー時に、BODY部に記述された内容を出力する。正常時には、このBODY部の記述は出力されません。]
 * &lt;og:columnCheck
 *     command = "{&#064;command}"
 *     columns = "AAA,BBB,CCC"      DB定義(DBColumnリソース)で定義した項目(桁数,タイプ等)でチェックします。
 *     nullCheck = "AAA,CCC"        NULLチェックを実行します。
 * &gt;
 *     &lt;form method=&quot;POST&quot; action=&quot;forward.jsp&quot; target=&quot;RESULT&quot;&gt;
 *         &lt;og:input type=&quot;button&quot; onClick=&quot;history.back()&quot; msg=&quot;MSG0049&quot; accesskey=&quot;R&quot; td=&quot;false&quot; /&gt;
 *     &lt;/form&gt;
 * &lt;/og:columnCheck&gt;
 *
 * mustAnyCheck 属性に、選択必須カラムを指定します。
 *      例：mustAnyCheck="AA|BB|CC"
 *          AA,BB,CC のカラムで選択必須(すべてがnullならエラー)
 *      例：mustAnyCheck="AA|BB|CC,XX|YY|ZZ"
 *          AA,BB,CC のセットと、XX,YY,ZZのセットでそれぞれ選択必須。
 *      例：mustAnyCheck="AA|XX,AA|YY,AA|ZZ"
 *          AA に値があればすべて成立。そうでない場合は、XX と YY と ZZ がすべて必須。
 *      例：mustAnyCheck="AA|BB,BB|CC,AA|CC"
 *          AA,BB,CC の内、どれか２つが必須。AAが成立すればBBかCCが必須。同様に、
 *          BBが成立すれば、AAかCCが必須。
 *
 *  例：query.jsp
 *        &lt;og:column name="AA" mustAny="true" /&gt;
 *        &lt;og:column name="BB" mustAny="true" /&gt;
 *        &lt;og:column name="XX" mustAny="XYZ"  /&gt;
 *        &lt;og:column name="YY" mustAny="XYZ"  /&gt;
 *      result.jsp
 *        &lt;og:columnCheck mustAnyCheck="AA|BB,XX|YY" /&gt;
 *      custom/custom.css
 *          .XYZ { background-color: Green; }
 *
 * @og.group 画面登録
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class ColumnCheckTag extends CommonTagSupport {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "4.0.0 (2005/08/31)" ;

	private static final long serialVersionUID = 4000 ;	// 4.0.0 (2005/01/31)

	/** command 引数に渡す事の出来る コマンド  新規 {@value} */
	public static final String CMD_NEW	= "NEW" ;
	/** command 引数に渡す事の出来る コマンド  再検索 {@value}  */
	public static final String CMD_RENEW = "RENEW" ;
	/** command 引数に渡す事の出来る コマンド  エントリー {@value} */
	public static final String CMD_ENTRY   = "ENTRY" ;
	/** command 引数に渡す事の出来る コマンド リスト  */

	// 4.3.1.1 (2008/08/23) transient 追加
	private transient DBTableModel table		= null;		// 4.1.2.1 (2008/03/13)

	private String		tableId		= HybsSystem.TBL_MDL_KEY;		// 3.5.4.3 (2004/01/05)
	private String		command		= CMD_NEW;	// 無指定時は、チェックを行う。
	private String[]	columns		= null;
	private String[]	nullCheck	= null;
	private String[]	mustAnyCheck= null;		// 3.8.0.9 (2005/10/17)
	private int			maxRowCount	= -1 ;		// 初期値として，無制限を指定
	private int			minRowCount	= -1 ;		// 初期値として，無制限を指定
	private String		checkType	= "AUTO";
	private String		bodyString	= null;		// 3.5.4.2 (2003/12/15)
	private String[]	matchKeys	= null;		// 4.0.0 (2005/11/30)
	private String[]	matchVals	= null;		// 4.0.0 (2005/11/30)
//	private boolean		isViewSimple= false;	// 4.3.3.0 (2008/10/01)
	private boolean		isRealTime	= false;	// 4.3.7.0 (2009/06/01)

	/**
	 * Taglibの開始タグが見つかったときに処理する doStartTag() を オーバーライドします。
	 *
	 * @og.rev 3.5.4.2 (2003/12/15) エラー時に、BODY部に記述された内容を出力する。
	 * @og.rev 5.0.0.2 (2009/09/15) xss対応->チェックする
	 *
	 * @return  int
	 */
	@Override
	public int doStartTag() {
		// 5.0.0.2 (2009/09/15) XSSチェックしない->する
		// useXssCheck( false );
		return ( EVAL_BODY_BUFFERED );		// Body を評価する
	}

	/**
	 * Taglibのタグ本体を処理する doAfterBody() を オーバーライドします。
	 *
	 * @og.rev 3.5.4.2 (2003/12/15) エラー時に、BODY部に記述された内容を出力する。
	 *
	 * @return  int
	 */
	@Override
	public int doAfterBody() {
		bodyString = getBodyString();

		return ( SKIP_BODY );
	}

	/**
	 * Taglibの終了タグが見つかったときに処理する doEndTag() を オーバーライドします。
	 *
	 * @og.rev 3.1.1.2 (2003/04/04) Tomcat4.1 対応。release2() を doEndTag()で呼ぶ。
	 * @og.rev 3.4.0.3 (2003/09/10) command="NEW" でエラー発生時には、DBTableModel をクリアする。
	 * @og.rev 3.5.4.2 (2003/12/15) HTMLTableViewForm クラス名変更（⇒ ViewForm_HTMLTable）
	 * @og.rev 3.5.4.2 (2003/12/15) エラー時に、BODY部に記述された内容を出力する。
	 * @og.rev 3.5.4.3 (2004/01/05) tableId 属性を追加。
	 * @og.rev 3.5.4.4 (2004/01/16) エラー結果を表示するテーブル形式のフォーム修正
	 * @og.rev 3.5.5.2 (2004/04/02) TaglibUtil.makeHTMLErrorTable メソッドを利用
	 * @og.rev 4.1.2.1 (2008/03/13) table 属性を追加。
	 * @og.rev 4.3.3.0 (2008/10/01) viewSimple属性追加
	 * @og.rev 4.3.7.0 (2009/06/01) viewSimple属性名称変更 -> isRealTime
	 *
	 * @return  int
	 */
	@Override
	public int doEndTag() {
		debugPrint();		// 4.0.0 (2005/02/28)
		int rtnCode = EVAL_PAGE;

		// 4.1.2.1 (2008/03/13)
		table = (DBTableModel)getObject( tableId );

		// 3.4.0.3 (2003/09/10) コマンドが、NEW,RENEW 以外の場合は、DBTableModel は必須。
		if( !CMD_NEW.equals( command ) &&
			!CMD_RENEW.equals( command ) &&
//			getObject( tableId ) == null ) {	// 3.5.4.3 (2004/01/05)
			table == null ) {					// 4.1.2.1 (2008/03/13)
				String errMsg = "検索結果のオブジェクトが存在しません。"
								+ " command=[" + command + "]" ;
				jspPrint( errMsg );
				rtnCode = SKIP_PAGE ;
		}
		else {
		// 3.5.5.2 (2004/04/02) TaglibUtil.makeHTMLErrorTable メソッドを利用
			ErrorMessage errMsg = makeErrorDBTable();
			if( errMsg.getKekka() >= ErrorMessage.NG ) {
				if( bodyString != null ) { jspPrint( bodyString ); }	// 3.5.4.2 (2003/12/15)

				// 4.3.3.0 (2008/10/01)
//				jspPrint( TaglibUtil.makeHTMLErrorTable( errMsg,getResource() ) );
				// 4.3.7.0 (2009/06/01)
				if( isRealTime ) { jspPrint( TaglibUtil.makeHTMLSimpleErrorList( errMsg,getResource() ) ); }
				else { jspPrint( TaglibUtil.makeHTMLErrorTable( errMsg, getResource() ) ); } 

				if( CMD_NEW.equals( command ) ) {
					removeObject( tableId );
				}
				rtnCode = SKIP_PAGE ;
			}
			else if( errMsg.getKekka() == ErrorMessage.WARNING ) {
				// 4.3.3.0 (2008/10/01)
//				jspPrint( TaglibUtil.makeHTMLErrorTable( errMsg,getResource() ) );
				// 4.3.7.0 (2009/06/01)
				if( isRealTime ) { jspPrint( TaglibUtil.makeHTMLSimpleErrorList( errMsg,getResource() ) ); }
				else { jspPrint( TaglibUtil.makeHTMLErrorTable( errMsg, getResource() ) ); } 			}
		}

		return( rtnCode );
	}

	/**
	 * タグリブオブジェクトをリリースします。
	 *
	 * キャッシュされて再利用されるので、フィールドの初期設定を行います。
	 *
	 * @og.rev 2.0.0.4 (2002/09/27) カスタムタグの release() メソッドを、追加
	 * @og.rev 3.1.1.2 (2003/04/04) Tomcat4.1 対応。release2() を doEndTag()で呼ぶ。
	 * @og.rev 3.5.4.2 (2003/12/15) エラー時に、BODY部に記述された内容を出力する。
	 * @og.rev 3.5.4.3 (2004/01/05) tableId 属性を追加。
	 * @og.rev 3.8.0.9 (2005/10/17) mustAnyCheck 属性を追加。
	 * @og.rev 4.1.2.1 (2008/03/13) table 属性を追加。
	 * @og.rev 4.3.3.0 (2008/10/01) viewSimple属性追加
	 * @og.rev 4.3.7.0 (2009/06/01) viewSimple属性名称変更 -> isRealTime
	 *
	 */
	@Override
	protected void release2() {
		super.release2();
		command		= CMD_NEW;	// 無指定時は、チェックを行う。
		columns		= null;
		nullCheck	= null;
		maxRowCount	= -1 ;		// 初期値として，無制限を指定
		minRowCount	= -1 ;		// 初期値として，無制限を指定
		checkType	= "AUTO";
		bodyString	= null ;	// 3.5.4.2 (2003/12/15)
		tableId		= HybsSystem.TBL_MDL_KEY;		// 3.5.4.3 (2004/01/05)
		mustAnyCheck= null;		// 3.8.0.9 (2005/10/17)
		matchKeys	= null;		// 4.0.0 (2005/11/30)
		matchVals	= null;		// 4.0.0 (2005/11/30)
		table		= null;		// 4.1.2.1 (2008/03/13)
//		isViewSimple= false;	// 4.3.3.0 (2008/10/01)
		isRealTime	= false;	// 4.3.7.0 (2009/06/01)
	}

	/**
	 * DBColumn オブジェクトを作成して、DBColumn#valueCheck( String )で
	 * チェックを行う。その結果の、ErrorMessage オブジェクトを DBTableModel に
	 * 取り込んで、エラーの場合は、その表示を行う。
	 * 表示方法は、そのまま ViewForm オブジェクトを作成して、表示を行う。
	 *
	 * @og.rev 3.5.5.2 (2004/04/02) リターン値を、DBTableModel から ErrorMessage に変更
	 * @og.rev 4.0.0 (2005/01/31) リターン値の ErrorMessage は、必ず存在する。
	 * @og.rev 4.1.2.1 (2008/03/13) command="MODIFY" , "DELETE" 時には、強制的に、minRowCount="1" で処理します。
	 *
	 * @return  DBTableModel エラーメッセージのテーブルモデル。<del> 無ければ null</del>
	 */
	private ErrorMessage makeErrorDBTable() {
		ErrorMessage errMsg = new ErrorMessage();

		if( "AUTO".equals( checkType ) ) { checkType = command; }
		if( CMD_ENTRY.equals( checkType ) ) {
			errMsg.setTitle( "Request Column Error!" );
			errMsg = makeEntryErrorMessage( errMsg );
			errMsg = makeMaxRowCountErrorMessage( errMsg );
			errMsg = makeMinRowCountErrorMessage( errMsg );
		}
		else if( CMD_NEW.equals( checkType ) ) {
			errMsg.setTitle( "Entry Column Error!" );
			errMsg = makeErrorMessage( errMsg );
			errMsg = makeMaxRowCountErrorMessage( errMsg );
			errMsg = makeMinRowCountErrorMessage( errMsg );
		}
		else if( minRowCount >= 0 || maxRowCount >= 0) {	// -1 は制限無し
			errMsg.setTitle( "Row Count Error Limited Error!" );
			errMsg = makeMaxRowCountErrorMessage( errMsg );
			errMsg = makeMinRowCountErrorMessage( errMsg );
		}
	 	// 4.1.2.1 (2008/03/13) command="MODIFY" , "DELETE" 時には、強制的に、minRowCount="1" で処理します。
		else if( "MODIFY,DELETE".indexOf( checkType ) >= 0 && minRowCount < 0 ) {
			minRowCount = 1;
			errMsg = makeMinRowCountErrorMessage( errMsg );
		}
		return errMsg;
	}

	/**
	 * ErrorMessageをセットします。
	 * 引数のカラム名配列よりリクエスト情報を取得して、値をチェックします。
	 * DBColumn#valueCheck( String ) の結果のErrorMessageをすべて append
	 * していきます。
	 *
	 * @og.rev 3.8.0.9 (2005/10/17) 選択必須 mustAnyCheck のチェック追加
	 * @og.rev 3.8.1.0 (2005/10/24) リクエスト情報の正規化変換(DBColumn#valueSet(String))中止
	 * @og.rev 3.8.5.3 (2006/06/30) リクエストに % , _ が含まれたときは、削除します。
	 * @og.rev 4.1.2.1 (2008/03/13) must , mustAny の自動取得追加
	 * @og.rev 5.0.0.2 (2009/09/15) 個別にxssチェックをfalse
	 *
	 * @param   errMsg ErrorMessage
	 * @return  ErrorMessage カラムキー ＋ 値 のエラーメッセージオブジェクト
	 */
	private ErrorMessage makeErrorMessage( final ErrorMessage errMsg ) {

	 	// 4.1.2.1 (2008/03/13) must , mustAny の自動取得追加
		// request から取出す 注意：mustAny 以外の自由形式の値は所得していません。
		if( nullCheck == null ) {
			nullCheck = getRequestValues( HybsSystem.MUST_KEY + "must" );
			if( nullCheck != null ) { Arrays.sort( nullCheck ); }
		}
		if( mustAnyCheck == null ) {
			String[] mustAnyReq = getRequestValues( HybsSystem.MUST_KEY + "mustAny" );
			if( mustAnyReq != null && mustAnyReq.length > 0 ) {
				mustAnyCheck = new String[] { StringUtil.array2line( mustAnyReq,"|" ) };
			}
		}

		if( nullCheck != null && nullCheck.length != 0 ) {
			for( int i=0; i<nullCheck.length; i++ ) {
				// String val  = getRequestValue( nullCheck[i] );
				String val  = getRequestValue( nullCheck[i], false ); // 5.0.0.2 (2009/09/15)
				if( val == null || val.length() == 0 ) {
					String label = getLabel( nullCheck[i] );	// 4.0.0 (2005/01/31)
					// ERR0012 : 指定のデータがセットされていません。（NULLエラー）。key={0}
					errMsg.addMessage( 0,ErrorMessage.NG,"ERR0012",label );
				}
			}
		}
		// 3.8.0.9 (2005/10/17) 選択必須 mustAnyCheck のチェック追加
		if( mustAnyCheck != null && mustAnyCheck.length != 0 ) {
			for( int i=0; i<mustAnyCheck.length; i++ ) {
				boolean flag = false;
				String[] mustSub = StringUtil.csv2Array( mustAnyCheck[i],'|' );
				for( int j=0; j<mustSub.length; j++ ) {
					// String val  = getRequestValue( mustSub[j] );
					String val  = getRequestValue( mustSub[j], false ); // 5.0.0.2 (2009/09/15)
					if( val != null && val.length() > 0 ) {
						flag = true; break;		// ひとつでもnullでなければ、OK
					}
				}
				if( ! flag ) {
					StringBuilder buf = new StringBuilder();
					for( int j=0; j<mustSub.length; j++ ) {
						buf.append( getResource().getLabel( mustSub[j] ) ).append( "," );
					}
					// ERR0036 : 選択必須エラー。以下のデータの内どれかは入力してください。key={0}
					errMsg.addMessage( 0,ErrorMessage.NG,"ERR0036",buf.toString() );
				}
			}
		}
		if( columns != null && columns.length != 0 ) {
			for( int i=0; i<columns.length; i++ ) {
	 			// 3.8.1.0 (2005/10/24) リクエスト情報の正規化変換(DBColumn#valueSet(String))中止
				// String clmVal = getRequestValue( columns[i] );
				String clmVal = getRequestValue( columns[i], false ); // 5.0.0.2 (2009/09/15)
				if( clmVal != null && clmVal.length() > 0 ) {
					clmVal = StringUtil.replace( clmVal,"%","" );		// 3.8.5.3 (2006/06/30)
					clmVal = StringUtil.replace( clmVal,"_","" );		// 3.8.5.3 (2006/06/30)
					DBColumn dbColumn = getDBColumn( columns[i] );
					errMsg.append( dbColumn.valueCheck( clmVal ) );
				}
			}
		}
		// 4.0.0 (2005/11/30) 正規表現チェックの追加
		if( matchKeys != null && matchKeys.length != 0 ) {
			for( int i=0; i<matchKeys.length; i++ ) {
				// String val  = getRequestValue( matchKeys[i] );
				String val  = getRequestValue( matchKeys[i], false ); // 5.0.0.2 (2009/09/15)
				if( val != null && ! val.matches( matchVals[i] ) ) {
					String label = getLabel( matchKeys[i] );
					// ERR0037 : データ整合性エラー。指定のキーは整合性チェックの結果マッチしませんでした。key={0} val={1} regex={2}
					errMsg.addMessage( 0,ErrorMessage.NG,"ERR0037",label,val,matchVals[i] );
				}
			}
		}

		return errMsg;
	}

	/**
	 * エントリーデータのErrorMessageをセットします。
	 * 引数のカラム名配列よりエントリーデータ形式のリクエスト情報を取得して、
	 * 値をチェックします。
	 * DBColumn#valueCheck( String ) の結果のErrorMessageをすべて append
	 * していきます。
	 *
	 * @og.rev 3.1.0.0 (2003/03/20) 名前と行番号の区切り記号を "^" から "__" に変更。
	 * @og.rev 3.5.5.0 (2004/03/12) 名前と行番号の区切り記号("__")を、HybsSystem.JOINT_STRING  に変更。
	 * @og.rev 3.8.0.9 (2005/10/17) 選択必須 mustAnyCheck のチェック追加
	 * @og.rev 4.1.2.1 (2008/03/13) must , mustAny の自動取得追加
	 * @og.rev 4.3.6.4 (2009/05/01) 削除時に必ずmustAnyチェックエラーになるバグを修正(書込み可能行のみを処理する
	 * @og.rev 4.3.7.0 (2009/06/01) リアルタイムチェックの場合は、must,mustAnyの自動チェックは行わない
	 * @og.rev 5.0.0.2 (2009/09/15) xssチェック
	 *
	 * @param   errMsg ErrorMessage
	 * @return  ErrorMessage カラムキー ＋ 値 のエラーメッセージオブジェクト
	 */
	private ErrorMessage makeEntryErrorMessage( final ErrorMessage errMsg ) {

		int[] rowNo = getParameterRows();
		if( rowNo.length == 0 ) { return errMsg; }

	 	// 4.1.2.1 (2008/03/13) must , mustAny の自動取得追加
		// table から取出す (ソート済み)
		// 4.3.7.0 (2009/06/01) リアルタイムチェックの場合は、must,mustAnyの自動チェックは行わない
		if( !isRealTime ) {
			if( nullCheck	 == null ) { nullCheck	  = table.getMustArray();	 }
			if( mustAnyCheck == null ) { mustAnyCheck = table.getMustAnyArray(); }
		}

		// 3.8.0.9 (2005/10/17) 選択必須 mustAnyCheck のチェック追加
		boolean[][] rowAnyOne = null;
		// 4.3.6.4 (2009/05/01)
		boolean[] rowForCheck = null;
		if( mustAnyCheck != null && mustAnyCheck.length != 0 ) {
			rowAnyOne = new boolean[rowNo.length][mustAnyCheck.length];
			rowForCheck = new boolean[rowNo.length];
	//		for( int i=0; i<rowNo.length; i++ ) {
	//			Arrays.fill( rowAnyOne[i],false );
	//		}
		}

		Enumeration<?> enume = getParameterNames();		// 4.3.3.6 (2008/11/15) Generics警告対応
		while( enume.hasMoreElements() ) {
			String key  = (String)(enume.nextElement());
			int    idx  = key.lastIndexOf(HybsSystem.JOINT_STRING);

			if( idx > 0 ) {
				String column = key.substring(0,idx);
				int    row    = Integer.parseInt( key.substring(idx + 2) );
				// String val    = getRequestValue( key );
				String val    = getRequestValue( key, false ); // 5.0.0.2 (2009/09/15)

				int i = Arrays.binarySearch( rowNo,row );
				if( i >= 0 ) {
					DBColumn dbColumn = getDBColumn( column );
					String val2 = dbColumn.valueSet( val );
					errMsg.append( row+1,dbColumn.valueCheck( val2 ) );
					if( nullCheck != null && nullCheck.length != 0 ) {
						int j = Arrays.binarySearch( nullCheck,column );
						if( j>=0 && ( val2 == null || val2.length() == 0 )) {
							// ERR0012 : 指定のデータがセットされていません。（NULLエラー）。key={0}
							errMsg.addMessage( row+1,ErrorMessage.NG,"ERR0012",dbColumn.getLabel() );
						}
					}
					// 3.8.0.9 (2005/10/17) 選択必須 mustAnyCheck のチェック追加
					if( rowAnyOne != null ) {
						for( int j=0; j<mustAnyCheck.length; j++ ) {
							if( !rowAnyOne[i][j] &&
								mustAnyCheck[j].indexOf( column ) >= 0 &&
								val2 != null && val2.length() > 0 ) {
									rowAnyOne[i][j] = true;	// どれかが存在
							}
						}
						// 4.3.6.4 (2009/05/01)
						rowForCheck[i] = true;
					}

					// 4.0.0 (2005/11/30) 正規表現チェックの追加
					if( matchKeys != null && matchKeys.length != 0 ) {
						for( int j=0; j<matchKeys.length; j++ ) {
							if( column.equals( matchKeys[j] ) ) {
								if( val2 != null && ! val2.matches( matchVals[j] ) ) {
									String label = getLabel( column );
									// ERR0037 : データ整合性エラー。指定のキーは整合性チェックの結果マッチしませんでした。key={0} val={1} regex={2}
									errMsg.addMessage( 0,ErrorMessage.NG,"ERR0037",label,val2,matchVals[i] );
								}
								break;
							}
						}
					}
				}
			}
		}

		// 3.8.0.9 (2005/10/17) 選択必須 mustAnyCheck のチェック追加
		if( rowAnyOne != null ) {
			for( int row=0; row<rowAnyOne.length; row++ ) {
				// 4.3.6.4 (2009/05/01) 書込み可能行のみを処理する
				if ( rowForCheck[row] ) {
					for ( int i = 0; i < mustAnyCheck.length; i++ ) {
						if ( !rowAnyOne[row][i] ) {
							String[] mustSub = StringUtil.csv2Array( mustAnyCheck[i], '|' );

							StringBuilder buf = new StringBuilder();
							buf.append( "row=[" ).append( rowNo[row] + 1 ).append( "] " );
							for ( int j = 0; j < mustSub.length; j++ ) {
								buf.append( getResource().getLabel( mustSub[j] ) ).append( "|" );
							}
							// ERR0036 : 選択必須エラー。以下のデータの内どれかは入力してください。key={0}
							errMsg.addMessage( 0, ErrorMessage.NG, "ERR0036", buf.toString() );
						}
					}
				}
			}
		}

		return errMsg;
	}

	/**
	 * ErrorMessageをセットします。
	 * リクエストされた件数の最大値に制限を加えます。
	 * １件だけにしたい場合は，通常はViewでチェックボックスを使用せずに
	 * ラジオボタンを使用してください。
	 *
	 * @param   errMsg ErrorMessage
	 * @return  ErrorMessage カラムキー ＋ 値 のエラーメッセージオブジェクト
	 */
	private ErrorMessage makeMaxRowCountErrorMessage( final ErrorMessage errMsg ) {
		if( maxRowCount < 0 ) { return errMsg; }	// 無制限

		// 件数制限のチェック
		int[] rowNo = getParameterRows();
		int rowCount = rowNo.length ;
		if( rowCount > maxRowCount ) {
			// ERR0017 : 選択エラー。選択行数（{0} 件）が、最大制限値（{1} 件）以上選ばれました。
			String arg0 = String.valueOf( rowCount );
			String arg1 = String.valueOf( maxRowCount );
			errMsg.addMessage( 0,ErrorMessage.NG,"ERR0017",arg0,arg1 );
		}

		return errMsg;
	}

	/**
	 * ErrorMessageをセットします。
	 * リクエストされた件数の最小値に制限を加えます。
	 * １件だけにしたい場合は，通常はViewでチェックボックスを使用せずに
	 * ラジオボタンを使用してください。
	 *
	 * @param   errMsg ErrorMessage
	 * @return  ErrorMessage カラムキー ＋ 値 のエラーメッセージオブジェクト
	 */
	private ErrorMessage makeMinRowCountErrorMessage( final ErrorMessage errMsg ) {
		if( minRowCount < 0 ) { return errMsg; }	// 無制限

		// 件数制限のチェック
		int[] rowNo = getParameterRows();
		int rowCount = rowNo.length ;
		if( rowCount < minRowCount ) {
			// ERR0018 : 選択エラー。選択行数（{0} 件）が、最小制限値（{1} 件）以下選ばれました。
			String arg0 = String.valueOf( rowCount );
			String arg1 = String.valueOf( minRowCount );
			errMsg.addMessage( 0,ErrorMessage.NG,"ERR0018",arg0,arg1 );
		}

		return errMsg;
	}

	/**
	 * 【TAG】チェックすべきカラム列をカンマ区切りで指定します。
	 *
	 * @og.tag columns="AAA,BBB,CCC,DDD"
	 * <del>先に配列に分解してからリクエスト変数の値を取得するようにします。
	 * こうする事で、リクエストにカンマ区切りの値を設定できるようになります。</del>
	 * 分解方法は、通常のパラメータ取得後に、CSV分解します。
	 *
	 * @og.rev 3.5.6.2 (2004/07/05) 先に配列に分解してからリクエスト変数の値を取得
	 * @og.rev 3.8.8.5 (2007/03/09) 通常のパラメータ取得後に、CSV分解に戻します。
	 *
	 * @param   clms String
	 */
	public void setColumns( final String clms ) {
		columns = StringUtil.csv2Array( getRequestParameter( clms ) );
		if( columns.length == 0 ) { columns = null; }
	}

	/**
	 * 【TAG】NULL チェックすべきカラム列をカンマ区切りで指定します。
	 *
	 * @og.tag
	 * (must 属性のセットにより、自動処理されます)
	 * nullCheck="AAA,BBB,CCC,DDD"
	 * <del>先に配列に分解してからリクエスト変数の値を取得するようにします。
	 * こうする事で、リクエストにカンマ区切りの値を設定できるようになります。</del>
	 * 分解方法は、通常のパラメータ取得後に、CSV分解します。
	 *
	 * @og.rev 3.5.6.2 (2004/07/05) 先に配列に分解してからリクエスト変数の値を取得
	 * @og.rev 3.8.8.5 (2007/03/09) 通常のパラメータ取得後に、CSV分解に戻します。
	 *
	 * @param   clms String
	 */
	public void setNullCheck( final String clms ) {
		nullCheck = StringUtil.csv2Array( getRequestParameter( clms ) );
		if( nullCheck.length == 0 ) { nullCheck = null; }
		else {
			Arrays.sort( nullCheck );
		}
	}

	/**
	 * 【TAG】選択必須カラム(指定のカラムの内最低ひとつがNULLでない)を"AA|BB|CC,XX|YY|ZZ" 形式で指定します。
	 *
	 * @og.tag
	 * (mustAny 属性のセットにより、自動処理されます)
	 * 複数のカラム属性の値のうち、どれかが null でない場合は正常とし、
	 * すべてが null の場合を警告します。
	 *
	 *  例：mustAnyCheck="AA|BB|CC"
	 *      AA,BB,CC のカラムで選択必須(すべてがnullならエラー)
	 *  例：mustAnyCheck="AA|BB|CC,XX|YY|ZZ"
	 *      AA,BB,CC のセットと、XX,YY,ZZのセットでそれぞれ選択必須。
	 *  例：mustAnyCheck="AA|XX,AA|YY,AA|ZZ"
	 *      AA に値があればすべて成立。そうでない場合は、XX と YY と ZZ がすべて必須。
	 *  例：mustAnyCheck="AA|BB,BB|CC,AA|CC"
	 *      AA,BB,CC の内、どれか２つが必須。AAが成立すればBBかCCが必須。同様に、
	 *      BBが成立すれば、AAかCCが必須。
	 *
	 * 選択必須は、must と同様に、色づけを行う(query.jsp)画面では、mustAny 属性を
	 * セットします。チェックを行う(result.jsp)画面では、columnCheck タグの
	 * mustAnyCheck 属性に、選択必須カラムを指定します。(上記参照)
	 * column タグ等の mustAny 属性に、mustAny="true" とセットすると、生成される
	 * HTMLは、class="mustAny" が出力されます。エンジン標準では、default.css に
	 * .mustAny が定義されています。( background-color: #CCFFFF; )
	 * なお、mustAny 属性に、true 以外の値をセットした場合(mustAny="XYZ")は、
	 * 生成されるHTMLに、class="XYZ" が出力されます。これは、複数のグループ間で
	 * 色を変えて、選択必須を指定する場合に使用します。色は、custom/custom.css
	 * で指定します。
	 * 分解方法は、通常のパラメータ取得後に、CSV分解します。
	 *
	 * @og.rev 3.8.0.9 (2005/10/17) 新規追加
	 * @og.rev 3.8.8.5 (2007/03/09) 通常のパラメータ取得後に、CSV分解に戻します。
	 *
	 * @param   clms String
	 */
	public void setMustAnyCheck( final String clms ) {
		mustAnyCheck = StringUtil.csv2Array( getRequestParameter( clms ) );
		if( mustAnyCheck.length == 0 ) { mustAnyCheck = null; }
		else {
			Arrays.sort( mustAnyCheck );
		}
	}

	/**
	 * 【TAG】コマンド(NEW,RENEW,ENTRY)をセットします(初期値:NEW)。
	 *
	 * @og.tag
	 * コマンドは,HTMLから（get/post)指定されますので,CMD_xxx で設定される
	 * フィールド定数値のいづれかを、指定できます。
	 * 何も設定されない、または、null の場合は、"NEW" が初期値にセットされます。
	 *
	 * @param	cmd コマンド（public static final 宣言されている文字列)
	 * @see		<a href="{@docRoot}/constant-values.html#org.opengion.hayabusa.taglib.ColumnCheckTag.CMD_NEW">コマンド定数</a>
	 */
	public void setCommand( final String cmd ) {
		String cmd2 = getRequestParameter( cmd );
		if( cmd2 != null && cmd2.length() > 0 ) { command = cmd2.toUpperCase(Locale.JAPAN); }
	}

	/**
	 * 【TAG】 チェック対象のデータ(AUTO/NEW/ENTRY)を指定します(初期値:AUTO)。
	 *
	 * @og.tag
	 * 通常のリクエストデータは、キーそのものですが,エントリデータは表形式の
	 * データを一括で登録する為、（キー＋"__" ＋ 行番号）形式を、バラす必要があります。
	 *
	 * ENTRY は、DBTableModelのリクエスト情報をチェックします。
	 * これは、全データが対象になります。（columns/nullCheck 属性は無効）
	 * AUTO は、command が、上記 NEW か ENTRY かを判断して処理を振り分けます。
	 * 初期値は、AUTO です。
	 *
	 * @param   flag チェック対象のデータ(AUTO/NEW/ENTRY)
	 */
	public void setCheckType( final String flag ) {
		String ct = getRequestParameter( flag );
		if( ct != null && ct.length() > 0 ) { checkType = ct ; }
	}

	/**
	 * 【TAG】<del>ENTRY時に</del>チェックで選択された行数の最大値を設定します。
	 *
	 * @og.tag
	 * 最大選択件数を超えた場合は,エラーメッセージを返します。
	 * 例えば、１件のみを正常とする場合は、maxRowCount="1" とします。
	 *
	 * @param   count 制限をかけたい行数（この件数と同じ場合は正常 ）
	 */
	public void setMaxRowCount( final String count ) {
		maxRowCount = nval( getRequestParameter( count ),maxRowCount );
	}

	/**
	 * 【TAG】<del>ENTRY時に</del>チェックで選択された行数の最小値を設定します。
	 *
	 * @og.tag
	 * 最小選択件数を超えない場合は、エラーメッセージを返します。
	 * 例えば、１件のみを正常とする場合は、minRowCount="1" とします。
	 *
	 * @param   count 制限をかけたい行数（この件数と同じ場合は正常 ）
	 */
	 public void setMinRowCount( final String count ) {
	 	minRowCount = nval( getRequestParameter( count ), minRowCount );
	 }

	/**
	 * 【TAG】(通常は使いません) sessionに登録されている DBTableModel を取り出すキーを指定します。
	 *
	 * @og.tag
	 * 検索結果のDBTableModelを別メモリーに分ける場合は、このキーを指定します。
	 * その場合、メモリ上から削除する方法がないため(session タイムアウトで破棄)大量の
	 * データを分けた場合は、ご注意ください。
	 * 初期値は、HybsSystem.TBL_MDL_KEY です。
	 *
	 * @og.rev 3.5.4.3 (2004/01/05) 新規追加
	 *
	 * @param	id sessionに登録する時の ID
	 */
	public void setTableId( final String id ) {
		tableId = nval( getRequestParameter( id ),tableId );
	}

	/**
	 * 【TAG】正規表現でのマッチングを行うカラム列をカンマ区切りで指定します。
	 *
	 * @og.tag matchKeys="AAA,BBB,CCC,DDD"
	 * matchKeys属性とmatchVals属性の個数は、同じでなければなりません。
	 * 先に配列に分解してからリクエスト変数の値を取得するようにします。
	 * こうする事で、リクエストにカンマ区切りの値を設定できるようになります。
	 *
	 * @og.rev 4.0.0 (2005/11/30) 新規作成
	 *
	 * @param   keys String
	 * @see     #setMatchVals( String )
	 */
	public void setMatchKeys( final String keys ) {
		matchKeys = getCSVParameter( keys );

		if( matchVals != null && matchKeys.length != matchVals.length ) {
			String errMsg = "matchKeys属性とmatchVals属性の個数が合いません。"
						+ HybsSystem.CR
						+ " matchKeys=[" + matchKeys.length + "]:KEYS="
						+ StringUtil.array2csv( matchKeys ) + HybsSystem.CR
						+ " matchVals=[" + matchVals.length + "]:VLAS="
						+ StringUtil.array2csv( matchVals ) + HybsSystem.CR ;
			throw new HybsSystemException( errMsg );
		}
	}

	/**
	 * 【TAG】正規表現でのマッチングを行うカラム列に対する値(正規表現)をカンマ区切りで指定します。
	 *
	 * @og.tag matchVals="AAA,BBB,CCC,DDD"
	 * matchKeys属性とmatchVals属性の個数は、同じでなければなりません。
	 * 先に配列に分解してからリクエスト変数の値を取得するようにします。
	 * こうする事で、リクエストにカンマ区切りの値を設定できるようになります。
	 *
	 * @og.rev 4.0.0 (2005/11/30) 新規作成
	 *
	 * @param   vals String
	 * @see     #setMatchKeys( String )
	 */
	public void setMatchVals( final String vals ) {
		matchVals = getCSVParameter( vals );

		if( matchKeys != null && matchKeys.length != matchVals.length ) {
			String errMsg = "matchKeys属性とmatchVals属性の個数が合いません。"
						+ HybsSystem.CR
						+ " matchKeys=[" + matchKeys.length + "]:KEYS="
						+ StringUtil.array2csv( matchKeys ) + HybsSystem.CR
						+ " matchVals=[" + matchVals.length + "]:VLAS="
						+ StringUtil.array2csv( matchVals ) + HybsSystem.CR ;
			throw new HybsSystemException( errMsg );
		}
	}
	
	/**
	 * 【TAG】(通常使いません)リアルタイムチェックを行う場合に有効にします。
	 *
	 * @og.tag
	 * リアルタイムチェックを行う場合に有効にする属性です。
	 * trueが指定された場合、通常のチェックと比較し、以下の差異があります。
	 * ①エラー結果を簡易フォーマットで出力します。
	 *   (ViewFormType="HTMLSimpleErrorList"で表示されます。)
	 *   この簡易フォーマットで出力した場合は、ラベル(短)の定義で出力され、
	 *   カラム名やデータ等は出力されません。
	 * ②must,mustAny属性のチェックを行いません。
	 *   必須及び選択必須は、視覚的に判別できるため、リアルタイムチェックの
	 *   対象外とします。
	 *
	 * @og.rev 4.3.3.0 (2008/10/01) 新規作成
	 * @og.rev 4.3.7.0 (2009/06/01) viewSimple属性名称変更 -> isRealTime
	 *
	 * @param   flg 簡易表示を行うかどうか
	 */
	public void setRealTime( final String flg ) {
		isRealTime = nval( getRequestParameter( flg ),isRealTime );
	}

	/**
	 * このオブジェクトの文字列表現を返します。
	 * 基本的にデバッグ目的に使用します。
	 *
	 * @return このクラスの文字列表現
	 */
	public String toString() {
		return org.opengion.fukurou.util.ToString.title( this.getClass().getName() )
				.println( "VERSION"			,VERSION		)
				.println( "tableId"			,tableId		)
				.println( "command"			,command		)
				.println( "columns"			,columns		)
				.println( "nullCheck"		,nullCheck		)
				.println( "mustAnyCheck"	,mustAnyCheck	)
				.println( "maxRowCount"		,maxRowCount	)
				.println( "minRowCount"		,minRowCount	)
				.println( "checkType"		,checkType		)
				.println( "matchKeys"		,matchKeys		)
				.println( "matchVals"		,matchVals		)
				.println( "realTime"		,isRealTime		)
				.println( "bodyString"		,StringUtil.htmlFilter( bodyString ) )
				.println( "Other..."		,getAttributes().getAttribute() )
				.fixForm().toString() ;
	}
}
