/**************************************************
opengate server
 module for Controling ipfw for IPv6 address 

Copyright (C) 2005 Opengate Project Team
Written by Katsuhiko Eguchi, 2005 

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

Email: watanaby@is.saga-u.ac.jp
**************************************************/

#include "opengatesrv.h"

char ruleNumber6[WORDMAXLN];  /* ipfw rule number in string form */

int getRuleNumber6(char *clientAddr6);
int GetRuleNumber6(char *clientAddr6);

static void sigFunc(int signo);

/******************************************************************/
/* open gate for clientAddr6 (nnnn:nnnn::nnnn:nnnn)               */
/******************************************************************/
int openClientGate6(char *clientAddr6, char *userid, char *macAddr6, char *userProperty)
{
  int fd;
  int ret=0;
  int retNum;

  Sigfunc *defaultSigFunc;

  /* exclusive exec of ipfw to avoid duplicated rule number */

  /**** prepare ****/
  /* open lockfile */
  fd=open(GetConfValue("LockFile"), O_RDWR|O_CREAT, 
	  S_IRUSR|S_IWUSR|S_IRGRP|S_IROTH);
  if(fd==-1){
    err_msg("ERR at %s#%d: lockfile open error",__FILE__,__LINE__);
    return 1;
  } 

  /* set timeout */
  if((defaultSigFunc=Signal(SIGALRM, sigFunc))==SIG_ERR) return 1;
  alarm(atoi(GetConfValue("LockTimeout")));

  /* lock */
  if(Lock(fd)<0){
    err_msg("ERR at %s#%d: lock error",__FILE__,__LINE__);
    return 1;
  }

  /* reset timeout */
  Signal(SIGALRM, defaultSigFunc);
  alarm(0);

  /**** read rules ****/
  if((retNum=GetRuleNumber6(clientAddr6))<0){
    Unlock(fd);
    Close(fd);
    return retNum;
  }

  /**** write rules ****/
  if(atoi(GetConfValue("IpfwScript/Enable"))){
    /********** use perl script to control firewall ************/

    if(Systeml(1, GetConfValue("IpfwScript/Path"),GetConfValue("IpfwPath"),
	       ruleNumber6,clientAddr6,
	       userid,macAddr6,userProperty,
	       GetConfValue("IpfwTagNumber"),(char *)0) != 0){
      err_msg("ERR at %s#%d: exec ipfw script error",__FILE__,__LINE__);
      ret=1; /* abmormal */
    }
  }
  else{
    /********** direct control of firewall **********************/
    /********** add outgoing ipfw rule for the client *************/
    if(Systeml(1, GetConfValue("IpfwPath"),"-q","add",ruleNumber6,
	       "count","tag",GetConfValue("IpfwTagNumber"),
	       "ip","from",clientAddr6,"to","any",
	       "//", userid, (char *)0) != 0){
	      err_msg("ERR at %s#%d: exec ipfw add error",__FILE__,__LINE__);
	      ret=1;
    }

    /********** add incoming ipfw rule for the client *************/
    if(Systeml(1, GetConfValue("IpfwPath"),"-q","add",ruleNumber6,
	       "count","tag",GetConfValue("IpfwTagNumber"),
	       "ip","from","any","to",clientAddr6,
	       "//", userid, (char *)0) != 0){
	      err_msg("ERR at %s#%d: exec ipfw add error",__FILE__,__LINE__);
	      ret=1; /* abnormal */
    }
  }

  /* uplock */
  Unlock(fd);
  Close(fd);

  return ret;
}


/******************************************************************/
/* close gate for clientAddr (nnnn:nnnn:nnnn::nnnn:nnnn:nnnn)     */
/******************************************************************/
void closeClientGate6(struct clientAddr *pClientAddr, char *userid, char *macAddr6)
{
  double time_l;
  int hour, min, sec;
  time_t timeOut;
  
  /********** del ipfw rule for the client *************/
  DelIp6fwRule(pClientAddr->ruleNumber);
  
  timeOut = time(NULL);
  time_l=difftime(timeOut,pClientAddr->timeIn);
  hour=time_l/60/60;
  min=(time_l-hour*60*60)/60;
  sec=(time_l-hour*60*60-min*60);
  err_msg("CLOS: user %s from %s at %s ( %02d:%02d:%02d )",
	  userid, pClientAddr->ipAddr, macAddr6, hour,min,sec);

  /* send message to opengatemd server to renew the info in md cache */
  PutMacAddressToOpengateMd(macAddr6);
 
  return;
}


/***********************************************/
/* delete ipfw rule                            */
/***********************************************/
void delIp6fwRule(char *ruleNumber)
{
  int ruleCount;

  /* get rule count */
  ruleCount = CountRuleNumber6(ruleNumber);

  /* delete rule */
  if(ruleCount>0){
    if(Systeml(1, GetConfValue("IpfwPath"),"delete",ruleNumber,(char *)0) != 0){
      err_msg("ERR at %s#%d: exec ipfw del error",__FILE__,__LINE__);
    }
  }
}
  
/**************************************/
/* get unused ipfw rule number        */
/* error if addr is already in rules */ 
/* return value ret>0: acquired rule number that can be used */
/*              ret=-1: no rule number available */
/*              ret=-2: some system error occured */
/*              ret=-num: the ip address is already registered in rule 'num' */
/**************************************/
int getRuleNumber6(char *clientAddr6)
{
  FILE *fpipe;
  char buf[BUFFMAXLN];
  int num,newNum,readinNum;
    char *p;
  int ip6fwmin;
  int ip6fwmax;
  int ip6fwinterval;
  int portStatus;
  int fileStatus;
  enum status {NORMAL, ABNORMAL, FOUND, NOTFOUND, DUP};

  if((fpipe=Popenl(1, "r", GetConfValue("IpfwPath"),"list",(char *)0)) == NULL){ 
      err_msg("ERR at %s#%d: exec ipfw list error",__FILE__,__LINE__);
  }
  
  /* search unused rule number in the list read from pipe */
  /* check duplication of clientAddr to existing rules */

  newNum=-1;
  readinNum=0;
  portStatus=NOTFOUND;
  fileStatus=NORMAL;

  /* get rule range from config */
  ip6fwmin=atoi(GetConfValue("IpfwRule/Min"));
  ip6fwmax=atoi(GetConfValue("IpfwRule/Max"));
  ip6fwinterval=atoi(GetConfValue("IpfwRule/Interval"));

  /* each port is checked whether it can be used for new rule or not */
  for(num=ip6fwmin;num<=ip6fwmax;num+=ip6fwinterval){

    /* skip rules smaller than num */
    while(readinNum<num){
      if(fgets(buf, BUFFMAXLN, fpipe)==NULL){
	if(feof(fpipe)==1) fileStatus=EOF; 
	else fileStatus=ABNORMAL;
	break;
      }
      if( sscanf(buf, "%d", &readinNum) !=1 ){
	err_msg("ERR at %s#%d: abnormal ipfw response[ %s ]",
		__FILE__,__LINE__,buf);
	fileStatus=ABNORMAL; /* abnormal responsem exit internal loop */
	break;
      }
    }
      
    if(fileStatus==ABNORMAL){
      /* abnormal file proc, exit external loop */ 
      break;
    }

    if(fileStatus==EOF){
      /* EOF before reading a rule that is larger or equal to num */
      /* it means that num can be used for new client */
      portStatus=FOUND;
      newNum=num;
      break;
    }

    /* at this point, readinNum is larger or equal to num */
    /* check number duplication */
    if(readinNum==num){

      /* if clientAddr is found in the existing rule, then err exit. */
      if(((p=(char*)strstr(buf+1,clientAddr6))!=NULL)
	 && isspace(*(p-1))
	 && !isalnum(*(p+strlen(clientAddr6)))){
	/* the clientAddr is found in the rule num */
	newNum=num;
	portStatus=DUP;
	break;
      }
      /* the num is used for other client */
      /* go to checking of next num */
      else{
	continue;
      }
    }
   
    /* at this point, readNum is larger than num */
    /* it means that num can be used for new client */
    newNum=num;
    portStatus=FOUND;
    break;
  }
  
  /* close pipe */
  Pclose(fpipe);

  if(fileStatus==ABNORMAL){
    err_msg("ERR at %s#%d: abnormal ipfw response ",__FILE__,__LINE__);
    return -2;
  }
  if(portStatus==NOTFOUND){
    err_msg("ERR at %s#%d: cannot get unused ipfw number",__FILE__,__LINE__);
    return -1;
  }
  if(portStatus==DUP){
    snprintf(ruleNumber6, WORDMAXLN, "%d", newNum); /* to string */
    return -newNum;
  }

  snprintf(ruleNumber6, WORDMAXLN, "%d", newNum); /* to string */

  return newNum;
}

/*******************************/
/* get packet count from ipfw  */
/*******************************/
int getPacketCount6(char *ruleNumber)
{
  FILE *fpipe;
  char buf[BUFFMAXLN];
  int rule;
  int packets,packetsSum;

  /* exec proc */
  if((fpipe=Popenl(1, "r", GetConfValue("IpfwPath"),"-a","list",ruleNumber,(char *)0)) == NULL){ 
    err_msg("ERR at %s#%d: exec ipfw -a list error",__FILE__,__LINE__);
    return 0; /* abnormal */
  }
  
  /* search unused number in the list read from pipe */
  packetsSum=0;
    
  while(fgets(buf, BUFFMAXLN, fpipe)!=NULL){
    sscanf(buf, "%d %d", &rule, &packets);   /* get packet count */
    packetsSum+=packets;
  }

  /* close pipe */
  Pclose(fpipe);

  return packetsSum;
}

/**********************************************/
/* get rule count registed to a rule number   */
/**********************************************/
int countRuleNumber6(char *ruleNumber)
{
  FILE *fpipe;
  char buf[BUFFMAXLN];
  int ruleCount;

  /* exec proc */
  if((fpipe=Popenl(1, "r", GetConfValue("IpfwPath"),"list",ruleNumber,(char *)0)) == NULL){ 
    err_msg("ERR at %s#%d: exec ipfw list error",__FILE__,__LINE__);
  }
  
  /* count line read from pipe */
  ruleCount = 0;
  while(fgets(buf, BUFFMAXLN, fpipe)!=0) ruleCount++;

  /* close pipe */
  Pclose(fpipe);

  return ruleCount;
}

/**********************************************/
/* function called by signal int              */
/**********************************************/
static void sigFunc(int signo)
{
  return;
}

/**********************************************/
/**********************************************/

int GetRuleNumber6(char *clientAddr6)
{
  int ret;

  if(debug>1) err_msg("DEBUG:=>getRuleNumber6(%s)",clientAddr6);
  ret=getRuleNumber6(clientAddr6);
  if(debug>1) err_msg("DEBUG:(%d)<=getRuleNumber6( )",ret);

  return ret;
}

int OpenClientGate6(char *clientAddr6, char *userid, char *macAddr6, char *userProperty)
{
  int ret;

  if(debug>1) err_msg("DEBUG:=>openClientGate6(%s,%s,%s,%s)",clientAddr6,userid,macAddr6,userProperty);
  ret=openClientGate6(clientAddr6, userid, macAddr6, userProperty);
  if(debug>1) err_msg("DEBUG:(%d)<=openClientGate6( )",ret);

  return ret;
}

void CloseClientGate6(struct clientAddr *pClientAddr, char *userid, char *macAddr6)
{
  if(debug>1) err_msg("DEBUG:=>closeClientGate6(%p,%s,%s)",pClientAddr,userid,macAddr6);
  closeClientGate6(pClientAddr,userid,macAddr6);
  if(debug>1) err_msg("DEBUG:<=closeClientGate6( )");
}

int GetPacketCount6(char *ruleNumber)
{
  int ret;

  if(debug>1) err_msg("DEBUG:=>getPacketCount6(%s)",ruleNumber);
  ret=getPacketCount6(ruleNumber);
  if(debug>1) err_msg("DEBUG:(%d)<=getPacketCount6( )",ret);

  return ret;
}

int CountRuleNumber6(char *ruleNumber)
{
  int ret;
  
  if(debug>1) err_msg("DEBUG:=>countRuleNumber6(%s)", ruleNumber);
  ret=countRuleNumber6(ruleNumber);
  if(debug>1) err_msg("DEBUG:(%d)<=countRuleNumber6( )",ret);
  
  return ret;
}

void DelIp6fwRule(char *ruleNumber){
  if(debug>1) err_msg("DEBUG:=>delIp6fwRule(%s)",ruleNumber);
  delIp6fwRule(ruleNumber);
  if(debug>1) err_msg("DEBUG:<=delIp6fwRule( )");
}
