/* ************************************************************** *
 *                                                                *
 * Copyright (c) 2005, Kota Mizushima, All rights reserved.       *
 *                                                                *
 *                                                                *
 * This software is distributed under the modified BSD License.   *
 * ************************************************************** */
package org.onion_lang.onion.tools;

import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.lang.reflect.*;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.net.*;
import java.text.MessageFormat;
import java.util.Map;

import org.onion_lang.onion.compiler.*;
import org.onion_lang.onion.compiler.CompiledClass;
import org.onion_lang.onion.compiler.CompilerConfig;
import org.onion_lang.onion.compiler.OnionCompiler;
import org.onion_lang.onion.compiler.utility.Messages;
import org.onion_lang.onion.compiler.utility.Systems;

/**
 * 
 * @author Kota Mizushima
 * 
 */
public class OnionCompilerFrontend {
  private static final String CLASSPATH = "-classpath";
  private static final String ENCODING = "-encoding";
  private static final String OUTPUT = "-d";
  private static final String MAX_ERROR = "-maxErrorReport";  
  
  private static final String[] DEFAULT_CLASSPATH = new String[]{"."};
  private static final String DEFAULT_ENCODING = System.getProperty("file.encoding");
  private static final String DEFAULT_OUTPUT = ".";
  private static final int DEFAULT_MAX_ERROR = 10;
  
  private CommandLineParser commandLineParser;
  
  private static OptionConfiguration conf(String option, boolean requireArg) {
    return new OptionConfiguration(option, requireArg);
  }
    
  public OnionCompilerFrontend(){
    commandLineParser = new CommandLineParser(
      new OptionConfiguration[]{
        conf(CLASSPATH, true),
        conf(ENCODING, true),
        conf(OUTPUT, true),
        conf(MAX_ERROR, true)
      }
    );
  }
  
  public int run(String[] commandLine) {
    if(commandLine.length == 0){
      printUsage();
      return -1;
    }
    ParseSuccess result = parseCommandLine(commandLine);
    if(result == null) return -1;
    CompilerConfig config = createConfig(result);
    String[] params = (String[]) result.getArguments().toArray(new String[0]);
    if(params.length == 0) {
      printUsage();
      return -1;
    }
    if(config == null) return -1;
    CompiledClass[] classes = compile(config, params);
    if(classes == null) return -1;
    return generateFile(classes) ? 0 : -1;
  }
  
  private String getSimpleName(String fqcn){
    int index = fqcn.lastIndexOf(".");
    if(index < 0){
      return fqcn;
    }else{
      return fqcn.substring(index + 1, fqcn.length());
    }
  }
  
  private String getOutputPath(String outDir, String fqcn) {
    String name = getSimpleName(fqcn);
    return outDir + Systems.getFileSeparator() + name + ".class";
  }
   
  private boolean generateFile(CompiledClass[] binaries){
    for (int i = 0; i < binaries.length; i++) {
      CompiledClass binary = binaries[i];
      String outDir = binary.getOutputPath();
      new File(outDir).mkdirs();
      String outPath = getOutputPath(outDir, binary.getClassName());
      File targetFile = new File(outPath);
      try{
        if(!targetFile.exists()) targetFile.createNewFile();
        BufferedOutputStream out = new BufferedOutputStream(
          new FileOutputStream(targetFile)
        );
        out.write(binary.getContent());
        out.close();
      }catch(FileNotFoundException e){
        e.printStackTrace();
        return false;
      }catch(IOException e){
        e.printStackTrace();
        return false;
      }
    }
    return true;
  }
  
  protected void printUsage() {
    printerr("Usage: onionc [-options] source_file ...");
    printerr("options: ");
    printerr("  -d <path>                   specify output directory");
    printerr("  -classpath <path>           specify classpath");
    printerr("  -encoding <encoding>        specify source file encoding");
    printerr("  -maxErrorReport <number>    set number of errors reported");
  }
    
  private ParseSuccess parseCommandLine(String[] commandLine) {
    ParseResult result = commandLineParser.parse(commandLine);
    if(result.getStatus() == ParseResult.FAILURE){
      ParseFailure failure = (ParseFailure)result;
      String[] lackedOptions = failure.getLackedOptions();
      String[] invalidOptions = failure.getInvalidOptions();
      for (int i = 0; i < invalidOptions.length; i++) {
        printerr(Messages.get("error.command.invalidArgument", invalidOptions[i]));
      }
      for (int i = 0; i < lackedOptions.length; i++) {
        printerr(Messages.get("error.command..noArgument", lackedOptions[i]));
      }
      return null;
    }
    return (ParseSuccess)result;
  }
  
  private CompilerConfig createConfig(ParseSuccess result) {
    Map option = result.getOptions();
    Map noargOption = result.getNoArgumentOptions();
    String[] classpath = checkClasspath((String)option.get(CLASSPATH));
    String encoding = checkEncoding((String)option.get(ENCODING));
    String outputDirectory = checkOutputDirectory((String)option.get(OUTPUT));
    Integer maxErrorReport = checkMaxErrorReport((String)option.get(MAX_ERROR));
    if(encoding == null || maxErrorReport == null || outputDirectory == null) {
      return null;
    }
    return new CompilerConfig(
      classpath,
      encoding,
      outputDirectory,
      maxErrorReport.intValue()
    );
  }  
  
  private CompiledClass[] compile(CompilerConfig config, String[] fileNames) {
    OnionCompiler compiler = new OnionCompiler(config);
    return compiler.compile(fileNames);
  }
  
  private String[] checkClasspath(String classpath) {
    if(classpath == null) return DEFAULT_CLASSPATH;
    String[] paths = pathArray(classpath);
    return paths;
  }
  
  private String checkOutputDirectory(String outputDirectory) {
    if(outputDirectory == null) return DEFAULT_OUTPUT;
    return outputDirectory;
  }
  
  private String checkEncoding(String encoding) {
    if(encoding == null) return System.getProperty("file.encoding");
    try {
      "".getBytes(encoding);
      return encoding;
    }catch(UnsupportedEncodingException e){
      printerr(Messages.get("error.command.invalidEncoding", ENCODING));
      return null;
    }
  }
  
  private Integer checkMaxErrorReport(String maxErrorReport) {
    if(maxErrorReport == null) return new Integer(DEFAULT_MAX_ERROR);
    try{
      int value = Integer.parseInt(maxErrorReport);
      if(value > 0){
        return new Integer(value);
      }
    }catch(NumberFormatException e){/* nothing to do */}
    printerr(Messages.get("error.command.requireNaturalNumber", MAX_ERROR));
    return null;
  }
    
  private static String[] pathArray(String path) {
    return path.split(Systems.getPathSeparator());
  }
  
  private static void printerr(String message) {
    System.err.println(message);
  }
  
  public static void main(String[] args) throws Throwable {
    try {
      new OnionCompilerFrontend().run(args);
    } catch (ScriptFailureException e) {
      throw e.getCause();
    }
  }
}
