/* ************************************************************** *
 *                                                                *
 * Copyright (c) 2005, Kota Mizushima, All rights reserved.       *
 *                                                                *
 *                                                                *
 * This software is distributed under the modified BSD License.   *
 * ************************************************************** */
package org.onion_lang.onion.compiler.phase.analysis;

import java.util.ArrayList;
import java.util.List;

import org.onion_lang.onion.compiler.environment.*;
import org.onion_lang.onion.compiler.phase.CodeAnalysisPhase;
import org.onion_lang.onion.compiler.problem.SemanticErrorReporter;
import org.onion_lang.onion.compiler.utility.Boxing;
import org.onion_lang.onion.lang.kernel.*;
import org.onion_lang.onion.lang.kernel.type.*;
import org.onion_lang.onion.lang.syntax.*;
import org.onion_lang.onion.lang.syntax.visitor.ASTVisitor;

/**
 * @author Kota Mizushima
 * Date: 2005/07/08
 */
public class TypeChecker extends ASTVisitor
	implements SemanticErrorReporter.Constants, BinaryExpressionNode.Constants,
	UnaryExpressionNode.Constants {
  private CodeAnalysisPhase state;
  public TypeChecker(CodeAnalysisPhase state){
    this.state = state;
  }
  
  public void process(CompilationUnit unit){
    accept(unit);
  }
//------------------------------ top level --------------------------------------//
  public Object visit(CompilationUnit unit, Object object) {
    state.setUnit(unit);
    TopLevelElement[] toplevels = unit.getTopLevels();
    LocalContext context = new LocalContext();
    List statements = new ArrayList();
    String implicitClassName = state.topClass();
    state.setCurrentResolver(state.getResolver(implicitClassName));
    ClassNode implicitClass = (ClassNode) state.loadTopClass();
    ArraySymbol argsType = state.loadArray(state.load("java.lang.String"), 1);
    
    MethodNode method = new MethodNode(
      Modifier.PUBLIC, implicitClass, 
      "start", new TypeSymbol[]{argsType}, BasicSymbol.VOID, null);
    context.addEntry("args", argsType);
    for(int i = 0; i < toplevels.length; i++){
      TopLevelElement element = toplevels[i];
      if(!(element instanceof TypeDeclaration)){
        state.setContextClass(implicitClass);
      }
      if(element instanceof Statement){
        context.setMethod(method);
        StatementNode statement = (StatementNode) accept(toplevels[i], context);
        statements.add(statement);
      }else{
        accept(toplevels[i], null);
      }
    }    
    
    if(implicitClass != null){
      statements.add(new ReturnNode(null));
      method.setBlock(
        new BlockNode(
          (StatementNode[])statements.toArray(new StatementNode[0])));
      method.setFrame(context.getContextFrame());
      implicitClass.addMethod(method);
      
      //create start method invoked from main method
      implicitClass.addMethod(
        createStartupMethod(
          implicitClass, method, "main",
          new TypeSymbol[]{argsType}, BasicSymbol.VOID));
    }
    return null;
  }
  
  private MethodNode createStartupMethod(
    ClassSymbol top, MethodSymbol ref, String name, TypeSymbol[] args,
    TypeSymbol returnType){
    MethodNode method = new MethodNode(
      Modifier.STATIC | Modifier.PUBLIC, top, name, args, returnType, null);
    LocalFrame frame = new LocalFrame(null);
    ExpressionNode[] params = new ExpressionNode[args.length];
    for(int i = 0; i < args.length; i++){
      int index = frame.addEntry("args" + i, args[i]);
      params[i] = new LocalRefNode(0, index, args[i]);
    }
    method.setFrame(frame);
    ConstructorSymbol c = top.findConstructor(new ExpressionNode[0])[0];
    ExpressionNode exp = new NewNode(c, new ExpressionNode[0]);
    exp = new MethodCallNode(exp, ref, params);
    BlockNode block = new BlockNode(
      new StatementNode[]{new ExpressionStatementNode(exp)});
    block = addReturnNode(block, BasicSymbol.VOID);
    method.setBlock(block);
    return method;
  }
  
  public Object visit(InterfaceDeclaration ast, Object context) {
    state.setContextClass((ClassNode) state.lookupKernelNode(ast));
    return null;
  }
  
  public Object visit(ClassDeclaration ast, Object context) {
    state.setContextClass((ClassNode) state.lookupKernelNode(ast));
    state.setCurrentResolver(state.getResolver(state.getContextClass().getName()));
    if(ast.getDefaultSection() != null){
      accept(ast.getDefaultSection(), context);
    }
    acceptEach(ast.getSections(), context);
    return null;
  }
  
  public Object visit(AccessSection ast, Object context) {
    acceptEach(ast.getMembers(), context);
    return null;
  }
//-------------------------------------------------------------------------------//

  
//------------------------- binary expressions ----------------------------------//
  public Object visit(Addition ast, Object context) {
    ExpressionNode left = processExpression(ast.getLeft(), context);
    ExpressionNode right = processExpression(ast.getRight(), context);
    if(left == null || right == null) return null;
    if(left.isBasicType() && right.isBasicType()){
      return processNumericalBinaryExpression(ADD, ast, left, right, context);
    }
    if(left.isBasicType()){
      if(left.type() == BasicSymbol.VOID){
        report(IS_NOT_BOXABLE_TYPE, ast.getLeft(), new Object[]{left.type()});
        return null;
      }else{
        left = Boxing.boxing(state.table(), left);
      }
    }
    if(right.isBasicType()){
      if(right.type() == BasicSymbol.VOID){
        report(IS_NOT_BOXABLE_TYPE, ast.getRight(), new Object[]{right.type()});
        return null;
      }else{
        right = Boxing.boxing(state.table(), right);
      }
    }
    MethodSymbol toString;
    toString = findMethod(ast.getLeft(), (ObjectSymbol)left.type(), "toString");
    left = new MethodCallNode(left, toString, new ExpressionNode[0]);
    toString = findMethod(ast.getRight(), (ObjectSymbol)right.type(), "toString");
    right = new MethodCallNode(right, toString, new ExpressionNode[0]);
    MethodSymbol concat =
      findMethod(ast, (ObjectSymbol)left.type(), "concat", new ExpressionNode[]{right});
    return new MethodCallNode(left, concat, new ExpressionNode[]{right});
  }  
  
  public Object visit(Subtraction ast, Object context) {
    ExpressionNode left = processExpression(ast.getLeft(), context);
    ExpressionNode right = processExpression(ast.getRight(), context);
    if(left == null || right == null) return null;
    return processNumericalBinaryExpression(SUBTRACT, ast, left, right, context);
  }
  
  public Object visit(Multiplication ast, Object context) {
    ExpressionNode left = processExpression(ast.getLeft(), context);
    ExpressionNode right = processExpression(ast.getRight(), context);
    if(left == null || right == null) return null;
    return processNumericalBinaryExpression(MULTIPLY,  ast, left, right, context);
  }
  
  public Object visit(Division ast, Object context) {
    ExpressionNode left = processExpression(ast.getLeft(), context);
    ExpressionNode right = processExpression(ast.getRight(), context);
    if(left == null || right == null) return null;
    return processNumericalBinaryExpression(DIVIDE, ast, left, right, context);
  }
  
  public Object visit(Modulo ast, Object context) {
    ExpressionNode left = processExpression(ast.getLeft(), context);
    ExpressionNode right = processExpression(ast.getRight(), context);
    if(left == null || right == null) return null;
    return processNumericalBinaryExpression(MOD, ast, left, right, context);
  }
    
  public Object visit(XOR ast, Object context) {
    return processBitExpression(XOR, ast, context);
  }
  
  public Object visit(Equal ast, Object context) {
    return processEqualExpression(EQUAL, ast, context);
  }
  
  public Object visit(NotEqual ast, Object context) {
    return processEqualExpression(NOT_EQUAL, ast, context);
  }
  
  public Object visit(ReferenceEqual ast, Object context) {
    return processReferenceEqualExpression(EQUAL, ast, context);
  }
  
  public Object visit(ReferenceNotEqual ast, Object context) {
    return processReferenceEqualExpression(NOT_EQUAL, ast, context);
  }
  
  public Object visit(LessOrEqual ast, Object context) {
    ExpressionNode[] ops = processComparableExpression(ast, context);
    if(ops == null){
      return null;
    }
    return new BinaryExpressionNode(LESS_OR_EQUAL, BasicSymbol.BOOLEAN, ops[0], ops[1]);
  }
  
  public Object visit(LessThan ast, Object context) {
    ExpressionNode[] ops = processComparableExpression(ast, context);
    if(ops == null) return null;
    return new BinaryExpressionNode(
      LESS_THAN, BasicSymbol.BOOLEAN, ops[0], ops[1]);
  }
  
  public Object visit(GreaterOrEqual ast, Object context) {
    ExpressionNode[] ops = processComparableExpression(ast, context);
    if(ops == null) return null;    
    return new BinaryExpressionNode(
      GREATER_OR_EQUAL, BasicSymbol.BOOLEAN, ops[0], ops[1]);
  }
  
  public Object visit(GreaterThan ast, Object context) {
    ExpressionNode[] ops = processComparableExpression(ast, context);
    if(ops == null) return null;
    return new BinaryExpressionNode(
      GREATER_THAN, BasicSymbol.BOOLEAN, ops[0], ops[1]);
  }
  
  public Object visit(LogicalAnd ast, Object context) {
    ExpressionNode[] ops = processLogicalExpression(ast, context);
    if(ops == null) return null;
    return new BinaryExpressionNode(
      LOGICAL_AND, BasicSymbol.BOOLEAN, ops[0], ops[1]);
  }
  
  public Object visit(LogicalOr ast, Object context) {
    ExpressionNode[] ops = processLogicalExpression(ast, context);
    if(ops == null) return null;
    return new BinaryExpressionNode(
      LOGICAL_OR, BasicSymbol.BOOLEAN, ops[0], ops[1]);
  }
  
  public Object visit(LogicalRightShift ast, Object context) {
    return processShiftExpression(BIT_SHIFT_R3, ast, context);
  }
  
  public Object visit(MathLeftShift ast, Object context) {
    return processShiftExpression(BIT_SHIFT_L2, ast, context);
  }
  
  public Object visit(MathRightShift ast, Object context) {
    return processShiftExpression(BIT_SHIFT_R2, ast, context);
  }
    
  public Object visit(BitAnd ast, Object context) {
    return processBitExpression(BIT_AND, ast, context);
  }
  
  public Object visit(BitOr expression, Object context) {
    return processBitExpression(BIT_AND, expression, context);
  }
  
  ExpressionNode[] processLogicalExpression(BinaryExpression ast, Object context){
    ExpressionNode left = processExpression(ast.getLeft(), context);
    ExpressionNode right = processExpression(ast.getRight(), context);
    if(left == null || right == null) return null;
    TypeSymbol leftType = left.type(), rightType = right.type();
    if((leftType != BasicSymbol.BOOLEAN) || (rightType != BasicSymbol.BOOLEAN)){
      report(
        INCOMPATIBLE_OPERAND_TYPE, ast,
        new Object[]{ast.getSymbol(), new TypeSymbol[]{left.type(), right.type()}});
      return null;
    }
    return new ExpressionNode[]{left, right};
  }
  
  ExpressionNode processShiftExpression(
    int kind, BinaryExpression ast, Object context){
    ExpressionNode left = processExpression(ast.getLeft(), context);
    ExpressionNode right = processExpression(ast.getRight(), context);
    if(left == null || right == null) return null;
    if(!left.type().isBasicType()){
      ExpressionNode[] params = new ExpressionNode[]{right};
      Object[] result = 
        tryFindMethod(ast, (ObjectSymbol)left.type(), "add", params);
      if(result[1] == null){
        report(METHOD_NOT_FOUND, ast, new Object[]{left.type(), "add", types(params)});
        return null;
      }
      return new MethodCallNode(left, (MethodSymbol)result[1], params);
    }
    if(!right.type().isBasicType()){
      report(
        INCOMPATIBLE_OPERAND_TYPE, ast,
        new Object[]{ast.getSymbol(), new TypeSymbol[]{left.type(), right.type()}});
      return null;
    }
    BasicSymbol leftType = (BasicSymbol)left.type();
    BasicSymbol rightType = (BasicSymbol)right.type();
    if((!leftType.isInteger()) || (!rightType.isInteger())){
      report(
        INCOMPATIBLE_OPERAND_TYPE, ast,
        new Object[]{ast.getSymbol(), new TypeSymbol[]{left.type(), right.type()}});
      return null;
    }
    TypeSymbol leftResultType = promoteInteger(leftType);
    if(leftResultType != leftType){
      left = new CastNode(left, leftResultType);
    }
    if(rightType != BasicSymbol.INT){
      right = new CastNode(right, BasicSymbol.INT);
    }
    return new BinaryExpressionNode(kind, BasicSymbol.BOOLEAN, left, right);
  }
  
  static TypeSymbol promoteInteger(TypeSymbol type){
    if(type == BasicSymbol.BYTE || type == BasicSymbol.SHORT ||
       type == BasicSymbol.CHAR || type == BasicSymbol.INT){
      return BasicSymbol.INT;
    }
    if(type == BasicSymbol.LONG){
      return BasicSymbol.LONG;
    }
    return null;
  }  
    
  ExpressionNode processBitExpression(
    int kind, BinaryExpression ast, Object context) {
    ExpressionNode left = processExpression(ast.getLeft(), context);
    ExpressionNode right = processExpression(ast.getRight(), context);
    if(left == null || right == null) return null;
    if((!left.isBasicType()) || (!right.isBasicType())){
      report(
        INCOMPATIBLE_OPERAND_TYPE, ast,
        new Object[]{ast.getSymbol(), new TypeSymbol[]{left.type(), right.type()}});
      return null;
    }
    BasicSymbol leftType = (BasicSymbol)left.type();
    BasicSymbol rightType = (BasicSymbol)right.type();
    TypeSymbol resultType = null;
    if(leftType.isInteger() && rightType.isInteger()){
      resultType = promoteNumericTypes(leftType, rightType);    
    }else if(leftType.isBoolean() && rightType.isBoolean()){
      resultType = BasicSymbol.BOOLEAN;
    }else{
      report(
        INCOMPATIBLE_OPERAND_TYPE, ast,
        new Object[]{ast.getSymbol(), new TypeSymbol[]{leftType, rightType}});
      return null;
    }
    if(left.type() != resultType){
      left = new CastNode(left, resultType);
    }
    if(right.type() != resultType){
      right = new CastNode(right, resultType);
    }
    return new BinaryExpressionNode(kind, resultType, left, right);
  }
  
  ExpressionNode processNumericalBinaryExpression(
    int kind, BinaryExpression ast,
    ExpressionNode left, ExpressionNode right, Object context) {
    if((!hasNumericType(left)) || (!hasNumericType(right))){
      report(
        INCOMPATIBLE_OPERAND_TYPE, ast,
        new Object[]{ast.getSymbol(), new TypeSymbol[]{left.type(), right.type()}});
      return null;
    }
    TypeSymbol resultType = promoteNumericTypes(left.type(), right.type());
    if(left.type() != resultType){
      left = new CastNode(left, resultType);
    }
    if(right.type() != resultType){
      right = new CastNode(right, resultType);
    }
    return new BinaryExpressionNode(kind, resultType, left, right);
  }
  
  ExpressionNode processReferenceEqualExpression(
    int kind, BinaryExpression ast, Object context){
    ExpressionNode left = processExpression(ast.getLeft(), context);
    ExpressionNode right = processExpression(ast.getRight(), context);
    if(left == null || right == null) return null;
    TypeSymbol leftType = left.type();
    TypeSymbol rightType = right.type();
    if(
      (left.isBasicType() && (!right.isBasicType())) ||
      ((!left.isBasicType()) && (right.isBasicType()))){
      report(
        INCOMPATIBLE_OPERAND_TYPE, ast,
        new Object[]{ast.getSymbol(), new TypeSymbol[]{leftType, rightType}});
      return null;
    }
    if(left.isBasicType() && right.isBasicType()){
      if(hasNumericType(left) && hasNumericType(right)){
        TypeSymbol resultType = promoteNumericTypes(leftType, rightType);
        if(resultType != left.type()){
          left = new CastNode(left, resultType);
        }
        if(resultType != right.type()){
          right = new CastNode(right, resultType);
        }
      }else if(leftType != BasicSymbol.BOOLEAN || rightType != BasicSymbol.BOOLEAN){
        report(
          INCOMPATIBLE_OPERAND_TYPE, ast,
          new Object[]{ast.getSymbol(), new TypeSymbol[]{leftType, rightType}});
        return null;
      }
    }
    return new BinaryExpressionNode(kind, BasicSymbol.BOOLEAN, left, right);
  }
  
  ExpressionNode processEqualExpression(
    int kind, BinaryExpression ast, Object context
  ){
    ExpressionNode left = processExpression(ast.getLeft(), context);
    ExpressionNode right = processExpression(ast.getRight(), context);
    if(left == null || right == null) return null;
    TypeSymbol leftType = left.type(), rightType = right.type();
    if(
      (left.isBasicType() && (!right.isBasicType())) ||
      ((!left.isBasicType()) && (right.isBasicType()))){
      report(
        INCOMPATIBLE_OPERAND_TYPE, ast,
        new Object[]{ast.getSymbol(), new TypeSymbol[]{leftType, rightType}}
      );
      return null;
    }
    if(left.isBasicType() && right.isBasicType()){      
      if(hasNumericType(left) && hasNumericType(right)){
        TypeSymbol resultType = promoteNumericTypes(leftType, rightType);
        if(resultType != left.type()){
          left = new CastNode(left, resultType);
        }
        if(resultType != right.type()){
          right = new CastNode(right, resultType);
        }
      }else if(leftType != BasicSymbol.BOOLEAN || rightType != BasicSymbol.BOOLEAN){
        report(
          INCOMPATIBLE_OPERAND_TYPE, ast,
          new Object[]{ast.getSymbol(), new TypeSymbol[]{leftType, rightType}});
        return null;
      }
    }else if(left.isReferenceType() && right.isReferenceType()){
      return createEquals(kind, left, right);
    }
    return new BinaryExpressionNode(kind, BasicSymbol.BOOLEAN, left, right);
  }
  
  ExpressionNode createEquals(int kind, ExpressionNode left, ExpressionNode right){
    right = new CastNode(right, state.rootClass());
    ExpressionNode[] params = {right};
    ObjectSymbol target = (ObjectSymbol) left.type();
    MethodSymbol[] methods = target.findMethod("equals", params);
    ExpressionNode node = new MethodCallNode(left, methods[0], params);
    if(kind == BinaryExpressionNode.Constants.NOT_EQUAL){
      node = new UnaryExpressionNode(
        NOT, BasicSymbol.BOOLEAN, node);
    }
    return node;
  }
  
  ExpressionNode[] processComparableExpression(BinaryExpression ast, Object context) {
    ExpressionNode left = processExpression(ast.getLeft(), context);
    ExpressionNode right = processExpression(ast.getRight(), context);
    if(left == null || right == null) return null;
    TypeSymbol leftType = left.type(), rightType = right.type();
    if((!numeric(left.type())) || (!numeric(right.type()))){
      report(
        INCOMPATIBLE_OPERAND_TYPE, ast,
        new Object[]{ast.getSymbol(), new TypeSymbol[]{left.type(), right.type()}});
      return null;
    }
    TypeSymbol resultType = promoteNumericTypes(leftType, rightType);
    if(leftType != resultType){
      left = new CastNode(left, resultType);
    }
    if(rightType != resultType){
      right = new CastNode(right, resultType);
    }
    return new ExpressionNode[]{left, right};
  }
//-------------------------------------------------------------------------------//  
//------------------------- literals --------------------------------------------//
  public Object visit(FloatLiteral ast, Object context) {
    return new FloatNode(ast.value);
  }
  
  public Object visit(SuperMethodCall ast, Object context) {
    ExpressionNode[] params;    
    params = processExpressions(ast.getParameters(), context);
    if(params == null) return null;
    ClassSymbol contextClass = state.getContextClass();
    Object[] result = 
      tryFindMethod(ast, contextClass.getSuperClass(), ast.getName(), params);
    if(result[1] == null){
      Boolean notFound = (Boolean) result[0];
      if(notFound.booleanValue()){
        report(
          METHOD_NOT_FOUND, ast, 
          new Object[]{contextClass, ast.getName(), types(params)});
      }
      return null;
    }
    MethodSymbol method = (MethodSymbol) result[1];
    return new SuperCallNode(new SelfNode(contextClass), method, params);
  }
  
  public Object visit(DoubleLiteral ast, Object context) {
    return new DoubleNode(ast.value);
  }
  
  public Object visit(IntegerLiteral ast, Object context) {
    return new IntegerNode(ast.value);
  }
  
  public Object visit(LongLiteral ast, Object context) {
    return new LongNode(ast.getValue());
  }
  
  public Object visit(BooleanLiteral ast, Object context) {
    return new BooleanNode(ast.getValue());
  }
  
  public Object visit(ListLiteral ast, Object context) {
    ExpressionNode[] elements = new ExpressionNode[ast.size()];
    for(int i = 0; i < ast.size(); i++){
      elements[i] = processExpression(ast.getExpression(i), context);
    }
    ListNode node = new ListNode(elements, state.load("java.util.List"));
    return node;
  }
  
  public Object visit(StringLiteral ast, Object context) {
    return new StringNode(ast.value, state.load("java.lang.String"));
  }
  
  public Object visit(NullLiteral ast, Object context) {
    return new NullNode();
  }
//-----------------------------------------------------------------------------//
  
//---------------------------- unary expressions ------------------------------//
  public Object visit(Posit ast, Object context) {
    ExpressionNode node = processExpression(ast.getTarget(), context);
    if(node == null) return null;
    if(!hasNumericType(node)){
      report(
        INCOMPATIBLE_OPERAND_TYPE, ast, 
        new Object[]{"+", new TypeSymbol[]{node.type()}});
      return null;
    }
    node = new UnaryExpressionNode(PLUS, node.type(), node);
    return node;
  }
  
  public Object visit(Negate ast, Object context) {
    ExpressionNode node = processExpression(ast.getTarget(), context);
    if(node == null) return null;
    if(!hasNumericType(node)){
      report(
        INCOMPATIBLE_OPERAND_TYPE, ast,
        new Object[]{"-", new TypeSymbol[]{node.type()}}
      );
      return null;
    }
    node = new UnaryExpressionNode(MINUS, node.type(), node);
    return node;
  }
  
  public Object visit(Not ast, Object context) {
    ExpressionNode node = processExpression(ast.getTarget(), context);
    if(node == null) return null;
    if(node.type() != BasicSymbol.BOOLEAN){
      report(
        INCOMPATIBLE_OPERAND_TYPE, ast,
        new Object[]{"!", new TypeSymbol[]{node.type()}});
      return null;
    }
    node = new UnaryExpressionNode(NOT, BasicSymbol.BOOLEAN, node);
    return node;
  }
//-----------------------------------------------------------------------------//
  
//---------------------------- assignment operators ---------------------------//
  public Object visit(Assignment ast, Object context) {
    Expression left = ast.getLeft();
    if(left instanceof IDExpression){
      return processLocalAssign(ast, context);
    }else if(left instanceof SelfFieldReference){
      return processSelfFieldAssign(ast, context);
    }else if(left instanceof Indexing){
      return processArrayAssign(ast, context);
    }else if(left instanceof FieldOrMethodRef){
      return processFieldOrMethodAssign(ast, context);
    }
    return null;
  }
  
  private ExpressionNode processLocalAssign(Assignment ast, Object context){
    ExpressionNode value = processExpression(ast.getRight(), context);
    if(value == null) return null;
    IDExpression id = (IDExpression) ast.getLeft();
    LocalContext local = ((LocalContext)context);
    ClosureLocalBinding bind = local.lookup(id.name);
    int frame, index;
    TypeSymbol leftType, rightType = value.type();
    if(bind != null){
      frame = bind.getFrame();
      index = bind.getIndex();
      leftType = bind.getType();
    }else{
      frame = 0;
      if(rightType.isNullType()){
        leftType = state.rootClass();
      }else{
        leftType = rightType;
      }
      index = local.addEntry(id.name, leftType);
    }
    value = processAssignable(ast.getRight(), leftType, value);
    if(value == null) return null;
    return new LocalAssignmentNode(frame, index, leftType, value);
  }
  
  private Object processSelfFieldAssign(Assignment ast, Object context){
    ExpressionNode value = processExpression(ast.getRight(), context);
    if(value == null) return null;
    SelfFieldReference ref = (SelfFieldReference) ast.getLeft();
    LocalContext local = (LocalContext) context;
    ClassSymbol selfClass;
    if(local.isGlobal()){
      selfClass = state.loadTopClass();
    }else {
      if(local.getMethod() != null){
        selfClass = local.getMethod().getClassType();
      }else{
        selfClass = local.getConstructor().getClassType();
      }
    }
    FieldSymbol field = findField(selfClass, ref.getName());
    if(field == null){
      report(FIELD_NOT_FOUND, ref, new Object[]{selfClass, ref.getName()});
      return null;
    }
    if(!isAccessible(field, selfClass)){
      report(
        FIELD_NOT_ACCESSIBLE, ast, 
        new Object[]{field.getClassType(), field.getName(), selfClass});
      return null;
    }
    value = processAssignable(ast.getRight(), field.getType(), value);
    if(value == null) return null;
    return new FieldAssignmentNode(new SelfNode(selfClass), field, value);
  }
  
  Object processArrayAssign(Assignment ast, Object context){
    ExpressionNode value = processExpression(ast.getRight(), context);
    Indexing indexing = (Indexing) ast.getLeft();
    ExpressionNode target = processExpression(indexing.getLeft(), context);
    ExpressionNode index = processExpression(indexing.getRight(), context);
    if(value == null || target == null || index == null) return null;
    if(target.isBasicType()){
      report(
        INCOMPATIBLE_TYPE,
        indexing.getLeft(), new Object[]{state.rootClass(), target.type()});
      return null;
    }
    if(target.isArrayType()){
      ArraySymbol targetType = ((ArraySymbol)target.type());
      if(!(index.isBasicType() && ((BasicSymbol)index.type()).isInteger())){
        report(
          INCOMPATIBLE_TYPE, 
          indexing.getRight(), new Object[]{BasicSymbol.INT, index.type()});
        return null;
      }
      TypeSymbol base = targetType.getBase();
      value = processAssignable(ast.getRight(), base, value);
      if(value == null) return null;
      return new ArrayAssignmentNode(target, index, value);      
    }
    ExpressionNode[] params;
    params = new ExpressionNode[]{index, value};
    Object[] result = tryFindMethod(
      ast, (ObjectSymbol)target.type(), 
      "set", new ExpressionNode[]{index, value});
    if(result[1] == null){
      report(
        METHOD_NOT_FOUND, ast, 
        new Object[]{target.type(), "set", types(params)});
      return null;
    }
    return new MethodCallNode(target, (MethodSymbol)result[1], params);
  }
  
  Object processFieldOrMethodAssign(Assignment ast, Object context){
    ExpressionNode right = processExpression(ast.getRight(), context);
    report(UNIMPLEMENTED_FEATURE, ast, new Object[0]);
    return null;
  }
  
  public Object visit(AdditionAssignment ast, Object context) {
    ExpressionNode right = processExpression(ast.getRight(), context);
    report(UNIMPLEMENTED_FEATURE, ast, new Object[0]);
    return null;
  }
  
  public Object visit(SubtractionAssignment ast, Object context) {
    ExpressionNode right = processExpression(ast.getRight(), context);
    report(UNIMPLEMENTED_FEATURE, ast, new Object[0]);
    return null;
  }
  
  public Object visit(MultiplicationAssignment ast, Object context) {
    ExpressionNode right = processExpression(ast.getRight(), context);
    report(UNIMPLEMENTED_FEATURE, ast, new Object[0]);    
    return null;
  }
  
  public Object visit(DivisionAssignment ast, Object context) {
    ExpressionNode right = processExpression(ast.getRight(), context);
    report(UNIMPLEMENTED_FEATURE, ast, new Object[0]);
    return null;
  }
  
  public Object visit(ModuloAssignment ast, Object context) {
    ExpressionNode right = processExpression(ast.getRight(), context);
    report(UNIMPLEMENTED_FEATURE, ast, new Object[0]);
    return null;
  }
//-----------------------------------------------------------------------------//

//---------------------------- other expressions ------------------------------//
  public Object visit(IDExpression ast, Object context) {
    LocalContext local = (LocalContext) context;
    ClosureLocalBinding bind = local.lookup(ast.name);
    if(bind == null){
      report(VARIABLE_NOT_FOUND, ast, new Object[]{ast.name});
      return null;
    }
    return new LocalRefNode(bind);
  }
  
  private MethodSymbol findMethod(
    ASTNode ast, ObjectSymbol type, String name
  ) {
    ExpressionNode[] params = new ExpressionNode[0];
    MethodSymbol[] methods = type.findMethod(name, params);    
    if(methods.length == 0){
      report(
        METHOD_NOT_FOUND, ast, 
        new Object[]{type, name, types(params)});
      return null;
    }
    return methods[0];
  }
  
  private MethodSymbol findMethod(
    ASTNode ast, ObjectSymbol type, String name, ExpressionNode[] params
  ) {
    MethodSymbol[] methods = type.findMethod(name, params);
    return methods[0];
  }
  
  public Object visit(CurrentInstance ast, Object context) {
    LocalContext local = (LocalContext) context;
    if(local.isStatic()) return null;
    ClassSymbol selfClass = state.getContextClass();
    return new SelfNode(selfClass);
  }
  
  boolean hasSamePackage(ClassSymbol a, ClassSymbol b) {
    String name1 = a.getName();
    String name2 = b.getName();
    int index;
    index = name1.lastIndexOf(".");
    if(index >= 0){
      name1 = name1.substring(0, index);
    }else{
      name1 = "";
    }
    index = name2.lastIndexOf(".");
    if(index >= 0){
      name2 = name2.substring(0, index);
    }else{
      name2 = "";
    }
    return name1.equals(name2);
  }
  
  boolean isAccessible(ClassSymbol target, ClassSymbol context) {
    if(hasSamePackage(target, context)){
      return true;
    }else{
      if(Modifier.isInternal(target.getModifier())){
        return false;
      }else{
        return true;
      }
    }
  }
  
  boolean isAccessible(MemberSymbol member, ClassSymbol context) {
    ClassSymbol targetType = member.getClassType();
    if(targetType == context) return true;
    int modifier = member.getModifier();
    if(TypeRules.isSuperType(targetType, context)){
      if(Modifier.isProtected(modifier) || Modifier.isPublic(modifier)){
        return true;
      }else{
        return false;
      }
    }else{
      if(Modifier.isPublic(modifier)){
        return true;
      }else{
        return false;
      }
    }
  }
  
  private FieldSymbol findField(ObjectSymbol target, String name) {
    if(target == null) return null;
    FieldSymbol[] fields = target.getFields();
    for (int i = 0; i < fields.length; i++) {
      if(fields[i].getName().equals(name)){
        return fields[i];
      }
    }
    FieldSymbol field = findField(target.getSuperClass(), name);
    if(field != null) return field;
    ClassSymbol[] interfaces = target.getInterfaces();
    for(int i = 0; i < interfaces.length; i++){
      field = findField(interfaces[i], name);
      if(field != null) return field;
    }
    return null;
  }
  
  private boolean checkAccessible(
    ASTNode ast, ObjectSymbol target, ClassSymbol context
  ) {
    if(target.isArrayType()){
      TypeSymbol component = ((ArraySymbol)target).getComponent();
      if(!component.isBasicType()){
        if(!isAccessible((ClassSymbol)component, state.getContextClass())){
          report(CLASS_NOT_ACCESSIBLE, ast, new Object[]{target, context});
          return false;
        }
      }
    }else{
      if(!isAccessible((ClassSymbol)target, context)){
        report(CLASS_NOT_ACCESSIBLE, ast, new Object[]{target, context});
        return false;
      }
    }
    return true;
  }
  
  public Object visit(FieldOrMethodRef ast, Object context) {
    ClassNode contextClass = state.getContextClass();
    ExpressionNode target = processExpression(ast.getTarget(), context);
    if(target == null) return null;
    if(target.type().isBasicType() || target.type().isNullType()){
      report(
        INCOMPATIBLE_TYPE, ast.getTarget(),
        new TypeSymbol[]{state.rootClass(), target.type()});
      return null;
    }
    ObjectSymbol targetType = (ObjectSymbol) target.type();
    if(!checkAccessible(ast, targetType, contextClass)) return null;
    String name = ast.getName();
    if(target.type().isArrayType()){
      if(name.equals("length") || name.equals("size")){
        return new ArrayLengthNode(target);
      }else{
        return null;
      }
    }
    FieldSymbol field = findField(targetType, name);
    if(field != null && isAccessible(field, state.getContextClass())){
      return new FieldRefNode(target, field);
    }
    Object[] result;
    boolean continuable;
    
    result = tryFindMethod(ast, targetType, name, new ExpressionNode[0]);
    if(result[1] != null){
      return new MethodCallNode(target, (MethodSymbol)result[1], new ExpressionNode[0]);
    }
    continuable = ((Boolean)result[0]).booleanValue();
    if(!continuable) return null;
    
    String getterName;
    getterName = createGetter(name);
    result = tryFindMethod(ast, targetType, getterName, new ExpressionNode[0]);
    if(result[1] != null){
      return new MethodCallNode(target, (MethodSymbol)result[1], new ExpressionNode[0]);
    }
    continuable = ((Boolean)result[0]).booleanValue();
    if(!continuable) return null;
    
    getterName = createBooleanGetter(name);
    result = tryFindMethod(ast, targetType, getterName, new ExpressionNode[0]);
    if(result[1] != null){
      return new MethodCallNode(target, (MethodSymbol)result[1], new ExpressionNode[0]);
    }
    
    if(field == null){
      report(FIELD_NOT_FOUND, ast, new Object[]{targetType, ast.getName()});
    }else{
      report(
        FIELD_NOT_ACCESSIBLE, ast, 
        new Object[]{targetType, ast.getName(), state.getContextClass()});
    }
    return null;
  }
  
  private Object[] tryFindMethod(
    ASTNode ast, ObjectSymbol target, String name, ExpressionNode[] params
  ) {
    MethodSymbol[] methods;
    methods = target.findMethod(name, params);
    if(methods.length > 0){
	    if(methods.length > 1){
	      report(
	        AMBIGUOUS_METHOD, ast,
	        new Object[]{
	          new Object[]{
	            methods[0].getClassType(), name, methods[0].getArguments()
	          },
	          new Object[]{
	            methods[1].getClassType(), name, methods[1].getArguments()
	          }
	        });
	      return new Object[]{Boolean.valueOf(false), null};
	    }
	    if(!isAccessible(methods[0], state.getContextClass())){
	      report(
	        METHOD_NOT_ACCESSIBLE, ast,
	        new Object[]{
	          methods[0].getClassType(), name, methods[0].getArguments(), 
	          state.getContextClass()
	        }
	      );
	      return new Object[]{Boolean.valueOf(false), null};
	    }
	    return new Object[]{Boolean.valueOf(false), methods[0]};
    }
    return new Object[]{Boolean.valueOf(true), null};
  }
  
  private String createGetter(String name) {
    return "get" + Character.toUpperCase(name.charAt(0)) + name.substring(1);
  }
  
  private String createBooleanGetter(String name) {
    return "is" + Character.toUpperCase(name.charAt(0)) + name.substring(1);
  }
  
  private String createSetter(String name) {
    return "set" + Character.toUpperCase(name.charAt(0)) + name.substring(1);
  }
  
  public Object visit(ArgumentDeclaration ast, Object context) {
    LocalContext local = ((LocalContext)context);
    String name = ast.getName();
    ClosureLocalBinding binding = local.lookupOnlyCurrentScope(name);
    if(binding != null){
      report(DUPLICATE_LOCAL_VARIABLE, ast, new Object[]{name});
      return null;
    }
    TypeSymbol type = resolve(ast.getType(), state.getCurrentResolver());
    if(type == null) return null;
    local.addEntry(name, type);
    return type;
  }
  
  public Object visit(ArrayCreation ast, Object context) {
    TypeSymbol type = resolve(ast.getType(), state.getCurrentResolver());
    ExpressionNode[] parameters = processExpressions(ast.getArguments(), context);
    if(type == null || parameters == null) return null;
    ArraySymbol resultType = state.loadArray(type, parameters.length);
    return new NewArrayNode(resultType, parameters);
  }
    
  public Object visit(Cast ast, Object context) {
    ExpressionNode node = processExpression(ast.getTarget(), context);
    if(node == null) return null;
    TypeSymbol conversion = resolve(ast.getConvertType(), state.getCurrentResolver());
    if(conversion == null) return null;
    node = new CastNode(node, conversion);
    return node;
  }
  
  public boolean equals(TypeSymbol[] types1, TypeSymbol[] types2) {
    if(types1.length != types2.length) return false;
    for(int i = 0; i < types1.length; i++){
      if(types1[i] != types2[i]) return false;
    }
    return true;
  }
  
  public Object visit(ClosureExpression ast, Object context) {
    LocalContext local = ((LocalContext)context);
    ClassSymbol type = (ClassSymbol) state.resolve(ast.getType());
    ArgumentDeclaration[] args = ast.getArguments();
    TypeSymbol[] argTypes = new TypeSymbol[args.length];
    String name = ast.getName();
    try {
	    local.openFrame();
	    boolean error = false;
	    for(int i = 0; i < args.length; i++){
	      argTypes[i] = (TypeSymbol)accept(args[i], context);
	      if(argTypes[i] == null){
		      error = true;
	      }
	    }	    
	    if(type == null) return null;
	    if(!type.isInterface()){
	      report(INTERFACE_REQUIRED, ast.getType(), new Object[]{type});
	      return null;
	    }
	    if(error) return null;
	    MethodSymbol[] methods = type.getMethods();
	    MethodSymbol method = null;
	    for(int i = 0; i < methods.length; i++){
	      TypeSymbol[] types = methods[i].getArguments();
	      if(name.equals(methods[i].getName()) && equals(argTypes, types)){
	        method = methods[i];
	        break;
	      }
	    }
	    if(method == null){
	      report(METHOD_NOT_FOUND, ast, new Object[]{type, name, argTypes});
	      return null;
	    }
	    local.setMethod(method);
	    local.getContextFrame().getParent().setAllClosed(true);
	    StatementNode block = processStatement(ast.getBlock(), context);
	    block = addReturnNode(block, method.getReturnType());
	    ClosureNode node = new ClosureNode(type, method, block);
	    node.setFrame(local.getContextFrame());
		  return node;
    }finally{
	    local.closeFrame();
    }	    
  }
  
  public Object visit(Indexing ast, Object context) {
    ExpressionNode target = processExpression(ast.getLeft(), context);
    ExpressionNode index = processExpression(ast.getRight(), context);
    if(target == null || index == null) return null;
    if(target.isArrayType()){
      if(!(index.isBasicType() && ((BasicSymbol)index.type()).isInteger())){
        report(
          INCOMPATIBLE_TYPE, ast, new Object[]{BasicSymbol.INT, index.type()});
        return null;
      }
      return new ArrayRefNode(target, index);
    }    
    if(target.isBasicType()){
      report(
        INCOMPATIBLE_TYPE,
        ast.getLeft(), new Object[]{state.rootClass(), target.type()});
      return null;
    }
    if(target.isArrayType()){
      if(!(index.isBasicType() && ((BasicSymbol)index.type()).isInteger())){
        report(
          INCOMPATIBLE_TYPE, 
          ast.getRight(), new Object[]{BasicSymbol.INT, index.type()}
        );
        return null;
      }
      return new ArrayRefNode(target, index);
    }    
    ExpressionNode[] params;
    params = new ExpressionNode[]{index};
    Object[] result = tryFindMethod(
      ast, (ObjectSymbol)target.type(), 
      "get", new ExpressionNode[]{index});
    if(result[1] == null){
      report(
        METHOD_NOT_FOUND, ast, 
        new Object[]{target.type(), "get", types(params)});
      return null;
    }
    return new MethodCallNode(target, (MethodSymbol)result[1], params);    
  }
  
  public Object visit(SelfFieldReference ast, Object context) {
    LocalContext local = (LocalContext) context;
    ClassSymbol selfClass = null;
    if(local.isStatic()) return null;
    selfClass = state.getContextClass();
    FieldSymbol field = findField(selfClass, ast.getName());
    if(field == null){
      report(FIELD_NOT_FOUND, ast, new Object[]{selfClass, ast.getName()});
      return null;
    }
    if(!isAccessible(field, selfClass)){
      report(
        FIELD_NOT_ACCESSIBLE, ast, 
        new Object[]{field.getClassType(), ast.getName(), selfClass});
      return null;
    }    
    return new FieldRefNode(new SelfNode(selfClass), field);
  }
  
  public Object visit(InstanceCreation ast, Object context) {
    ClassSymbol type = (ClassSymbol) state.resolve(ast.getType());
    ExpressionNode[] parameters = processExpressions(ast.getArguments(), context);
    if(parameters == null || type == null) return null;
    ConstructorSymbol[] constructors = type.findConstructor(parameters);
    if(constructors.length == 0){
      report(CONSTRUCTOR_NOT_FOUND, ast, new Object[]{type, types(parameters)});
      return null;
    }
    if(constructors.length > 1){
      report(
        AMBIGUOUS_CONSTRUCTOR, ast,
        new Object[]{
          new Object[]{
            constructors[0].getClassType(), 
            constructors[0].getArguments()
          },
          new Object[]{
            constructors[1].getClassType(),
            constructors[1].getArguments()
          }
        });
      return null;
    }
    return new NewNode(constructors[0], parameters);
  }
      
  public Object visit(IsInstance ast, Object context) {
    ExpressionNode target = processExpression(ast.target, context);
    TypeSymbol checkType = resolve(ast.type, state.getCurrentResolver());
    if(target == null || checkType == null) return null;
    return new IsInstanceNode(target, checkType);      
  }
  
  private TypeSymbol[] types(ExpressionNode[] parameters){
    TypeSymbol[] types = new TypeSymbol[parameters.length];
    for(int i = 0; i < types.length; i++){
      types[i] = parameters[i].type();
    }
    return types;
  }
  
  public Object visit(SelfMethodCall ast, Object context) {
    ExpressionNode[] params = processExpressions(ast.getArguments(), context);
    if(params == null) return null;
    ClassNode targetType = state.getContextClass();
    String name = ast.getName();
    MethodSymbol[] methods = targetType.findMethod(ast.getName(), params);
    
    if(methods.length == 0){
      report(
        METHOD_NOT_FOUND, ast, 
        new Object[]{targetType, name, types(params)});
      return null;
    }
    
    if(methods.length > 1){
      report(
        AMBIGUOUS_METHOD, ast,
        new Object[]{
          new Object[]{
            methods[0].getClassType(), name, methods[0].getArguments()
          },
          new Object[]{
            methods[1].getClassType(), name, methods[1].getArguments()
          }
        });
      return null;
    }
    
    /*
     * TODO check illegal method call
     * ex. instance method call in the static context 
     */
    
    params = convert(methods[0].getArguments(), params);
    
    if((methods[0].getModifier() & Modifier.STATIC) != 0){
      return new StaticMethodCallNode(targetType, methods[0], params);
    }else {
      return new MethodCallNode(new SelfNode(targetType), methods[0], params);
    }
  }
  
  private ExpressionNode[] convert(TypeSymbol[] arguments, ExpressionNode[] params){
    for(int i = 0; i < params.length; i++){
      if(arguments[i] != params[i].type()){
        params[i] = new CastNode(params[i], arguments[i]);
      }
    }
    return params;
  }
  
  public Object visit(MethodCall ast, Object context) {
    ExpressionNode target = processExpression(ast.getTarget(), context);
    if(target == null) return null;
    ExpressionNode[] params = processExpressions(ast.getArguments(), context);
    if(params == null) return null;
    ObjectSymbol targetType = (ObjectSymbol) target.type();
    final String name = ast.getName();
    MethodSymbol[] methods = targetType.findMethod(name, params);
    
    if(methods.length == 0){
      report(
        METHOD_NOT_FOUND, ast, new Object[]{targetType, name, types(params)});
      return null;
    }
    if(methods.length > 1){
      report(
        AMBIGUOUS_METHOD, ast,
        new Object[]{
          new Object[]{
            methods[0].getClassType(), name, methods[0].getArguments()
          },
          new Object[]{
            methods[1].getClassType(), name, methods[1].getArguments()
          }
        });
      return null;
    }    
    if((methods[0].getModifier() & Modifier.STATIC) != 0){
      report(
        ILLEGAL_METHOD_CALL, ast,
        new Object[]{
          methods[0].getClassType(), name, methods[0].getArguments()
        });
      return null;
    }    
    params = convert(methods[0].getArguments(), params);
    return new MethodCallNode(target, methods[0], params);
  }
  
  public Object visit(StaticIDExpression ast, Object context) {
    ClassSymbol type = (ClassSymbol) state.resolve(ast.classType);
    if(type == null) return null;
    FieldSymbol field = findField(type, ast.fieldName);
    if(field == null){
      report(FIELD_NOT_FOUND, ast, new Object[]{type, ast.fieldName});
      return null;
    }
    return new StaticFieldRefNode(type, field);
  }
  
  public Object visit(StaticMethodCall ast, Object context) {
    ClassSymbol type = (ClassSymbol) state.resolve(ast.getTarget());
    ExpressionNode[] params = processExpressions(ast.getArguments(), context);
    if(params == null) return null;
    MethodSymbol[] methods = type.findMethod(ast.getMethodName(), params);
    
    if(methods.length == 0){
      report(
        METHOD_NOT_FOUND, ast, 
        new Object[]{type, ast.getMethodName(), types(params)});
      return null;
    }
    if(methods.length > 1){
      report(
        AMBIGUOUS_METHOD, ast,
        new Object[]{
          ast.getMethodName(),
          typeNames(methods[0].getArguments()), typeNames(methods[1].getArguments())
        });
      return null;
    }
    
    params = convert(methods[0].getArguments(), params);
    return new StaticMethodCallNode(type, methods[0], params);
  }
    
  private String[] typeNames(TypeSymbol[] types) {
    String[] names = new String[types.length];
    for(int i = 0; i < names.length; i++){
      names[i] = types[i].getName();
    }
    return names;
  }
  
  private String[] parameterTypeNames(ExpressionNode[] parameters) {
    String[] names = new String[parameters.length];
    for(int i = 0; i < names.length; i++){
      names[i] = parameters[i].type().getName();
    }
    return names;
  }
  
  private ExpressionNode[] processExpressions(Expression[] ast, Object context){
    ExpressionNode[] expressions = new ExpressionNode[ast.length];
    boolean success = true;
    for(int i = 0; i < ast.length; i++){
      expressions[i] = processExpression(ast[i], context);
      if(expressions[i] == null){
        success = false;
      }
    }
    if(success){
      return expressions;
    }else{
      return null;
    }
  }
//-------------------------------------------------------------------------//

//------------------------- statements ------------------------------------//
  public Object visit(ForeachStatement ast, Object context) {
    Expression collectionAST = ast.getCollection();
    LocalContext local = (LocalContext) context;
    try {
      local.openScope();
	    ExpressionNode collection = processExpression(collectionAST, context);
	    ArgumentDeclaration arg = ast.getDeclaration();
	    accept(arg, context);
	    ClosureLocalBinding bind = local.lookupOnlyCurrentScope(arg.getName());
	    StatementNode block = processStatement(ast.getStatement(), context);
	    
	    if(collection.isBasicType()){
	      report(
	        INCOMPATIBLE_TYPE, collectionAST,
	        new Object[]{state.load("java.util.Collection"), collection.type()});
	      return null;
	    }
	    ClosureLocalBinding bind2 = new ClosureLocalBinding(
	      0, local.addEntry(local.generateName(), collection.type()), collection.type());
	    StatementNode init = 
	      new ExpressionStatementNode(new LocalAssignmentNode(bind2, collection));
	    if(collection.isArrayType()){
	      ClosureLocalBinding bind3 = new ClosureLocalBinding(
	        0, local.addEntry(local.generateName(), BasicSymbol.INT), BasicSymbol.INT);
	      init = new BlockNode(
	        new StatementNode[]{
	          init, 
	          new ExpressionStatementNode(
	            new LocalAssignmentNode(bind3, new IntegerNode(0)))
	        });      
	      block = new LoopNode(
	        new BinaryExpressionNode(
	          LESS_THAN, BasicSymbol.BOOLEAN,
	          ref(bind3),
	          new ArrayLengthNode(ref(bind2))
	        ),
	        new BlockNode(
	          new StatementNode[]{
	            assign(bind, indexref(bind2, ref(bind3))),
	            block,
	            assign(
	              bind3, 
	              new BinaryExpressionNode(
	                ADD, BasicSymbol.INT, ref(bind3), new IntegerNode(1)))
	        }));
	      return new BlockNode(new StatementNode[]{init, block});
	    }else{
	      ObjectSymbol iterator = state.load("java.util.Iterator");
	      ClosureLocalBinding bind3 = new ClosureLocalBinding(
	        0, local.addEntry(local.generateName(), iterator), iterator);
	      MethodSymbol getIterator, getNext, hasNext;
	      getIterator =
	        findMethod(collectionAST, 
	          (ObjectSymbol) collection.type(), "iterator");
	      getNext = findMethod(ast.getCollection(), iterator, "next");
	      hasNext = findMethod(ast.getCollection(), iterator, "hasNext");
	      init = new BlockNode(
	        new StatementNode[]{
	          init, 
	          assign(
	            bind3,
	            new MethodCallNode(ref(bind2), getIterator, new ExpressionNode[0]))
	        });
	      ExpressionNode callGetNext = new MethodCallNode(
	        ref(bind3), getNext, new ExpressionNode[0]
	      );
	      if(bind.getType() != state.rootClass()){
	        callGetNext = new CastNode(callGetNext, bind.getType());
	      }
	      block = new LoopNode(
	        new MethodCallNode(ref(bind3), hasNext, new ExpressionNode[0]),
	        new BlockNode(
	          new StatementNode[]{
	            assign(bind, callGetNext),
	            block,
	        }));
	      return new BlockNode(new StatementNode[]{init, block});
	    }
    }finally{
      local.closeScope();
    }
  }
  
  private ExpressionNode indexref(ClosureLocalBinding bind, ExpressionNode value) {
    return new ArrayRefNode(new LocalRefNode(bind), value);
  }
  
  private StatementNode assign(ClosureLocalBinding bind, ExpressionNode value) {
    return new ExpressionStatementNode(new LocalAssignmentNode(bind, value));
  }
  
  private ExpressionNode ref(ClosureLocalBinding bind) {
    return new LocalRefNode(bind);
  }
  
  public Object visit(ExpressionStatement ast, Object context) {
    ExpressionNode expression = processExpression(ast.expression, context);
    return new ExpressionStatementNode(expression);
  }
  
  public Object visit(CondStatement ast, Object context) {
    LocalContext local = (LocalContext) context;
    local.openScope();
    IfNode node = processCond(ast, 0, context);
    local.closeScope();
    return node;
  }
  
  IfNode processCond(CondStatement ast, int index, Object context){
    Expression astCondition = ast.getCondition(index);
    ExpressionNode condition = processExpression(astCondition, context);
    if(condition != null && condition.type() != BasicSymbol.BOOLEAN){
      TypeSymbol expected = BasicSymbol.BOOLEAN;
      TypeSymbol appeared = condition.type();
      report(INCOMPATIBLE_TYPE, astCondition, new Object[]{expected, appeared});
    }
    StatementNode thenBlock = processStatement(ast.getBlock(index), context);
    StatementNode elseBlock;
    if(index >= ast.size() - 1){
      if(ast.getElseBlock() != null){
        elseBlock = processStatement(ast.getElseBlock(), context);
      }else{
        elseBlock = null;
      }
    }else{
      elseBlock = processCond(ast, index + 1, context);      
    }
    return new IfNode(condition, thenBlock, elseBlock);
  }
  
  public Object visit(ForStatement ast, Object context) {
    LocalContext local = (LocalContext) context;
    try{
	    local.openScope();
	    
	    StatementNode init = null;
	    if(ast.getInitializer() != null){
	      init = processStatement(ast.getInitializer(), context);
	    }else{
	      init = new EmptyNode();
	    }
	    ExpressionNode condition;
	    Expression astCondition = ast.getCondition();
	    if(astCondition != null){
	      condition = processExpression(ast.getCondition(), context);
	      TypeSymbol expected = BasicSymbol.BOOLEAN;
	      if(condition != null && condition.type() != expected){
		      TypeSymbol appeared = condition.type();
	        report(INCOMPATIBLE_TYPE, astCondition, new Object[]{expected, appeared});
	      }
	    }else{
	      condition = new BooleanNode(true);
	    }
	    ExpressionNode update = null;
	    if(ast.getIncrement() != null){
	      update = processExpression(ast.getIncrement(), context);
	    }
	    StatementNode loop = processStatement(
	      ast.getBlock(), context);
	    if(update != null){
	      loop = new BlockNode(
	        new StatementNode[]{loop, new ExpressionStatementNode(update)});
	    }
	    StatementNode result = new LoopNode(condition, loop);
	    result = new BlockNode(new StatementNode[]{init, result});
	    return result;
    }finally{
	    local.closeScope();
    }
  }
  
  public Object visit(BlockStatement ast, Object context) {
    Statement[] astStatements = ast.getStatements();
    StatementNode[] statements = new StatementNode[astStatements.length];
    LocalContext local = (LocalContext) context;
    try{
      local.openScope();
      for(int i = 0; i < astStatements.length; i++){
        statements[i] = processStatement(astStatements[i], context);
      }
      return new BlockNode(statements);
    }finally{
      local.closeScope();
    }
  }
  
  public Object visit(IfStatement ast, Object context) {
    LocalContext local = ((LocalContext)context);
    try{
	    local.openScope();
	    ExpressionNode condition = processExpression(ast.condition, context);
	    TypeSymbol expected = BasicSymbol.BOOLEAN;
	    if(condition != null && condition.type() != expected){
	      TypeSymbol appeared = condition.type();
	      report(INCOMPATIBLE_TYPE, ast.condition, new Object[]{expected, appeared});
	    }
	    StatementNode thenBlock = processStatement(ast.thenBlock, context);
	    StatementNode elseBlock = null;
	    if(ast.elseBlock != null){
	      elseBlock = processStatement(ast.elseBlock, context);
	    }
	    return new IfNode(condition, thenBlock, elseBlock);
    }finally{	    
	    local.closeScope();
    }
  }
  
  public Object visit(WhileStatement ast, Object context) {
    LocalContext local = ((LocalContext)context);
    try{
	    local.openScope();
	    ExpressionNode condition = processExpression(ast.condition, context);
	    TypeSymbol expected = BasicSymbol.BOOLEAN;
	    if(condition != null && condition.type() != expected){
	      TypeSymbol appeared = condition.type();
	      report(INCOMPATIBLE_TYPE, ast, new Object[]{expected, appeared});
	    }
	    StatementNode thenBlock = processStatement(ast.thenBlock, context);
	    return new LoopNode(condition, thenBlock);
    }finally{
	    local.closeScope();
    }	    
  }
  
  public Object visit(ReturnStatement ast, Object context) {
    TypeSymbol returnType = ((LocalContext)context).getReturnType();
    if(ast.returned == null){
      TypeSymbol expected = BasicSymbol.VOID;
      if(returnType != expected){
        report(CANNOT_RETURN_VALUE, ast, new Object[0]);
      }
      return new ReturnNode(null);
    }else{
	    ExpressionNode returned = processExpression(ast.returned, context);
	    if(returned == null) return new ReturnNode(null);
	    if(returned.type() == BasicSymbol.VOID){
	      report(CANNOT_RETURN_VALUE, ast, new Object[0]);
	    }else {
	      returned = processAssignable(ast.returned, returnType, returned);
	      if(returned == null) return new ReturnNode(null);
	    }
	    return new ReturnNode(returned);
    }
  }
  
  ExpressionNode processAssignable(ASTNode ast, TypeSymbol a, ExpressionNode b){
    if(b == null) return null;
    if(a == b.type()) return b;
    if(!TypeRules.isAssignable(a, b.type())){
      report(INCOMPATIBLE_TYPE, ast, new Object[]{ a, b.type() });
      return null;
    }
    b = new CastNode(b, a);
    return b;
  }
  
  public Object visit(SelectStatement ast, Object context) {
    LocalContext local = (LocalContext) context;
    try{
	    local.openScope();
	    ExpressionNode condition = processExpression(ast.getCondition(), context);
	    if(condition == null){
	      return new EmptyNode();
	    }
	    String name = local.generateName();
	    int index = local.addEntry(name, condition.type());
	    StatementNode statement;
	    if(ast.getCases().length == 0){
	      if(ast.getElseBlock() != null){
	        statement = processStatement(ast.getElseBlock(), context);
	      }else{
	        statement = new EmptyNode();
	      }
	    }else{
	      statement = processCases(ast, condition, name, (LocalContext) context);
	    }
	    BlockNode block = new BlockNode(
	      new ExpressionStatementNode(
	        new LocalAssignmentNode(0, index, condition.type(), condition)),
	        statement);
	    return block;
    }finally{
	    local.closeScope();
    }
  }
  
  private StatementNode processCases(
    SelectStatement ast, ExpressionNode cond, String var, LocalContext context
  ) {
    CaseBranch[] cases = ast.getCases();
    List nodes = new ArrayList();
    List thens = new ArrayList();
    for(int i = 0; i < cases.length; i++){
      Expression[] astExpressions = cases[i].getExpressions();
      ClosureLocalBinding bind = context.lookup(var);
      nodes.add(processNodes(astExpressions, cond.type(), bind, context));
      thens.add(processStatement(cases[i].getBlock(), context));
    }
    StatementNode statement;
    if(ast.getElseBlock() != null){
      statement = processStatement(ast.getElseBlock(), context);
    }else{
      statement = null;
    }
    for(int i = cases.length - 1; i >= 0; i--){
      ExpressionNode value = (ExpressionNode) nodes.get(i);
      StatementNode then = (StatementNode) thens.get(i);
      statement = new IfNode(value, then, statement);
    }
    return statement;
  }
  
  ExpressionNode processNodes(
    Expression[] asts, TypeSymbol type, ClosureLocalBinding bind, Object context
  ) {
    ExpressionNode[] nodes = new ExpressionNode[asts.length];
    boolean error = false;
    for(int i = 0; i < asts.length; i++){
      nodes[i] = processExpression(asts[i], context);
      if(nodes[i] == null){
        error = true;
        continue;
      }
      if(!TypeRules.isAssignable(type, nodes[i].type())){
        report(INCOMPATIBLE_TYPE, asts[i], new TypeSymbol[]{type, nodes[i].type()});
        error = true;
        continue;
      }
      if(nodes[i].isBasicType() && nodes[i].type() != type){
        nodes[i] = new CastNode(nodes[i], type);
      }
      if(nodes[i].isReferenceType() && nodes[i].type() != state.rootClass()){
        nodes[i] = new CastNode(nodes[i], state.rootClass());
      }
    }
    if(!error){
      ExpressionNode node = new BinaryExpressionNode(
        EQUAL, BasicSymbol.BOOLEAN, new LocalRefNode(bind), nodes[0]);
      for(int i = 1; i < nodes.length; i++){
        node = new BinaryExpressionNode(
          LOGICAL_OR, BasicSymbol.BOOLEAN, node,
          new BinaryExpressionNode(
            EQUAL, BasicSymbol.BOOLEAN, new LocalRefNode(bind), nodes[i]));
      }
      return node;
    }else{
      return null;
    }
  }
  
  public Object visit(ThrowStatement ast, Object context) {
    ExpressionNode expression = processExpression(ast.expression, context);
    if(expression != null){
      TypeSymbol expected = state.load("java.lang.Throwable");
      TypeSymbol detected = expression.type();
      if(!TypeRules.isSuperType(expected, detected)){
        report(
          INCOMPATIBLE_TYPE, ast.expression, 
          new Object[]{ expected, detected}
        );
      }
    }
    return new ThrowNode(expression);
  }
  
  public Object visit(LocalVariableDeclaration ast, Object context) {
    Expression initializer = ast.getInitializer();
    LocalContext localContext = ((LocalContext)context);
    ClosureLocalBinding 
    	binding = localContext.lookupOnlyCurrentScope(ast.getName());
    if(binding != null){
      report(
        DUPLICATE_LOCAL_VARIABLE, ast,
        new Object[]{ast.getName()});
      return new EmptyNode();
    }
    TypeSymbol leftType = state.resolve(ast.getType());
    if(leftType == null) return new EmptyNode();
    int index = localContext.addEntry(ast.getName(), leftType);
    LocalAssignmentNode node;
    if(initializer != null){
      ExpressionNode valueNode = processExpression(initializer, context);
      if(valueNode == null) return new EmptyNode();
      valueNode = processAssignable(initializer, leftType, valueNode);
      if(valueNode == null) return new EmptyNode();
      node = new LocalAssignmentNode(0, index, leftType, valueNode);
    }else{
      node = new LocalAssignmentNode(0, index, leftType, defaultValue(leftType));
    }
    return new ExpressionStatementNode(node);
  }
  
  public static ExpressionNode defaultValue(TypeSymbol type) {
    return ExpressionNode.defaultValue(type);
  }
  
  public Object visit(EmptyStatement ast, Object context) {
    return new EmptyNode();
  }
  
  public Object visit(TryStatement ast, Object context) {
    LocalContext local = (LocalContext) context;   
    StatementNode tryStatement = processStatement(ast.tryBlock, context);
    ClosureLocalBinding[] binds = new ClosureLocalBinding[ast.arguments.length];
    StatementNode[] catchBlocks = new StatementNode[ast.arguments.length];
    for(int i = 0; i < ast.arguments.length; i++){
      local.openScope();
      TypeSymbol arg = (TypeSymbol)accept(ast.arguments[i], context);
      TypeSymbol expected = state.load("java.lang.Throwable");
      if(!TypeRules.isSuperType(expected, arg)){
        report(INCOMPATIBLE_TYPE, ast.arguments[i], new Object[]{expected, arg});
      }
      binds[i] = local.lookupOnlyCurrentScope(ast.arguments[i].getName());
      catchBlocks[i] = processStatement(ast.recBlocks[i], context);
      local.closeScope();
    }
    return new TryNode(tryStatement, binds, catchBlocks);
  }
  
  public Object visit(SynchronizedStatement ast, Object context) {
    ExpressionNode lock = processExpression(ast.target, context);
    StatementNode block = processStatement(ast.block, context);
    report(UNIMPLEMENTED_FEATURE, ast, new Object[0]);
    return new SynchronizedNode(lock, block);
  }
  
  public Object visit(BreakStatement ast, Object context) {
    report(UNIMPLEMENTED_FEATURE, ast, new Object[0]);
    return new BreakNode();
  }
  
  public Object visit(ContinueStatement ast, Object context) {
    report(UNIMPLEMENTED_FEATURE, ast, new Object[0]);
    return new ContinueNode();
  }
//-------------------------------------------------------------------------//
  
//----------------------------- members ----------------------------------------//  
  public Object visit(FunctionDeclaration ast, Object context) {
    MethodNode function = (MethodNode) state.lookupKernelNode(ast);
    if(function == null) return null;
    LocalContext local = new LocalContext();
    if(Modifier.isStatic(function.getModifier())){
      local.setStatic(true);
    }
    local.setMethod(function);
    TypeSymbol[] arguments = function.getArguments();
    for(int i = 0; i < arguments.length; i++){
      local.addEntry(ast.getArguments()[i].getName(), arguments[i]);
    }
    BlockNode block = (BlockNode) accept(ast.getBlock(), local);
    block = addReturnNode(block, function.getReturnType());
    function.setBlock(block);
    function.setFrame(local.getContextFrame());
    return null;
  }
  
  public BlockNode addReturnNode(StatementNode node, TypeSymbol returnType) {
    return new BlockNode(node, new ReturnNode(defaultValue(returnType)));
  }
  
  public Object visit(GlobalVariableDeclaration ast, Object context) {
    return null;
  }
  
  public Object visit(FieldDeclaration ast, Object context) {
    return null;
  }
  
  public Object visit(DelegationDeclaration ast, Object context) {
    return null;
  }
  
  public Object visit(InterfaceMethodDeclaration ast, Object context) {
    return null;
  }
  
  public Object visit(MethodDeclaration ast, Object context) {
    MethodNode method = (MethodNode) state.lookupKernelNode(ast);
    if(method == null) return null;
    LocalContext local = new LocalContext();
    if(Modifier.isStatic(method.getModifier())){
      local.setStatic(true);
    }
    local.setMethod(method);
    TypeSymbol[] arguments = method.getArguments();
    for(int i = 0; i < arguments.length; i++){
      local.addEntry(ast.getArguments()[i].getName(), arguments[i]);
    }    
    BlockNode block = (BlockNode) accept(ast.getBlock(), local);
    block = addReturnNode(block, method.getReturnType());
    method.setBlock(block);
    method.setFrame(local.getContextFrame());
    return null;
  }
  
  public Object visit(ConstructorDeclaration ast, Object context) {
    ConstructorNode constructor = (ConstructorNode) state.lookupKernelNode(ast);
    if(constructor == null) return null;
    LocalContext local = new LocalContext();
    local.setConstructor(constructor);
    TypeSymbol[] arguments = constructor.getArguments();
    for(int i = 0; i < arguments.length; i++){
      local.addEntry(ast.getArguments()[i].getName(), arguments[i]);
    }
    ExpressionNode[] params = processExpressions(ast.getInitializers(), context);
    BlockNode block = (BlockNode) accept(ast.getBlock(), local);
    ClassNode currentClass = state.getContextClass();
    ClassSymbol superClass = currentClass.getSuperClass();
    ConstructorSymbol[] matched = superClass.findConstructor(params);
    if(matched.length == 0){
      report(CONSTRUCTOR_NOT_FOUND, ast, new Object[]{superClass, types(params)});
      return null;
    }
    if(matched.length > 1){
      report(
        AMBIGUOUS_CONSTRUCTOR, ast,
        new Object[]{
          new Object[]{superClass, types(params)},
          new Object[]{superClass, types(params)}
        });
      return null;
    }
    SuperInitNode init = new SuperInitNode(
      superClass, matched[0].getArguments(), params
    );
    constructor.setSuperInitializer(init);
    block = addReturnNode(block, BasicSymbol.VOID);
    constructor.setBlock(block);
    constructor.setFrame(local.getContextFrame());
    return null;
  }
//----------------------------------------------------------------------------// 
//----------------------------------------------------------------------------//

  public TypeSymbol resolve(TypeSpecifier type, NameResolution resolver) {
    TypeSymbol resolvedType = (TypeSymbol) resolver.resolve(type);
    if(resolvedType == null){
      report(CLASS_NOT_FOUND, type, new Object[]{type.getComponentName()});      
    }
    return resolvedType;
  }
  
  public void report(int error, ASTNode node, Object[] items) {
    state.report(error, node, items);
  }

  public static TypeSymbol promoteNumericTypes(
    TypeSymbol left,  TypeSymbol right
  ) {
    if(!numeric(left) || !numeric(right)) return null;
    if(left == BasicSymbol.DOUBLE || right == BasicSymbol.DOUBLE){
      return BasicSymbol.DOUBLE;
    }
    if(left == BasicSymbol.FLOAT || right == BasicSymbol.FLOAT){
      return BasicSymbol.FLOAT;
    }
    if(left == BasicSymbol.LONG || right == BasicSymbol.LONG){
      return BasicSymbol.LONG;
    }
    return BasicSymbol.INT;
  }
  
  public static boolean hasNumericType(ExpressionNode expression){
    return numeric(expression.type());
  }
  
  private static boolean numeric(TypeSymbol symbol){
    return 
  	(symbol.isBasicType()) &&
  	(	symbol == BasicSymbol.BYTE || symbol == BasicSymbol.SHORT ||
  	  symbol == BasicSymbol.CHAR || symbol == BasicSymbol.INT || 
  	  symbol == BasicSymbol.LONG || symbol == BasicSymbol.FLOAT || 
  	  symbol == BasicSymbol.DOUBLE);
  }
  
  ExpressionNode processExpression(Expression expression, Object context){
    return (ExpressionNode) expression.accept(this, context);
  }
    
  StatementNode processStatement(Statement statement, Object context){
    return (StatementNode) statement.accept(this, context);
  }
}
