/* ************************************************************** *
 *                                                                *
 * Copyright (c) 2005, Kota Mizushima, All rights reserved.       *
 *                                                                *
 *                                                                *
 * This software is distributed under the modified BSD License.   *
 * ************************************************************** */
package org.onion_lang.onion.compiler.phase;

import java.util.*;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.bcel.Constants;
import org.apache.bcel.classfile.JavaClass;
import org.apache.bcel.generic.*;
import org.apache.bcel.generic.DADD;
import org.apache.bcel.generic.DSUB;
import org.apache.bcel.generic.FADD;
import org.apache.bcel.generic.FSUB;
import org.apache.bcel.generic.IADD;
import org.apache.bcel.generic.ISUB;
import org.apache.bcel.generic.LADD;
import org.apache.bcel.generic.LSUB;
import org.onion_lang.onion.compiler.*;
import org.onion_lang.onion.compiler.CompilerConfig;
import org.onion_lang.onion.compiler.CompiledClass;
import org.onion_lang.onion.compiler.environment.*;
import org.onion_lang.onion.compiler.environment.LocalFrame;
import org.onion_lang.onion.compiler.phase.codegeneration.CodeProxy;
import org.onion_lang.onion.compiler.utility.*;
import org.onion_lang.onion.compiler.utility.Classes;
import org.onion_lang.onion.compiler.utility.Systems;
import org.onion_lang.onion.lang.kernel.*;
import org.onion_lang.onion.lang.kernel.type.*;
import org.onion_lang.onion.lang.syntax.Modifier;

/**
 * @author Kota Mizushima Date: 2005/04/10
 */
public class CodeGenerationPhase 
	implements BinaryExpressionNode.Constants, UnaryExpressionNode.Constants {
  private CompilerConfig config;
  private List compiledClasses = new ArrayList();
  private SymbolGenerator generator;
  
  private static Map basicTypeTable = new HashMap() {{
    put(BasicSymbol.BYTE, BasicType.BYTE);
    put(BasicSymbol.SHORT, BasicType.SHORT);
    put(BasicSymbol.CHAR, BasicType.CHAR);
    put(BasicSymbol.INT, BasicType.INT);
    put(BasicSymbol.LONG, BasicType.LONG);
    put(BasicSymbol.FLOAT, BasicType.FLOAT);
    put(BasicSymbol.DOUBLE, BasicType.DOUBLE);
    put(BasicSymbol.BOOLEAN, BasicType.BOOLEAN);
    put(BasicSymbol.VOID, BasicType.VOID);
  }};
  
  private static Map unboxingMethods = new HashMap(){{
    put("java.lang.Byte", "byteValue");
    put("java.lang.Short", "shortValue");
    put("java.lang.Character", "charValue");
    put("java.lang.Integer", "intValue");
    put("java.lang.Long", "longValue");
    put("java.lang.Float", "floatValue");
    put("java.lang.Double", "doubleValue");
    put("java.lang.Boolean", "booleanValue");
  }};
  
  private static final String FRAME_PREFIX = "frame";
  private static final String CLOSURE_CLASS_SUFFIX = "Closure";
  private VMTypeBridge bridge;
  
  public CodeGenerationPhase(CompilerConfig config) {
    this.config = config;
    this.bridge = new VMTypeBridge();
  }

  public CompiledClass[] process(ClassNode[] classes) {
    compiledClasses.clear();
    String base = config.getOutputDirectory();
    base = base != null ? base : ".";
    base += Systems.getFileSeparator();
    for (int i = 0; i < classes.length; i++) {
      codeClass(classes[i]);
    }
    CompiledClass[] classFiles = new CompiledClass[compiledClasses.size()];
    for(int i = 0; i < compiledClasses.size(); i++){
      JavaClass clazz = (JavaClass) compiledClasses.get(i);
      String outDir = getOutputDir(base, clazz.getClassName());
      classFiles[i] = new CompiledClass(clazz.getClassName(), outDir, clazz.getBytes());
    }
    return classFiles;
  }
  
  private String getOutputDir(String base, String fqcn){
    String packageName = getPackageName(fqcn);
    return base + packageName.replaceAll(".", Systems.getFileSeparator());
  }
  
  private String getPackageName(String fqcn){
    int index = fqcn.lastIndexOf("\\.");
    if(index < 0){
      return "";
    }else{
      return fqcn.substring(0, index);
    }
  }
  
  private static Integer id(int number) {
    return new Integer(number);
  }
  
  private int classModifier(ClassNode node){
    int modifier = toJavaModifier(node.getModifier());
    modifier |= node.isInterface() ? Constants.ACC_INTERFACE : modifier;
    modifier |= (!Modifier.isInternal(modifier)) ? Constants.ACC_PUBLIC : modifier;
    return modifier;
  }

  public void codeClass(ClassNode node) {
    int modifier = classModifier(node);
    String className = node.getName();
    generator = new SymbolGenerator(className + CLOSURE_CLASS_SUFFIX);
    String superClass = node.getSuperClass().getName();
    String[] interfaces = namesOf(node.getInterfaces());
    String file = node.getSourceFile();
    ClassGen gen = new ClassGen(className, superClass, file, modifier, interfaces);
    ConstructorSymbol[] constructors = node.getConstructors();
    for (int i = 0; i < constructors.length; i++) {
      codeConstructor(gen, ((ConstructorNode) constructors[i]));
    }
    MethodSymbol[] methods = node.getMethods();
    for (int i = 0; i < methods.length; i++) {
      codeMethod(gen, ((MethodNode) methods[i]));
    }
    FieldSymbol[] fields = node.getFields();
    for (int i = 0; i < fields.length; i++) {
      codeField(gen, ((FieldNode) fields[i]));
    }
    compiledClasses.add(gen.getJavaClass());
  }
  
  public InstructionHandle codeExpressions(ExpressionNode[] nodes, CodeProxy code){
    InstructionHandle start;
    if(nodes.length > 0){
      start = codeExpression(nodes[0], code);
      for(int i = 1; i < nodes.length; i++){
        codeExpression(nodes[i], code);
      }
    }else{
      start = code.append(InstructionConstants.NOP);
    }
    return start;
  }

  public void codeConstructor(ClassGen gen, ConstructorNode node) {
    CodeProxy code = new CodeProxy(gen.getConstantPool());
    LocalFrame frame = node.getFrame();
    code.setFrame(frame);
    String[] args = new String[node.getArguments().length];
    for (int i = 0; i < args.length; i++) {
      args[i] = "arg" + i;
    }
    ObjectType classType = (ObjectType) typeOf(node.getClassType());
    int modifier = toJavaModifier(node.getModifier());
    Type[] arguments = typesOf(node.getArguments());
    MethodGen method = new MethodGen(
      modifier, Type.VOID, arguments, args, "<init>",
      classType.getClassName(), code.getCode(), gen.getConstantPool());
    if(frame.isClosed()){
      int frameObjectIndex = frameObjectIndex(1, node.getArguments());
      code.setFrameObjectIndex(frameObjectIndex);
      code.setIndexTable(indexTableFor(frame));   
      appendInitialCode(code, frame, arguments, 1);
    }else{
      code.setIndexTable(indexTableFor(1, frame));
    }
    code.setMethod(method);
    SuperInitNode init = node.getSuperInitializer();
    classType = (ObjectType) typeOf(init.getClassType());
    arguments = typesOf(init.getArguments());    
    code.append(InstructionConstants.ALOAD_0);
    codeExpressions(init.getExpressions(), code);
    code.appendCallConstructor(classType, arguments);
    codeBlock(node.getBlock(), code);
    method.setMaxLocals();
    method.setMaxStack();
    code.appendReturn(typeOf(BasicSymbol.VOID));
    gen.addMethod(method.getMethod());
  }

  public void codeMethod(ClassGen gen, MethodNode node) {
    CodeProxy code = new CodeProxy(gen.getConstantPool());
    LocalFrame frame = node.getFrame();
    code.setFrame(frame);
    
    int modifier = toJavaModifier(node.getModifier());
    Type returned = typeOf(node.getReturnType());
    Type[] arguments = typesOf(node.getArguments());
    String[] argNames = names(arguments.length);
    String name = node.getName();
    String className = node.getClassType().getName();
    MethodGen method = new MethodGen(
      modifier, returned, arguments, argNames, name, className,
      code.getCode(), gen.getConstantPool());
    code.setMethod(method);
    if (!Modifier.isAbstract(node.getModifier())) {   
      if(frame.isClosed()){
        int origin;
        if(Modifier.isStatic(node.getModifier())){
          code.setFrameObjectIndex(
            frameObjectIndex(0, node.getArguments()));
          origin = 0;
        }else{
          code.setFrameObjectIndex(
            frameObjectIndex(1, node.getArguments()));
          origin = 1;
        }
        code.setIndexTable(indexTableFor(frame));
        appendInitialCode(code, frame, arguments, origin);
      }else{
        if (Modifier.isStatic(node.getModifier())) {
          code.setIndexTable(indexTableFor(0, frame));
        } else {
          code.setIndexTable(indexTableFor(1, frame));
        }
      }
      codeBlock(node.getBlock(), code);
      method.setMaxLocals();
      method.setMaxStack();
    }
    gen.addMethod(method.getMethod());
  }
  
  private void appendInitialCode(CodeProxy code, LocalFrame frame, Type[] arguments, int origin) {
    int frameObjectIndex = code.getFrameObjectIndex();
    code.appendConstant(new Integer(frame.entries().length));
    code.appendNewArray(Type.OBJECT, (short)1);
    code.appendDup(1);
    code.appendStore(new ArrayType(Type.OBJECT, 1), frameObjectIndex);
    int index = origin;
    for(int i = 0; i < arguments.length;){
      Type arg = arguments[i];
      code.appendDup(1);
      code.appendConstant(new Integer(i));
      if(arguments[i] instanceof BasicType){
        ObjectType boxed = code.boxingType(arg);
        code.appendNew(boxed);
        code.appendDup(1);
        code.appendLoad(arg, index + i);
        code.appendCallConstructor(boxed, new Type[]{arg});
      }else{
        code.appendLoad(arg, index + i);
      }
      code.appendArrayStore(Type.OBJECT);
      if(arg == Type.DOUBLE || arg == Type.LONG){
        i += 2;
      }else{
        i++;
      }
    }
  }

  private static Set intefaceMethods(ClassSymbol type){
    Set methods =  new TreeSet(new MethodSymbolComparator());
    collectInterfaceMethods(type, methods);
    return methods;
  }
  
  private static void collectInterfaceMethods(ClassSymbol type, Set set){
    MethodSymbol[] methods = type.getMethods();
    for(int i = 0; i < methods.length; i++){
      set.add(methods[i]);
    }
    ClassSymbol[] interfaces = type.getInterfaces();
    for(int i = 0 ; i < interfaces.length; i++){
      collectInterfaceMethods(interfaces[i], set);
    }
  }
  
  private void implementsMethods(ClassGen gen, MethodSymbol[] methods){
    for(int i = 0; i < methods.length; i++){
      MethodSymbol method = methods[i];
      Type returnType = typeOf(method.getReturnType());
      String name = method.getName();
      Type[] args = typesOf(method.getArguments());
      String[] argNames = names(args.length);
      CodeProxy code = new CodeProxy(gen.getConstantPool());
      MethodGen mgen = new MethodGen(
        Constants.ACC_PUBLIC, returnType, args, argNames, name, 
        gen.getClassName(), code.getCode(), gen.getConstantPool());
      code.appendDefaultValue(returnType);
      code.appendReturn(returnType);
      mgen.setMaxLocals();
      mgen.setMaxStack();
      gen.addMethod(mgen.getMethod());
    }
  }
  
  public InstructionHandle codeClosure(ClosureNode node, CodeProxy code){
    ClassSymbol classType = node.getClassType();
    String closureName = generator.generate();
    Type[] arguments = typesOf(node.getArguments());
    ClassGen gen = new ClassGen(
      closureName, "java.lang.Object", "<generated>", Constants.ACC_PUBLIC,
      new String[]{classType.getName()});
    
    Set methods = Classes.getInterfaceMethods(classType);
    methods.remove(node.getMethod());
    implementsMethods(
      gen, (MethodSymbol[]) methods.toArray(new MethodSymbol[0]));
    
    LocalFrame frame = node.getFrame();
    int depth = frame.depth();
    for(int i = 1; i <= depth; i++){
      FieldGen field = new FieldGen(
        Constants.ACC_PRIVATE, new ArrayType("java.lang.Object", 1),
        FRAME_PREFIX + i,
        gen.getConstantPool());
      gen.addField(field.getField());
    }
    Type[] types = closureArguments(depth);
    MethodGen method = 
      createClosureConstructor(closureName, types, gen.getConstantPool());
    gen.addMethod(method.getMethod());
    
    CodeProxy closureCode = new CodeProxy(gen.getConstantPool());
    method = new MethodGen(
      Constants.ACC_PUBLIC, typeOf(node.getReturnType()),
      arguments, names(arguments.length), node.getName(), 
      closureName, closureCode.getCode(), gen.getConstantPool());
    closureCode.setMethod(method);
    closureCode.setFrame(frame);
    if(frame.isClosed()){
      int frameObjectIndex = frameObjectIndex(1, node.getArguments());
      closureCode.setFrameObjectIndex(frameObjectIndex);
      closureCode.setIndexTable(indexTableFor(frame));      
      appendInitialCode(closureCode, frame, arguments, 1);
    }else{
      closureCode.setIndexTable(indexTableFor(1, frame));
    }
    codeStatement(node.getBlock(), closureCode);
    method.setMaxLocals();
    method.setMaxStack();
    gen.addMethod(method.getMethod());
    compiledClasses.add(gen.getJavaClass());
    
    InstructionHandle start = code.appendNew(new ObjectType(closureName));
    code.appendDup(1);
    String name = code.getMethod().getClassName();
    int index = code.getFrameObjectIndex();
    code.appendLoad(new ArrayType("java.lang.Object", 1), index);
    for(int i = 1; i < depth; i++){
      code.appendThis();
      code.appendGetField(
        name, FRAME_PREFIX + i, new ArrayType("java.lang.Object", 1));
    }
    code.appendCallConstructor(
      new ObjectType(closureName), closureArguments(depth));
    return start;
  }
  
  private Type[] closureArguments(int size){
    Type[] arguments = new Type[size];
    for(int i = 0; i < arguments.length; i++){
      arguments[i] = new ArrayType("java.lang.Object", 1);
    }
    return arguments;
  }
  
  private InstructionHandle codeList(ListNode node, CodeProxy code){
    ObjectType listType = (ObjectType) typeOf(node.type());
    InstructionHandle start = code.appendNew("java.util.ArrayList");
    code.appendDup(1);
    code.appendCallConstructor(
      new ObjectType("java.util.ArrayList"), new Type[0]);
    ExpressionNode[] elements = node.getElements();
    for(int i = 0; i < elements.length; i++){
      code.appendDup(1);
      codeExpression(elements[i], code);
      code.appendInvoke(
        listType.getClassName(), "add", Type.BOOLEAN, new Type[]{Type.OBJECT}, 
        Constants.INVOKEINTERFACE);
      code.appendPop(1);
    }
    return start;
  }
  
  public InstructionHandle codeSuperCall(SuperCallNode node, CodeProxy code){
    InstructionHandle start = codeExpression(node.target, code);
    codeExpressions(node.parameters, code);
    MethodSymbol method = node.method;
    code.appendInvoke(
      method.getClassType().getName(),
      method.getName(),
      typeOf(method.getReturnType()),
      typesOf(method.getArguments()),
      Constants.INVOKESPECIAL);
    return start;
  }
  
  private String[] names(int size){
    String[] names = new String[size];
    for(int i = 0; i < names.length; i++){
      names[i] = "args" + size;
    }
    return names;
  }
  
  private MethodGen createClosureConstructor(
    String className, Type[] types, ConstantPoolGen pool){
    String[] argNames = new String[types.length];
    for(int i = 0; i < types.length; i++){
      argNames[i] = FRAME_PREFIX + i;
    }
    CodeProxy code = new CodeProxy(pool);
    MethodGen constructor = new MethodGen(
      Constants.ACC_PUBLIC, Type.VOID, types, argNames, "<init>",
      className, code.getCode(), pool);
    int index = 1;
    code.appendThis();
    code.appendCallConstructor(Type.OBJECT, new Type[0]);
    for(int i = 0; i < types.length; i++){
      code.appendThis();
      code.appendLoad(types[i], i + 1);
      code.appendPutField(className, FRAME_PREFIX + (i + 1), types[i]);
    }
    code.append(InstructionConstants.RETURN);
    constructor.setMaxLocals();
    constructor.setMaxStack();
    return constructor;
  }
  
  public void codeField(ClassGen gen, FieldNode node) {
    FieldGen field = new FieldGen(
      toJavaModifier(node.getModifier()), 
      typeOf(node.getType()), node.getName(), gen.getConstantPool());
    gen.addField(field.getField());
  }

  public InstructionHandle codeBlock(BlockNode node, CodeProxy code) {
    InstructionHandle start;
    if(node.getStatements().length > 0){
      start = codeStatement(node.getStatements()[0], code);
      for (int i = 1; i < node.getStatements().length; i++) {
        codeStatement(node.getStatements()[i], code);
      }
    }else{
      start = code.append(InstructionConstants.NOP);
    }
    return start;
  }

  public InstructionHandle codeExpressionStatement(ExpressionStatementNode node, CodeProxy code) {
    InstructionHandle start = codeExpression(node.expression, code);
    TypeSymbol type = node.expression.type();
    if (type != BasicSymbol.VOID) {
      if (isWideType(type)) {
        code.append(InstructionConstants.POP2);
      } else {
        code.append(InstructionConstants.POP);
      }
    }
    return start;
  }

  public InstructionHandle codeStatement(StatementNode node, CodeProxy code) {
    InstructionHandle start;
    if (node instanceof BlockNode) {
      start = codeBlock((BlockNode) node, code);
    } else if (node instanceof ExpressionStatementNode) {
      start = codeExpressionStatement((ExpressionStatementNode) node, code);
    } else if (node instanceof IfNode) {
      start = codeIf((IfNode) node, code);
    } else if (node instanceof LoopNode) {
      start = codeLoop((LoopNode) node, code);
    } else if (node instanceof EmptyNode) {
      start = codeEmpty((EmptyNode) node, code);
    } else if (node instanceof ReturnNode) {
      start = codeReturn((ReturnNode) node, code);
    } else if (node instanceof SynchronizedNode) {
      start = codeSynchronized((SynchronizedNode) node, code);
    } else if (node instanceof ThrowNode) {
      start = codeThrowNode((ThrowNode)node, code);
    } else if (node instanceof TryNode){
      start = codeTry((TryNode)node, code);
    } else {
      start = code.append(InstructionConstants.NOP);
    }
    return start;
  }

  public InstructionHandle codeReturn(ReturnNode node, CodeProxy code) {
    InstructionHandle start;
    if (node.expression != null) {
      start = codeExpression(node.expression, code);
      Type type = typeOf(node.expression.type());
      code.appendReturn(type);
    } else {
      start = code.append(InstructionConstants.RETURN);
    }
    return start;
  }

  public InstructionHandle codeSynchronized(SynchronizedNode node, CodeProxy code) {
    return null;
  }
  
  public InstructionHandle codeThrowNode(ThrowNode node, CodeProxy code){
    InstructionHandle start = codeExpression(node.expression, code);
    code.append(InstructionConstants.ATHROW);
    return start;
  }
  
  public InstructionHandle codeTry(TryNode node, CodeProxy code){
    InstructionHandle start = codeStatement(node.tryStatement, code);    
    BranchHandle to = code.append(new GOTO(null));
    int length = node.catchTypes.length;
    BranchHandle[] catchEnds = new BranchHandle[length];
    for(int i = 0; i < length; i++){
      ClosureLocalBinding bind = node.catchTypes[i];
      int index = code.getIndexTable()[bind.getIndex()];
      ObjectType type = (ObjectType) typeOf(bind.getType());
      InstructionHandle target = code.appendStore(type, index);
      code.addExceptionHandler(start, to, target, type);
      codeStatement(node.catchStatements[i], code);
      catchEnds[i] = code.append(new GOTO(null));
    }
    InstructionHandle end = code.append(InstructionConstants.NOP);
    to.setTarget(end);
    for(int i = 0; i < catchEnds.length; i++){
      catchEnds[i].setTarget(end);
    }
    return start;
  }

  public InstructionHandle codeEmpty(EmptyNode node, CodeProxy code) {
    return code.append(InstructionConstants.NOP);
  }

  public InstructionHandle codeIf(IfNode node, CodeProxy code) {
    InstructionHandle start = codeExpression(node.getCondition(), code);
    BranchHandle toThen = code.append(new IFNE(null));
    if (node.getElseStatement() != null) {
      codeStatement(node.getElseStatement(), code);
    }
    BranchHandle toEnd = code.append(new GOTO(null));
    toThen.setTarget(codeStatement(node.getThenStatement(), code));
    toEnd.setTarget(code.append(new NOP()));
    return start;
  }

  public InstructionHandle codeLoop(LoopNode node, CodeProxy code) {
    InstructionHandle start = codeExpression(node.condition, code);
    BranchHandle branch = code.append(new IFEQ(null));
    codeStatement(node.stmt, code);
    code.append(new GOTO(start));    
    InstructionHandle end = code.append(InstructionConstants.NOP);
    branch.setTarget(end);
    return start;
  }

  private static int toJavaModifier(int src) {
    int modifier = 0;
    modifier |= Modifier.isPrivate(src) ? Constants.ACC_PRIVATE : modifier;
    modifier |= Modifier.isProtected(src) ? Constants.ACC_PROTECTED : modifier;
    modifier |= Modifier.isPublic(src) ? Constants.ACC_PUBLIC : modifier;
    modifier |= Modifier.isStatic(src) ? Constants.ACC_STATIC : modifier;
    modifier |= Modifier.isSynchronized(src) ? Constants.ACC_SYNCHRONIZED : modifier;
    modifier |= Modifier.isAbstract(src) ? Constants.ACC_ABSTRACT : modifier;
    modifier |= Modifier.isFinal(src) ? Constants.ACC_FINAL : modifier;
    return modifier;
  }

  private String nameOf(ClassSymbol symbol) {
    return symbol.getName();
  }

  private String[] namesOf(ClassSymbol[] symbols) {
    String[] names = new String[symbols.length];
    for (int i = 0; i < names.length; i++) {
      names[i] = nameOf(symbols[i]);
    }
    return names;
  }

  public InstructionHandle codeExpression(
    ExpressionNode node, CodeProxy code) {
    InstructionHandle start;
    if (node instanceof BinaryExpressionNode) {
      start = codeBinaryExpression((BinaryExpressionNode) node, code);
    } else if(node instanceof UnaryExpressionNode) {
      start = codeUnaryExpression((UnaryExpressionNode) node, code);
    } else if(node instanceof LocalAssignmentNode) {
      start = codeLocalAssign((LocalAssignmentNode) node, code);
    } else if(node instanceof LocalRefNode) {
      start = codeLocalRef((LocalRefNode) node, code);
    } else if(node instanceof StaticFieldRefNode) {
      start = codeStaticFieldRef((StaticFieldRefNode) node, code);
    } else if(node instanceof FieldRefNode) {
      start = codeFieldRef((FieldRefNode) node, code);
    } else if(node instanceof FieldAssignmentNode) {
      start = codeFieldAssign((FieldAssignmentNode) node, code);
    } else if(node instanceof MethodCallNode) {
      start = codeMethodCall((MethodCallNode) node, code);
    } else if(node instanceof ArrayRefNode){
      start = codeArrayRef((ArrayRefNode)node, code);
    } else if(node instanceof ArrayLengthNode){
      start = codeArrayLengthNode((ArrayLengthNode)node, code);
    } else if(node instanceof ArrayAssignmentNode){
      start = codeArrayAssignment((ArrayAssignmentNode)node, code);
    } else if(node instanceof NewNode){
      start = codeNew((NewNode)node, code);
    } else if(node instanceof NewArrayNode){
      start = codeNewArray((NewArrayNode)node, code);
    } else if(node instanceof ArrayRefNode){
      start = codeArrayRef((ArrayRefNode)node, code);
    } else if(node instanceof StaticMethodCallNode){
      start = codeStaticMethodCall((StaticMethodCallNode)node, code);
    } else if(node instanceof StringNode) {
      start = codeString((StringNode) node, code);
    } else if(node instanceof IntegerNode) {
      start = codeInteger((IntegerNode)node, code);
    } else if(node instanceof LongNode){
      start = codeLong((LongNode)node, code);
  	} else if(node instanceof FloatNode) {
  	  start = codeFloat((FloatNode)node ,code);
  	} else if(node instanceof DoubleNode) {
  	  start = codeDouble((DoubleNode)node, code);
  	} else if(node instanceof BooleanNode) {
  	  start = codeBoolean((BooleanNode)node, code);
  	} else if(node instanceof NullNode) {
  	  start = codeNull((NullNode)node, code);
  	} else if(node instanceof CastNode) {
  	  start = codeCast((CastNode)node, code);
  	} else if(node instanceof SelfNode) {
  	  start = codeSelf((SelfNode)node, code);
  	} else if(node instanceof IsInstanceNode){
  	  start = codeIsInstance((IsInstanceNode)node, code);
  	} else if(node instanceof ClosureNode){
  	  start = codeClosure((ClosureNode)node, code);
  	} else if(node instanceof ListNode){
  	  start = codeList((ListNode)node, code);
  	} else if(node instanceof SuperCallNode){
  	  start = codeSuperCall((SuperCallNode)node, code);
  	} else {  	  
  	  throw new RuntimeException();
    }
    return start;
  }
  
  public InstructionHandle codeLocalAssign(LocalAssignmentNode node, CodeProxy code) {
    InstructionHandle start = null;
    Type type = typeOf(node.type());
    if(node.getFrame() == 0 && !code.getFrame().isClosed()){
      start = codeExpression(node.getValue(), code);
      if(isWideType(node.type())){
        code.append(InstructionConstants.DUP2);
      }else{
        code.append(InstructionConstants.DUP);
      }
      code.appendStore(type, code.getIndexTable()[node.getIndex()]);
    }else{
      if(node.getFrame() == 0 && code.getFrame().isClosed()){
        int index = code.getFrameObjectIndex();
        start = code.appendLoad(new ArrayType("java.lang.Object", 1), index);
        code.appendConstant(new Integer(code.index(node.getIndex())));
      }else{
        start = code.appendThis();
        code.appendGetField(
          code.getMethod().getClassName(),
          FRAME_PREFIX + node.getFrame(), new ArrayType("java.lang.Object", 1));
        code.appendConstant(new Integer(node.getIndex()));
      }
      if(node.isBasicType()){
        ObjectType boxed = code.boxingType(type);          
        code.appendNew(boxed);
        code.appendDup(1);
        codeExpression(node.getValue(), code);
        code.appendInvoke(
          boxed.getClassName(), "<init>", Type.VOID, new Type[]{type},
          Constants.INVOKESPECIAL);
        code.appendDup_2(1);
        code.appendArrayStore(Type.OBJECT);
        String method = (String)unboxingMethods.get(boxed.getClassName());
        code.appendInvoke(
          boxed.getClassName(), method, 
          type, new Type[0], Constants.INVOKEVIRTUAL);
      }else{
        codeExpression(node.getValue(), code);
        code.appendDup_2(1);
        code.appendArrayStore(Type.OBJECT);          
      }
    }
    return start;
  }
  


  public InstructionHandle codeLocalRef(LocalRefNode node, CodeProxy code) {
    InstructionHandle start = null;
    Type type = typeOf(node.type());
    if(node.frame() == 0 && !code.getFrame().isClosed()){
      start = code.appendLoad(type, code.index(node.index()));
    }else{
      if(node.frame() == 0 && code.getFrame().isClosed()){
        int index = code.getFrameObjectIndex();
        start = code.appendLoad(new ArrayType("java.lang.Object", 1), index);
        code.appendConstant(new Integer(code.index(node.index())));
      }else{
        start = code.appendThis();
        code.appendGetField(
          code.getMethod().getClassName(),
          FRAME_PREFIX + node.frame(), new ArrayType("java.lang.Object", 1));
        code.appendConstant(new Integer(node.index()));
      }
      code.appendArrayLoad(Type.OBJECT);
      if(node.isBasicType()){
        ObjectType boxed = code.boxingType(type);      
        String method = (String)unboxingMethods.get(boxed.getClassName());
        code.appendCast(Type.OBJECT, boxed);
        code.appendInvoke(
          boxed.getClassName(), method, type, new Type[0],
          Constants.INVOKEVIRTUAL);
      }else{
        code.appendCast(Type.OBJECT, type);
      }
    }
    return start;
  }

  public InstructionHandle codeStaticFieldRef(StaticFieldRefNode node, CodeProxy code) {
    String classType = node.field.getClassType().getName();
    String name = node.field.getName();
    Type type = typeOf(node.type());
    return code.appendGetStatic(classType, name, type);
  }

  public InstructionHandle codeMethodCall(MethodCallNode node, CodeProxy code) {
    InstructionHandle start = codeExpression(node.target, code);
    for (int i = 0; i < node.parameters.length; i++) {
      codeExpression(node.parameters[i], code);
    }    
    ObjectSymbol classType = (ObjectSymbol) node.target.type();
    short kind;
    if(classType.isInterface()){
      kind = Constants.INVOKEINTERFACE;
    }else{
      kind = Constants.INVOKEVIRTUAL;
    }
    String className = classType.getName();
    String name = node.method.getName();
    Type ret = typeOf(node.type());
    Type[] args = typesOf(node.method.getArguments());
    code.appendInvoke(className, name, ret, args, kind);
    return start;
  }
  
  public InstructionHandle codeArrayRef(ArrayRefNode node, CodeProxy code){
    ArraySymbol targetType = (ArraySymbol)node.getObject().type();
    InstructionHandle start = codeExpression(node.getObject(), code);
    codeExpression(node.getIndex(), code);
    code.appendArrayLoad(typeOf(targetType.getBase()));
    return start;
  }
  
  public InstructionHandle codeArrayLengthNode(ArrayLengthNode node, CodeProxy code){
    InstructionHandle start = codeExpression(node.getTarget(), code);
    code.append(InstructionConstants.ARRAYLENGTH);
    return start;
  }
  
  public InstructionHandle codeArrayAssignment(
    ArrayAssignmentNode node, CodeProxy code){
    ArraySymbol targetType = (ArraySymbol)node.getObject().type();
    InstructionHandle start = codeExpression(node.getObject(), code);
    code.appendDup(1);
    codeExpression(node.getIndex(), code);
    codeExpression(node.getValue(), code);
    code.appendArrayStore(typeOf(targetType.getBase()));
    return start;
  }
  
  public InstructionHandle codeNew(NewNode node, CodeProxy code) {
    ClassSymbol type = node.constructor.getClassType();
    InstructionHandle start = code.appendNew((ObjectType)typeOf(type));
    code.append(InstructionConstants.DUP);
    for (int i = 0; i < node.parameters.length; i++) {
      codeExpression(node.parameters[i], code);
    }
    String className = type.getName();
    Type[] arguments = typesOf(node.constructor.getArguments());
    short kind = Constants.INVOKESPECIAL;
    code.appendInvoke(className, "<init>", Type.VOID, arguments, kind);
    return start;
  }
  
  public InstructionHandle codeNewArray(NewArrayNode node, CodeProxy code){
    InstructionHandle start = codeExpressions(node.parameters, code);
    ArraySymbol type = node.arrayType;
    code.appendNewArray(typeOf(type.getComponent()), (short)node.parameters.length);
    return start;
  }
  
  public InstructionHandle codeStaticMethodCall(
    StaticMethodCallNode node, CodeProxy code
  ){
    InstructionHandle start;
    if(node.parameters.length > 0){
      start = codeExpression(node.parameters[0], code);
      for (int i = 1; i < node.parameters.length; i++) {
        codeExpression(node.parameters[i], code);
      }      
    }else{
      start = code.append(InstructionConstants.NOP);
    }
    String className = node.target.getName();
    String name = node.method.getName();
    Type returnType = typeOf(node.type());
    Type[] arguments = typesOf(node.method.getArguments());
    short kind = Constants.INVOKESTATIC;
    code.appendInvoke(className, name, returnType, arguments, kind);
    return start;
  }

  public InstructionHandle codeBinaryExpression(
    BinaryExpressionNode node, CodeProxy code
  ){
    if(node.getKind() == LOGICAL_AND){
      return codeLogicalAnd(node, code);
    }else if(node.getKind() == LOGICAL_OR){
      return codeLogicalOr(node, code);
    }
    ExpressionNode left = node.getLeft();
    ExpressionNode right = node.getRight();
    InstructionHandle start = codeExpression(left, code);
    codeExpression(right, code);
    switch (node.getKind()) {
      case ADD: add(code, left.type()); break;
      case SUBTRACT: sub(code, left.type()); break;
      case MULTIPLY: mul(code, left.type()); break;
      case DIVIDE: div(code, left.type()); break;
      case MOD: mod(code, left.type()); break;
      case EQUAL: eq(code, left.type()); break;
      case NOT_EQUAL: noteq(code, left.type()); break;
      case LESS_OR_EQUAL: lte(code, left.type()); break;
      case GREATER_OR_EQUAL: gte(code, left.type()); break;
      case LESS_THAN: lt(code, left.type()); break;
      case GREATER_THAN: gt(code, left.type()); break;
      case BIT_AND: bitAnd(code, left.type()); break;
      case BIT_OR: bitOr(code, right.type()); break;
      case XOR: xor(code, right.type()); break;
      case BIT_SHIFT_L2: bitShiftL2(code, left.type()); break;
      case BIT_SHIFT_R2: bitShiftR2(code, left.type()); break;
      case BIT_SHIFT_R3: bitShiftR3(code, left.type()); break;
      default: break;
    }
    return start;
  }
  
  public void bitShiftR2(CodeProxy code, TypeSymbol type){
    if(type == BasicSymbol.INT){
      code.append(InstructionConstants.ISHR);
    }else if(type == BasicSymbol.LONG){
      code.append(InstructionConstants.LSHR);
    }else{
      throw new RuntimeException();
    }
  }
  
  public void bitShiftL2(CodeProxy code, TypeSymbol type){
    if(type == BasicSymbol.INT){
      code.append(InstructionConstants.ISHL);
    }else if(type == BasicSymbol.LONG){
      code.append(InstructionConstants.LSHL);
    }else{
      throw new RuntimeException();
    }
  }
  
  public void bitShiftR3(CodeProxy code, TypeSymbol type){
    if(type == BasicSymbol.INT){
      code.append(InstructionConstants.IUSHR);
    }else if(type == BasicSymbol.LONG){
      code.append(InstructionConstants.LUSHR);
    }else{
      throw new RuntimeException();
    }
  }
  
  public InstructionHandle codeLogicalAnd(BinaryExpressionNode node, CodeProxy code) {
    InstructionHandle start = codeExpression(node.getLeft(), code);
    BranchHandle b1 = null, b2 = null, b3 = null;
    
    b1 = code.append(new IFEQ(null));
    codeExpression(node.getRight(), code);
    b2 = code.append(new IFEQ(null));
    code.append(InstructionConstants.ICONST_1);
    b3 = code.append(new GOTO(null));
    InstructionHandle failure = code.append(InstructionConstants.ICONST_0);
    b1.setTarget(failure);
    b2.setTarget(failure);
    b3.setTarget(code.append(InstructionConstants.NOP));      
    return start;
  }

  public InstructionHandle codeLogicalOr(BinaryExpressionNode node, CodeProxy code) {
    InstructionHandle start = codeExpression(node.getLeft(), code);
    BranchHandle b1 = null, b2 = null, b3 = null;
    b1 = code.append(new IFNE(null));
    codeExpression(node.getRight(), code);
    b2 = code.append(new IFNE(null));
    code.append(InstructionConstants.ICONST_0);
    b3 = code.append(new GOTO(null));
    InstructionHandle success = code.append(InstructionConstants.ICONST_1);
    b1.setTarget(success);
    b2.setTarget(success);
    b3.setTarget(code.append(new NOP()));
    return start;
  }
  
  public void bitAnd(CodeProxy code, TypeSymbol type) {
    if (type == BasicSymbol.INT || type == BasicSymbol.BOOLEAN) {
      code.append(new IAND());
    } else if(type == BasicSymbol.LONG) {
      code.append(new LAND());
    } else {
      throw new RuntimeException();
    }
  }
  
  public void bitOr(CodeProxy code, TypeSymbol type) {
    if(type == BasicSymbol.INT || type == BasicSymbol.BOOLEAN) {
      code.append(new IOR());
    } else if (type == BasicSymbol.LONG) {
      code.append(new LOR());
    } else {
      throw new RuntimeException();
    }
  }
  
  public void xor(CodeProxy code, TypeSymbol type) {
    if(type == BasicSymbol.INT || type == BasicSymbol.BOOLEAN) {
      code.append(new IXOR());
    } else if (type == BasicSymbol.LONG) {
      code.append(new LXOR());
    } else {
      throw new RuntimeException();
    }
  }

  public void eq(CodeProxy code, TypeSymbol type) {
    BranchHandle b1 = null;
    if(type == BasicSymbol.INT || type == BasicSymbol.CHAR ||
       type == BasicSymbol.BOOLEAN) {
      b1 = code.append(new IF_ICMPEQ(null));
    } else if (type == BasicSymbol.LONG) {
      code.append(new LCMP());
      b1 = code.append(new IFEQ(null));
    } else if (type == BasicSymbol.FLOAT) {
      code.append(new FCMPL());
      b1 = code.append(new IFEQ(null));
    } else if (type == BasicSymbol.DOUBLE) {
      code.append(new DCMPL());
      b1 = code.append(new IFEQ(null));
    } else {
      b1 = code.append(new IF_ACMPEQ(null));
    }
    processBranch(code, b1);
  }
  
  public void noteq(CodeProxy code, TypeSymbol type) {
    BranchHandle b1 = null;
    if(type == BasicSymbol.INT || type == BasicSymbol.CHAR ||
       type == BasicSymbol.BOOLEAN) {
      b1 = code.append(new IF_ICMPNE(null));
    } else if (type == BasicSymbol.LONG) {
      code.append(new LCMP());
      b1 = code.append(new IFNE(null));
    } else if (type == BasicSymbol.FLOAT) {
      code.append(new FCMPL());
      b1 = code.append(new IFNE(null));
    } else if (type == BasicSymbol.DOUBLE) {
      code.append(new DCMPL());
      b1 = code.append(new IFNE(null));
    } else {
      b1 = code.append(new IF_ACMPNE(null));
    }
    processBranch(code, b1);
  }
  
  public void gt(CodeProxy code, TypeSymbol type) {
    BranchHandle b1 = null;
    if (type == BasicSymbol.INT){
      b1 = code.append(new IF_ICMPGT(null));
    } else if (type == BasicSymbol.LONG){
      code.append(new LCMP());
      b1 = code.append(new IFGT(null));
    } else if (type == BasicSymbol.FLOAT){
      code.append(new FCMPL());
      b1 = code.append(new IFGT(null));
    } else if (type == BasicSymbol.DOUBLE){
      code.append(new DCMPL());
      b1 = code.append(new IFGT(null));
    } else {
      throw new RuntimeException("");
    }
    processBranch(code, b1);
  }
  
  public void gte(CodeProxy code, TypeSymbol type) {
    BranchHandle comparation = null;
    if(type == BasicSymbol.INT) {
      comparation = code.append(new IF_ICMPGE(null));
    } else if (type == BasicSymbol.LONG) {
      code.append(new LCMP());
      comparation = code.append(new IFGE(null));
    } else if (type == BasicSymbol.FLOAT) {
      code.append(new FCMPL());
      comparation = code.append(new IFGE(null));
    } else if (type == BasicSymbol.DOUBLE) {
      code.append(new DCMPL());
      comparation = code.append(new IFGE(null));
    } else {
      throw new RuntimeException("");
    }
    processBranch(code, comparation);
  }

  public void lte(CodeProxy code, TypeSymbol type) {
    BranchHandle b1 = null;
    if (type == BasicSymbol.INT) {
      b1 = code.append(new IF_ICMPLE(null));
    } else if (type == BasicSymbol.LONG) {
      code.append(new LCMP());
      b1 = code.append(new IFLT(null));
    } else if (type == BasicSymbol.FLOAT) {
      code.append(new FCMPL());
      b1 = code.append(new IFLE(null));
    } else if (type == BasicSymbol.DOUBLE) {
      code.append(new DCMPL());
      b1 = code.append(new IFLE(null));
    } else {
      throw new RuntimeException("");
    }
    processBranch(code, b1);
  }

  
  public void lt(CodeProxy code, TypeSymbol type) {
    BranchHandle comparation = null;
    if(type == BasicSymbol.INT) {
      comparation = code.append(new IF_ICMPLT(null));
    } else if (type == BasicSymbol.LONG) {
      code.append(new LCMP());
      comparation = code.append(new IFLT(null));
    } else if (type == BasicSymbol.FLOAT){
      code.append(new FCMPL());
      comparation = code.append(new IFLT(null));
    } else if (type == BasicSymbol.DOUBLE) {
      code.append(new DCMPL());
      comparation = code.append(new IFLT(null));
    } else {
      throw new RuntimeException("");
    }
    processBranch(code, comparation);
  }

  
  private void processBranch(CodeProxy code, BranchHandle b1) {
    code.append(InstructionConstants.ICONST_0);
    BranchHandle b2 = code.append(new GOTO(null));
    b1.setTarget(code.append(InstructionConstants.ICONST_1));
    b2.setTarget(code.append(InstructionConstants.NOP));
  }


  public InstructionHandle codeString(StringNode node, CodeProxy code) {
    return code.appendConstant(node.getValue());
  }
  
  public InstructionHandle codeInteger(IntegerNode node, CodeProxy code){
    return code.appendConstant(new Integer(node.getValue()));
  }
  
  public InstructionHandle codeLong(LongNode node, CodeProxy code){
    return code.appendConstant(new Long(node.getValue()));
  }
  
  public InstructionHandle codeFloat(FloatNode node, CodeProxy code){
    return code.appendConstant(new Float(node.getValue()));
  }
  
  public InstructionHandle codeDouble(DoubleNode node, CodeProxy code){
    return code.appendConstant(new Double(node.getValue()));
  }
  
  public InstructionHandle codeBoolean(BooleanNode node, CodeProxy code){
    return code.appendConstant(Boolean.valueOf(node.getValue()));
  }
  
  public InstructionHandle codeNull(NullNode node, CodeProxy code){
    return code.append(InstructionConstants.ACONST_NULL);
  }
  
  public InstructionHandle codeUnaryExpression(
    UnaryExpressionNode node, CodeProxy code) {
    InstructionHandle start = codeExpression(node.getOperand(), code);
    TypeSymbol type = node.getOperand().type();
    switch(node.getKind()){
    	case PLUS: plus(code, type); break;
    	case MINUS: minus(code, type); break;
    	case NOT: not(code, type); break;
    	case BIT_NOT: bitNot(code, type); break;
    	default: throw new RuntimeException();
    }
    return start;
  }
  
  private void plus(CodeProxy code, TypeSymbol type){
    if(
     type != BasicSymbol.INT && type != BasicSymbol.LONG &&
     type != BasicSymbol.FLOAT && type != BasicSymbol.DOUBLE){
      throw new RuntimeException();
    }
    /*nothing to do*/
  }

  private void minus(CodeProxy code, TypeSymbol type){
    if(type == BasicSymbol.INT){
      code.append(InstructionConstants.INEG);
    }else if(type == BasicSymbol.LONG){
      code.append(InstructionConstants.LNEG);
    }else if(type == BasicSymbol.FLOAT){
      code.append(InstructionConstants.FNEG);
    }else if(type == BasicSymbol.DOUBLE){
      code.append(InstructionConstants.DNEG);
    }else{
      throw new RuntimeException();
    }
  }

  private void not(CodeProxy code, TypeSymbol type){
    if(type == BasicSymbol.BOOLEAN){
      BranchHandle b1 = code.append(new IFNE(null));
      BranchHandle b2;
      code.append(new ICONST(1));
      b2 = code.append(new GOTO(null));
      b1.setTarget(code.append(new ICONST(0)));
      b2.setTarget(code.append(new NOP()));
    }else{
      throw new RuntimeException();
    }
  }

  private void bitNot(CodeProxy code, TypeSymbol type){
    if(type == BasicSymbol.INT){
      code.append(new ICONST(-1));
      code.append(new IXOR());
    }else if(type == BasicSymbol.LONG){
      code.append(new LCONST(-1));
      code.append(new LXOR());
    }else{
      throw new RuntimeException();
    }
  }

  public InstructionHandle codeCast(CastNode node, CodeProxy code) {
    ExpressionNode target = node.getTarget();
    InstructionHandle start = codeExpression(target, code);
    code.appendCast(typeOf(target.type()), typeOf(node.getConversion()));
    return start;
  }
  
  public InstructionHandle codeIsInstance(IsInstanceNode node, CodeProxy code){
    InstructionHandle start = codeExpression(node.target, code);
    code.appendInstanceOf((ReferenceType)typeOf(node.getCheckType()));
    return start;
  }
  
  public InstructionHandle codeSelf(SelfNode node, CodeProxy code){
    return code.append(InstructionConstants.ALOAD_0);
  }

  public InstructionHandle codeFieldRef(FieldRefNode node, CodeProxy code) {
    InstructionHandle start = codeExpression(node.target, code);
    ClassSymbol symbol = (ClassSymbol) node.target.type();
    code.appendGetField(symbol.getName(), node.field.getName(), typeOf(node.type()));
    return start;
  }

  public InstructionHandle codeFieldAssign(FieldAssignmentNode node, CodeProxy code) {
    InstructionHandle start = codeExpression(node.getObject(), code);    
    codeExpression(node.getValue(), code);
    if(isWideType(node.getValue().type())){
      code.append(InstructionConstants.DUP2_X1);
    }else{
      code.append(InstructionConstants.DUP_X1);
    }
    ClassSymbol symbol = (ClassSymbol) node.getObject().type();
    code.appendPutField(symbol.getName(), node.getField().getName(), typeOf(node.type()));
    return start;
  }

  private void add(CodeProxy code, TypeSymbol type) {
    if (type == BasicSymbol.INT) {
      code.append(new IADD());
    } else if (type == BasicSymbol.LONG) {
      code.append(new LADD());
    } else if (type == BasicSymbol.FLOAT) {
      code.append(new FADD());
    } else {
      code.append(new DADD());
    }
  }

  private void sub(CodeProxy code, TypeSymbol type) {
    if (type == BasicSymbol.INT) {
      code.append(new ISUB());
    } else if (type == BasicSymbol.LONG) {
      code.append(new LSUB());
    } else if (type == BasicSymbol.FLOAT) {
      code.append(new FSUB());
    } else {
      code.append(new DSUB());
    }
  }
  
  private void mul(CodeProxy code, TypeSymbol type) {
    if (type == BasicSymbol.INT) {
      code.append(new IMUL());
    } else if (type == BasicSymbol.LONG) {
      code.append(new LMUL());
    } else if (type == BasicSymbol.FLOAT) {
      code.append(new FMUL());
    } else {
      code.append(new DMUL());
    }
  }

  private void div(CodeProxy code, TypeSymbol type) {
    if (type == BasicSymbol.INT) {
      code.append(new IDIV());
    } else if (type == BasicSymbol.LONG) {
      code.append(new LDIV());
    } else if (type == BasicSymbol.FLOAT) {
      code.append(new FDIV());
    } else {
      code.append(new DDIV());
    }
  }

  private void mod(CodeProxy code, TypeSymbol type) {
    if (type == BasicSymbol.INT) {
      code.append(new IREM());
    } else if (type == BasicSymbol.LONG) {
      code.append(new LREM());
    } else if (type == BasicSymbol.FLOAT) {
      code.append(new FREM());
    } else {
      code.append(new DREM());
    }
  }
  
  private int frameObjectIndex(int origin, TypeSymbol[] arguments){
    int maxIndex = origin;
    for(int i = 0; i < arguments.length; i++) {
      if (isWideType(arguments[i])) {
        maxIndex += 2;
      } else {
        maxIndex++;
      }
    }
    return maxIndex;
  }

  private int[] indexTableFor(int origin, LocalFrame frame) {
    LocalBinding[] bindings = frame.entries();
    int[] indexTable = new int[bindings.length];
    int maxIndex = origin;
    for(int i = 0; i < bindings.length; i++) {
      indexTable[i] = maxIndex;
      if (isWideType(bindings[i].getType())) {
        maxIndex += 2;
      } else {
        maxIndex++;
      }
    }
    return indexTable;
  }
  
  private int[] indexTableFor(LocalFrame frame) {
    LocalBinding[] bindings = frame.entries();
    int[] indexTable = new int[bindings.length];
    int maxIndex = 0;
    for(int i = 0; i < bindings.length; i++) {
      indexTable[i] = maxIndex;
      maxIndex++;
    }
    return indexTable;
  }

  private boolean isWideType(TypeSymbol symbol) {
    if (symbol == BasicSymbol.DOUBLE || symbol == BasicSymbol.LONG)
      return true;
    return false;
  }

  private Type typeOf(TypeSymbol type) {
    return bridge.toVMType(type);
  }

  private Type[] typesOf(TypeSymbol[] types) {
    Type[] destinationTypes = new Type[types.length];
    for (int i = 0; i < destinationTypes.length; i++) {
      destinationTypes[i] = bridge.toVMType(types[i]);
    }
    return destinationTypes;
  }
}