/* ************************************************************** *
 *                                                                *
 * Copyright (c) 2005, Kota Mizushima, All rights reserved.       *
 *                                                                *
 *                                                                *
 * This software is distributed under the modified BSD License.   *
 * ************************************************************** */
package org.onion_lang.onion.compiler.phase;

import java.util.*;
import java.util.Arrays;

import org.onion_lang.onion.compiler.CompilerConfig;
import org.onion_lang.onion.compiler.environment.*;
import org.onion_lang.onion.compiler.phase.analysis.*;
import org.onion_lang.onion.compiler.phase.analysis.ClassTableBuilder;
import org.onion_lang.onion.compiler.phase.analysis.TypeHeaderAnalysis;
import org.onion_lang.onion.compiler.problem.*;
import org.onion_lang.onion.compiler.problem.CompilationFailureException;
import org.onion_lang.onion.compiler.problem.SemanticErrorReporter;
import org.onion_lang.onion.compiler.utility.Paths;
import org.onion_lang.onion.compiler.utility.Systems;
import org.onion_lang.onion.lang.kernel.*;
import org.onion_lang.onion.lang.kernel.type.*;
import org.onion_lang.onion.lang.syntax.*;

/**
 * @author Kota Mizushima
 * Date: 2005/04/16
 */
public class CodeAnalysisPhase
	implements SemanticErrorReporter.Constants, BinaryExpressionNode.Constants {
  private SemanticErrorReporter reporter;
  private CompilerConfig configuration;
  private ClassTable table;
  private Map toAST;
  private Map toKernelNode;
  private Map resolutions;
  
  private ClassTableBuilder builder;
  private TypeHeaderAnalysis analysis;
  private DuplicationChecker duplicationChecker;
  private TypeChecker checker;
  
  private CompilationUnit currentUnit;
  private ImportList currentImport;
  private ClassNode contextClass;
  private NameResolution currentResolver;
  private int access;
    
  public CodeAnalysisPhase(CompilerConfig configuration) {
    this.configuration = configuration;
    this.table = new ClassTable(classpath(configuration.getClassPath()));
    this.builder = new ClassTableBuilder(this);
    this.analysis = new TypeHeaderAnalysis(this);
    this.duplicationChecker = new DuplicationChecker(this);
    this.checker = new TypeChecker(this);
    this.toAST = new HashMap();
    this.toKernelNode = new HashMap();
    this.resolutions = new HashMap();
    this.reporter = new SemanticErrorReporter(configuration.getMaxErrorReports()); 
  }
  
  public String topClass(){
    ModuleDeclaration module = currentUnit.getModuleDeclaration();
    String moduleName = module != null ? module.getName() : null;
    return createName(
      moduleName, Paths.extCutName(currentUnit.getSourceFileName()) + "Main");
  }
  
  public ClassTable table(){
    return table;
  }
  
  public void setCurrentResolver(NameResolution resolver){
    this.currentResolver = resolver;
  }
  
  public NameResolution getCurrentResolver(){
    return currentResolver;
  }
  
  public void setUnit(CompilationUnit unit){
    currentUnit = unit;
  }
  
  public CompilationUnit getUnit(){
    return currentUnit;
  }
  
  public void setImport(ImportList imports){
    currentImport = imports;
  }
  
  public ImportList getImport(){
    return currentImport;
  }
  
  public void put(ASTNode astNode, KernelNode kernelNode){
    toKernelNode.put(astNode, kernelNode);
    toAST.put(kernelNode, astNode);
  }
  
  public ASTNode lookupAST(KernelNode kernelNode){
    return (ASTNode) toAST.get(kernelNode);
  }
  
  public KernelNode lookupKernelNode(ASTNode astNode){
    return (KernelNode) toKernelNode.get(astNode);
  }
  
  public void setContextClass(ClassNode contextClass){
    this.contextClass = contextClass;
  }
  
  public ClassNode getContextClass(){
    return contextClass;
  }
  
  public void setAccess(int access){
    this.access = access;
  }
  
  public int getAccess(){
    return access;
  }
  
  public void putResolver(String className, NameResolution table) {
    resolutions.put(className, table);
  }
  
  public NameResolution getResolver(String className){
    return (NameResolution) resolutions.get(className);
  }
  
  private String createName(String moduleName, String simpleName){
    return (moduleName != null ? moduleName + "." : "") + simpleName;
  }
    
  public ClassNode[] process(CompilationUnit[] units){
    for(int i = 0; i < units.length; i++){
      builder.process(units[i]);
    }
    for(int i = 0; i < units.length; i++){
      analysis.process(units[i]);
    }
    for(int i = 0; i < units.length; i++){
      checker.process(units[i]);
    }
    for(int i = 0; i < units.length; i++){
      duplicationChecker.process(units[i]);
    }
    CompilationProblem[] problems = reporter.getProblems();
    if(problems.length > 0){
      throw new CompilationFailureException(Arrays.asList(problems));
    }
    return table.getSourceClasses();
  }
  
  private static String classpath(String[] classPaths){
    StringBuffer path = new StringBuffer();
    if(classPaths.length > 0){
      path.append(classPaths[0]);
      for(int i = 1; i < classPaths.length; i++){
        path.append(Systems.getPathSeparator());
        path.append(classPaths[i]);
      }
    }
    return new String(path);
  }
//----------------------------------------------------------------------------// 
//----------------------------------------------------------------------------//

  public TypeSymbol resolve(TypeSpecifier type, NameResolution resolver) {
    TypeSymbol resolvedType = (TypeSymbol) resolver.resolve(type);
    if(resolvedType == null){
      report(CLASS_NOT_FOUND, type, new Object[]{type.getComponentName()});      
    }
    return resolvedType;
  }
  
  public TypeSymbol resolve(TypeSpecifier type){
    return resolve(type, getCurrentResolver());
  }

  public void report(int error, ASTNode node, Object[] items){
    reporter.setSourceFile(currentUnit.getSourceFileName());
    reporter.report(error, node.getSourcePosition(), items);
  }

  public static TypeSymbol promoteNumericTypes(
    TypeSymbol left,  TypeSymbol right){
    if(!numeric(left) || !numeric(right)) return null;
    if(left == BasicSymbol.DOUBLE || right == BasicSymbol.DOUBLE){
      return BasicSymbol.DOUBLE;
    }
    if(left == BasicSymbol.FLOAT || right == BasicSymbol.FLOAT){
      return BasicSymbol.FLOAT;
    }
    if(left == BasicSymbol.LONG || right == BasicSymbol.LONG){
      return BasicSymbol.LONG;
    }
    return BasicSymbol.INT;
  }
  
  public static boolean hasNumericType(ExpressionNode expression){
    return numeric(expression.type());
  }
  
  private static boolean numeric(TypeSymbol symbol){
    return 
  	(symbol.isBasicType()) &&
  	(	symbol == BasicSymbol.BYTE || symbol == BasicSymbol.SHORT ||
  		symbol == BasicSymbol.INT || symbol == BasicSymbol.LONG ||
  		symbol == BasicSymbol.FLOAT || symbol == BasicSymbol.DOUBLE);
  }
  
  public ClassSymbol load(String name){
    return table.load(name);
  }
  
  public ClassSymbol loadTopClass(){
    return table.load(topClass());
  }
  
  public ArraySymbol loadArray(TypeSymbol type, int dimension){
    return table.loadArray(type, dimension);
  }
  
  public ClassSymbol rootClass(){
    return table.rootClass();
  }
  
}