unit OleDnD;

// $Id: OleDnD.pas,v 1.11 2002/04/04 18:37:44 takao Exp $

{$ObjExportAll On}

interface

uses
  Classes, Controls, Messages, Windows, SysUtils, ActiveX
  , MSGSnatchers, OleDnDImpl, OleDnDUtils;

type
  { Event Types for TDragGestureRecognizer }
  TRecognizeStartingMode = (rmContinue, rmContinueAndThrough, rmCancel);
  TRecognizeStartEvent = procedure(
    aSender: TObject; aButton: TMouseButton;
    aScreenMousePos: TPoint; var aMode: TRecognizeStartingMode) of Object;
  TRecognizedEvent = procedure(
    aSender: TObject; aButton: TMouseButton) of Object;

  { TCustomDragGestureRecognizer }
  TCustomDragGestureRecognizer = class(TCustomMessageSnatcher)
  private
    FThreshold: Integer;
    FThroughMessage: Boolean;

    FOnRecognizeStart: TRecognizeStartEvent;
    FOnRecognized: TRecognizedEvent;
    FOnRecognizeEnd: TNotifyEvent;

    tracking_: Boolean;
    startPos_: TPoint;
    button_: TMouseButton;

  protected
    procedure MessageHandler(var aMessage: TMessage; var aHandled: Boolean);
      override;

    procedure ButtonDownHandler(
      var aMessage: TMessage; var aHandled: Boolean);
    procedure ButtonUpHandler(
      var aMessage: TMessage; var aHandled: Boolean);
    procedure MouseMoveHandler(
      var aMessage: TMessage; var aHandled: Boolean);

    procedure DoRecognizeStart(var aMode: TRecognizeStartingMode); virtual;
    procedure DoRecognized; virtual;
    procedure DoRecognizeEnd; virtual;

    property Control;

    property Threshold: Integer read FThreshold write FThreshold default 5;
    property ThroughMessage: Boolean read FThroughMessage
      write FThroughMessage default True;

    property OnRecognizeStart: TRecognizeStartEvent read FOnRecognizeStart
      write FOnRecognizeStart;
    property OnRecognized: TRecognizedEvent read FOnRecognized
      write FOnRecognized;
    property OnRecognizeEnd: TNotifyEvent read FOnRecognizeEnd
      write FOnRecognizeEnd;

  public
    constructor Create(anOwner: TComponent); override;

  end;

  { TDragGestureRecognizer }
  TDragGestureRecognizer = class(TCustomDragGestureRecognizer)
  published
    { TCustomMessageSnatcher }
    property Control;

    { TCustomDragGestureRecognizer }
    property Threshold;
    property ThroughMessage;

    property OnRecognizeStart;
    property OnRecognized;
    property OnRecognizeEnd;
  end;

  { Event Types for TOleDropSource }
  TOleQueryContinueDragEvent = procedure(aSender: TObject;
    aEscapePressed: Boolean; aKeyState: Longint; var aResult: HResult) of Object;
  TOleGiveFeedbackEvent = procedure(aSender: TObject;
    aDropEffect: TDropEffect; var aResult: HResult) of Object;

  { TOleDropSource }
  TOleDropSource = class(TCustomDragGestureRecognizer)
  private
    FDropSource: IDropSourceBroker;
    FDropEffect: TDropEffect;
    FValidDropEffects: TDropEffects;

    FOnQueryContinueDrag: TOleQueryContinueDragEvent;
    FOnGiveFeedback: TOleGiveFeedbackEvent;

  protected
    procedure QueryContinueDrag(anEscapePressed: Boolean;
      aKeyState: Integer; var aResult: HResult);
    procedure GiveFeedback(aDropEffect: Longint; var aResult: HResult);

  public
    constructor Create(anOwner: TComponent); override;
    destructor Destroy; override;

    function DoDragDrop(anObject: IDataObject): Boolean;

    property DropEffect: TDropEffect read FDropEffect;
    property DropSource: IDropSourceBroker read FDropSource;

  published
    { TCustomMessageSnatcher }
    property Control;

    { TCustomDragGestureRecognizer }
    property Threshold;
    property ThroughMessage;

    property OnRecognizeStart;
    property OnRecognized;
    property OnRecognizeEnd;

    { Original Properties and Events }
    property ValidDropEffects: TDropEffects read FValidDropEffects
      write FValidDropEffects default [ekCopy];

    property OnQueryContinueDrag: TOleQueryContinueDragEvent
      read FOnQueryContinueDrag write FOnQueryContinueDrag;
    property OnGiveFeedback: TOleGiveFeedbackEvent
      read FOnGiveFeedback write FOnGiveFeedback;
  end;

  { TFileDropSource }
  TFileDropSource = class(TOleDropSource)
  private
    FDirectory: string;
    FFiles: TStringList;
  protected
    { TCustomDragGestureRecognizer }
    procedure DoRecognized; override;

    function GetFiles: TStrings;
    procedure SetFiles(files: TStrings);
  public
    constructor Create(anOwner: TComponent); override;
    destructor Destroy; override;
  published
    property Directory: string read FDirectory write FDirectory;
    property Files: TStrings read GetFiles write SetFiles;
  end;

  { TDragContext }
  TDragContext = record
    DataObject: IDataObject;
    KeyState: DWord;
    Point: TPoint;
    DropEffect: Longint;
    Result: HResult;
  end;

  { Event Types for TOleDropTarget }
  TOleDragEvent = procedure(aSender: TObject; var aContext: TDragContext) of Object;
  TOleDragLeaveEvent = procedure(aSender: TObject; var aResult: HResult)
    of Object;

  { TDFItem }
  TDFItem = class(TCollectionItem)
  private
    FDropEffects: TDropEffects;
    FCFName: string;
    FCFValue: Word;
    FTag: Integer;

    FOnDragEnter: TOleDragEvent;
    FOnDragOver: TOleDragEvent;
    FOnDrop: TOleDragEvent;

  protected
    procedure SetFormatName(aName: string);
    function GetDisplayName: string; override;

  public
    constructor Create(anOwner: TCollection); override;

    procedure Register;
    procedure FireEnter(var aContext: TDragContext);
    procedure FireOver(var aContext: TDragContext);
    procedure FireDrop(var aContext: TDragContext);

    property Format: Word read FCFValue;

  published
    property DropEffects: TDropEffects read FDropEffects write FDropEffects
      default [ekCopy];
    property FormatName: string read FCFName write SetFormatName;
    property Tag: Integer read FTag write FTag default 0;

    property OnDragEnter: TOleDragEvent read FOnDragEnter write FOnDragEnter;
    property OnDragOver: TOleDragEvent read FOnDragOver write FOnDragOver;
    property OnDrop: TOleDragEvent read FOnDrop write FOnDrop;
  end;

  { TDFItems }
  TDFItems = class(TCollection)
  private
    FOwner: TComponent;
  protected
    function GetOwner: TPersistent; override;
  public
    constructor Create(anOwner: TComponent);
    procedure RegisterAll;
  end;

  { TCustomOleDropTarget }
  TCustomOleDropTarget = class(TCustomWinMessageSnatcher)
  private
    FFormats: TDFItems;

    FFoundItem: TDFItem;
    FDropTarget: IDropTargetBroker;

    FOnDragEnter: TOleDragEvent;
    FOnDragOver: TOleDragEvent;
    FOnDragLeave: TOleDragLeaveEvent;
    FOnDrop: TOleDragEvent;

  protected
    { TCustomWinMessageSnatcher }
    procedure InitializeHWND(aHandle: HWND); override;
    procedure FinalizeHWND(aHandle: HWND); override;

    { Original Members }
    function GetDropEffect(aDataObject: IDataObject; aKeyState: DWord;
      aValidEffect: DWord): TDropEffect;
    function GetClientPoint(aPoint: TPoint): TPoint;

    procedure SetFormats(items: TDFItems);

    procedure DragEnter(const aDataObject: IDataObject; aKeyState: Longint;
      aPoint: TPoint; var aDropEffect: Longint; var aResult: HResult);
    procedure DragOver(aKeyState: Longint; aPoint: TPoint;
      var aDropEffect: Longint; var aResult: HResult);
    procedure DragLeave(var aResult: HResult);
    procedure Drop(const aDataObj: IDataObject; aKeyState: Longint;
      aPoint: TPoint; var aDropEffect: Longint; var aResult: HResult);

    procedure Loaded; override;

    property Formats: TDFItems read FFormats write SetFormats;
    property FoundItem: TDFItem read FFoundItem;

    property OnDragEnter: TOleDragEvent read FOnDragEnter write FOnDragEnter;
    property OnDragOver: TOleDragEvent read FOnDragOver write FOnDragOver;
    property OnDragLeave: TOleDragLeaveEvent read FOnDragLeave write FOnDragLeave;
    property OnDrop: TOleDragEvent read FOnDrop write FOnDrop;

  public
    constructor Create(anOwner: TComponent); override;
    destructor Destroy; override;

    property DropTarget: IDropTargetBroker read FDropTarget;
  end;

  { TOleDropTarget }
  TOleDropTarget = class(TCustomOleDropTarget)
  published
    property WinControl;
    property Formats;

    property OnDragEnter;
    property OnDragOver;
    property OnDragLeave;
    property OnDrop;
  end;

  { TFileDropTarget }
  TFileDropTarget = class(TCustomOleDropTarget)
  private
    fileItem_: TDFItem;
    FDropEffects: TDropEffects;
    FDroppedFiles: TStringList;

    FOnFileDragEnter: TOleDragEvent;
    FOnFileDragOver: TOleDragEvent;
    FOnFileDragLeave: TOleDragLeaveEvent;
    FOnFileDrop: TOleDragEvent;

  protected
    procedure SetDropEffects(dropEffects: TDropEffects);

    procedure FileDragEnter(aSender: TObject; var aContext: TDragContext);
    procedure FileDragOver(aSender: TObject; var aContext: TDragContext);
    procedure FileDragLeave(aSender: TObject; var aResult: HResult);
    procedure FileDrop(aSender: TObject; var aContext: TDragContext);

  public
    constructor Create(anOwner: TComponent); override;
    destructor Destroy; override;

  published
    property DropEffects: TDropEffects read FDropEffects write SetDropEffects
      default [ekCopy];
    property DroppedFiles: TStringList read FDroppedFiles;

    property WinControl;

    property OnFileDragEnter: TOleDragEvent
     read FOnFileDragEnter write FOnFileDragEnter;
    property OnFileDragOver: TOleDragEvent
     read FOnFileDragOver write FOnFileDragOver;
    property OnFileDragLeave: TOleDragLeaveEvent
     read FOnFileDragLeave write FOnFileDragLeave;
    property OnFileDrop: TOleDragEvent
     read FOnFileDrop write FOnFileDrop;
  end;


implementation

uses
  Forms, DataObjects;

{ TCustomDragGestureRecognizer }

constructor TCustomDragGestureRecognizer.Create(anOwner: TComponent);
begin
  inherited;

  FThreshold := 5;
  FThroughMessage := True;

  tracking_ := False;
end;

procedure TCustomDragGestureRecognizer.MessageHandler(var aMessage: TMessage;
  var aHandled: Boolean);
begin
  inherited;

  case aMessage.Msg of
    WM_LBUTTONDOWN: ButtonDownHandler(aMessage, aHandled);
    WM_RBUTTONDOWN: ButtonDownHandler(aMessage, aHandled);
    WM_LBUTTONUP: ButtonUpHandler(aMessage, aHandled);
    WM_RBUTTONUP: ButtonUpHandler(aMessage, aHandled);
    WM_MOUSEMOVE: MouseMoveHandler(aMessage, aHandled);
  end;
end;

procedure TCustomDragGestureRecognizer.ButtonDownHandler(
  var aMessage: TMessage; var aHandled: Boolean);
var
  msg: TWMLButtonDown;
  mode: TRecognizeStartingMode;
begin
  if tracking_ then begin
    tracking_ := False;
    Exit;
  end;

  msg := TWMLButtonDown(aMessage);
  if (msg.Keys and MK_LBUTTON) <> 0 then button_ := mbLeft;
  if (msg.Keys and MK_RBUTTON) <> 0 then button_ := mbRight;
  if (msg.Keys and MK_MBUTTON) <> 0 then button_ := mbMiddle;

  mode := rmContinue;
  DoRecognizeStart(mode);
  if mode = rmCancel then Exit;

  if mode = rmContinueAndThrough then aHandled := False
  else aHandled := not FThroughMessage;

  startPos_.x := msg.XPos;
  startPos_.y := msg.YPos;
  tracking_ := True;
end;

procedure TCustomDragGestureRecognizer.ButtonUpHandler(
  var aMessage: TMessage; var aHandled: Boolean);
const
  buttonMessages: array[0..2] of Integer = (
    WM_LBUTTONDOWN, WM_RBUTTONDOWN, WM_MBUTTONDOWN
  );
var
  msg: TMessage;
begin
  if not tracking_ then Exit;

  tracking_ := False;
  if not FThroughMessage then begin
    msg := aMessage;
    msg.Msg := buttonMessages[Integer(button_)];
    OldWndProc(msg);
  end;

  DoRecognizeEnd;
end;

procedure TCustomDragGestureRecognizer.MouseMoveHandler(
  var aMessage: TMessage; var aHandled: Boolean);
var
  msg: TWMMouseMove;
  dx: Integer;
  dy: Integer;
  d: Integer;
begin
  if not tracking_ then Exit;
  aHandled := not FThroughMessage;

  msg := TWMMouseMove(aMessage);

  dx := startPos_.x - msg.XPos;
  dy := startPos_.y - msg.YPos;
  d := Round(Sqrt(Sqr(dx) + Sqr(dy)));

  if d <= FThreshold then Exit;

  DoRecognized;

  tracking_ := False;

  DoRecognizeEnd;
end;

procedure TCustomDragGestureRecognizer.DoRecognizeStart(
  var aMode: TRecognizeStartingMode);
var
  point: TPoint;
begin
  if Assigned(FOnRecognizeStart) then begin
    GetCursorPos(point);
    FOnRecognizeStart(Self, button_, point, aMode);
  end;
end;

procedure TCustomDragGestureRecognizer.DoRecognized;
begin
  if Assigned(FOnRecognized) then FOnRecognized(Self, button_);
end;

procedure TCustomDragGestureRecognizer.DoRecognizeEnd;
begin
  if Assigned(FOnRecognizeEnd) then FOnRecognizeEnd(Self);
end;


{ TOleDropSource }

constructor TOleDropSource.Create(anOwner: TComponent);
begin
  inherited Create(anOwner);

  FDropEffect := ekNone;
  Include(FValidDropEffects, ekCopy);

  FDropSource := TDropSourceImpl.Create;
  FDropSource.Events.OnGiveFeedback := GiveFeedback;
  FDropSource.Events.OnQueryContinueDrag := QueryContinueDrag;
end;

destructor TOleDropSource.Destroy;
begin
  inherited;
end;

function TOleDropSource.DoDragDrop(anObject: IDataObject): Boolean;
var
  dropEffect: DWord;
  ret: HResult;
begin
  dropEffect := DROPEFFECT_NONE;
  ret := ActiveX.DoDragDrop(anObject, FDropSource
    , GetDropEffectValue(FValidDropEffects), Longint(dropEffect));
  FDropEffect := GetDropEffect(dropEffect);
  Result := ret = DRAGDROP_S_DROP;
end;

procedure TOleDropSource.GiveFeedback(aDropEffect: Longint; var aResult: HResult);
begin
  if Assigned(FOnGiveFeedback) then
    FOnGiveFeedback(Self, GetDropEffect(aDropEffect), aResult);
end;

procedure TOleDropSource.QueryContinueDrag(anEscapePressed: Boolean;
      aKeyState: Integer; var aResult: HResult);
begin
  if Assigned(FOnQueryContinueDrag) then
    FOnQueryContinueDrag(Self, anEscapePressed, aKeyState, aResult);
end;


{ TFileDropSource }

constructor TFileDropSource.Create(anOwner: TComponent);
begin
  inherited;

  FFiles := TStringList.Create;
end;

destructor TFileDropSource.Destroy;
begin
  FFiles.Free;

  inherited;
end;

procedure TFileDropSource.DoRecognized;
var
  dataObject: IDataObject;
  directory: string;
begin
  inherited;

  if FFiles.Count = 0 then Exit;

  directory := ExpandFileName(FDirectory);
  if Length(directory) = 0 then begin
    directory := ExtractFilePath(ExpandFileName(FFiles[0]));
  end;

  dataObject := GetFileDataObject(Application.Handle, directory, FFiles);
  DoDragDrop(dataObject);
end;

function TFileDropSource.GetFiles: TStrings;
begin
  Result := FFiles;
end;

procedure TFileDropSource.SetFiles(files: TStrings);
begin
  FFiles.Assign(files);
end;


{ TDFItem }

constructor TDFItem.Create(anOwner: TCollection);
begin
  inherited;

   Include(FDropEffects, ekCopy);
end;

procedure TDFItem.FireDrop(var aContext: TDragContext);
begin
  if Assigned(FOnDrop) then
    FOnDrop(Self, aContext);
end;

procedure TDFItem.FireEnter(var aContext: TDragContext);
begin
  if Assigned(FOnDragEnter) then
    FOnDragEnter(Self, aContext);
end;

procedure TDFItem.FireOver(var aContext: TDragContext);
begin
  if Assigned(FOnDragOver) then
    FOnDragOver(Self, aContext);
end;

function TDFItem.GetDisplayName: string;
begin
  if Length(FCFName) = 0 then begin
    Result := inherited GetDisplayName;
  end else begin
    Result := FCFName;
  end;
end;

procedure TDFItem.Register;
begin
  if Length(FCFName) > 0 then
    FCFValue := GetCFValue(FCFName);
end;

procedure TDFItem.SetFormatName(aName: string);
begin
  FCFName := aName;
  DisplayName := aName;
end;


{ TDFItems }

constructor TDFItems.Create(anOwner: TComponent);
begin
  inherited Create(TDFItem);

  FOwner := anOwner;
end;

function TDFItems.GetOwner: TPersistent;
begin
  Result := FOwner;
end;

procedure TDFItems.RegisterAll;
var
  i: Integer;
begin
  for i := 0 to Count - 1 do begin
    (Items[i] as TDFItem).Register;
  end;
end;


{ TOleDropTarget }

constructor TCustomOleDropTarget.Create(anOwner: TComponent);
begin
  inherited;

  FFormats := TDFItems.Create(Self);

  FDropTarget := TDropTargetImpl.Create;
  FDropTarget.Events.OnDragEnter := DragEnter;
  FDropTarget.Events.OnDragOver := DragOver;
  FDropTarget.Events.OnDragLeave := DragLeave;
  FDropTarget.Events.OnDrop := Drop;
end;

destructor TCustomOleDropTarget.Destroy;
begin
  WinControl := Nil;
  FFormats.Free;

  inherited;
end;

procedure TCustomOleDropTarget.DragEnter(const aDataObject: IDataObject;
  aKeyState: Integer; aPoint: TPoint; var aDropEffect: Integer;
  var aResult: HResult);
var
  dropEffect: TDropEffect;
  point: TPoint;
  dragContext: TDragContext;
begin
  FFoundItem := Nil;
  dropEffect := GetDropEffect(aDataObject, aKeyState, aDropEffect);
  aDropEffect := GetDropEffectValue([dropEffect]);

  point := GetClientPoint(aPoint);
  dragContext.DataObject := aDataObject;
  dragContext.KeyState := aKeyState;
  dragContext.Point := point;
  dragContext.DropEffect := aDropEffect;
  dragContext.Result := aResult;

  if FFoundItem <> Nil then FFoundItem.FireEnter(dragContext);
  if Assigned(FOnDragEnter) then FOnDragEnter(Self, dragContext);

  aDropEffect := dragContext.DropEffect;
  aResult := dragContext.Result;
end;

procedure TCustomOleDropTarget.DragLeave(var aResult: HResult);
begin
  if Assigned(FOnDragLeave) then
    FOnDragLeave(Self, aResult);
end;

procedure TCustomOleDropTarget.DragOver(aKeyState: Integer; aPoint: TPoint;
  var aDropEffect: Integer; var aResult: HResult);
var
  dropEffect: TDropEffect;
  dragContext: TDragContext;
begin
  FFoundItem := Nil;
  dropEffect := GetDropEffect(FDropTarget.DataObject, aKeyState, aDropEffect);
  aDropEffect := GetDropEffectValue([dropEffect]);

  dragContext.DataObject := FDropTarget.DataObject;
  dragContext.KeyState := aKeyState;
  dragContext.Point := GetClientPoint(aPoint);
  dragContext.DropEffect := aDropEffect;
  dragContext.Result := aResult;

  if FFoundItem <> Nil then FFoundItem.FireOver(dragContext);
  if Assigned(FOnDragOver) then FOnDragOver(Self, dragContext);

  aDropEffect := dragContext.DropEffect;
  aResult := dragContext.Result;
end;

procedure TCustomOleDropTarget.Drop(const aDataObj: IDataObject;
  aKeyState: Integer; aPoint: TPoint; var aDropEffect: Integer;
  var aResult: HResult);
var
  dragContext: TDragContext;
begin
  aDropEffect := FDropTarget.DropEffect;

  dragContext.DataObject := FDropTarget.DataObject;
  dragContext.KeyState := aKeyState;
  dragContext.Point := GetClientPoint(aPoint);
  dragContext.DropEffect := aDropEffect;
  dragContext.Result := aResult;

  if FFoundItem <> Nil then FFoundItem.FireDrop(dragContext);
  if Assigned(FOnDrop) then FOnDrop(Self, dragContext);

  aDropEffect := dragContext.DropEffect;
  aResult := dragContext.Result;
end;

procedure TCustomOleDropTarget.FinalizeHWND(aHandle: HWND);
begin
  if csDesigning in ComponentState then Exit;
  RevokeDragDrop(WinControl.Handle);
end;

function TCustomOleDropTarget.GetClientPoint(aPoint: TPoint): TPoint;
begin
  Result := aPoint;
  ScreenToClient(WinControl.Handle, Result);
end;

function TCustomOleDropTarget.GetDropEffect(aDataObject: IDataObject; aKeyState,
  aValidEffect: DWord): TDropEffect;
var
  validEffects: TDropEffects;
  reader: TDataObjectReader;
  empty, intersection: TDropEffects;
  i, n: Integer;
  item: TDFItem;
begin
  validEffects := GetDropEffects(aValidEffect);

  reader := TDataObjectReader.Create(aDataObject);
  try
    Result := ekNone;
    n := FFormats.Count;
    for i := 0 to n - 1 do begin
      item := FFormats.Items[i] as TDFItem;
      if not reader.HasFormat(item.Format) then Continue;

       intersection := item.DropEffects * validEffects;
      if intersection = empty then Continue;

      Result := GetStdDropEffect(aKeyState);
      if not (Result in intersection) then begin
        if ekCopy in intersection then Result := ekCopy
        else if ekMove in intersection then Result := ekMove
        else if ekLink in intersection then Result := ekLink
        else begin
          Result := ekNone;
          Exit;
        end;
      end;
      FFoundItem := item;
      Break;
    end;
  finally
    reader.Free;
  end;
end;

procedure TCustomOleDropTarget.InitializeHWND(aHandle: HWND);
begin
  if csDesigning in ComponentState then Exit;
  RegisterDragDrop(WinControl.Handle, FDropTarget);
end;

procedure TCustomOleDropTarget.Loaded;
begin
  inherited;

  if csDesigning in ComponentState then Exit;

  FFormats.RegisterAll;
end;

procedure TCustomOleDropTarget.SetFormats(items: TDFItems);
begin
  raise Exception.Create('̓̓T|[gĂ܂.');
end;


{ TFileDropTarget }

constructor TFileDropTarget.Create(anOwner: TComponent);
begin
  inherited;

  FDropEffects := [ekCopy];
  FDroppedFiles := TStringList.Create;

  fileItem_ := Formats.Add as TDFItem;
  fileItem_.DropEffects := FDropEffects;
  fileItem_.FormatName := 'CF_HDROP';
  fileItem_.OnDragEnter := FileDragEnter;
  fileItem_.OnDragOver := FileDragOver;
  fileItem_.OnDrop := FileDrop;
  OnDragLeave := FileDragLeave;
end;

destructor TFileDropTarget.Destroy;
begin
  WinControl := Nil;
  FDroppedFiles.Free;

  inherited;
end;

procedure TFileDropTarget.FileDragEnter(aSender: TObject;
  var aContext: TDragContext);
begin
  if Assigned(FOnFileDragEnter) then FOnFileDragEnter(Self, aContext);
end;

procedure TFileDropTarget.FileDragLeave(aSender: TObject;
  var aResult: HResult);
begin
  if (FoundItem <> Nil) and Assigned(FOnFileDragEnter) then
    FOnFileDragLeave(Self, aResult);
end;

procedure TFileDropTarget.FileDragOver(aSender: TObject;
  var aContext: TDragContext);
begin
  if Assigned(FOnFileDragOver) then FOnFileDragOver(Self, aContext);
end;

procedure TFileDropTarget.FileDrop(aSender: TObject;
  var aContext: TDragContext);
var
  reader: TDataObjectReader;
begin
  reader := TDataObjectReader.Create(aContext.DataObject);
  try
    FDroppedFiles.Clear;
    reader.GetFileList(FDroppedFiles);
  finally
    reader.Free;
  end;

  if Assigned(FOnFileDrop) then FOnFileDrop(Self, aContext);
end;

procedure TFileDropTarget.SetDropEffects(dropEffects: TDropEffects);
begin
  FDropEffects := dropEffects;
  fileItem_.DropEffects := dropEffects;
end;


initialization
  OleInitialize(Nil);

finalization
  OleUninitialize;

end.

