/*
 * Copyright 2009 Funambol, Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/* $Id$ */

#include "posix/Timer.h"
#include "Logger/LoggerMacroses.h"
//#include <sys/types.h>
#include <time.h>
#include <pthread.h>


namespace NS_DM_Client
{
    namespace NS_Common
    {
        void TimerAction(union sigval sigval);
    }
}

using namespace NS_DM_Client;
using namespace NS_DM_Client::NS_Common;

const char* const c_LogName = "Timer";

Timer::Timer() :
    m_isRunning(false), m_ticksCount(0), m_ticksToCount(0), m_timerInterval(0), m_callback(NULL), m_timerid(0)
{
}


Timer::~Timer()
{
	Stop();
}


bool Timer::IsRunning()
{
	return m_isRunning;
}


void Timer::SetCallback(Callback *c)
{
	GDLDEBUG("Callback %p", c);
	m_callback = c;
}


void Timer::SetInterval(uint t)
{
	m_timerInterval = t;
}


void Timer::SetTicksCount(int count)
{
	m_ticksCount = count;
}


void Timer::Start()
{
	if (m_isRunning) return;

	if (m_timerInterval)
	{
		if (m_ticksCount == -1 || m_ticksCount > 0)
			startTimer();
	}
}


void Timer::Stop()
{
	if (!m_isRunning) return;

	timer_delete(m_timerid);
	m_timerid = 0;
	m_isRunning = false;
}


void Timer::notifyCallback()
{
	if (m_callback) m_callback->Call();
}


void Timer::startTimer()
{
	m_ticksToCount = m_ticksCount;
	
	// tick once before starting the timer
	if (m_ticksToCount)
		tick();
	
	if (!m_ticksToCount) return;
	
	struct sigevent sigev;
	struct itimerspec itval;
	struct itimerspec oitval;

	memset(&sigev, 0, sizeof(sigevent));

	// Create the POSIX timer and pass pointer to self
	sigev.sigev_notify = SIGEV_THREAD;
	sigev.sigev_value.sival_ptr = this;
	sigev.sigev_notify_function = TimerAction;

	// CLOCK_REALTIME - counts systemwide time
	if (timer_create(CLOCK_REALTIME, &sigev, &m_timerid) == 0)
	{
		GDLDEBUG("created timer: %x", m_timerid);
		itval.it_value.tv_sec = m_timerInterval;
		itval.it_value.tv_nsec = 0;

		itval.it_interval.tv_sec = itval.it_value.tv_sec;
		itval.it_interval.tv_nsec = itval.it_value.tv_nsec;

		if (timer_settime(m_timerid, 0, &itval, &oitval) == 0)
		{
			m_isRunning = true;
		}
		else
		{
			GDLDEBUG("time_settime error!");
		}
	}
	else
	{
		GDLDEBUG("timer_create failed !");
	}
}


void Timer::tick()
{
	if (m_ticksToCount == 0)
	{
		Stop();
	}
	else
	{
		if (m_ticksToCount > 0)
			m_ticksToCount--;
		notifyCallback();				
	}
}


void NS_DM_Client::NS_Common::TimerAction(union sigval val)
{
	Timer *pTimer = (Timer*) val.sival_ptr;
	pTimer->tick();
	pthread_exit(NULL);
}
