/*
 * Copyright 2009 Funambol, Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/* $Id$ */
#pragma once
////#ifndef IDATASTORAGE_H_INCLUDED
////#define IDATASTORAGE_H_INCLUDED

#include <cstdio>
#include <vector>
#include "commontypes.h"
#include "platform.h"

namespace NS_DM_Client
{

class Buffer;

namespace NS_DataStorage
{
/// platform independent interface for handle stream data
class IStreamHandler
{
public:
    virtual ~IStreamHandler() { }
    IStreamHandler() { }
    /// write portion of data to stream
    virtual bool Write(const void* buffer, size_t size) = 0;
    virtual bool Write(const Buffer& buffer) = 0;

    /// indicate that writing is finished and we can read from stream
    virtual bool FinishWriting() = 0;
    /// release stream resources
    virtual void Release() = 0;
private:
    IStreamHandler(const IStreamHandler&);
    IStreamHandler& operator=(const IStreamHandler&);
};

/// platform independent interface for store program data
class IDataStorage
{
public:
    virtual ~IDataStorage() { }
    IDataStorage() { }

    virtual bool Init(const String& base_path = "") = 0;
    virtual void Release() = 0;

    /// functions for work with stream data
    /// return stream handler which uniquely identify stream
    /// NOTE: need call Release() for release object resources
    virtual IStreamHandler* CreateStream() = 0;

    /// functions for daemon's private data
    /// key - identify unique private data
    virtual bool SavePrivateData(const String& key, const void* buffer, size_t size, bool profileSpecific = true) = 0;
    virtual bool SavePrivateData(const String& key, const Buffer& buffer, bool profileSpecific = true) = 0;
    virtual bool LoadPrivateData(const String& key, Buffer& buffer, bool profileSpecific = true) = 0;
    virtual bool RemovePrivateData(const String& key) = 0;

    // get child list
    virtual bool GetChildList(const String& key, StringArray& list, bool profileSpecific = true) = 0;

    // check peresistence
    virtual bool Exist(const String& key, bool profileSpecific = true) = 0;
    virtual const char* GetBasePath() = 0;

    virtual bool GetPrivateDataSize(const String& key, size_t& size, bool profileSpecific = true) = 0;

private:
    IDataStorage(const IDataStorage& data);
    IDataStorage& operator=(const IDataStorage& data);
};

class IConfigurationStorage
{
public:
    virtual ~IConfigurationStorage() { }
    IConfigurationStorage() { }

    virtual bool Init() = 0;
    virtual void Release() = 0;

    /// functions for daemon's configuration
    virtual bool SaveConfiguration(const void* buffer, size_t size) = 0;
    virtual bool SaveConfiguration(const Buffer& buffer) = 0;
    virtual bool LoadConfiguration(Buffer& buffer) = 0;

private:
    IConfigurationStorage(const IConfigurationStorage& data);
    IConfigurationStorage& operator=(const IConfigurationStorage& data);
};

/// factory method for IDataStorage instance creation
IDataStorage* CreateDataStorage(const String& profile, const String& base_path = "");

/// factory method for IConfigurationStorage instance creation
IConfigurationStorage* CreateConfigurationStorage();

const char* GetBasePath();
}

}
////#endif // IDATASTORAGE_H_INCLUDED
