package cn.com.qimingx.spring;

import java.sql.Connection;
import java.sql.Date;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Types;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * 従来のPreparedStatementとは異なり、プレースホルダ"?"にパラメータ名をつけたSQLを使ってパラメータ設定を行う。
 * 使い方は以下の通り。
 * <code>
 * String sql = "SELECT * FROM EXAMPLE_TBL WHERE NAME = ?name AND AGE = ?age";
 * NamedPreparedStatement stmt = new NamedPreparedStatement(sql);
 * stmt.setString("name", "田中");
 * stmt.setInt("age", 20);
 * ReusltSet rs = stmt.executeQuery(connection);
 * </code>
 */
public class NamedPreparedStatement {

	/**
	 * 実行すべきSQL。
	 */
	private String sql = null;

	/**
	 * 直前に実行したSQL。
	 */
	private String preExecuteSql = "";

	/**
	 * パラメータ名をキーとして設定値を保持するマップ。
	 */
	private HashMap<String, ArrayList<IParamHolder>> paramListMap
		= new HashMap<String, ArrayList<IParamHolder>>();

	/**
	 * パラメータ名に使用できる文字列を表した正規表現。
	 */
	private static final String PARAM_NAME_CHAR_PATTERN = "\\:[a-zA-Z0-9_]*";

	/**
	 * 日付型データの文字列書式
	 * デバッグ時に使用。
	 */
	private static final SimpleDateFormat dateFormat = new SimpleDateFormat("yyyy-MM-dd");

	/**
	 * ステートメント。
	 */
	private PreparedStatement pstmt;

	/**
	 * コンストラクタ。
	 * @param sql SQL
	 */
	public NamedPreparedStatement(String sql) {

		Pattern pattern = Pattern.compile(PARAM_NAME_CHAR_PATTERN);
		Matcher matcher = pattern.matcher(sql);

		// SQL中にパラメータ名の無い"?"だけのプレースホルダがあれば例外スローする
		while(matcher.find()) {
			if (matcher.group().length() == 1) {
				throw new IllegalArgumentException(
						String.format(
								"No parameter name. Parameter name is required after ':'. (%d column): %s",
								matcher.start(),
								sql)
								);
			}
		}

		this.sql = sql;
	}

	/**
	 * 文字列型パラメータの設定
	 * @param name パラメータ名
	 * @param values 設定値
	 */
	public void setString(String name, String...values) {

		if (isEmpty(values)) {
			return;
		}

		ArrayList<IParamHolder> holderlist = new ArrayList<IParamHolder>();

		for (String value : values) {

			IParamHolder paramHolder = new IParamHolder(){

				/**
				 * SQLに設定する値
				 */
				private String value;

				/**
				 * SQL設定値の設定
				 * @param value 設定値
				 */
				public void setValue(Object value) {
					this.value = (String)value;
				}

				/**
				 * 文字列型の値を指定位置に設定します。
				 */
				public void set(PreparedStatement pstmt, int parameterIndex) throws SQLException {
					if (value == null) {
						pstmt.setNull(parameterIndex, Types.CHAR);
					} else {
						pstmt.setString(parameterIndex, value);
					}
				}

				/**
				 * デバッグ用パラメータ出力
				 */
				public String debug() {
					if (value == null) {
						return "null";
					}
					return String.format("'%s'", value);
				}
			};

			paramHolder.setValue(value);
			holderlist.add(paramHolder);
		}

		paramListMap.put(name, holderlist);
	}

	/**
	 * 数値型パラメータの設定
	 * @param name パラメータ名
	 * @param values 設定値
	 */
	public void setInt(String name, Integer...values) {

		if (isEmpty(values)) {
			return;
		}

		ArrayList<IParamHolder> holderlist = new ArrayList<IParamHolder>();

		for (Integer value : values) {

			IParamHolder paramHolder = new IParamHolder(){

				/**
				 * SQLに設定する値
				 */
				private Integer value;

				/**
				 * SQL設定値の設定
				 * @param value 設定値
				 */
				public void setValue(Object value) {
					this.value = (Integer)value;
				}

				/**
				 * 数値型の値を指定位置に設定します。
				 */
				public void set(PreparedStatement pstmt, int parameterIndex) throws SQLException {
					if (value == null) {
						pstmt.setNull(parameterIndex, Types.INTEGER);
					} else {
						pstmt.setInt(parameterIndex, value);
					}
				}

				/**
				 * デバッグ用パラメータ出力
				 */
				public String debug() {
					if (value == null) {
						return "null";
					}
					return String.valueOf(value);
				}
			};

			paramHolder.setValue(value);
			holderlist.add(paramHolder);
		}

		paramListMap.put(name, holderlist);
	}

	/**
	 * 日付型パラメータの設定
	 * @param name パラメータ名
	 * @param values 設定値
	 */
	public void setDate(String name, Date...values) {

		if (isEmpty(values)) {
			return;
		}

		ArrayList<IParamHolder> holderlist = new ArrayList<IParamHolder>();

		for (Date value : values) {

			IParamHolder paramHolder = new IParamHolder(){

				/**
				 * SQLに設定する値
				 */
				private Date value;

				/**
				 * SQL設定値の設定
				 * @param value 設定値
				 */
				public void setValue(Object value) {
					this.value = (Date)value;
				}

				/**
				 * 日付型の値を指定位置に設定します。
				 */
				public void set(PreparedStatement pstmt, int parameterIndex) throws SQLException {
					if (value == null) {
						pstmt.setNull(parameterIndex, Types.DATE);
					} else {
						pstmt.setDate(parameterIndex, value);
					}
				}

				/**
				 * デバッグ用パラメータ出力
				 */
				public String debug() {
					if (value == null) {
						return "null";
					}
					return String.format("TO_DATE('%s', 'yyyy-MM-dd'')", dateFormat.format(value));
				}
			};

			paramHolder.setValue(value);
			holderlist.add(paramHolder);
		}

		paramListMap.put(name, holderlist);
	}


	/**
	 * 検索処理の実行
	 * @param con コネクション
	 * @return 検索結果
	 * @throws SQLException パラメータ名に設定値がない場合に発生。
	 */
	public ResultSet executeQuery(Connection con) throws SQLException{
		createStatment(con);
		return pstmt.executeQuery();
	}

	/**
	 * 更新処理の実行
	 * @param con コネクション
	 * @return 更新結果件数
	 * @throws SQLException パラメータ名に設定値がない場合に発生。
	 */
	public int executeUpdate(Connection con) throws SQLException{
		createStatment(con);
		return pstmt.executeUpdate();
	}

	/**
	 * 値設定済みの実行直前ステートメントを作成する。
	 * @param con コネクション
	 * @return ステートメント
	 * @throws SQLException パラメータ名に設定値がない場合に発生。
	 */
	private void createStatment(Connection con) throws SQLException {

		// コンソールにデバッグ用SQLを出力
		System.out.println(debug());

		String statementSql = this.sql;

		ArrayList<IParamHolder> settingOrderList = new ArrayList<IParamHolder>();

		Pattern pattern = Pattern.compile(PARAM_NAME_CHAR_PATTERN);
		Matcher matcher = pattern.matcher(this.sql);

		// "?"のパラメータの数だけ繰り返し
		while(matcher.find()) {

			// マッチしたパラメータ名には?が含まれるので、"?"以降の2文字目からのパラメータ名だけ取得する
			String name = matcher.group().substring(1);

			if (!paramListMap.containsKey(name)) {
				throw new SQLException(String.format("Parameter value unknown. Parameter name: ?%s", name));
			}

			// パラメータ名に対して設定するパラメータリストを取得
			ArrayList<IParamHolder> paramHolderList = paramListMap.get(name);

			// パラメータの数だけプレースホルダを作成
			StringBuilder sb = new StringBuilder("?");
			for (int i = 1; i < paramHolderList.size(); i++) {
				sb.append(", ?");
			}

			// パラメータ名を"?"だけに変えて、PreparedStatementに設定するSQLに変更する
			statementSql = statementSql.replaceFirst("\\:" + name, sb.toString());

			// ステートメントへ設定する順にパラメータを保持しなおす
			settingOrderList.addAll(paramHolderList);
		}

		// 実行すべきSQLが前回実行時のSQLと同じであるか検査
		if (this.preExecuteSql.equals(statementSql)) {

			// 実行SQLが同じであればステートメントの使いまわしができるので
			// ステートメントに設定したパラメータのクリアだけして、ステートメント自体はそのまま利用する
			this.pstmt.clearParameters();

		} else {

			// 前回実行したSQLが異なる場合はステートメントに保持されたSQLが使えないので
			// 旧ステートメントをクローズして、新規ステートメントを生成する。
			// 初回実行もこちらで処理されるがクローズ処理は実行されないだけで同じ流れ。
			close();
			this.pstmt = con.prepareStatement(statementSql);
		}

		// 設定順に並べたパラメータリストをそのままステートメントへ設定していく
		for (int i = 0; i < settingOrderList.size(); i++ ) {
			IParamHolder paramHolder = settingOrderList.get(i);

			// ステートメントの設定開始番号は、"0"でなく"1"からなので+1した番号を渡す
			paramHolder.set(pstmt, i + 1);
		}
	}

	/**
	 * 実行可能な形式のデバッグ用SQLを出力する。
	 * @return デバッグ用SQL
	 */
	public String debug() {

		String degugSql = this.sql;

		Pattern pattern = Pattern.compile(PARAM_NAME_CHAR_PATTERN);
		Matcher matcher = pattern.matcher(this.sql);

		// "?"のパラメータの数だけ繰り返し
		while(matcher.find()) {

			// マッチしたパラメータ名には?が含まれるので、"?"以降の2文字目からのパラメータ名だけ取得する
			String name = matcher.group().substring(1);

			// ステートメント作成時にはエラーになるSQLも出力したいので
			// パラメータが設定されてない場合も適当な文字を設定
			if (!paramListMap.containsKey(name)) {
				degugSql = degugSql.replaceFirst("\\:" + name, "<no-setting>");
				continue;
			}

			// パラメータ名に対して設定するパラメータリストを取得
			ArrayList<IParamHolder> paramHolderList = paramListMap.get(name);

			// パラメータの数だけ値を挿入
			StringBuilder sb = new StringBuilder(paramHolderList.get(0).debug());
			for (int i = 1; i < paramHolderList.size(); i++) {
				sb.append(", " + paramHolderList.get(i).debug());
			}

			// パラメータ名を"?"だけに変えて、PreparedStatementに設定するSQLに変更する
			degugSql = degugSql.replaceFirst("\\:" + name, sb.toString());
		}

		return "[Debug SQL] "+ degugSql;
	}

	/**
	 * パラメータのクリア
	 * @throws SQLException 通常のクリア処理と同様原因で発生。
	 */
	public void clearParameters() throws SQLException {
		this.pstmt.clearParameters();
		this.paramListMap.clear();
		this.preExecuteSql = "";
	}

	/**
	 * ステートメントのクローズ。
	 * @throws SQLException 通常のクローズ処理と同様原因で発生。
	 */
	public void close() throws SQLException {

		if (!isClosed()) {
			this.pstmt.close();
			this.pstmt = null;
		}
	}

	/**
	 * ステートメントのクローズ確認。
	 * @return クローズ確認結果
	 * @throws SQLException 通常のクローズ確認処理と同様原因で発生。
	 */
	public boolean isClosed() throws SQLException {
		return this.pstmt == null || this.pstmt.isClosed();
	}

	/**
	 * 配列の空チェック。
	 * null or 要素０個 ならば真、それ以外ならば偽を返却する。
	 * @param list 配列
	 * @return チェック結果
	 */
	private boolean isEmpty(Object[] list) {
		if (list == null || list.length == 0) {
			return true;
		}
		return false;
	}

	/**
	 * 設定値の保持とステートメントへの値設定処理を持つインターフェース
	 */
	private interface IParamHolder {
		public abstract void setValue(Object pram);
		public abstract void set(PreparedStatement pstmt, int parameterIndex) throws SQLException;
		public abstract String debug();
	}

}
