/*
 * Copyright (c) 2003-2004 The Ochusha Project.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Id: ochusha_bbs_thread.c,v 1.27 2004/01/18 04:50:26 fuyu Exp $
 */

#include "config.h"

#include "ochusha_private.h"
#include "ochusha_bbs_thread.h"
#include "ochusha_bulletin_board.h"

#include "utils.h"

#include "marshal.h"

#include <glib.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <zlib.h>


static void ochusha_bbs_thread_class_init(OchushaBBSThreadClass *klass);
static void ochusha_bbs_thread_init(OchushaBBSThread *thread);
static void ochusha_bbs_thread_finalize(GObject *object);

static void ochusha_bbs_thread_set_property(GObject *object, guint prop_id,
					    const GValue *value,
					    GParamSpec *pspec);
static void ochusha_bbs_thread_get_property(GObject *object, guint prop_id,
					    GValue *value,
					    GParamSpec *pspec);
static void ochusha_bbs_thread_read_threadlist_element(
						OchushaBBSThread *thread,
						GHashTable *thread_attributes);
static void ochusha_bbs_thread_write_threadlist_element(
						OchushaBBSThread *thread,
						gzFile *threadlist_xml);

GType
ochusha_bbs_thread_get_type(void)
{
  static GType bbs_thread_type = 0;

  if (bbs_thread_type == 0)
    {
      static const GTypeInfo bbs_thread_info =
	{
	  sizeof(OchushaBBSThreadClass),
	  NULL, /* base_init */
	  NULL, /* base_finalize */
	  (GClassInitFunc)ochusha_bbs_thread_class_init,
	  NULL, /* class_finalize */
	  NULL, /* class_data */
	  sizeof(OchushaBBSThread),
	  0,	/* n_preallocs */
	  (GInstanceInitFunc)ochusha_bbs_thread_init,
	};

      bbs_thread_type = g_type_register_static(G_TYPE_OBJECT,
					       "OchushaBBSThread",
					       &bbs_thread_info, 0);
    }

  return bbs_thread_type;
}


enum {
  READ_THREADLIST_ELEMENT_SIGNAL,
  WRITE_THREADLIST_ELEMENT_SIGNAL,
  LAST_SIGNAL
};


enum {
  PROP_0,
  PROP_BOARD,
  PROP_ID,
  PROP_TITLE
};


static GObjectClass *parent_class = NULL;
static int bbs_thread_signals[LAST_SIGNAL] = { 0, 0 };


static void
ochusha_bbs_thread_class_init(OchushaBBSThreadClass *klass)
{
  GObjectClass *o_class = G_OBJECT_CLASS(klass);

  parent_class = g_type_class_peek_parent(klass);

  o_class->set_property = ochusha_bbs_thread_set_property;
  o_class->get_property = ochusha_bbs_thread_get_property;
  o_class->finalize = ochusha_bbs_thread_finalize;

  g_object_class_install_property(o_class,
				  PROP_BOARD,
				  g_param_spec_object("board",
						      _("Board Object"),
						      _("The Board to Which This Thread Belongs"),
						      OCHUSHA_TYPE_BULLETIN_BOARD,
						      G_PARAM_READWRITE));
  g_object_class_install_property(o_class,
				  PROP_ID,
				  g_param_spec_string("id",
						      _("Thread ID"),
						      _("String Identifies Each Threads"),
						      NULL,
						      G_PARAM_READWRITE));
  g_object_class_install_property(o_class,
				  PROP_TITLE,
				  g_param_spec_string("title",
						      _("Title"),
						      _("The Title of the Thread"),
						      _("Untitled Thread"),
						      G_PARAM_READWRITE));

  bbs_thread_signals[READ_THREADLIST_ELEMENT_SIGNAL] =
    g_signal_new("read_threadlist_element",
		 G_TYPE_FROM_CLASS(klass),
		 G_SIGNAL_RUN_FIRST,
		 G_STRUCT_OFFSET(OchushaBBSThreadClass,
				 read_threadlist_element),
		 NULL, NULL,
		 libochusha_marshal_VOID__POINTER,
		 G_TYPE_NONE, 1,
		 G_TYPE_POINTER);
  bbs_thread_signals[WRITE_THREADLIST_ELEMENT_SIGNAL] =
    g_signal_new("write_threadlist_element",
		 G_TYPE_FROM_CLASS(klass),
		 G_SIGNAL_RUN_FIRST,
		 G_STRUCT_OFFSET(OchushaBBSThreadClass,
				 write_threadlist_element),
		 NULL, NULL,
		 libochusha_marshal_VOID__POINTER,
		 G_TYPE_NONE, 1,
		 G_TYPE_POINTER);

  klass->read_threadlist_element = ochusha_bbs_thread_read_threadlist_element;
  klass->write_threadlist_element
    = ochusha_bbs_thread_write_threadlist_element;

  klass->get_board = NULL;
  klass->get_number_of_responses_on_server = NULL;
  klass->get_number_of_responses_read = NULL;
  klass->set_number_of_responses_read = NULL;
  klass->get_flags = NULL;
  klass->set_flags = NULL;

  klass->get_responses_source = NULL;
  klass->parse_responses = NULL;
  klass->get_last_modified_utc = NULL;
  klass->get_url = NULL;
  klass->get_url_for_response = NULL;
  klass->get_url_to_post = NULL;
  klass->check_url = NULL;
  klass->remove_cache = NULL;
}


static void
ochusha_bbs_thread_init(OchushaBBSThread *thread)
{
  thread->board = NULL;

  thread->id = NULL;

  thread->title = NULL;

  thread->number_of_responses_on_server = 0;
  thread->number_of_responses_read = 0;

  thread->flags = 0;
}


static void
ochusha_bbs_thread_finalize(GObject *object)
{
  OchushaBBSThread *thread = OCHUSHA_BBS_THREAD(object);

  if (thread->board != NULL)
    {
#if 0	/* Ȥ롼פΤǱѤǥС */
      OCHU_OBJECT_UNREF(thread->board);
#endif
      thread->board = NULL;
    }

  if (thread->id != NULL)
    {
      G_FREE(thread->id);
      thread->id = NULL;
    }

  if (thread->title != NULL)
    {
      G_FREE(thread->title);
      thread->title = NULL;
    }

  if (G_OBJECT_CLASS(parent_class)->finalize)
    (*G_OBJECT_CLASS(parent_class)->finalize)(object);
}


static void
ochusha_bbs_thread_set_property(GObject *object, guint prop_id,
				const GValue *value, GParamSpec *pspec)
{
  OchushaBBSThread *thread = OCHUSHA_BBS_THREAD(object);

  switch (prop_id)
    {
    case PROP_BOARD:
      ochusha_bbs_thread_set_board(thread,
			OCHUSHA_BULLETIN_BOARD(g_value_get_object(value)));
      break;

    case PROP_ID:
      ochusha_bbs_thread_set_id(thread, g_value_get_string(value));
      break;

    case PROP_TITLE:
      ochusha_bbs_thread_set_title(thread, g_value_get_string(value));
      break;

    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
      break;
    }
}


static void
ochusha_bbs_thread_get_property(GObject *object, guint prop_id,
				GValue *value, GParamSpec *pspec)
{
  OchushaBBSThread *thread = OCHUSHA_BBS_THREAD(object);

  switch (prop_id)
    {
    case PROP_BOARD:
      g_value_set_object(value, thread->board);
      break;

    case PROP_ID:
      g_value_set_string(value, thread->id);
      break;

    case PROP_TITLE:
      g_value_set_string(value, thread->title);
      break;

    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
      break;
    }
}


OchushaBBSThread *
ochusha_bbs_thread_new(OchushaBulletinBoard *board, const char *id,
		       const gchar *title)
{
  return OCHUSHA_BBS_THREAD(g_object_new(OCHUSHA_TYPE_BBS_THREAD,
					 "board", board,
					 "id", id,
					 "title", title,
					 NULL));
}


void
ochusha_bbs_thread_set_board(OchushaBBSThread *thread,
			     OchushaBulletinBoard *board)
{
  g_return_if_fail(OCHUSHA_IS_BBS_THREAD(thread));

  if (thread->board == board)
    return;

#if 0	/* Ȥ롼פƤޤΤǱѤǥС */
  if (thread->board != NULL)
    OCHU_OBJECT_UNREF(thread->board);

  if (board != NULL)
    OCHU_OBJECT_REF(board);
#endif

  thread->board = board;

  g_object_notify(G_OBJECT(thread), "board");
}


OchushaBulletinBoard *
ochusha_bbs_thread_get_board(OchushaBBSThread *thread)
{
  OchushaBBSThreadClass *klass;
  g_return_val_if_fail(OCHUSHA_IS_BBS_THREAD(thread), NULL);

  klass = OCHUSHA_BBS_THREAD_GET_CLASS(thread);
  if (klass->get_board)
    return (*klass->get_board)(thread);

  return thread->board;
}


void
ochusha_bbs_thread_set_id(OchushaBBSThread *thread, const char *id)
{
  g_return_if_fail(OCHUSHA_IS_BBS_THREAD(thread));

  if (thread->id != NULL)
    G_FREE(thread->id);
  thread->id = (id != NULL ? G_STRDUP(id) : NULL);

  g_object_notify(G_OBJECT(thread), "id");
}


const char *
ochusha_bbs_thread_get_id(OchushaBBSThread *thread)
{
  g_return_val_if_fail(OCHUSHA_IS_BBS_THREAD(thread), NULL);

  return thread->id;
}


void
ochusha_bbs_thread_set_title(OchushaBBSThread *thread, const gchar *title)
{
  g_return_if_fail(OCHUSHA_IS_BBS_THREAD(thread));

  if (thread->title != NULL)
    G_FREE(thread->title);

  thread->title = (title != NULL ? wipe_string(title) : NULL);

  g_object_notify(G_OBJECT(thread), "title");
}


const gchar *
ochusha_bbs_thread_get_title(OchushaBBSThread *thread)
{
  g_return_val_if_fail(OCHUSHA_IS_BBS_THREAD(thread), NULL);

  return thread->title;
}


int
ochusha_bbs_thread_get_number_of_responses_on_server(OchushaBBSThread *thread)
{
  OchushaBBSThreadClass *klass;
  g_return_val_if_fail(OCHUSHA_IS_BBS_THREAD(thread), 0);

  klass = OCHUSHA_BBS_THREAD_GET_CLASS(thread);
  if (klass->get_number_of_responses_on_server)
    return (*klass->get_number_of_responses_on_server)(thread);
  return thread->number_of_responses_on_server;
}


int
ochusha_bbs_thread_get_number_of_responses_read(OchushaBBSThread *thread)
{
  OchushaBBSThreadClass *klass;
  g_return_val_if_fail(OCHUSHA_IS_BBS_THREAD(thread), 0);

  klass = OCHUSHA_BBS_THREAD_GET_CLASS(thread);
  if (klass->get_number_of_responses_read)
    return (*klass->get_number_of_responses_read)(thread);
  return thread->number_of_responses_read;
}


void
ochusha_bbs_thread_set_number_of_responses_read(OchushaBBSThread *thread,
						int number)
{
  OchushaBBSThreadClass *klass;
  g_return_if_fail(OCHUSHA_IS_BBS_THREAD(thread));

  klass = OCHUSHA_BBS_THREAD_GET_CLASS(thread);
  if (klass->set_number_of_responses_read)
    (*klass->set_number_of_responses_read)(thread, number);
  else
    thread->number_of_responses_read = number;
}


int
ochusha_bbs_thread_get_flags(OchushaBBSThread *thread)
{
  OchushaBBSThreadClass *klass;
  g_return_val_if_fail(OCHUSHA_IS_BBS_THREAD(thread), 0);

  klass = OCHUSHA_BBS_THREAD_GET_CLASS(thread);
  if (klass->get_flags)
    return (*klass->get_flags)(thread);
  return thread->flags;
}


void
ochusha_bbs_thread_set_flags(OchushaBBSThread *thread, int flags)
{
  OchushaBBSThreadClass *klass;
  g_return_if_fail(OCHUSHA_IS_BBS_THREAD(thread));

  klass = OCHUSHA_BBS_THREAD_GET_CLASS(thread);
  if (klass->set_flags)
    (*klass->set_flags)(thread, flags);
  else
    thread->flags = flags;
}


OchushaAsyncBuffer *
ochusha_bbs_thread_get_responses_source(OchushaBBSThread *thread,
					OchushaNetworkBroker *broker,
					OchushaAsyncBuffer *buffer,
					OchushaNetworkBrokerCacheMode mode)
{
  OchushaBBSThreadClass *klass;

  g_return_val_if_fail(OCHUSHA_IS_BBS_THREAD(thread)
		       && OCHUSHA_IS_NETWORK_BROKER(broker), NULL);

  klass = OCHUSHA_BBS_THREAD_GET_CLASS(thread);
  g_return_val_if_fail(klass->get_responses_source != NULL, NULL);

  return (*klass->get_responses_source)(thread, broker, buffer, mode);
}


const char *
ochusha_bbs_thread_get_response_character_encoding(OchushaBBSThread *thread)
{
  OchushaBulletinBoard *board;
  g_return_val_if_fail(OCHUSHA_IS_BBS_THREAD(thread), NULL);

  board = ochusha_bbs_thread_get_board(thread);
  return ochusha_bulletin_board_get_response_character_encoding(OCHUSHA_BULLETIN_BOARD(board));
}


iconv_helper *
ochusha_bbs_thread_get_response_iconv_helper(OchushaBBSThread *thread)
{
  OchushaBulletinBoard *board;
  g_return_val_if_fail(OCHUSHA_IS_BBS_THREAD(thread), NULL);

  board = ochusha_bbs_thread_get_board(thread);
  return ochusha_bulletin_board_get_response_iconv_helper(OCHUSHA_BULLETIN_BOARD(board));
}


gboolean
ochusha_bbs_thread_parse_responses(OchushaBBSThread *thread,
				   OchushaAsyncBuffer *buffer,
				   int start, int number,
				   gboolean no_wait,
				   StartThreadCallback *start_thread_cb,
				   EachResponseCallback *each_response_cb,
				   BrokenResponseCallback *broken_response_cb,
				   EndThreadCallback *end_thread_cb,
				   StartParsingCallback *start_parsing_cb,
				   BeforeWaitCallback *before_wait_cb,
				   AfterWaitCallback *after_wait_cb,
				   EndParsingCallback *end_parsing_cb,
				   gpointer callback_data)
{
  OchushaBBSThreadClass *klass;

  g_return_val_if_fail(OCHUSHA_IS_BBS_THREAD(thread) && buffer != NULL, FALSE);

  klass = OCHUSHA_BBS_THREAD_GET_CLASS(thread);
  g_return_val_if_fail(klass->parse_responses != NULL, FALSE);

  return (*klass->parse_responses)(thread, buffer, start, number, no_wait,
				   start_thread_cb, each_response_cb,
				   broken_response_cb, end_thread_cb,
				   start_parsing_cb, before_wait_cb,
				   after_wait_cb, end_parsing_cb,
				   callback_data);
}


time_t
ochusha_bbs_thread_get_last_modified_utc(OchushaBBSThread *thread)
{
  OchushaBBSThreadClass *klass;

  g_return_val_if_fail(OCHUSHA_IS_BBS_THREAD(thread), 0);

  klass = OCHUSHA_BBS_THREAD_GET_CLASS(thread);

  if (klass->get_last_modified_utc == NULL)
    return 0;

  return (*klass->get_last_modified_utc)(thread);
}


gboolean
ochusha_bbs_thread_is_preview_supported(OchushaBBSThread *thread)
{
  OchushaBBSThreadClass *klass;

  g_return_val_if_fail(OCHUSHA_IS_BBS_THREAD(thread), FALSE);

  klass = OCHUSHA_BBS_THREAD_GET_CLASS(thread);
  return klass->preview_response != NULL;
}


gboolean
ochusha_bbs_thread_preview_response(OchushaBBSThread *thread,
				    const OchushaBBSResponse *response,
				    StartThreadCallback *start_cb,
				    EachResponseCallback *response_cb,
				    EndThreadCallback *end_cb,
				    gpointer callback_data)
{
  OchushaBBSThreadClass *klass;

  g_return_val_if_fail(OCHUSHA_IS_BBS_THREAD(thread) && response != NULL,
		       FALSE);
  klass = OCHUSHA_BBS_THREAD_GET_CLASS(thread);

  g_return_val_if_fail(klass->preview_response != NULL, FALSE);

  return (*klass->preview_response)(thread, response,
				    start_cb, response_cb, end_cb,
				    callback_data);
}


gboolean
ochusha_bbs_thread_is_post_supported(OchushaBBSThread *thread)
{
  OchushaBBSThreadClass *klass;

  g_return_val_if_fail(OCHUSHA_IS_BBS_THREAD(thread), FALSE);

  klass = OCHUSHA_BBS_THREAD_GET_CLASS(thread);
  if (klass->post_supported != NULL)
    return (*klass->post_supported)(thread);

  return FALSE;
}


gboolean
ochusha_bbs_thread_post_response(OchushaBBSThread *thread,
				 OchushaNetworkBroker *broker,
				 const OchushaBBSResponse *response)
{
  OchushaBBSThreadClass *klass;

  g_return_val_if_fail(OCHUSHA_IS_BBS_THREAD(thread)
		       && OCHUSHA_IS_NETWORK_BROKER(broker)
		       && response != NULL, FALSE);

  klass = OCHUSHA_BBS_THREAD_GET_CLASS(thread);

  g_return_val_if_fail(klass->post_response != NULL, FALSE);

  return (*klass->post_response)(thread, broker, response);
}


const char *
ochusha_bbs_thread_get_url(OchushaBBSThread *thread)
{
  OchushaBBSThreadClass *klass;
  g_return_val_if_fail(OCHUSHA_IS_BBS_THREAD(thread), NULL);

  klass = OCHUSHA_BBS_THREAD_GET_CLASS(thread);
  g_return_val_if_fail(klass->get_url != NULL, NULL);

  return (*klass->get_url)(thread);
}


char *
ochusha_bbs_thread_get_url_for_response(OchushaBBSThread *thread, int from,
					int to)
{
  OchushaBBSThreadClass *klass;
  g_return_val_if_fail(OCHUSHA_IS_BBS_THREAD(thread), NULL);

  klass = OCHUSHA_BBS_THREAD_GET_CLASS(thread);

  if (klass->get_url_for_response != NULL)
    return (*klass->get_url_for_response)(thread, from, to);
  else
    return NULL;
}


const char *
ochusha_bbs_thread_get_url_to_post_response(OchushaBBSThread *thread)
{
  OchushaBBSThreadClass *klass;
  g_return_val_if_fail(OCHUSHA_IS_BBS_THREAD(thread), NULL);

  klass = OCHUSHA_BBS_THREAD_GET_CLASS(thread);
  if (klass->get_url_to_post != NULL)
    return (*klass->get_url_to_post)(thread);

  g_return_val_if_fail(klass->get_url != NULL, NULL);
  return (*klass->get_url)(thread);
}


gboolean
ochusha_bbs_thread_check_url(OchushaBBSThread *thread, const char *url,
			     unsigned int *from_p, unsigned int *to_p)
{
  OchushaBBSThreadClass *klass;
  g_return_val_if_fail(OCHUSHA_IS_BBS_THREAD(thread), FALSE);

  klass = OCHUSHA_BBS_THREAD_GET_CLASS(thread);
  g_return_val_if_fail(klass->check_url != NULL, FALSE);

  return (*klass->check_url)(thread, url, from_p, to_p);
}


void
ochusha_bbs_thread_remove_cache(OchushaBBSThread *thread,
				OchushaConfig *config)
{
  OchushaBBSThreadClass *klass;
  g_return_if_fail(OCHUSHA_IS_BBS_THREAD(thread));

  klass = OCHUSHA_BBS_THREAD_GET_CLASS(thread);
  g_return_if_fail(klass->remove_cache != NULL);

  thread->number_of_responses_read = 0;
  thread->flags &= ~OCHUSHA_BBS_THREAD_STOPPED;

  return (*klass->remove_cache)(thread, config);
}


static void
ochusha_bbs_thread_read_threadlist_element(OchushaBBSThread *thread,
					   GHashTable *thread_attributes)
{
  thread->number_of_responses_read
    = ochusha_utils_get_attribute_int(thread_attributes,
				      "number_of_responses_read");
  if (thread->number_of_responses_read == 0)
    thread->flags &= ~OCHUSHA_BBS_THREAD_STOPPED;
  thread->flags = ochusha_utils_get_attribute_int(thread_attributes,
						  "flags");
}


/*
 * threadlist.xmlϢ
 */
#define OUTPUT_THREAD_ATTRIBUTE_INT(gzfile, thread, attribute)		\
  do									\
    {									\
      if ((thread)->attribute)						\
	{								\
	  gzprintf(gzfile,						\
		  "      <attribute name=\"" #attribute	"\">\n"		\
		  "        <int val=\"%d\"/>\n"				\
		  "      </attribute>\n", (thread)->attribute);		\
	}								\
    } while (0)


#define OUTPUT_THREAD_ATTRIBUTE_BOOLEAN(gzfile, thread, attribute)	\
  do									\
    {									\
      if ((thread)->attribute)						\
	{								\
	  gzprintf(gzfile,						\
		  "      <attribute name=\"" #attribute	"\">\n"		\
		  "        <boolean val=\"%s\"/>\n"			\
		  "      </attribute>\n",				\
		  (thread)->attribute ? "true" : "false");		\
	}								\
    } while (0)


#define OUTPUT_THREAD_ATTRIBUTE_STRING(gzfile, thread, attribute)	\
  do									\
    {									\
      if ((thread)->attribute)						\
	{								\
	  gchar *text = g_markup_escape_text((thread)->attribute, -1);	\
	  gzprintf(gzfile,						\
		  "      <attribute name=\"" #attribute	"\">\n"		\
		  "        <string>%s</string>\n"			\
		  "      </attribute>\n", text);			\
	  g_free(text);							\
	}								\
    } while (0)


static void
ochusha_bbs_thread_write_threadlist_element(OchushaBBSThread *thread,
					    gzFile *threadlist_xml)
{
  OUTPUT_THREAD_ATTRIBUTE_STRING(threadlist_xml, thread, id);
  OUTPUT_THREAD_ATTRIBUTE_STRING(threadlist_xml, thread, title);
  OUTPUT_THREAD_ATTRIBUTE_INT(threadlist_xml, thread,
			      number_of_responses_read);
  OUTPUT_THREAD_ATTRIBUTE_INT(threadlist_xml, thread, flags);
}
