/*
 * Copyright (c) 2003-2004 The Ochusha Project.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Id: xsmp.c,v 1.4 2004/01/15 02:18:05 fuyu Exp $
 */

#include "config.h"

#include "ochusha_private.h"
#include "ochusha.h"
#include "ochusha_async_buffer.h"

#include "ochusha_ui.h"
#include "xsmp.h"

#if ENABLE_XSMP

#include "session.h"

#include <glib.h>

#include <X11/SM/SMlib.h>
#include <X11/ICE/ICElib.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#define DEBUG_XSMP	0


static gboolean
ice_connection_poll_cb(GIOChannel *channel, GIOCondition condition,
		       OchushaApplication *application)
{
  if (application->ice_connection == NULL)
    return FALSE;

  if (condition & (G_IO_IN | G_IO_PRI))
    {
      IceProcessMessagesStatus status
	= IceProcessMessages((IceConn)application->ice_connection, NULL, NULL);

      if (status == IceProcessMessagesIOError)
	{
	  IceCloseConnection(application->ice_connection);
	  application->ice_connection = NULL;
	  return FALSE;
	}
    }

  return TRUE;
}


static void
ice_connection_watch_proc(IceConn ice_conn, IcePointer client_data,
			  Bool opening, IcePointer *watch_data)
{
  OchushaApplication *application = (OchushaApplication *)client_data;

#if DEBUG_XSMP
  FILE *log_file = application->log_file;
  fprintf(log_file, "ice_connection_watch_proc: client_data=%p, opening=%d\n",
	  client_data, opening);
  fprintf(log_file, "*watch_data=%p\n", *watch_data);
  fflush(log_file);
#endif

  application->ice_connection = ice_conn;

  if (application->ice_connection_poll_id != 0)
    {
      g_source_remove(application->ice_connection_poll_id);
      application->ice_connection_poll_id = 0;
    }

  if (opening)
    {
      int fd = IceConnectionNumber(ice_conn);
      if (fd >= 0)
	{
	  GIOChannel *channel = g_io_channel_unix_new(fd);
	  *watch_data = application;
	  application->ice_connection_poll_id
	    = g_io_add_watch_full(channel,
				  G_PRIORITY_HIGH,
				  G_IO_IN|G_IO_PRI|G_IO_ERR|G_IO_HUP|G_IO_NVAL,
				  (GIOFunc)ice_connection_poll_cb,
				  application, NULL);
	  g_io_channel_unref(channel);
	}
#if DEBUG_XSMP
      else
	{
	  fprintf(log_file, "ice_connection_watch_proc: fd=%d\n", fd);
	  fflush(log_file);
	}
#endif
    }
}


static void
save_yourself_cb(SmcConn smc_conn, SmPointer client_data, int save_type,
		 Bool shutdown, int interact_style, Bool fast)
{
  OchushaApplication *application = (OchushaApplication *)client_data;
  SmProp prop = { NULL, NULL, 0, NULL };
  SmProp *props = { &prop };
  SmPropValue *prop_values = g_new0(SmPropValue, application->argc + 1);
  int i;

  /* Initialize common parts of prop_values */
  for (i = 0; i < application->argc; i++)
    {
      prop_values[i].length = strlen(application->argv[i]);
      prop_values[i].value = application->argv[i];
    }
  prop.type = (char *)SmLISTofARRAY8;
  prop.num_vals = i + 1;
  prop.vals = prop_values;


  /* set SmCloneCommand */
  prop_values[i].value = (char *)"--session";
  prop_values[i].length = 9;
  prop.name = (char *)SmCloneCommand;
  SmcSetProperties(smc_conn, 1, &props);


  /* set SmRestartCommand */
  prop_values[i].value = g_strconcat("--restart=", application->sm_client_id,
				     NULL);
  prop_values[i].length = strlen(prop_values[i].value);
  prop.name = (char *)SmRestartCommand;
  SmcSetProperties(smc_conn, 1, &props);
  g_free(prop_values[i].value);


  /* set SmDiscardCommand */
  prop_values[i].value = g_strconcat("--discard=", application->sm_client_id,
				     NULL);
  prop_values[i].length = strlen(prop_values[i].value);
  prop.name = (char *)SmDiscardCommand;
  SmcSetProperties(smc_conn, 1, &props);
  g_free(prop_values[i].value);


  /* set SmProgram */
  prop.name = (char *)SmProgram;
  prop.type = (char *)SmARRAY8;
  prop.num_vals = 1;
  prop.vals = prop_values;
  SmcSetProperties(smc_conn, 1, &props);


  /* set SmUserID */
  prop.name = (char *)SmUserID;
  prop.type = (char *)SmARRAY8;
  prop.num_vals = 1;
  prop.vals = prop_values;
  prop_values[0].value = (char *)g_get_user_name();
  prop_values[0].length = strlen(prop_values[0].value);
  SmcSetProperties(smc_conn, 1, &props);


  /* set SmCurrentDirectory */
  prop.name = (char *)SmCurrentDirectory;
  prop.type = (char *)SmARRAY8;
  prop.num_vals = 1;
  prop.vals = prop_values;
  prop_values[0].value = g_get_current_dir();
  prop_values[0].length = strlen(prop_values[0].value);
  SmcSetProperties(smc_conn, 1, &props);
  g_free(prop_values[0].value);

  g_free(prop_values);

#if DEBUG_XSMP
  fprintf(application->log_file,
	  "save_yourself_cb: pre suspend_all()\n");
  fflush(application->log_file);
#endif

  ochusha_async_buffer_suspend_all();

#if DEBUG_XSMP
  fprintf(application->log_file,
	  "save_yourself_cb: post suspend_all()\n");
  fflush(application->log_file);
#endif

  ochusha_session_save_states(application);

#if DEBUG_XSMP
  fprintf(application->log_file,
	  "save_yourself_cb: pre resume_all()\n");
  fflush(application->log_file);
#endif
  ochusha_async_buffer_resume_all();

#if DEBUG_XSMP
  fprintf(application->log_file,
	  "save_yourself_cb: post resume_all()\n");
  fflush(application->log_file);
#endif

  SmcSaveYourselfDone(smc_conn, True);

#if DEBUG_XSMP
  fprintf(application->log_file,
	  "save_yourself_cb: type=%d, shutdown=%d, style=%d, fast=%d\n",
	  save_type, shutdown, interact_style, fast);
  fflush(application->log_file);
#endif
}


static void
die_cb(SmcConn smc_conn, SmPointer client_data)
{
  OchushaApplication *application = (OchushaApplication *)client_data;
#if DEBUG_XSMP
  fprintf(application->log_file, "die_cb\n");
  fflush(application->log_file);
#endif
  SmcCloseConnection(application->smc_connection, 0, NULL);
  application->smc_connection = NULL;

  ochusha_async_buffer_suspend_all();
  ochusha_async_buffer_terminate_all();

  gtk_main_quit();
}


static void
save_complete_cb(SmcConn smc_conn, SmPointer client_data)
{
#if DEBUG_XSMP
  OchushaApplication *application = (OchushaApplication *)client_data;
  fprintf(application->log_file, "save_complete_cb\n");
  fflush(application->log_file);
#endif
}


static void
shutdown_cancelled_cb(SmcConn smc_conn, SmPointer client_data)
{
#if DEBUG_XSMP
  OchushaApplication *application = (OchushaApplication *)client_data;
  fprintf(application->log_file, "shutdown_cancelled_cb\n");
  fflush(application->log_file);
#endif
}


void
initialize_xsmp_handlers(OchushaApplication *application)
{
  char *sm_client_id;
  char error_string[4096];
  SmcCallbacks smc_callbacks =
    {
      {
	save_yourself_cb,
	application
      },
      {
	die_cb,
	application
      },
      {
	save_complete_cb,
	application
      },
      {
	shutdown_cancelled_cb,
	application
      }
    };

  IceInitThreads();	/* ᥤ󥹥åɤΤߤSMlibȤΤ¿ʬ */

  if (IceAddConnectionWatch(ice_connection_watch_proc, application) == 0)
    {
#if DEBUG_XSMP
      fprintf(application->log_file,
	      "Couldn't register watch proc for ICE connection.\n");
      fflush(application->log_file);
#endif
      return;
    }

#if DEBUG_XSMP
  if (application->sm_client_id != NULL)
    fprintf(application->log_file, "previous sm_client_id=\"%s\"\n",
	    application->sm_client_id);
  else
    fprintf(application->log_file, "starting new session\n");
  fflush(application->log_file);
#endif

  application->smc_connection
    = SmcOpenConnection(NULL, NULL, 1, 0,
			(SmcSaveYourselfProcMask
			 | SmcDieProcMask
			 | SmcSaveCompleteProcMask
			 | SmcShutdownCancelledProcMask),
			&smc_callbacks,
			application->sm_client_id,
			&sm_client_id,
			4096, error_string);

  application->previous_sm_client_id = application->sm_client_id;

  if (application->smc_connection == NULL)
    {
#if DEBUG_XSMP
      fprintf(application->log_file, "Couldn't open SMC connection.\n");
      fflush(application->log_file);
#endif
      application->sm_client_id = NULL;
      return;
    }

#if TRACE_MEMORY_USAGE
  application->sm_client_id = G_STRDUP(sm_client_id);
  free(sm_client_id);
#else
  application->sm_client_id = sm_client_id;
#endif

#if DEBUG_XSMP
  fprintf(application->log_file, "sm_client_id=\"%s\"\n",
	  application->sm_client_id);
  fflush(application->log_file);
#endif
}


void
finalize_xsmp_handlers(OchushaApplication *application)
{
  if (application->smc_connection != NULL)
    {
      SmcCloseConnection(application->smc_connection, 0, NULL);
      application->smc_connection = NULL;
    }
}


#else
void
initialize_xsmp_handlers(OchushaApplication *application)
{
  /* Do nothing, XSMP disabled */
  return;
}


void
finalize_xsmp_handlers(OchushaApplication *application)
{
  /* Do nothing, XSMP disabled */
  return;
}
#endif	/* ENABLE_XSMP */
