/*
 * Copyright (c) 2003 The Ochusha Project.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Id: ochusha_network_broker.c,v 1.35 2003/12/15 17:19:22 fuyu Exp $
 */

#include "config.h"

#include "ochusha_private.h"
#include "ochusha.h"
#include "ochusha_network_broker.h"

#include "gziputils.h"
#include "worker.h"

#include "marshal.h"

#include <ghttp.h>
#include <glib.h>

#include <errno.h>
#include <fcntl.h>

#include <pthread.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <sys/types.h>
#include <sys/mman.h>

#include <unistd.h>


#ifndef MAP_NOCORE
# define MAP_NOCORE	0
#endif


#define DISABLE_PERSISTENT_CONNECTION		0
#define ENABLE_GZIPPED_TRANSFER			0
#define ENABLE_GZIPPED_DIFFERENCIAL_READ	0

#define CACHE_COMPARE_SIZE	128	/* Ĥѹǽˤ٤ */
#define HEADER_BUFFER_LENGTH	256
#define LOG_MESSAGE_SIZE	4096

#define POLLING_INTERVAL_MILLIS	100


#define DEBUG_MMAP		0


static void ochusha_network_broker_class_init(OchushaNetworkBrokerClass *klass);
static void ochusha_network_broker_init(OchushaNetworkBroker *broker);
static void ochusha_network_broker_finalize(GObject *object);


GType
ochusha_network_broker_get_type(void)
{
  static GType broker_type = 0;

  if (broker_type == 0)
    {
      static const GTypeInfo broker_info =
	{
	  sizeof(OchushaNetworkBrokerClass),
	  NULL,	/* base_init */
	  NULL,	/* base_finalize */
	  (GClassInitFunc)ochusha_network_broker_class_init,
	  NULL,	/* class_finalize */
	  NULL,	/* class_data */
	  sizeof(OchushaNetworkBroker),
	  0,	/* n_preallocs */
	  (GInstanceInitFunc)ochusha_network_broker_init,
	};

      broker_type = g_type_register_static(G_TYPE_OBJECT,
					   "OchushaNetworkBroker",
					   &broker_info, 0);
    }

  return broker_type;
}


enum {
  ACCESS_STARTED_SIGNAL,
  ACCESS_PROGRESSED_SIGNAL,
  ACCESS_COMPLETED_SIGNAL,
  ACCESS_TERMINATED_SIGNAL,
  ACCESS_FAILED_SIGNAL,
  OUTPUT_LOG_SIGNAL,
  LAST_SIGNAL
};


static GObjectClass *parent_class = NULL;
static gint broker_signals[LAST_SIGNAL] = { 0, 0, 0, 0, 0, 0 };
static GQuark broker_buffer_status_id;
static GQuark broker_job_args_id;
static GQuark worker_sync_object_id;


static void
ochusha_network_broker_class_init(OchushaNetworkBrokerClass *klass)
{
  GObjectClass *o_class = G_OBJECT_CLASS(klass);

  parent_class = g_type_class_peek_parent(klass);

  o_class->finalize = ochusha_network_broker_finalize;

  broker_signals[ACCESS_STARTED_SIGNAL] =
    g_signal_new("access_started",
		 G_TYPE_FROM_CLASS(klass),
		 G_SIGNAL_RUN_LAST,
		 G_STRUCT_OFFSET(OchushaNetworkBrokerClass, access_started),
		 NULL, NULL,
		 libochusha_marshal_BOOLEAN__OBJECT,
		 G_TYPE_BOOLEAN, 1,
		 OCHUSHA_TYPE_ASYNC_BUFFER);
  broker_signals[ACCESS_PROGRESSED_SIGNAL] =
    g_signal_new("access_progressed",
		 G_TYPE_FROM_CLASS(klass),
		 G_SIGNAL_RUN_LAST,
		 G_STRUCT_OFFSET(OchushaNetworkBrokerClass, access_progressed),
		 NULL, NULL,
		 libochusha_marshal_BOOLEAN__OBJECT_INT_INT,
		 G_TYPE_BOOLEAN, 3,
		 OCHUSHA_TYPE_ASYNC_BUFFER,
		 G_TYPE_INT,
		 G_TYPE_INT);
  broker_signals[ACCESS_COMPLETED_SIGNAL] =
    g_signal_new("access_completed",
		 G_TYPE_FROM_CLASS(klass),
		 G_SIGNAL_RUN_LAST,
		 G_STRUCT_OFFSET(OchushaNetworkBrokerClass, access_completed),
		 NULL, NULL,
		 libochusha_marshal_BOOLEAN__OBJECT,
		 G_TYPE_BOOLEAN, 1,
		 OCHUSHA_TYPE_ASYNC_BUFFER);
  broker_signals[ACCESS_TERMINATED_SIGNAL] =
    g_signal_new("access_terminated",
		 G_TYPE_FROM_CLASS(klass),
		 G_SIGNAL_RUN_LAST,
		 G_STRUCT_OFFSET(OchushaNetworkBrokerClass, access_terminated),
		 NULL, NULL,
		 libochusha_marshal_BOOLEAN__OBJECT,
		 G_TYPE_BOOLEAN, 1,
		 OCHUSHA_TYPE_ASYNC_BUFFER);
  /*
   * XXX: failˤOchushaAsyncBufferʤȤ⤢롣
   *      OBJECTΥʥΰNULLݥ󥿤ϤȡGLibٹ
   *      Ф⤷ʤΤǡٹ𤬽Ф褦ʤ顢OBJECTǤʤ
   *      POINTERѹ٤
   */
  broker_signals[ACCESS_FAILED_SIGNAL] =
    g_signal_new("access_failed",
		 G_TYPE_FROM_CLASS(klass),
		 G_SIGNAL_RUN_LAST,
		 G_STRUCT_OFFSET(OchushaNetworkBrokerClass, access_failed),
		 NULL, NULL,
		 libochusha_marshal_BOOLEAN__OBJECT_INT_STRING,
		 G_TYPE_BOOLEAN, 3,
		 OCHUSHA_TYPE_ASYNC_BUFFER,
		 G_TYPE_INT,
		 G_TYPE_STRING);
  broker_signals[OUTPUT_LOG_SIGNAL] =
    g_signal_new("output_log",
		 G_TYPE_FROM_CLASS(klass),
		 G_SIGNAL_RUN_LAST,
		 G_STRUCT_OFFSET(OchushaNetworkBrokerClass, output_log),
		 NULL, NULL,
		 libochusha_marshal_VOID__STRING,
		 G_TYPE_NONE, 1,
		 G_TYPE_STRING);

  broker_buffer_status_id
    = g_quark_from_static_string("OchushaNetworkBroker::BufferStatus");
  broker_job_args_id
    = g_quark_from_static_string("OchushaNetworkBroker::JobArgs");
  worker_sync_object_id
    = g_quark_from_static_string("OchushaNetworkBroker::SyncObject");

  klass->access_started = NULL;
  klass->access_progressed = NULL;
  klass->access_completed = NULL;
  klass->access_terminated = NULL;
  klass->access_failed = NULL;
  klass->output_log = NULL;
}


static void
ochusha_network_broker_init(OchushaNetworkBroker *broker)
{
  broker->config = NULL;
}


static void
ochusha_network_broker_finalize(GObject *object)
{
  if (G_OBJECT_CLASS(parent_class)->finalize)
    (*G_OBJECT_CLASS(parent_class)->finalize)(object);
}


OchushaNetworkBroker *
ochusha_network_broker_new(OchushaConfig *config)
{
  OchushaNetworkBroker *broker
    = OCHUSHA_NETWORK_BROKER(g_object_new(OCHUSHA_TYPE_NETWORK_BROKER, NULL));
  broker->config = config;

  return broker;
}


void
ochusha_network_broker_output_log(OchushaNetworkBroker *broker,
				  const gchar *message)
{
  g_signal_emit(G_OBJECT(broker),
		broker_signals[OUTPUT_LOG_SIGNAL],
		0,
		message);
}


static void
munmap_when_finished(OchushaAsyncBuffer *buffer)
{
  munmap((void *)buffer->buffer, buffer->length);
#if DEBUG_MMAP
  fprintf(stderr, "mmapped buffer(at %p) is unmapped.\n",
	  buffer->buffer);
#endif
}


static void
ochusha_network_broker_buffer_status_free(OchushaNetworkBrokerBufferStatus *status)
{
  if (status->last_modified != NULL)
    G_FREE(status->last_modified);
  if (status->date != NULL)
    G_FREE(status->date);
  G_FREE(status);
}


static OchushaAsyncBuffer *
get_mmapped_buffer(int fd)
{
  char *buf;
  off_t len;
  OchushaAsyncBuffer *buffer;
  OchushaNetworkBrokerBufferStatus *status;

  if (fd < 0)
    return NULL;

  status = G_NEW0(OchushaNetworkBrokerBufferStatus, 1);
  status->state = OCHUSHA_NETWORK_BROKER_BUFFER_STATE_CACHE_HIT;

  len = lseek(fd, 0, SEEK_END);
  buf = mmap(NULL, len, PROT_READ, MAP_NOCORE | MAP_PRIVATE, fd, 0);
  if (buf == MAP_FAILED)
    {
      fprintf(stderr, "mmap failed due to: %s (%d)\n", strerror(errno), errno);
      G_FREE(status);
      return NULL;
    }
  close(fd);

#if DEBUG_MMAP
  fprintf(stderr, "allocating mmapped buffer at %p\n", buf);
#endif

  buffer = ochusha_async_buffer_new(buf, len, munmap_when_finished);
  g_object_set_qdata_full(G_OBJECT(buffer), broker_buffer_status_id, status,
		(GDestroyNotify)ochusha_network_broker_buffer_status_free);
  if (!ochusha_async_buffer_fix(buffer,
			"ochusha_network_broker.c: get_mmapped_buffer()"))
    {
#if DEBUG_ASYNC_BUFFER_MOST
      fprintf(stderr, "buffer has been terminated.\n");
#endif
      /* ȤϤɤ褦ʤ */
    }
#if DEBUG_NETWORK_MOST
  fprintf(stderr, "get_mmapped_buffer(): buffer=%p\n", buffer);
#endif

  return buffer;
}


static int
read_cache_to_buffer(const OchushaConfig *config, const char *url,
		     OchushaAsyncBuffer *buffer)
{
  int len;
  int fd = ochusha_config_cache_open_file(config, url, O_RDONLY);

  if (fd < 0)
    return 0;	/* åʤξ */

  /* å夢ꡣ*/
  len = lseek(fd, 0, SEEK_END);
  lseek(fd, 0, SEEK_SET);
  if (ochusha_async_buffer_resize(buffer, len,
			"ochusha_network_broker.c: read_cache_to_buffer()"))
    {
      len = read(fd, (void *)buffer->buffer, len);
      if (!ochusha_async_buffer_update_length(buffer, len,
			"ochusha_network_broker.c: read_cache_to_buffer()"))
	{
#if DEBUG_ASYNC_BUFFER
	  fprintf(stderr, "read_cache_to_buffer(): buffer may have been terminated.\n");
#endif
	  /* ξϤɤ褦ʤ */
	}
    }
  else
    {
#if DEBUG_ASYNC_BUFFER
      fprintf(stderr, "read_cache_to_buffer(): buffer may have been terminated.\n");
#endif
      len = 0;	/* Out of memory? */
    }
  close(fd);

  return len;
}


static void
write_buffer_to_cache(OchushaNetworkBroker *broker, const char *url,
		      OchushaAsyncBuffer *buffer)
{
  gchar message[LOG_MESSAGE_SIZE];
  int fd = ochusha_config_cache_open_file(broker->config, url,
					  O_WRONLY | O_TRUNC | O_CREAT);
  if (fd >= 0)
    {
      ssize_t len = write(fd, (void *)buffer->buffer, buffer->length);
      close(fd);
      if (len != buffer->length)
	{
	  ochusha_config_cache_unlink_file(broker->config, url);
	  snprintf(message, LOG_MESSAGE_SIZE,
		   _("Couldn't update cache file for %s: %s (%d)\n"),
		   url, strerror(errno), errno);
	  ochusha_network_broker_output_log(broker, message);
#if DEBUG_NETWORK_MOST
	  fprintf(stderr, "Couldn't update cache file for %s: %s (%d)\n",
		  url, strerror(errno), errno);
#endif
	}
    }
  else
    {
      snprintf(message, LOG_MESSAGE_SIZE,
	       _("Couldn't open cache file for %s: %s (%d)\n"),
	       url, strerror(errno), errno);
      ochusha_network_broker_output_log(broker, message);
#if DEBUG_NETWORK_MOST
      fprintf(stderr, "Couldn't open cache file for %s: %s (%d)\n",
	      url, strerror(errno), errno);
#endif
    }
}


/*
 * ºݤnetwork˥뵡ǽ
 */


static gboolean
setup_common_request_headers(OchushaNetworkBroker *broker,
			     ghttp_request *request,
			     gboolean posting)
{
  gchar message[LOG_MESSAGE_SIZE];
  gboolean result = TRUE;
  const OchushaConfig *config = broker->config;

  if ((config->enable_proxy
       || (config->enable_proxy_only_for_posting && posting))
      && config->proxy_url != NULL)
    {
      int result = ghttp_set_proxy(request, config->proxy_url);
      if (result == 0)
	{
	  if (config->enable_proxy_auth
	      && config->proxy_user != NULL && config->proxy_password != NULL)
	    {
	      result = ghttp_set_proxy_authinfo(request, config->proxy_user,
						config->proxy_password);
	      if (result != 0)
		{
		  snprintf(message, LOG_MESSAGE_SIZE,
			   _("Invalid proxy auth info: user=\"%s\", password=\"%s\"\n"),
			   config->proxy_user, config->proxy_password);
		  ochusha_network_broker_output_log(broker, message);
#if DEBUG_NETWORK_MOST
		  fprintf(stderr, "Invalid proxy auth info.\n");
#endif
		  result = FALSE;
		}
	    }
	}
      else
	{
	  snprintf(message, LOG_MESSAGE_SIZE,
		   _("Invalid proxy URL: \"%s\"\n"),
		   config->proxy_url);
	  ochusha_network_broker_output_log(broker, message);
#if DEBUG_NETWORK_MOST
	  fprintf(stderr, "Invalid proxy URL: %s\n", config->proxy_url);
#endif
	  result = FALSE;
	}
    }

  ghttp_set_header(request, http_hdr_User_Agent, OCHUSHA_USER_AGENT);

  return result;
}


typedef enum
{
  UNKNOWN_ENCODING = -1,
  IDENTITY_ENCODING,
  GZIP_ENCODING,
  DEFLATE_ENCODING,
} ContentEncodingType;


static ContentEncodingType
get_content_encoding_type(ghttp_request *request)
{
  const char *encoding = ghttp_get_header(request, http_hdr_Content_Encoding);

  if (encoding != NULL)
    {
      if (strcmp(encoding, "gzip") == 0)
	return GZIP_ENCODING;
#if DEBUG_NETWORK_MOST
      fprintf(stderr, "Content-Encoding: %s\n", encoding);
#endif
      if (strcmp(encoding, "deflate") == 0)
	return DEFLATE_ENCODING;
      if (strcmp(encoding, "identity") == 0)
	return IDENTITY_ENCODING;
      return UNKNOWN_ENCODING;
    }

  return IDENTITY_ENCODING;
}


/*
 * OchushaAsyncBufferbroker_job_args_idqdataȤƤäĤ롣
 */
typedef struct _NetworkBrokerJobArgs
{
  OchushaNetworkBroker *broker;

  char *url;
  char *if_modified_since;

  ghttp_request *request;
} NetworkBrokerJobArgs;


/*
 * OchushaAsyncBufferworker_sync_object_idqdataȤƤäĤ롣
 */
#define OCHUSHA_TYPE_WORKER_SYNC_OBJECT			(worker_sync_object_get_type())
#define OCHUSHA_WORKER_SYNC_OBJECT(obj)			(G_TYPE_CHECK_INSTANCE_CAST((obj), OCHUSHA_TYPE_WORKER_SYNC_OBJECT, WorkerSyncObject))
#define OCHUSHA_WORKER_SYNC_OBJECT_CLASS(klass)		(G_TYPE_CHECK_CLASS_CAST((klass), OCHUSHA_TYPE_WORKER_SYNC_OBJECT, WorkerSyncObjectClass))
#define OCHUSHA_IS_WORKER_SYNC_OBJECT(obj)		(G_TYPE_CHECK_INSTANCE_TYPE((obj), OCHUSHA_TYPE_WORKER_SYNC_OBJECT))
#define OCHUSHA_IS_WORKER_SYNC_OBJECT_CLASS(klass)	(G_TYPE_CHECK_CLASS_TYPE((klass), OCHUSHA_TYPE_WORKER_SYNC_OBJECT))
#define OCHUSHA_WORKER_SYNC_OBJECT_GET_CLASS(obj)	(G_TYPE_INSTANCE_GET_CLASS((obj), OCHUSHA_TYPE_WORKER_SYNC_OBJECT, WorkerSyncObjectClass))


typedef struct _WorkerSyncObject WorkerSyncObject;
typedef struct _WorkerSyncObjectClass WorkerSyncObjectClass;

struct _WorkerSyncObject
{
  GObject parent_object;

  pthread_mutex_t lock;
  pthread_cond_t cond;
  guint timeout_id;
  guint poll_id;
  volatile gboolean read_ok;
  volatile gboolean finished;
  volatile gboolean interrupted;
  volatile gboolean lock_and_cond_initialized;
};


struct _WorkerSyncObjectClass
{
  GObjectClass parent_class;
};


static void worker_sync_object_class_init(WorkerSyncObjectClass *klass);
static void worker_sync_object_init(WorkerSyncObject *sync_object);
static void worker_sync_object_finalize(GObject *object);


static GType
worker_sync_object_get_type(void)
{
  static GType sync_object_type = 0;

  if (sync_object_type == 0)
    {
      static const GTypeInfo sync_object_info =
	{
	  sizeof(WorkerSyncObjectClass),
	  NULL,	/* base_init */
	  NULL,	/* base_finalize */
	  (GClassInitFunc)worker_sync_object_class_init,
	  NULL,	/* class_finalize */
	  NULL,	/* class_data */
	  sizeof(WorkerSyncObject),
	  0,	/* n_preallocs */
	  (GInstanceInitFunc)worker_sync_object_init,
	};

      sync_object_type = g_type_register_static(G_TYPE_OBJECT,
						"WorkerSyncObject",
						&sync_object_info, 0);
    }

  return sync_object_type;
}


GObjectClass *sync_object_parent_class = NULL;


static void
worker_sync_object_class_init(WorkerSyncObjectClass *klass)
{
  GObjectClass *o_class = G_OBJECT_CLASS(klass);
  sync_object_parent_class = g_type_class_peek_parent(klass);
  o_class->finalize = worker_sync_object_finalize;
}


static void
worker_sync_object_init(WorkerSyncObject *sync_object)
{
  if (pthread_mutex_init(&sync_object->lock, NULL) != 0)
    {
      fprintf(stderr, "Couldn't init a mutex.\n");
      abort();
    }

  if (pthread_cond_init(&sync_object->cond, NULL) != 0)
    {
      fprintf(stderr, "Couldn't init a condition.\n");
      abort();
    }
  sync_object->lock_and_cond_initialized = TRUE;
}


static void
worker_sync_object_finalize(GObject *object)
{
  WorkerSyncObject *sync_object = OCHUSHA_WORKER_SYNC_OBJECT(object);

  if (sync_object->lock_and_cond_initialized)
    {
      if (pthread_mutex_destroy(&sync_object->lock) != 0)
	{
	  fprintf(stderr, "Couldn't destroy a mutex: errno=%d\n", errno);
	  abort();
	}
      if (pthread_cond_destroy(&sync_object->cond) != 0)
	{
	  fprintf(stderr, "Couldn't destroy a condition: errno=%d\n", errno);
	  abort();
	}
      sync_object->lock_and_cond_initialized = FALSE;
    }

  if (sync_object_parent_class->finalize)
    (*sync_object_parent_class->finalize)(object);
}


static WorkerSyncObject *
worker_sync_object_new(void)
{
  return OCHUSHA_WORKER_SYNC_OBJECT(g_object_new(OCHUSHA_TYPE_WORKER_SYNC_OBJECT, NULL));
}


#define LOCK_WORKER_SYNC_OBJECT(sync_object)			\
  do								\
    {								\
      if (pthread_mutex_lock(&(sync_object)->lock) != 0)	\
	{							\
	  fprintf(stderr, "Couldn't lock a mutex.\n");		\
	  abort();						\
	}							\
    }								\
  while (0)

#define UNLOCK_WORKER_SYNC_OBJECT(sync_object)			\
  do								\
    {								\
      if (pthread_mutex_unlock(&(sync_object)->lock) != 0)	\
	{							\
	  fprintf(stderr, "Couldn't unlock a mutex.\n");	\
	  abort();						\
	}							\
    }								\
  while (0)


static gboolean
timeout_cb(WorkerSyncObject *sync_object)
{
  gboolean finished;
#if DEBUG_NETWORK_MOST
  fprintf(stderr, "entering: timeout_cb(%p)\n", sync_object);
#endif
  g_return_val_if_fail(OCHUSHA_IS_WORKER_SYNC_OBJECT(sync_object), FALSE);

  LOCK_WORKER_SYNC_OBJECT(sync_object);
  finished = sync_object->finished;
  if (pthread_cond_signal(&sync_object->cond) != 0)
    {
      fprintf(stderr, "Couldn't signal a condition.\n");
      abort();
    }
  UNLOCK_WORKER_SYNC_OBJECT(sync_object);

#if DEBUG_NETWORK_MOST
  fprintf(stderr, "leaving: timeout_cb(%p)\n", sync_object);
#endif

  if (!finished)
    return TRUE;

  g_object_unref(G_OBJECT(sync_object));

  return FALSE;
}


static gboolean
poll_cb(GIOChannel *channel, GIOCondition condition,
	WorkerSyncObject *sync_object)
{
#if DEBUG_NETWORK_MOST
  fprintf(stderr, "entering: poll_cb(%p)\n", sync_object);
#endif
  g_return_val_if_fail(OCHUSHA_IS_WORKER_SYNC_OBJECT(sync_object), FALSE);

  LOCK_WORKER_SYNC_OBJECT(sync_object);
  if (condition & (G_IO_IN | G_IO_PRI))
    sync_object->read_ok = TRUE;

  if (pthread_cond_signal(&sync_object->cond) != 0)
    {
      fprintf(stderr, "Couldn't signal a condition.\n");
      abort();
    }
  sync_object->poll_id = 0;
  UNLOCK_WORKER_SYNC_OBJECT(sync_object);

#if DEBUG_NETWORK_MOST
  fprintf(stderr, "leaving: poll_cb(%p)\n", sync_object);
#endif
  g_object_unref(G_OBJECT(sync_object));

  return FALSE;
}


static void
register_polling_function_for_read(WorkerSyncObject *sync_object, int fd)
{
  LOCK_WORKER_SYNC_OBJECT(sync_object);
  g_object_ref(G_OBJECT(sync_object));
  if (sync_object->poll_id == 0)
    {
      GIOChannel *channel = g_io_channel_unix_new(fd);
      sync_object->poll_id
	= g_io_add_watch_full(channel,
			      G_PRIORITY_DEFAULT,
			      G_IO_IN|G_IO_PRI|G_IO_ERR|G_IO_HUP|G_IO_NVAL,
			      (GIOFunc)poll_cb, sync_object, NULL);
      g_io_channel_unref(channel);
    }
  sync_object->read_ok = FALSE;
  UNLOCK_WORKER_SYNC_OBJECT(sync_object);
}


static gboolean
http_read_from_url(OchushaNetworkBroker *broker, OchushaAsyncBuffer *buffer)
{
  gboolean signal_result;
  NetworkBrokerJobArgs *args = g_object_get_qdata(G_OBJECT(buffer),
					      broker_job_args_id);
  WorkerSyncObject *sync_object = g_object_get_qdata(G_OBJECT(buffer),
						     worker_sync_object_id);
  OchushaNetworkBrokerBufferStatus *status
    = g_object_get_qdata(G_OBJECT(buffer), broker_buffer_status_id);
  const char *url = args->url;
  const char *if_modified_since = args->if_modified_since;
  const char *error_message = NULL;
  gchar message[LOG_MESSAGE_SIZE];

  int sock_fd = -1;
  int status_code = 0;
  ghttp_request *request = NULL;
  ContentEncodingType content_encoding = UNKNOWN_ENCODING;
  GzipBuffer *gzip_buffer = NULL;
  ghttp_status state = ghttp_not_done;
  ghttp_current_status current_status;

  request = ghttp_request_new();
  args->request = request;	/* 塹Τˡġ */

  ghttp_set_uri(request, (char *)url);
  ghttp_set_type(request, ghttp_type_get);
#if ENABLE_GZIPPED_TRANSFER
  ghttp_set_header(request, http_hdr_Accept_Encoding, "deflate, gzip");
#endif

#if DISABLE_PERSISTENT_CONNECTION
  ghttp_set_header(request, http_hdr_Connection, "close");
#endif

  if (if_modified_since != NULL
      && ochusha_config_cache_file_exist(broker->config, url))
    {
      ghttp_set_header(request, http_hdr_If_Modified_Since, if_modified_since);
#if DEBUG_NETWORK_MOST
      fprintf(stderr, "If-Modified-Since: %s\n", if_modified_since);
#endif
    }
  else
    if_modified_since = NULL;

  if (!setup_common_request_headers(broker, request, FALSE))
    {
      g_signal_emit(G_OBJECT(broker),
		    broker_signals[ACCESS_FAILED_SIGNAL],
		    0,
		    buffer,
		    OCHUSHA_NETWORK_BROKER_FAILURE_REASON_INVALID_PROXY,
		    _("Proxy setting may be wrong."),
		    &signal_result);

      LOCK_WORKER_SYNC_OBJECT(sync_object);
      sync_object->finished = TRUE;
      UNLOCK_WORKER_SYNC_OBJECT(sync_object);

      ghttp_request_destroy(request);
      args->request = NULL;

      return FALSE;
    }    

  g_signal_emit(G_OBJECT(broker),
		broker_signals[ACCESS_STARTED_SIGNAL],
		0,
		buffer,
		&signal_result);

  snprintf(message, LOG_MESSAGE_SIZE, _("Starting GET request: %s\n"), url);
  ochusha_network_broker_output_log(broker, message);

  ghttp_set_sync(request, ghttp_async);
  ghttp_prepare(request);

  current_status = ghttp_get_status(request);

  sock_fd = -1;
  while (TRUE)
    {
      int body_len;
      char *body;
      ghttp_proc prev_proc;
      gboolean read_ok;

      prev_proc = current_status.proc;

      LOCK_WORKER_SYNC_OBJECT(sync_object);
      if (sync_object->interrupted)
	{
	  sync_object->interrupted = FALSE;
	  UNLOCK_WORKER_SYNC_OBJECT(sync_object);
	  read_ok = FALSE;
	}
      else if ((read_ok = sync_object->read_ok)
	       || (state == ghttp_not_done && sock_fd == -1))
	{
	  sync_object->read_ok = FALSE;
	  UNLOCK_WORKER_SYNC_OBJECT(sync_object);
	  state = ghttp_process(request);
	  current_status = ghttp_get_status(request);
	}
      else
	UNLOCK_WORKER_SYNC_OBJECT(sync_object);

#if DEBUG_NETWORK_MOST
      if (state == ghttp_error)
	{
	  fprintf(stderr, "\n* ghttp_process() returns ghttp_error\n");
	  fprintf(stderr, "sock_fd = %d\n", ghttp_get_socket(request));
	  fprintf(stderr, "error: \"%s\"\n", ghttp_get_error(request));
	  fprintf(stderr, "current_status.proc = %d\n", current_status.proc);
	  fprintf(stderr, "current_status.bytes_read = %d\n", current_status.bytes_read);
	  fprintf(stderr, "current_status.bytes_total = %d\n", current_status.bytes_total);
	}
      else
	{
	  fprintf(stderr, "prev_proc = %d\n", prev_proc);
	  fprintf(stderr, "current_status.proc = %d\n", current_status.proc);
	}
#endif

      if (current_status.proc == ghttp_proc_response
	  || (sock_fd > 0 && current_status.proc == ghttp_proc_none))
	{
	  status_code = ghttp_status_code(request);
	  status->status_code = status_code;

	  if (status_code != 200)
	    {
	      if (status_code == 304)
		{
		  const char *tmp_header
		    = ghttp_get_header(request, http_hdr_Date);
		  if (tmp_header != NULL)
		    status->date = G_STRDUP(tmp_header);
		  else
		    status->date = NULL;

		  LOCK_WORKER_SYNC_OBJECT(sync_object);
		  sync_object->finished = TRUE;
		  UNLOCK_WORKER_SYNC_OBJECT(sync_object);

		  ghttp_request_destroy(request);
		  args->request = NULL;

		  snprintf(message, LOG_MESSAGE_SIZE,
			   _("Cache is fresh: %s\n"), url);
		  ochusha_network_broker_output_log(broker, message);

		  return FALSE;
		}
	      goto error_exit;
	    }
	}

      if (state == ghttp_not_done)
	{
#if 1
	  if (prev_proc == ghttp_proc_response_hdrs)
	    read_ok = FALSE;
#else
	  const char *tmp_header;
	  if (prev_proc == ghttp_proc_response_hdrs
	      && (tmp_header = ghttp_get_header(request,
						http_hdr_Transfer_Encoding))
	      && strcasecmp(tmp_header, "chunked") == 0)
	    read_ok = FALSE;
#endif
	  if (read_ok)
	    ghttp_flush_response_buffer(request);
	}
      else if (state == ghttp_error)
	{
#if DEBUG_NETWORK_MOST
	  fprintf(stderr, "ghttp_process() returns ghttp_error.\n");
#endif
	  error_message = _("Unknown I/O error.");
	  goto error_exit;
	}

      if (read_ok)
	{
	  body = ghttp_get_body(request);
	  body_len = ghttp_get_body_len(request);
	}
      else
	{
	  body = NULL;
	  body_len = 0;
	}
      
      if (body_len > 0)
	{
	  if (current_status.bytes_total > 0)
	    g_signal_emit(G_OBJECT(broker),
			  broker_signals[ACCESS_PROGRESSED_SIGNAL],
			  0,
			  buffer,
			  current_status.bytes_read,
			  current_status.bytes_total,
			  &signal_result);

	  if (content_encoding == UNKNOWN_ENCODING)
	    {
	      /* ˤ1٤ʤ */
	      content_encoding = get_content_encoding_type(request);

	      switch (content_encoding)
		{
		case GZIP_ENCODING:
		  gzip_buffer = gzip_buffer_new(buffer);
		  if (gzip_buffer == NULL)
		    {
		      error_message = _("Out of memory.");
		      goto error_exit;
		    }
		  break;

		case DEFLATE_ENCODING:
		  gzip_buffer = gzip_buffer_new(buffer);
		  if (gzip_buffer == NULL)
		    {
		      error_message = _("Out of memory.");
		      goto error_exit;
		    }
		  gzip_buffer_skip_gzip_header(gzip_buffer);
		  break;

		case UNKNOWN_ENCODING:
		  error_message = _("Unknown transfer encoding.");
		  goto error_exit;

		case IDENTITY_ENCODING:
		  break;
		}
	    }

	  if (content_encoding == GZIP_ENCODING
	      || content_encoding == DEFLATE_ENCODING)
	    {
	      GzipBufferStatus result
		= gzip_buffer_append_data(gzip_buffer, body, body_len);
	      if (result == GZIP_BUFFER_ERROR)
		{
		  error_message = _("Inflation failed.");
		  gzip_buffer_free(gzip_buffer);

		  goto error_exit;
		}
	      if (result == GZIP_BUFFER_INFLATION_DONE)
		{
		  gzip_buffer_free(gzip_buffer);
		  gzip_buffer = NULL;
		  if (state != ghttp_done)
		    {
		      snprintf(message, LOG_MESSAGE_SIZE,
			       _("Requested file seems to contain garbage: %s\n"),
			       url);
		      ochusha_network_broker_output_log(broker, message);
#if DEBUG_NETWORK_MOST
		      fprintf(stderr, "Requested file seems to contain garbage: %s\n", url);
#endif
		    }
		  break;
		}
	    }
	  else
	    {
	      if (!ochusha_async_buffer_append_data(buffer, body, body_len,
			"ochusha_network_broker.c: http_read_from_url"))
		{	/* content_encoding == IDENTITY_ENCODING */
		  g_signal_emit(G_OBJECT(broker),
				broker_signals[ACCESS_TERMINATED_SIGNAL],
				0,
				buffer,
				&signal_result);
		  ghttp_request_destroy(request);
		  args->request = NULL;

		  status->state
		    = OCHUSHA_NETWORK_BROKER_BUFFER_STATE_ACCESS_FAILED;

		  snprintf(message, LOG_MESSAGE_SIZE,
			   _("Access terminated: %s\n"),
			   url);
		  ochusha_network_broker_output_log(broker, message);

		  LOCK_WORKER_SYNC_OBJECT(sync_object);
		  sync_object->finished = TRUE;
		  UNLOCK_WORKER_SYNC_OBJECT(sync_object);

		  return FALSE;
		}
	    }
	}
      else
	{
	  if (!ochusha_async_buffer_signal(buffer,
			"ochusha_network_broker.c: http_read_from_url"))
	    {
	      g_signal_emit(G_OBJECT(broker),
			    broker_signals[ACCESS_TERMINATED_SIGNAL],
			    0,
			    buffer,
			    &signal_result);
	      ghttp_request_destroy(request);
	      args->request = NULL;

	      status->state
		= OCHUSHA_NETWORK_BROKER_BUFFER_STATE_ACCESS_FAILED;

	      snprintf(message, LOG_MESSAGE_SIZE,
		       _("Access terminated: %s\n"),
		       url);
	      ochusha_network_broker_output_log(broker, message);

	      LOCK_WORKER_SYNC_OBJECT(sync_object);
	      sync_object->finished = TRUE;
	      UNLOCK_WORKER_SYNC_OBJECT(sync_object);

	      return FALSE;
	    }
	}

      if (state == ghttp_done)
	break;

      if (current_status.proc > ghttp_proc_request)
	{
	  sock_fd = ghttp_get_socket(request);
	  if (sock_fd > 0)
	    register_polling_function_for_read(sync_object, sock_fd);
	}

      LOCK_WORKER_SYNC_OBJECT(sync_object);
      if (!sync_object->interrupted && !sync_object->read_ok
	  && pthread_cond_wait(&sync_object->cond, &sync_object->lock) != 0)
	{
	  fprintf(stderr, "Couldn't wait a condition.\n");
	  abort();
	}
      UNLOCK_WORKER_SYNC_OBJECT(sync_object);
    }

  if (gzip_buffer != NULL)
    {
#if DEBUG_NETWORK_MOST
      fprintf(stderr, "What's happen?  Connection was closed but inflation hasn't been completed.\n");
#endif
      gzip_buffer_free(gzip_buffer);
    }

  if (status_code == 200)
    {
      const char *tmp_header
	= ghttp_get_header(request, http_hdr_Last_Modified);
      if (tmp_header != NULL)
	status->last_modified = G_STRDUP(tmp_header);
      else
	status->last_modified = NULL;
      tmp_header = ghttp_get_header(request, http_hdr_Date);
      if (tmp_header != NULL)
	status->date = G_STRDUP(tmp_header);
      else
	status->date = NULL;

      g_signal_emit(G_OBJECT(broker),
		    broker_signals[ACCESS_COMPLETED_SIGNAL],
		    0,
		    buffer,
		    &signal_result);

      ghttp_request_destroy(request);
      args->request = NULL;

      snprintf(message, LOG_MESSAGE_SIZE,
	       _("Access completed: %s\n"),
	       url);
      ochusha_network_broker_output_log(broker, message);

      LOCK_WORKER_SYNC_OBJECT(sync_object);
      sync_object->finished = TRUE;
      UNLOCK_WORKER_SYNC_OBJECT(sync_object);

      return TRUE;
    }

 error_exit:
  LOCK_WORKER_SYNC_OBJECT(sync_object);
  sync_object->finished = TRUE;
  UNLOCK_WORKER_SYNC_OBJECT(sync_object);

  if (error_message != NULL)
    {
      snprintf(message, LOG_MESSAGE_SIZE,
	       _("Access failed(%s): status_code=%d (%s)\n"),
	       url, status_code, error_message);
      g_signal_emit(G_OBJECT(broker),
		    broker_signals[ACCESS_FAILED_SIGNAL],
		    0,
		    buffer,
		    OCHUSHA_NETWORK_BROKER_FAILURE_REASON_UNKNOWN,
		    message,
		    &signal_result);
      ochusha_network_broker_output_log(broker, message);
#if DEBUG_NETWORK_MOST
      fprintf(stderr, "access failed: status_code=%d %s\n",
	      status_code, error_message);
#endif
#if DEBUG_NETWORK
      {
	int i;
	char **headers = NULL;
	int num_headers = 0;
	ghttp_get_header_names(request, &headers, &num_headers);
	for (i = 0; i < num_headers; i++)
	  {
	    snprintf(message, LOG_MESSAGE_SIZE, "%s: %s\n",
		     headers[i], ghttp_get_header(request, headers[i]));
	    ochusha_network_broker_output_log(broker, message);
	    free(headers[i]);
	  }
	if (headers != NULL)
	  free(headers);
      }
#endif
    }
  else if (status_code != 0)
    {
      snprintf(message, LOG_MESSAGE_SIZE, _("Access failed(%s): %d (%s)"),
	       url, status_code, ghttp_reason_phrase(request));
      g_signal_emit(G_OBJECT(broker),
		    broker_signals[ACCESS_FAILED_SIGNAL],
		    0,
		    buffer,
		    OCHUSHA_NETWORK_BROKER_FAILURE_REASON_UNKNOWN,
		    message,
		    &signal_result);
      ochusha_network_broker_output_log(broker, message);
#if DEBUG_NETWORK
      {
	int i;
	char **headers = NULL;
	int num_headers = 0;
	ghttp_get_header_names(request, &headers, &num_headers);
	for (i = 0; i < num_headers; i++)
	  {
	    snprintf(message, LOG_MESSAGE_SIZE, "%s: %s\n",
		     headers[i], ghttp_get_header(request, headers[i]));
	    ochusha_network_broker_output_log(broker, message);
	    free(headers[i]);
	  }
	if (headers != NULL)
	  free(headers);
      }
#endif
#if DEBUG_NETWORK_MOST
      fprintf(stderr, "%s\n", message);
#endif
    }
  else
    {
      snprintf(message, LOG_MESSAGE_SIZE,
	       _("Access failed(%s): unknown reason.\n"),
	       url);
      g_signal_emit(G_OBJECT(broker),
		    broker_signals[ACCESS_FAILED_SIGNAL],
		    0,
		    buffer,
		    OCHUSHA_NETWORK_BROKER_FAILURE_REASON_UNKNOWN,
		    message,
		    &signal_result);
      ochusha_network_broker_output_log(broker, message);
#if DEBUG_NETWORK_MOST
      fprintf(stderr, "%s\n", message);
#endif
    }

  ghttp_request_destroy(request);
  args->request = NULL;

  return FALSE;

}


static void
destruct_job_args(NetworkBrokerJobArgs *job_args)
{
  if (job_args == NULL)
    return;

  if (job_args->broker != NULL)
    g_object_unref(G_OBJECT(job_args->broker));

  if (job_args->url != NULL)
    G_FREE(job_args->url);

  if (job_args->if_modified_since != NULL)
    G_FREE(job_args->if_modified_since);

  if (job_args->request != NULL)
    ghttp_request_destroy(job_args->request);

  G_FREE(job_args);
}


static void
wakeup_now_cb(OchushaAsyncBuffer *buffer, WorkerSyncObject *sync_object)
{
#if DEBUG_NETWORK_MOST
  fprintf(stderr, "entering: wakeup_now_cb\n");
#endif

  LOCK_WORKER_SYNC_OBJECT(sync_object);
  sync_object->interrupted = TRUE;
  if (pthread_cond_signal(&sync_object->cond) != 0)
    {
      fprintf(stderr, "Couldn't signal a condition.\n");
      abort();
    }
  UNLOCK_WORKER_SYNC_OBJECT(sync_object);

#if DEBUG_NETWORK_MOST
  fprintf(stderr, "leaving: wakeup_now_cb\n");
#endif
}


static void
ochusha_network_broker_worker_sync_object_free(WorkerSyncObject *sync_object)
{
  g_return_if_fail(OCHUSHA_IS_WORKER_SYNC_OBJECT(sync_object));

#if DEBUG_NETWORK_MOST
  fprintf(stderr, "entering: sync_object_free(%p)\n", sync_object);
#endif

  g_object_unref(G_OBJECT(sync_object));
}


static OchushaAsyncBuffer *
ochusha_network_broker_employ_worker_thread(OchushaNetworkBroker *broker,
					    const char *url,
					    const char *if_modified_since,
					    JobFunc *job_function)
{
  NetworkBrokerJobArgs *args;
  WorkerJob *job;
  WorkerSyncObject *sync_object = worker_sync_object_new();
  OchushaNetworkBrokerBufferStatus *status
    = G_NEW0(OchushaNetworkBrokerBufferStatus, 1);
  OchushaAsyncBuffer *buffer = ochusha_async_buffer_new(NULL, 0, NULL);

  g_object_set_qdata_full(G_OBJECT(buffer), worker_sync_object_id, sync_object,
		(GDestroyNotify)ochusha_network_broker_worker_sync_object_free);
  g_object_set_qdata_full(G_OBJECT(buffer), broker_buffer_status_id, status,
		(GDestroyNotify)ochusha_network_broker_buffer_status_free);

  g_signal_connect(G_OBJECT(buffer), "wakeup_now",
		   G_CALLBACK(wakeup_now_cb), sync_object);

  sync_object->timeout_id = g_timeout_add_full(G_PRIORITY_DEFAULT - 1,
					       POLLING_INTERVAL_MILLIS,
					       (GSourceFunc)timeout_cb,
					       sync_object, NULL);

  status->state = OCHUSHA_NETWORK_BROKER_BUFFER_STATE_UNKNOWN;

  args = G_NEW0(NetworkBrokerJobArgs, 1);
  args->broker = broker;
  args->url = G_STRDUP(url);
  if (if_modified_since != NULL
      && ochusha_config_cache_file_exist(broker->config, url))
    args->if_modified_since = G_STRDUP(if_modified_since);

  g_object_set_qdata_full(G_OBJECT(buffer), broker_job_args_id,
			  args, (GDestroyNotify)destruct_job_args);

  job = G_NEW0(WorkerJob, 1);
  job->canceled = FALSE;
  job->job = job_function;
  job->args = buffer;

  /* åɤΤ˳ */
  g_object_ref(G_OBJECT(sync_object));
  g_object_ref(G_OBJECT(broker));
  g_object_ref(G_OBJECT(buffer));

  commit_job(job);

  return buffer;
}


static void
try_update_cache(WorkerThread *employee, OchushaAsyncBuffer *buffer)
{
  gboolean signal_result;
  NetworkBrokerJobArgs *args = g_object_get_qdata(G_OBJECT(buffer),
						  broker_job_args_id);
  OchushaNetworkBroker *broker = args->broker;
  WorkerSyncObject *sync_object = g_object_get_qdata(G_OBJECT(buffer),
						     worker_sync_object_id);
  OchushaNetworkBrokerBufferStatus *status
    = g_object_get_qdata(G_OBJECT(buffer), broker_buffer_status_id);
  char *url = args->url;
  int len;
  ghttp_request *request = NULL;
  ContentEncodingType content_encoding = UNKNOWN_ENCODING;
  int status_code = 0;
  const char *error_message = NULL;
  gchar message[LOG_MESSAGE_SIZE];

  if (!ochusha_async_buffer_active_ref(buffer,
			"ochusha_network_broker.c: try_update_cache()"))
    {
      g_signal_emit(G_OBJECT(broker),
		    broker_signals[ACCESS_TERMINATED_SIGNAL],
		    0,
		    buffer,
		    &signal_result);

      status->state = OCHUSHA_NETWORK_BROKER_BUFFER_STATE_ACCESS_FAILED;
      g_signal_emit(G_OBJECT(broker),
		    broker_signals[ACCESS_FAILED_SIGNAL],
		    0,
		    buffer,
		    OCHUSHA_NETWORK_BROKER_FAILURE_REASON_ACCESS_TERMINATED,
		    _("Access terminated."),
		    &signal_result);

      ochusha_async_buffer_fix(buffer,
			       "ochusha_network_broker.c: try_update_cache()");
      g_object_set_qdata(G_OBJECT(buffer), broker_job_args_id, NULL);
      g_object_unref(G_OBJECT(buffer));

      snprintf(message, LOG_MESSAGE_SIZE,
	       _("Access terminated: %s\n"), url);
      ochusha_network_broker_output_log(broker, message);

      LOCK_WORKER_SYNC_OBJECT(sync_object);
      sync_object->finished = TRUE;
      UNLOCK_WORKER_SYNC_OBJECT(sync_object);

      return;
    }

  len = read_cache_to_buffer(broker->config, url, buffer);

  if (len > 0)
    {	/* å夢ξ */
      ghttp_status state = ghttp_not_done;
      int sock_fd = -1;
      char header[HEADER_BUFFER_LENGTH];
      char cache_compare_buffer[CACHE_COMPARE_SIZE];
      int offset = len - CACHE_COMPARE_SIZE;
      GzipBuffer *gzip_buffer = NULL;
      const char *tmp_header;

      if (offset < 0)
	{
	  /* å微 */
	  goto cache_is_dirty;
	}

      if (snprintf(header, HEADER_BUFFER_LENGTH, "bytes=%d-", offset)
	  >= HEADER_BUFFER_LENGTH)
	goto cache_is_dirty;	/* ľͭʤκƼ */

      status->state = OCHUSHA_NETWORK_BROKER_BUFFER_STATE_CACHE_HIT;
      request = ghttp_request_new();
      args->request = request;
      if (request == NULL)
	{
	  /* Out of memory? */
	  status->state = OCHUSHA_NETWORK_BROKER_BUFFER_STATE_CACHE_IS_OK;
	  error_message = _("Out of memory.");
	  goto error_exit;
	}

      ghttp_set_uri(request, url);

      ghttp_set_type(request, ghttp_type_get);
      ghttp_set_header(request, http_hdr_Connection, "close");

#if ENABLE_GZIPPED_TRANSFER
# if ENABLE_GZIPPED_DIFFERENCIAL_READ
      ghttp_set_header(request, http_hdr_Accept_Encoding, "deflate, gzip");
# else
      ghttp_set_header(request, http_hdr_Accept_Encoding, "deflate");
# endif
#endif
      ghttp_set_header(request, http_hdr_Range, header);

      if (args->if_modified_since != NULL)
	{
	  ghttp_set_header(request, http_hdr_If_Modified_Since,
			   args->if_modified_since);
#if DEBUG_NETWORK_MOST
	  fprintf(stderr, "If-Modified-Since: %s\n", args->if_modified_since);
#endif
	}
      if (!setup_common_request_headers(broker, request, FALSE))
	{
	  /* ۤäƥåϤɤΤ */
	  status->state = OCHUSHA_NETWORK_BROKER_BUFFER_STATE_CACHE_IS_OK;
	  g_signal_emit(G_OBJECT(broker),
			broker_signals[ACCESS_FAILED_SIGNAL],
			0,
			buffer,
			OCHUSHA_NETWORK_BROKER_FAILURE_REASON_INVALID_PROXY,
			_("Proxy setting may be wrong."),
			&signal_result);

	  ghttp_request_destroy(request);
	  args->request = NULL;

	  goto finish_try_update_cache;
	}

      g_signal_emit(G_OBJECT(broker),
		    broker_signals[ACCESS_STARTED_SIGNAL],
		    0,
		    buffer,
		    &signal_result);

      snprintf(message, LOG_MESSAGE_SIZE, _("Updating cache file: %s\n"),
	       url);
      ochusha_network_broker_output_log(broker, message);

      ghttp_set_sync(request, ghttp_async);
      ghttp_prepare(request);

      memcpy(cache_compare_buffer, (char *)buffer->buffer + offset,
	     CACHE_COMPARE_SIZE);

      if (!ochusha_async_buffer_update_length(buffer, offset,
			"ochusha_network_broker.c: try_update_cache()"))
	{
	  g_signal_emit(G_OBJECT(broker),
			broker_signals[ACCESS_TERMINATED_SIGNAL],
			0,
			buffer,
			&signal_result);
	  status->state = OCHUSHA_NETWORK_BROKER_BUFFER_STATE_CACHE_IS_OK;
	  g_signal_emit(G_OBJECT(broker),
			broker_signals[ACCESS_FAILED_SIGNAL],
			0,
			buffer,
			OCHUSHA_NETWORK_BROKER_FAILURE_REASON_ACCESS_TERMINATED,
			_("Access terminated."),
			&signal_result);
	  ghttp_request_destroy(request);
	  args->request = NULL;

	  snprintf(message, LOG_MESSAGE_SIZE,
		   _("Access terminated: %s\n"), url);
	  ochusha_network_broker_output_log(broker, message);

	  goto finish_try_update_cache;
	}

      while (TRUE)
	{
	  int body_len;
	  char *body;
	  gboolean before_cache_check = TRUE;
	  ghttp_current_status current_status;
	  gboolean read_ok;

	  LOCK_WORKER_SYNC_OBJECT(sync_object);
	  if (sync_object->interrupted)
	    {
	      sync_object->interrupted = FALSE;
	      UNLOCK_WORKER_SYNC_OBJECT(sync_object);
	      read_ok = FALSE;
	    }
	  else if ((read_ok = sync_object->read_ok)
		   || (state == ghttp_not_done && sock_fd == -1))
	    {
	      sync_object->read_ok = FALSE;
	      UNLOCK_WORKER_SYNC_OBJECT(sync_object);
	      state = ghttp_process(request);
	    }
	  else
	    UNLOCK_WORKER_SYNC_OBJECT(sync_object);

	  current_status = ghttp_get_status(request);

	  if (status_code == 0
	      && (current_status.proc == ghttp_proc_response
		  || (sock_fd > 0
		      && current_status.proc == ghttp_proc_none)))
	    {
	      /* ˤ1٤ʤϤ */
	      status_code = ghttp_status_code(request);
	      if (status_code == 304)
		{
		  tmp_header = ghttp_get_header(request, http_hdr_Date);
		  if (tmp_header != NULL)
		    status->date = G_STRDUP(tmp_header);
		  else
		    status->date = NULL;

		  ochusha_async_buffer_update_length(buffer, len,
			"ochusha_network_broker.c: try_update_cache()");

		  status->status_code = status_code;
		  status->state
		    = OCHUSHA_NETWORK_BROKER_BUFFER_STATE_CACHE_IS_OK;
		  g_signal_emit(G_OBJECT(broker),
				broker_signals[ACCESS_COMPLETED_SIGNAL],
				0,
				buffer,
				&signal_result);
		  ghttp_request_destroy(request);
		  args->request = NULL;

		  snprintf(message, LOG_MESSAGE_SIZE,
			   _("Cache is fresh: %s\n"), url);
		  ochusha_network_broker_output_log(broker, message);

		  goto finish_try_update_cache;
		}
	      if (status_code != 206)
		{
		  goto cache_is_dirty;
		}

	      content_encoding = get_content_encoding_type(request);

	      switch (content_encoding)
		{
		case GZIP_ENCODING:
		  gzip_buffer = gzip_buffer_new(buffer);
		  if (gzip_buffer == NULL)
		    {
		      error_message = _("Out of memory.");
		      goto error_exit;
		    }
		  break;

		case DEFLATE_ENCODING:
		  gzip_buffer = gzip_buffer_new(buffer);
		  if (gzip_buffer == NULL)
		    {
		      error_message = _("Out of memory.");
		      goto error_exit;
		    }
		  gzip_buffer_skip_gzip_header(gzip_buffer);
		  break;

		case UNKNOWN_ENCODING:
		  goto cache_is_dirty;	/* ĩ */

		case IDENTITY_ENCODING:
		  break;
		}
	    }

	  if (state == ghttp_not_done)
	    {
	      if (read_ok)
		ghttp_flush_response_buffer(request);
	    }
	  else if (state == ghttp_error)
	    {
	      error_message = _("Unknown I/O error.");
	      goto error_exit;
	    }

	  if (read_ok)
	    {
	      body = ghttp_get_body(request);
	      body_len = ghttp_get_body_len(request);
	    }
	  else
	    {
	      body = NULL;
	      body_len = 0;
	    }

	  if (body_len > 0)
	    {
	      if (current_status.bytes_total > 0)
		g_signal_emit(G_OBJECT(broker),
			      broker_signals[ACCESS_PROGRESSED_SIGNAL],
			      0,
			      buffer,
			      current_status.bytes_read,
			      current_status.bytes_total,
			      &signal_result);

	      if (content_encoding == GZIP_ENCODING
		  || content_encoding == DEFLATE_ENCODING)
		{
		  GzipBufferStatus result
		    = gzip_buffer_append_data(gzip_buffer, body, body_len);
		  if (result == GZIP_BUFFER_ERROR)
		    {
		      error_message = _("Inflation failed.");
		      gzip_buffer_free(gzip_buffer);
		      goto error_exit;
		    }
		  if (result == GZIP_BUFFER_INFLATION_DONE)
		    {
		      gzip_buffer_free(gzip_buffer);
		      gzip_buffer = NULL;
#if DEBUG_NETWORK_MOST
		      if (state != ghttp_done)
			fprintf(stderr, "Requested file contains garbage?\n");
#endif
		      break;
		    }
		}
	      else if (!ochusha_async_buffer_append_data(buffer, body,
							 body_len,
			      "ochusha_network_broker.c: try_update_cache()"))
		{	/* content_encoding == IDENTITY_ENCODING */
		  g_signal_emit(G_OBJECT(broker),
				broker_signals[ACCESS_TERMINATED_SIGNAL],
				0,
				buffer,
				&signal_result);

		  status->state
		    = OCHUSHA_NETWORK_BROKER_BUFFER_STATE_CACHE_IS_OK;
		  g_signal_emit(G_OBJECT(broker),
				broker_signals[ACCESS_FAILED_SIGNAL],
				0,
				buffer,
				OCHUSHA_NETWORK_BROKER_FAILURE_REASON_ACCESS_TERMINATED,
				_("Access terminated."),
				&signal_result);

		  ghttp_request_destroy(request);
		  args->request = NULL;

		  snprintf(message, LOG_MESSAGE_SIZE,
			   _("Access terminated: %s\n"), url);
		  ochusha_network_broker_output_log(broker, message);

		  goto finish_try_update_cache;
		}

	      /* buffer->length >= lenˤʤä饭å */
	      if (before_cache_check && buffer->length >= len)
		{
		  /*
		   * MEMO: ޤ褿顢consumeråɤȶĴ
		   *       ȻפäƤΤμȤߤconsumer
		   *       åɤ鸫ǽΤ ġĤȡ
		   *       դΤǡα
		   */
		  if (memcmp(cache_compare_buffer,
			     (char *)buffer->buffer + offset,
			     CACHE_COMPARE_SIZE) != 0)
		    {
		      goto cache_is_dirty;	/* åäƤ */
		    }
		  else
		    before_cache_check = FALSE;	/* åϿ */
		}
	    }
	  else
	    {
	      if (!ochusha_async_buffer_signal(buffer,
			"ochusha_network_broker.c: try_update_cache()"))
		{
		  g_signal_emit(G_OBJECT(broker),
				broker_signals[ACCESS_TERMINATED_SIGNAL],
				0,
				buffer,
				&signal_result);

		  status->state
		    = OCHUSHA_NETWORK_BROKER_BUFFER_STATE_CACHE_IS_OK;
		  g_signal_emit(G_OBJECT(broker),
				broker_signals[ACCESS_FAILED_SIGNAL],
				0,
				buffer,
				OCHUSHA_NETWORK_BROKER_FAILURE_REASON_ACCESS_TERMINATED,
				_("Access terminated."),
				&signal_result);

		  ghttp_request_destroy(request);
		  args->request = NULL;

		  snprintf(message, LOG_MESSAGE_SIZE,
			   _("Access terminated: %s\n"),
			   url);
		  ochusha_network_broker_output_log(broker, message);

		  goto finish_try_update_cache;
		}
	    }

	  if (state == ghttp_done)
	    break;

	  if (current_status.proc > ghttp_proc_request)
	    {
	      sock_fd = ghttp_get_socket(request);
	      if (sock_fd > 0)
		register_polling_function_for_read(sync_object, sock_fd);
	    }

	  LOCK_WORKER_SYNC_OBJECT(sync_object);
	  if (!sync_object->interrupted && !sync_object->read_ok
	      && pthread_cond_wait(&sync_object->cond,
				   &sync_object->lock) != 0)
	    {
	      fprintf(stderr, "Couldn't wait a condition.\n");
	      abort();
	    }
	  UNLOCK_WORKER_SYNC_OBJECT(sync_object);
	}

      if (buffer->length < len)
	goto cache_is_dirty;	/* ͽǡʤäΤ
				 * åäƤ롣
				 */
      tmp_header = ghttp_get_header(request, http_hdr_Last_Modified);
      if (tmp_header != NULL)
	status->last_modified = G_STRDUP(tmp_header);
      else
	status->last_modified = NULL;
      tmp_header = ghttp_get_header(request, http_hdr_Date);
      if (tmp_header != NULL)
	status->date = G_STRDUP(tmp_header);
      else
	status->date = NULL;

      status->status_code = status_code;
      g_signal_emit(G_OBJECT(broker),
		    broker_signals[ACCESS_COMPLETED_SIGNAL],
		    0,
		    buffer,
		    &signal_result);
      ghttp_request_destroy(request);
      args->request = NULL;
      write_buffer_to_cache(broker, url, buffer);

      snprintf(message, LOG_MESSAGE_SIZE, _("Cache updated: %s\n"), url);
      ochusha_network_broker_output_log(broker, message);

      goto finish_try_update_cache;

    cache_is_dirty:
      status->state = OCHUSHA_NETWORK_BROKER_BUFFER_STATE_CACHE_IS_DIRTY;
      if (request != NULL)
	{
	  ghttp_request_destroy(request);
	  args->request = NULL;
	}

      snprintf(message, LOG_MESSAGE_SIZE,
	       _("Cache is dirty: %s\n"), url);
      ochusha_network_broker_output_log(broker, message);

      if (!ochusha_async_buffer_update_length(buffer, 0,
			"ochusha_network_broker.c: try_update_cache()"))
	{
	  g_signal_emit(G_OBJECT(broker),
			broker_signals[ACCESS_TERMINATED_SIGNAL],
			0,
			buffer,
			&signal_result);
	  status->state = OCHUSHA_NETWORK_BROKER_BUFFER_STATE_ACCESS_FAILED;

	  g_signal_emit(G_OBJECT(broker),
		broker_signals[ACCESS_FAILED_SIGNAL],
		0,
		buffer,
		OCHUSHA_NETWORK_BROKER_FAILURE_REASON_ACCESS_TERMINATED,
		_("Access terminated."),
		&signal_result);

	  snprintf(message, LOG_MESSAGE_SIZE,
		   _("Access terminated: %s\n"), url);
	  ochusha_network_broker_output_log(broker, message);

	  goto finish_try_update_cache;
	}
    }
  else
    {
      /* åʤξ */
      status->state = OCHUSHA_NETWORK_BROKER_BUFFER_STATE_DIRECT_ACCESS;
    }

  /* å夬ʤääƤ */
  if (http_read_from_url(broker, buffer))
    {
      /* ̿ */
      if (status->state == OCHUSHA_NETWORK_BROKER_BUFFER_STATE_DIRECT_ACCESS)
	status->state = OCHUSHA_NETWORK_BROKER_BUFFER_STATE_ACCESS_OK;

      write_buffer_to_cache(broker, url, buffer);
    }
  ochusha_async_buffer_fix(buffer,
			   "ochusha_network_broker.c: try_update_cache");
  ochusha_async_buffer_active_unref(buffer,
				"ochusha_network_broker.c: try_update_cache");
  g_object_set_qdata(G_OBJECT(buffer), broker_job_args_id, NULL);
  g_object_unref(G_OBJECT(buffer));
  return;

 error_exit:
  if (error_message != NULL)
    {
      snprintf(message, LOG_MESSAGE_SIZE,
	       _("Access failed(%s): %s\n"), url, error_message);
      g_signal_emit(G_OBJECT(broker),
		    broker_signals[ACCESS_FAILED_SIGNAL],
		    0,
		    buffer,
		    OCHUSHA_NETWORK_BROKER_FAILURE_REASON_UNKNOWN,
		    error_message,
		    &signal_result);
      ochusha_network_broker_output_log(broker, message);
    }
  else
    {
      status_code = ghttp_status_code(request);
      if (status_code != 0)
	{
	  snprintf(message, LOG_MESSAGE_SIZE, _("Access failed(%s): %d (%s)"),
		   url, status_code, ghttp_reason_phrase(request));
	  g_signal_emit(G_OBJECT(broker),
			broker_signals[ACCESS_FAILED_SIGNAL],
			0,
			buffer,
			OCHUSHA_NETWORK_BROKER_FAILURE_REASON_UNKNOWN,
			message,
			&signal_result);
	  ochusha_network_broker_output_log(broker, message);
	}
      else
	{
	  snprintf(message, LOG_MESSAGE_SIZE,
		   _("Access failed(%s): unknown reason.\n"),
		   url);
	  g_signal_emit(G_OBJECT(broker),
			broker_signals[ACCESS_FAILED_SIGNAL],
			0,
			buffer,
			OCHUSHA_NETWORK_BROKER_FAILURE_REASON_UNKNOWN,
			message,
			&signal_result);
	  ochusha_network_broker_output_log(broker, message);
	}
#if DEBUG_NETWORK
      {
	int i;
	char **headers = NULL;
	int num_headers = 0;
	ghttp_get_header_names(request, &headers, &num_headers);
	for (i = 0; i < num_headers; i++)
	  {
	    snprintf(message, LOG_MESSAGE_SIZE, "%s: %s\n",
		     headers[i], ghttp_get_header(request, headers[i]));
	    ochusha_network_broker_output_log(broker, message);
	    free(headers[i]);
	  }
	if (headers != NULL)
	  free(headers);
      }
#endif
    }

  if (request != NULL)
    ghttp_request_destroy(request);
  args->request = NULL;

 finish_try_update_cache:
  LOCK_WORKER_SYNC_OBJECT(sync_object);
  sync_object->finished = TRUE;
  UNLOCK_WORKER_SYNC_OBJECT(sync_object);

  ochusha_async_buffer_fix(buffer,
			   "ochusha_network_broker.c: try_update_cache");
  ochusha_async_buffer_active_unref(buffer,
				"ochusha_network_broker.c: try_update_cache");
  g_object_set_qdata(G_OBJECT(buffer), broker_job_args_id, NULL);
  g_object_unref(G_OBJECT(buffer));
}


static void
refresh_cache_after_read(WorkerThread *employee, OchushaAsyncBuffer *buffer)
{
  gboolean signal_result;
  NetworkBrokerJobArgs *args = g_object_get_qdata(G_OBJECT(buffer),
					      broker_job_args_id);
  OchushaNetworkBroker *broker = args->broker;
  OchushaNetworkBrokerBufferStatus *status
    = g_object_get_qdata(G_OBJECT(buffer), broker_buffer_status_id);
  gchar message[LOG_MESSAGE_SIZE];

#if DEBUG_THREAD_MOST
  fprintf(stderr, "refresh_cache_after_read() thread is invoked.\n");
#endif

  if (!ochusha_async_buffer_active_ref(buffer,
		"ochusha_network_broker.c: refresh_cache_after_read()"))
    {
      status->state = OCHUSHA_NETWORK_BROKER_BUFFER_STATE_ACCESS_FAILED;
      g_signal_emit(G_OBJECT(broker),
		    broker_signals[ACCESS_TERMINATED_SIGNAL],
		    0,
		    buffer,
		    &signal_result);
      goto finish_refresh_cache_after_read;
    }

  status->state = OCHUSHA_NETWORK_BROKER_BUFFER_STATE_DIRECT_ACCESS;

  if (http_read_from_url(broker, buffer))
    {
      /* ̿ */
      status->state = OCHUSHA_NETWORK_BROKER_BUFFER_STATE_ACCESS_OK;
      /* ХåեƤ򥭥å˽񤭹ࡣ*/
      write_buffer_to_cache(broker, args->url, buffer);
    }
  else
    {
      /* ̿Ԥå夬(if_modified_sinceNULL)ʾ */
      if (read_cache_to_buffer(broker->config, args->url, buffer))
	{
	  status->state = OCHUSHA_NETWORK_BROKER_BUFFER_STATE_CACHE_HIT;
	  if (status->status_code == 304)
	    {	/* å夬ä */
	      if (args->if_modified_since != NULL)
		status->last_modified = G_STRDUP(args->if_modified_since);
	      g_signal_emit(G_OBJECT(broker),
			    broker_signals[ACCESS_COMPLETED_SIGNAL],
			    0,
			    buffer,
			    &signal_result);
	    }
	  else
	    {	/* ̿˼ԤΤǻʤåȤ */
	      g_signal_emit(G_OBJECT(broker),
			    broker_signals[ACCESS_FAILED_SIGNAL],
			    0,
			    buffer,
			    OCHUSHA_NETWORK_BROKER_FAILURE_REASON_UNKNOWN,
			    _("Couldn't read from network, cache used."),
			    &signal_result);
	      snprintf(message, LOG_MESSAGE_SIZE,
		       _("Use cached file(%s): due to networking error.\n"),
		       args->url);
	      ochusha_network_broker_output_log(broker, message);
	    }
	}
      else
	{	/* åɤʤ */
	  if (status->status_code == 304)
	    {
	      /* ٤å夬Ĥʤ */
	      g_signal_emit(G_OBJECT(broker),
			broker_signals[ACCESS_FAILED_SIGNAL],
			0,
			buffer,
			OCHUSHA_NETWORK_BROKER_FAILURE_REASON_CACHE_NOT_FOUND,
			_("Couldn't find cache that should exist."),
			&signal_result);
	      snprintf(message, LOG_MESSAGE_SIZE,
		       _("Couldn't find cache file: %s\n"),
		       args->url);
	      ochusha_network_broker_output_log(broker, message);
	    }
	  else
	    {
	      /* ̤˼ */
	      if (buffer->state != OCHUSHA_ASYNC_BUFFER_TERMINATED)
		g_signal_emit(G_OBJECT(broker),
			      broker_signals[ACCESS_FAILED_SIGNAL],
			      0,
			      buffer,
			      OCHUSHA_NETWORK_BROKER_FAILURE_REASON_UNKNOWN,
			      _("Couldn't read data via the network."),
			      &signal_result);
	      else
		g_signal_emit(G_OBJECT(broker),
			      broker_signals[ACCESS_TERMINATED_SIGNAL],
			      0,
			      buffer,
			      &signal_result);
	    }
	}
    }

  ochusha_async_buffer_active_unref(buffer,
		"ochusha_network_broker.c: refresh_cache_after_read()");

 finish_refresh_cache_after_read:
  ochusha_async_buffer_fix(buffer,
		"ochusha_network_broker.c: refresh_cache_after_read()");
  g_object_set_qdata(G_OBJECT(buffer), broker_job_args_id, NULL);
  g_object_unref(G_OBJECT(buffer));
}


static void
force_read(WorkerThread *employee, OchushaAsyncBuffer *buffer)
{
  gboolean signal_result;
  NetworkBrokerJobArgs *args = g_object_get_qdata(G_OBJECT(buffer),
					      broker_job_args_id);
  OchushaNetworkBroker *broker = args->broker;
  OchushaNetworkBrokerBufferStatus *status
    = g_object_get_qdata(G_OBJECT(buffer), broker_buffer_status_id);

#if DEBUG_THREAD_MOST
  fprintf(stderr, "force_read() thread is invoked.\n");
#endif

  if (!ochusha_async_buffer_active_ref(buffer,
				"ochusha_network_broker.c: force_read()"))
    {
      status->state = OCHUSHA_NETWORK_BROKER_BUFFER_STATE_ACCESS_FAILED;
      g_signal_emit(G_OBJECT(broker),
		    broker_signals[ACCESS_TERMINATED_SIGNAL],
		    0,
		    buffer,
		    &signal_result);
      goto finish_force_read;
    }

  status->state = OCHUSHA_NETWORK_BROKER_BUFFER_STATE_DIRECT_ACCESS;

  if (args->if_modified_since != NULL)
    {
      G_FREE(args->if_modified_since);
      args->if_modified_since = NULL;
    }

  if (http_read_from_url(broker, buffer))
    {
      /* ̿ */
      status->state = OCHUSHA_NETWORK_BROKER_BUFFER_STATE_ACCESS_OK;
#if DEBUG_NETWORK_MOST
      fprintf(stderr, "reading from %s succeeded.\n", args->url);
#endif
    }
  else
    {
      /* ̿ */
      status->state = OCHUSHA_NETWORK_BROKER_BUFFER_STATE_ACCESS_FAILED;
#if DEBUG_NETWORK_MOST
      fprintf(stderr, "reading from %s failed.\n", args->url);
#endif
    }

  ochusha_async_buffer_active_unref(buffer,
				    "ochusha_network_broker.c: force_read()");

 finish_force_read:
  ochusha_async_buffer_fix(buffer, "ochusha_network_broker.c: force_read()");
  g_object_set_qdata(G_OBJECT(buffer), broker_job_args_id, NULL);
  g_object_unref(G_OBJECT(buffer));
}


/*
 * ochusha_network_broker_read_from_url
 *
 * Ϳ줿URLǡɤ߹ࡣ
 * offline⡼ɤǥå夬¸ߤʤNULL֤
 *
 * modeOCHUSHA_NETWORK_BROKER_CACHE_IGNOREǤ硢ɬͥåȥ
 * ǡɤ߹ߡ̤򥭥å夷ʤ
 *
 * ʳξˤɤ߹߻˲餫ηǥå夬ФȤ
 * ˥ͥåȥɹǤˤϥǡ򥭥å¸롣
 *
 * OCHUSHA_NETWORK_BROKER_CACHE_AS_ISξ硢å夬¸ߤˤϤ
 * Τޤ޻Ȥ
 *
 * OCHUSHA_NETWORK_BROKER_CACHE_TRY_UPDATEξ硢å夵Ƥǡ
 * append onlyǹΤǤȸʤå夵Ƥǡ
 * ǿǤκʬΤߤͥåȥɤ߹褦ĩ魯롣⤷append only
 * ǹǤʤäˤɤ߹ľ
 *
 * OCHUSHA_NETWORK_BROKER_CACHE_TRY_REFRESHξ硢å夬ͥå
 * 鴰˥ǡɤ߹ߡͥåȥɹԲǽǤä
 * ˤΤߡå夵ƤǡȤ
 *
 * ͥåȥ˥硢̿˼ԤǤĹ0ΥХåե
 * ֤Τա
 */
OchushaAsyncBuffer *
ochusha_network_broker_read_from_url(OchushaNetworkBroker *broker,
				     const char *url,
				     const char *if_modified_since,
				     OchushaNetworkBrokerCacheMode mode)
{
  gboolean signal_result;

  g_return_val_if_fail(OCHUSHA_IS_NETWORK_BROKER(broker)
		       && broker->config != NULL && url != NULL, NULL);
#if DEBUG_NETWORK_MOST
  fprintf(stderr, "ochusha_network_broker_read_from_url: %s\n", url);
#endif

  if (mode == OCHUSHA_NETWORK_BROKER_CACHE_IGNORE && broker->config->offline)
    {
      g_signal_emit(G_OBJECT(broker),
		    broker_signals[ACCESS_FAILED_SIGNAL],
		    0,
		    NULL,
		    OCHUSHA_NETWORK_BROKER_FAILURE_REASON_NOT_CONNECTED,
		    _("Network not connected on offline mode."),
		    &signal_result);
      return NULL;
    }

  if (broker->config->offline
      || mode == OCHUSHA_NETWORK_BROKER_CACHE_AS_IS
      || mode == OCHUSHA_NETWORK_BROKER_CACHE_ONLY)
    {
      /* å夬ä餽򤽤Τޤ޻Ȥ */
      int fd = ochusha_config_cache_open_file(broker->config,
					      url, O_RDONLY);
      if (fd >= 0)
	{
	  OchushaAsyncBuffer *buffer = get_mmapped_buffer(fd);

	  if (buffer != NULL)
	    g_signal_emit(G_OBJECT(broker),
			  broker_signals[ACCESS_COMPLETED_SIGNAL],
			  0,
			  buffer,
			  &signal_result);
	  else
	    g_signal_emit(G_OBJECT(broker),
			  broker_signals[ACCESS_FAILED_SIGNAL],
			  0,
			  NULL,
			  OCHUSHA_NETWORK_BROKER_FAILURE_REASON_MMAP_FAILED,
			  _("Couldn't get mmapped buffer."),
			  &signal_result);
	  return buffer;
	}

      if (mode == OCHUSHA_NETWORK_BROKER_CACHE_ONLY)
	{
	  /* åʤĥå夷Ȥʤ */
	  g_signal_emit(G_OBJECT(broker),
			broker_signals[ACCESS_FAILED_SIGNAL],
			0,
			NULL,
			OCHUSHA_NETWORK_BROKER_FAILURE_REASON_CACHE_NOT_FOUND,
			_("Couldn't find cache that should exist."),
			&signal_result);
	  return NULL;
	}

      if (broker->config->offline)
	{
	  g_signal_emit(G_OBJECT(broker),
			broker_signals[ACCESS_FAILED_SIGNAL],
			0,
			NULL,
			OCHUSHA_NETWORK_BROKER_FAILURE_REASON_NOT_CONNECTED,
			_("Network not connected on offline mode."),
			&signal_result);
	  return NULL;
	}
    }

  switch (mode)
    {
    case OCHUSHA_NETWORK_BROKER_CACHE_TRY_UPDATE:
      return ochusha_network_broker_employ_worker_thread(broker,
						url, if_modified_since,
						(JobFunc *)try_update_cache);

    case OCHUSHA_NETWORK_BROKER_CACHE_TRY_REFRESH:
    case OCHUSHA_NETWORK_BROKER_CACHE_AS_IS:
      return ochusha_network_broker_employ_worker_thread(broker,
					url, if_modified_since,
					(JobFunc *)refresh_cache_after_read);

    case OCHUSHA_NETWORK_BROKER_CACHE_IGNORE:
      return ochusha_network_broker_employ_worker_thread(broker,
						    url, NULL,
						    (JobFunc *)force_read);

    default:
      break;
    }

  /* ߥͭʤΤabort */
  abort();
}


static gboolean
http_read_from_url_synchronously(OchushaNetworkBroker *broker,
				 OchushaAsyncBuffer *buffer)
{
  gboolean signal_result;
  NetworkBrokerJobArgs *args = g_object_get_qdata(G_OBJECT(buffer),
					      broker_job_args_id);
  OchushaNetworkBrokerBufferStatus *status
    = g_object_get_qdata(G_OBJECT(buffer), broker_buffer_status_id);
  const char *url = args->url;
  const char *error_message = NULL;
  const char *body;
  int body_len;
  int status_code = 0;
  ghttp_request *request = NULL;
  ContentEncodingType content_encoding = UNKNOWN_ENCODING;
  GzipBuffer *gzip_buffer = NULL;
  ghttp_status state;

  request = ghttp_request_new();
  args->request = request;	/* 塹Τˡġ */

  ghttp_set_uri(request, (char *)url);
  ghttp_set_type(request, ghttp_type_get);
#if ENABLE_GZIPPED_TRANSFER
  ghttp_set_header(request, http_hdr_Accept_Encoding, "deflate, gzip");
#endif
  ghttp_set_header(request, http_hdr_Connection, "close");

  if (!setup_common_request_headers(broker, request, FALSE))
    {
      g_signal_emit(G_OBJECT(broker),
		    broker_signals[ACCESS_FAILED_SIGNAL],
		    0,
		    buffer,
		    OCHUSHA_NETWORK_BROKER_FAILURE_REASON_INVALID_PROXY,
		    _("Proxy setting may be wrong."),
		    &signal_result);

      ghttp_request_destroy(request);
      args->request = NULL;
      return FALSE;
    }    

  g_signal_emit(G_OBJECT(broker),
		broker_signals[ACCESS_STARTED_SIGNAL],
		0,
		buffer,
		&signal_result);

  ghttp_set_sync(request, ghttp_sync);
  ghttp_prepare(request);
  state = ghttp_process(request);

  if (state == ghttp_error)
    {
      error_message = _("Unknown I/O error.");
      goto error_exit;
    }

  status_code = ghttp_status_code(request);
  status->status_code = status_code;

  if (status_code == 200)
    {
      const char *tmp_header
	= ghttp_get_header(request, http_hdr_Last_Modified);
      if (tmp_header != NULL)
	status->last_modified = G_STRDUP(tmp_header);
      else
	status->last_modified = NULL;
      tmp_header = ghttp_get_header(request, http_hdr_Date);
      if (tmp_header != NULL)
	status->date = G_STRDUP(tmp_header);
      else
	status->date = NULL;
#if DEBUG_NETWORK_MOST
      fprintf(stderr, "Synchronous read done.\n");
#endif
    }
  else
    {
      goto error_exit;
    }

  body = ghttp_get_body(request);
  body_len = ghttp_get_body_len(request);
      
  if (body_len > 0)
    {
      ghttp_current_status current_status = ghttp_get_status(request);
      if (current_status.bytes_total > 0)
	g_signal_emit(G_OBJECT(broker),
		      broker_signals[ACCESS_PROGRESSED_SIGNAL],
		      0,
		      buffer,
		      current_status.bytes_read,
		      current_status.bytes_total,
		      &signal_result);

      content_encoding = get_content_encoding_type(request);

      switch (content_encoding)
	{
	case GZIP_ENCODING:
	  gzip_buffer = gzip_buffer_new(buffer);
	  if (gzip_buffer == NULL)
	    {
	      error_message = _("Out of memory.");
	      goto error_exit;
	    }
	  break;

	case DEFLATE_ENCODING:
	  gzip_buffer = gzip_buffer_new(buffer);
	  if (gzip_buffer == NULL)
	    {
	      error_message = _("Out of memory.");
	      goto error_exit;
	    }
	  gzip_buffer_skip_gzip_header(gzip_buffer);
	  break;

	case UNKNOWN_ENCODING:
	  error_message = _("Unknown transfer encoding.");
	  goto error_exit;
  
	case IDENTITY_ENCODING:
	  break;
	}

      if (content_encoding == GZIP_ENCODING
	  || content_encoding == DEFLATE_ENCODING)
	{
	  GzipBufferStatus result
	    = gzip_buffer_append_data(gzip_buffer, body, body_len);
	  if (result == GZIP_BUFFER_ERROR)
	    {
	      error_message = _("Inflation failed.");
	      gzip_buffer_free(gzip_buffer);
	      goto error_exit;
	    }
	  if (result == GZIP_BUFFER_INFLATION_DONE)
	    {
	      gzip_buffer_free(gzip_buffer);
	      gzip_buffer = NULL;
#if DEBUG_NETWORK_MOST
	      if (state != ghttp_done)
		fprintf(stderr, "Requested file contains garbage?\n");
#endif
	    }
	}
      else if (!ochusha_async_buffer_append_data(buffer, body, body_len,
		"ochusha_network_broker.c: http_read_from_url_synchronously"))
	{	/* content_encoding == IDENTITY_ENCODING */
	  g_signal_emit(G_OBJECT(broker),
			broker_signals[ACCESS_TERMINATED_SIGNAL],
			0,
			buffer,
			&signal_result);
	  ghttp_request_destroy(request);
	  args->request = NULL;

	  status->state
	    = OCHUSHA_NETWORK_BROKER_BUFFER_STATE_ACCESS_FAILED;

	  return FALSE;
	}
    }

  if (gzip_buffer != NULL)
    {
#if DEBUG_NETWORK_MOST
      fprintf(stderr, "What's happen?  Connection was closed but inflation hasn't been completed.\n");
#endif
      gzip_buffer_free(gzip_buffer);
    }

#if DEBUG_NETWORK_MOST
  fprintf(stderr, "http_read_from_url_synchronously succeeded.\n");
#endif
  g_signal_emit(G_OBJECT(broker),
		broker_signals[ACCESS_COMPLETED_SIGNAL],
		0,
		buffer,
		&signal_result);

  return TRUE;

 error_exit:
#if DEBUG_NETWORK_MOST
  fprintf(stderr, "force_read_synchronously(): error.\n");
#endif
  if (error_message != NULL)
    {
      g_signal_emit(G_OBJECT(broker),
		    broker_signals[ACCESS_FAILED_SIGNAL],
		    0,
		    buffer,
		    OCHUSHA_NETWORK_BROKER_FAILURE_REASON_UNKNOWN,
		    error_message,
		    &signal_result);
      fprintf(stderr, "access failed: status_code=%d %s\n",
	      status_code, error_message);
    }
  else if (status_code != 0)
    {
      char message[LOG_MESSAGE_SIZE];
      snprintf(message, LOG_MESSAGE_SIZE, "Access failed: %d (%s)",
	       status_code, ghttp_reason_phrase(request));
      g_signal_emit(G_OBJECT(broker),
		    broker_signals[ACCESS_FAILED_SIGNAL],
		    0,
		    buffer,
		    OCHUSHA_NETWORK_BROKER_FAILURE_REASON_UNKNOWN,
		    message,
		    &signal_result);
      fprintf(stderr, "%s\n", message);
    }
  else
    g_signal_emit(G_OBJECT(broker),
		  broker_signals[ACCESS_FAILED_SIGNAL],
		  0,
		  buffer,
		  OCHUSHA_NETWORK_BROKER_FAILURE_REASON_UNKNOWN,
		  _("Access failed: unknown reason."),
		  &signal_result);
  if (request != NULL)
    ghttp_request_destroy(request);
  args->request = NULL;

  return FALSE;

}


static void
force_read_synchronously(WorkerThread *employee, OchushaAsyncBuffer *buffer)
{
  gboolean signal_result;
  NetworkBrokerJobArgs *args = g_object_get_qdata(G_OBJECT(buffer),
					      broker_job_args_id);
  OchushaNetworkBroker *broker = args->broker;
  OchushaNetworkBrokerBufferStatus *status
    = g_object_get_qdata(G_OBJECT(buffer), broker_buffer_status_id);

#if DEBUG_THREAD_MOST
  fprintf(stderr, "force_read_synchronously() thread is invoked.\n");
#endif

  if (!ochusha_async_buffer_active_ref(buffer,
		"ochusha_network_broker.c: force_read_synchronously()"))
    {
      status->state = OCHUSHA_NETWORK_BROKER_BUFFER_STATE_ACCESS_FAILED;
      g_signal_emit(G_OBJECT(broker),
		    broker_signals[ACCESS_TERMINATED_SIGNAL],
		    0,
		    buffer,
		    &signal_result);
      goto finish_force_read;
    }

  status->state = OCHUSHA_NETWORK_BROKER_BUFFER_STATE_DIRECT_ACCESS;

  if (args->if_modified_since != NULL)
    {
      G_FREE(args->if_modified_since);
      args->if_modified_since = NULL;
    }

  if (http_read_from_url_synchronously(broker, buffer))
    {
      /* ̿ */
      status->state = OCHUSHA_NETWORK_BROKER_BUFFER_STATE_ACCESS_OK;
#if DEBUG_NETWORK_MOST
      fprintf(stderr, "reading from %s succeeded.\n", args->url);
#endif
    }
  else
    {
      /* ̿ */
      status->state = OCHUSHA_NETWORK_BROKER_BUFFER_STATE_ACCESS_FAILED;
#if DEBUG_NETWORK_MOST
      fprintf(stderr, "reading from %s failed.\n", args->url);
#endif
    }

  ochusha_async_buffer_active_unref(buffer,
		"ochusha_network_broker.c: force_read_synchronously()");

 finish_force_read:
  ochusha_async_buffer_fix(buffer,
		"ochusha_network_broker.c: force_read_synchronously()");
  g_object_set_qdata(G_OBJECT(buffer), broker_job_args_id, NULL);
  g_object_unref(G_OBJECT(buffer));
}


/*
 * ochusha_network_broker_read_from_cgi
 *
 * Ϳ줿URLǡɤ߹ࡣ
 * offline⡼ɤǤNULL֤
 *
 * åˤϿʤޤCGIξHEAD쥹ݥ󥹤Ƥʤ
 * ȤΤǡƱ⡼ɤȤGETȯɤ߹ࡣ
 */
OchushaAsyncBuffer *
ochusha_network_broker_read_from_cgi(OchushaNetworkBroker *broker,
				     const char *url)
{
  g_return_val_if_fail(OCHUSHA_IS_NETWORK_BROKER(broker)
		       && broker->config != NULL && url != NULL, NULL);
#if DEBUG_NETWORK_MOST
  fprintf(stderr, "ochusha_network_broker_read_from_cgi: %s\n", url);
#endif

  if (broker->config->offline)
    return NULL;

  return ochusha_network_broker_employ_worker_thread(broker, url, NULL,
					(JobFunc *)force_read_synchronously);
}


/*
 * ochusha_network_broker_get_response_header:
 *
 * bufferHTTPˤ륢η̤ݻƤˡ
 * HTTP쥹ݥ󥹤λꤷإåƤ֤бإåʤ
 * HTTP쥹ݥ󥹥إåΤΤ̵ˤNULL֤
 *
 * ̾buffer̿åɤνλޤǥ쥹ݥ󥹥إåݻ
 * Ƥ롣
 */
const char *
ochusha_network_broker_get_response_header(OchushaNetworkBroker *broker,
					   OchushaAsyncBuffer *buffer,
					   const char *header)
{
  NetworkBrokerJobArgs *args;
  g_return_val_if_fail(OCHUSHA_IS_NETWORK_BROKER(broker)
		       && OCHUSHA_IS_ASYNC_BUFFER(buffer), NULL);

  args = g_object_get_qdata(G_OBJECT(buffer), broker_job_args_id);
  if (args == NULL || args->request == NULL)
    return NULL;

  return ghttp_get_header(args->request, header);
}


/*
 * ochusha_network_broker_get_header_names:
 *
 * bufferHTTPˤ륢η̤ݻƤˡ
 * HTTP쥹ݥ󥹤˴ޤޤƤΥإå̾ȡθĿͿ
 * 줿ݥ󥿤˳Ǽ롣
 * ˤ0֤顼ˤ-1֤
 *
 * ̾buffer̿åɤνλޤǥ쥹ݥ󥹥إåݻ
 * Ƥ롣
 *
 * ƤӽФheadersפˤʤäfreeǤ餦
 */
int
ochusha_network_broker_get_header_names(OchushaNetworkBroker *broker,
					OchushaAsyncBuffer *buffer,
					char ***headers, int *num_headers)
{
  NetworkBrokerJobArgs *args;
  g_return_val_if_fail(OCHUSHA_IS_NETWORK_BROKER(broker)
		       && OCHUSHA_IS_ASYNC_BUFFER(buffer)
		       && headers != NULL && num_headers != NULL, -1);
  args = g_object_get_qdata(G_OBJECT(buffer), broker_job_args_id);
  if (args == NULL || args->request == NULL)
    return -1;

  return ghttp_get_header_names(args->request, headers, num_headers);
}


#if 0
typedef struct _OchushaNetworkBrokerPostStatus
{
  int status_code;
  char *body;
  char *set_cookie;
} OchushaNetworkBrokerPostStatus;
#endif

gboolean
ochusha_network_broker_try_post(OchushaNetworkBroker *broker,
				char *url, const char *server,
				const char *referer,
				const char *cookie, const char *body,
				OchushaNetworkBrokerPostStatus *status)
{
  ghttp_request *request = ghttp_request_new();
#if DEBUG_NETWORK
  char message[LOG_MESSAGE_SIZE];
#endif

  g_return_val_if_fail(request != NULL, FALSE);

  ghttp_set_uri(request, url);
  ghttp_set_type(request, ghttp_type_post);

  setup_common_request_headers(broker, request, TRUE);
  ghttp_set_header(request, http_hdr_Host, server);

  /* ɬפʤΤ */
#if 0
  ghttp_set_header(request, http_hdr_Accept, "text/xml,application/xml,application/xhtml+xml,text/html;q=0.9,text/plain;q=0.8,video/x-mng,image/png,image/jpeg,image/gif;q=0.2,text/css,*/*;q=0.1");
  ghttp_set_header(request, http_hdr_Accept_Language, "ja,en-us;q=0.66,en;q=0.33");
#endif
  ghttp_set_header(request, http_hdr_Accept_Charset, "Shift_JIS,EUC-JP,utf-8;q=0.66,*;q=0.66");

  ghttp_set_header(request, http_hdr_Connection, "close");
  ghttp_set_header(request, http_hdr_Referrer, referer);
  ghttp_set_header(request, "Referer", referer);

  if (cookie != NULL)
    ghttp_set_header(request, "Cookie", cookie);

#if DEBUG_POST
  snprintf(message, LOG_MESSAGE_SIZE, _("Posting a response: %s\n"), url);
  ochusha_network_broker_output_log(broker, message);
#endif

  ghttp_set_body(request, (char *)body, strlen(body));
  ghttp_prepare(request);

  if (ghttp_process(request) == ghttp_error)
    {
#if DEBUG_POST
      ochusha_network_broker_output_log(broker, _("Posting failed: ghttp_process() returns ghttp_error\n"));
#endif
#if DEBUG_POST_MOST
      fprintf(stderr, "ghttp_process() returns ghttp_error\n");
#endif
      ghttp_request_destroy(request);
      if (status != NULL)
	{
	  status->status_code = 0;
	  status->body = NULL;
	  status->set_cookie = NULL;
	}

      return FALSE;
    }

  if (status != NULL)
    {
      const char *set_cookie;
      status->status_code = ghttp_status_code(request);
      status->body = G_STRNDUP(ghttp_get_body(request),
			       ghttp_get_body_len(request));
      set_cookie = ghttp_get_header(request, http_hdr_Set_Cookie);
      if (set_cookie != NULL)
	status->set_cookie = G_STRDUP(set_cookie);
      else
	status->set_cookie = NULL;
#if DEBUG_NETWORK
      {
	int i;
	char **headers = NULL;
	int num_headers = 0;
	snprintf(message, LOG_MESSAGE_SIZE, "Status: %s (%d)\n",
		 ghttp_reason_phrase(request), status->status_code);
	ochusha_network_broker_output_log(broker, message);
	ghttp_get_header_names(request, &headers, &num_headers);
	for (i = 0; i < num_headers; i++)
	  {
	    snprintf(message, LOG_MESSAGE_SIZE, "%s: %s\n",
		     headers[i], ghttp_get_header(request, headers[i]));
	    ochusha_network_broker_output_log(broker, message);
	    free(headers[i]);
	  }
	if (headers != NULL)
	  free(headers);
      }
#endif
    }

  ghttp_request_destroy(request);

#if DEBUG_POST
  ochusha_network_broker_output_log(broker, _("Posting done.\n"));
#endif

  return TRUE;
}

