/*
 * Copyright (c) 2003 The Ochusha Project.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Id: ochusha_bbs_table.c,v 1.13 2003/11/20 03:14:37 fuyu Exp $
 */

#include "config.h"

#include "ochusha_private.h"
#include "ochusha.h"
#include "ochusha_bbs_table.h"
#include "ochusha_board_category.h"
#include "ochusha_bulletin_board.h"
#include "ochusha_board_2ch.h"

#include "marshal.h"

#include <glib.h>
#include <libxml/SAX.h>
#include <libxml/parser.h>

#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#if ENABLE_REGEXP
# if HAVE_ONIGURUMA
#  if HAVE_ONIG_ONIGPOSIX_H
#    include <onig/onigposix.h>
#  elif HAVE_ONIGPOSIX_H
#    include <onigposix.h>
#  else
#    error "Use of oniguruma without onig/onigposix.h header isn't considered."
#  endif
# else
#  if HAVE_POSIX_REGEX
#    if HAVE_SYS_TYPES_H
#      include <sys/types.h>
#    endif
#    if HAVE_REGEX_H
#      include <regex.h>
#    else
#      warning "I don't know suitable header file of your posix regex library."
#    endif
#  endif
# endif
#endif


static void ochusha_bbs_table_class_init(OchushaBBSTableClass *klass);
static void ochusha_bbs_table_init(OchushaBBSTable *table);
static void ochusha_bbs_table_finalize(GObject *object);


GType
ochusha_bbs_table_get_type(void)
{
  static GType bbs_table_type = 0;

  if (bbs_table_type == 0)
    {
      static const GTypeInfo bbs_table_info =
	{
	  sizeof(OchushaBBSTableClass),
	  NULL, /* base_init */
	  NULL, /* base_finalize */
	  (GClassInitFunc)ochusha_bbs_table_class_init,
	  NULL, /* class_finalize */
	  NULL, /* class_data */
	  sizeof(OchushaBBSTable),
	  0,	/* n_preallocs */
	  (GInstanceInitFunc)ochusha_bbs_table_init,
	};

      bbs_table_type = g_type_register_static(G_TYPE_OBJECT,
					      "OchushaBBSTable",
					      &bbs_table_info, 0);
    }

  return bbs_table_type;
}


enum {
  BOARDLIST_READ_CATEGORY_ELEMENT_SIGNAL,
  BOARDLIST_WRITE_CATEGORY_ELEMENT_SIGNAL,
  BOARDLIST_READ_BOARD_ELEMENT_SIGNAL,
  BOARDLIST_WRITE_BOARD_ELEMENT_SIGNAL,
  LAST_SIGNAL,
};


static GObjectClass *parent_class = NULL;
static gint bbs_table_signals[LAST_SIGNAL] = { 0, 0, 0, 0 };

#if ENABLE_REGEXP
static regex_t cgi_query_pattern;
#endif

static void
ochusha_bbs_table_class_init(OchushaBBSTableClass *klass)
{
  GObjectClass *o_class = (GObjectClass *)klass;

  parent_class = g_type_class_peek_parent(klass);
  o_class->finalize = ochusha_bbs_table_finalize;

  bbs_table_signals[BOARDLIST_READ_CATEGORY_ELEMENT_SIGNAL] =
    g_signal_new("boardlist_read_category_element",
		 G_TYPE_FROM_CLASS(klass),
		 G_SIGNAL_RUN_LAST,
		 G_STRUCT_OFFSET(OchushaBBSTableClass,
				 boardlist_read_category_element),
		 NULL, NULL,
		 libochusha_marshal_VOID__OBJECT_POINTER,
		 G_TYPE_NONE, 2,
		 OCHUSHA_TYPE_BOARD_CATEGORY,
		 G_TYPE_POINTER);
  bbs_table_signals[BOARDLIST_WRITE_CATEGORY_ELEMENT_SIGNAL] =
    g_signal_new("boardlist_write_category_element",
		 G_TYPE_FROM_CLASS(klass),
		 G_SIGNAL_RUN_LAST,
		 G_STRUCT_OFFSET(OchushaBBSTableClass,
				 boardlist_write_category_element),
		 NULL, NULL,
		 libochusha_marshal_VOID__OBJECT_POINTER,
		 G_TYPE_NONE, 2,
		 OCHUSHA_TYPE_BOARD_CATEGORY,
		 G_TYPE_POINTER);
  bbs_table_signals[BOARDLIST_READ_BOARD_ELEMENT_SIGNAL] =
    g_signal_new("boardlist_read_board_element",
		 G_TYPE_FROM_CLASS(klass),
		 G_SIGNAL_RUN_LAST,
		 G_STRUCT_OFFSET(OchushaBBSTableClass,
				 boardlist_read_board_element),
		 NULL, NULL,
		 libochusha_marshal_VOID__OBJECT_POINTER,
		 G_TYPE_NONE, 2,
		 OCHUSHA_TYPE_BULLETIN_BOARD,
		 G_TYPE_POINTER);
  bbs_table_signals[BOARDLIST_WRITE_BOARD_ELEMENT_SIGNAL] =
    g_signal_new("boardlist_write_board_element",
		 G_TYPE_FROM_CLASS(klass),
		 G_SIGNAL_RUN_LAST,
		 G_STRUCT_OFFSET(OchushaBBSTableClass,
				 boardlist_write_board_element),
		 NULL, NULL,
		 libochusha_marshal_VOID__OBJECT_POINTER,
		 G_TYPE_NONE, 2,
		 OCHUSHA_TYPE_BULLETIN_BOARD,
		 G_TYPE_POINTER);

  klass->boardlist_read_category_element = NULL;
  klass->boardlist_write_category_element = NULL;
  klass->boardlist_read_board_element = NULL;
  klass->boardlist_write_board_element = NULL;

#if ENABLE_REGEXP
  if (regcomp(&cgi_query_pattern, "&?([^=]+)=([^&]*)", REG_EXTENDED))
    fprintf(stderr, "invalid regular expression\n");
#endif
}


static void
ochusha_bbs_table_init(OchushaBBSTable *table)
{
  table->category_table
    = g_hash_table_new_full(g_str_hash, g_str_equal, NULL,
			    (GDestroyNotify)g_object_unref);
  table->board_table
    = g_hash_table_new_full(g_str_hash, g_str_equal, NULL,
			    (GDestroyNotify)g_object_unref);
  table->board_name_table
    = g_hash_table_new_full(g_str_hash, g_str_equal, NULL,
			    (GDestroyNotify)g_object_unref);
}


static void
ochusha_bbs_table_finalize(GObject *object)
{
  OchushaBBSTable *table;
  g_return_if_fail(OCHUSHA_IS_BBS_TABLE(object));

  table = (OchushaBBSTable *)object;

  if (table->category_list != NULL)
    {
      g_slist_free(table->category_list);
      table->category_list = NULL;
    }

  if (table->category_table != NULL)
    {
      g_hash_table_destroy(table->category_table);
      table->category_table = NULL;
    }

  if (table->board_table != NULL)
    {
      g_hash_table_destroy(table->board_table);
      table->board_table = NULL;
    }

  if (table->board_name_table != NULL)
    {
      g_hash_table_destroy(table->board_name_table);
      table->board_name_table = NULL;
    }

  if (G_OBJECT_CLASS(parent_class)->finalize)
    (*G_OBJECT_CLASS(parent_class)->finalize)(object);
}


OchushaBBSTable *
ochusha_bbs_table_new(void)
{
  return OCHUSHA_BBS_TABLE(g_object_new(OCHUSHA_TYPE_BBS_TABLE, NULL));
}


void
ochusha_bbs_table_add_category(OchushaBBSTable *table,
			       OchushaBoardCategory *category)
{
  g_return_if_fail(OCHUSHA_IS_BBS_TABLE(table)
		   && OCHUSHA_IS_BOARD_CATEGORY(category));

  if (g_hash_table_lookup(table->category_table, category->name) != NULL)
    return;	/* ϿѤξ */

  g_object_ref(G_OBJECT(category));
  g_hash_table_insert(table->category_table, category->name, category);

  table->category_list = g_slist_append(table->category_list, category);
}


void
ochusha_bbs_table_remove_category(OchushaBBSTable *table,
				  OchushaBoardCategory *category)
{
  g_return_if_fail(OCHUSHA_IS_BBS_TABLE(table)
		   && OCHUSHA_IS_BOARD_CATEGORY(category));
  g_return_if_fail(g_hash_table_lookup(table->category_table, category->name)
		   != NULL);
  g_hash_table_remove(table->category_table, category);
}


OchushaBoardCategory *
ochusha_bbs_table_lookup_category(OchushaBBSTable *table, const gchar *name)
{
  g_return_val_if_fail(OCHUSHA_IS_BBS_TABLE(table), NULL);

  return g_hash_table_lookup(table->category_table, name);
}


void
ochusha_bbs_table_add_board(OchushaBBSTable *table,
			    OchushaBulletinBoard *board)
{
  g_return_if_fail(OCHUSHA_IS_BBS_TABLE(table)
		   && OCHUSHA_IS_BULLETIN_BOARD(board));

  g_object_ref(G_OBJECT(board));
  g_hash_table_insert(table->board_table, board->base_url, board);

  g_object_ref(G_OBJECT(board));
  g_hash_table_insert(table->board_name_table, board->name, board);

  return;
}


void
ochusha_bbs_table_remove_board(OchushaBBSTable *table,
			       OchushaBulletinBoard *board)
{
  OchushaBulletinBoard *entry;

  g_return_if_fail(OCHUSHA_IS_BBS_TABLE(table)
		   && OCHUSHA_IS_BULLETIN_BOARD(board));

  g_object_ref(G_OBJECT(board));

  entry = g_hash_table_lookup(table->board_table, board->base_url);
  if (entry == board)
    g_hash_table_remove(table->board_table, board->base_url);

  entry = g_hash_table_lookup(table->board_name_table, board->name);
  if (entry == board)
    g_hash_table_remove(table->board_name_table, board->name);
}


OchushaBulletinBoard *
ochusha_bbs_table_lookup_board_by_url(OchushaBBSTable *table, const char *url)
{
  g_return_val_if_fail(OCHUSHA_IS_BBS_TABLE(table) && url != NULL, NULL);

  return g_hash_table_lookup(table->board_table, url);
}


OchushaBulletinBoard *
ochusha_bbs_table_lookup_board_by_name(OchushaBBSTable *table,
				       const gchar *name)
{
  g_return_val_if_fail(OCHUSHA_IS_BBS_TABLE(table) && name != NULL, NULL);

  return g_hash_table_lookup(table->board_name_table, name);
}


static gboolean
parse_thread_path(char *path, char **thread_id_p,
		  unsigned int *from_p, unsigned int *to_p)
{
  char *thread_id = NULL;
  unsigned int from = 0;
  unsigned int to = 0;
  char *slash_pos = strchr(path + 1, '/');

#if 0
  fprintf(stderr, "parse_thread_path()\n");
  fprintf(stderr, "  path: %s\n", path);
#endif

  if (slash_pos != NULL)
    {
      if (thread_id_p != NULL)
	thread_id = G_STRNDUP(path, slash_pos - path);
      else
	thread_id = path;
      path = slash_pos + 1;

      if (*path != '\0')
	{
	  char tmp_char;
	  char *hyphen_pos = strchr(path, '-');
	  if (hyphen_pos != NULL)
	    {
	      tmp_char = *hyphen_pos;
	      *hyphen_pos = '\0';
	    }
	  else
	    tmp_char = '\0';

	  if (hyphen_pos != path)
	    {
	      if (sscanf(path, "%d", &from) != 1)
		from = 0;
	    }

	  if (hyphen_pos != NULL && hyphen_pos[1] != '\0')
	    {
	      if (sscanf(hyphen_pos + 1, "%d", &to) != 1)
		to = from;
	    }
	}
    }
  else
    {
      if (thread_id_p != NULL && *path != '\0')
	thread_id = G_STRDUP(path);
      else
	thread_id = path;
    }

  if (thread_id_p != NULL)
    *thread_id_p = thread_id;

  if (from_p != NULL)
    *from_p = from;

  if (to_p != NULL)
    *to_p = to;

#if 0
  fprintf(stderr, "  thread_id: %s\n", thread_id);
  fprintf(stderr, "  from=%d, to=%d\n", from, to);
#endif

  return thread_id != NULL;
}


#if ENABLE_REGEXP
static gboolean
parse_cgi_query(const char *query, char **board_id_p, char **thread_id_p,
		unsigned int *from_p, unsigned int *to_p)
{
  char board_id[PATH_MAX];
  char thread_id[PATH_MAX];
  char tmp_buf[PATH_MAX];
  unsigned int from = 0;
  unsigned int to = 0;
  regmatch_t match[3];
  char *tmp_pos;

#if 0
  fprintf(stderr, "parse_cgi_query()\n");
  fprintf(stderr, "  query: %s\n", query);
#endif

  board_id[0] = '\0';
  thread_id[0] = '\0';

  tmp_pos = strchr(query, '?');
  if (tmp_pos != NULL)
    query = tmp_pos + 1;	/* ?ޤǤФ */

  while (regexec(&cgi_query_pattern, query, 2, match, 0) == 0)
    {
      int match_len = match[0].rm_eo - match[0].rm_so;
      const char *name_head = query + match[1].rm_so;
      int name_len = match[1].rm_eo - match[1].rm_so;
      const char *value_head = query + match[2].rm_so;
      int value_len = match[2].rm_eo - match[2].rm_so;

      if (g_ascii_strncasecmp("BBS", name_head, name_len) == 0)
	{
	  if (value_len < (PATH_MAX - 1))
	    {
	      memcpy(board_id, value_head, value_len);
	      board_id[value_len] = '\0';
	    }
	}
      else if (g_ascii_strncasecmp("KEY", name_head, name_len) == 0)
	{
	  if (value_len < (PATH_MAX - 1))
	    {
	      memcpy(thread_id, value_head, value_len);
	      thread_id[value_len] = '\0';
	    }
	}
      else if (g_ascii_strncasecmp("START", name_head, name_len) == 0
	       || g_ascii_strncasecmp("ST", name_head, name_len) == 0)
	{
	  if (value_len > 0 && value_len < (PATH_MAX - 1))
	    {
	      memcpy(tmp_buf, value_head, value_len);
	      tmp_buf[value_len] = '\0';
	      if (sscanf(tmp_buf, "%d", &from) != 1)
		from = 0;
	    }
	}
      else if (g_ascii_strncasecmp("END", name_head, name_len) == 0
	       || g_ascii_strncasecmp("TO", name_head, name_len) == 0)
	{
	  if (value_len > 0 && value_len < (PATH_MAX - 1))
	    {
	      memcpy(tmp_buf, value_head, value_len);
	      tmp_buf[value_len] = '\0';
	      if (sscanf(tmp_buf, "%d", &to) != 1)
		to = 0;
	    }
	}

      query += match_len;
    }

  if (from == 0 && to != 0)
    from = to;
  else if (to == 0 && from != 0)
    to = from;
  else if (from == 0 && to == 0)
    {
      from = 1;
      to = 1;
    }

  if (from_p != NULL)
    *from_p = from;

  if (to_p != NULL)
    *to_p = to;

  if (board_id[0] == '\0')
    thread_id[0] = '\0';

  if (board_id_p != NULL)
    {
      if (board_id[0] != '\0')
	*board_id_p = G_STRDUP(board_id);
      else
	*board_id_p = NULL;
    }

  if (thread_id_p != NULL)
    {
      if (thread_id[0] != '\0')
	*thread_id_p = G_STRDUP(thread_id);
      else
	*thread_id_p = NULL;
    }

#if 0
  fprintf(stderr, "  board_id: %s\n", board_id);
  fprintf(stderr, "  thread_id: %s\n", thread_id);
  fprintf(stderr, "  from=%d, to=%d\n", from, to);
#endif

  return board_id[0] != '\0';
}
#endif


/*
 * Ϳ줿URLϤбƤʷǼĤΤΤɤ
 * Ƚꤹ롣
 *
 * Ƚ˲äʲΤ褦Ѥ򵯤
 *
 * ȤͿ줿board_pNULLʾ:
 *   URLǼ줿ڡǼºݤбƤǼĤΤΤǡ
 *   ġĤФOchushaBulletinBoard֥ȤtableϿ
 *   ˤϡboard_pˤΥݥ󥿤Ǽ롣
 * ȤͿ줿board_url_pNULLʾ:
 *   URLǼ줿ڡбǽʷǼĤΤΤäݤ
 *   tableˤ̤Ͽʾ硢ĤURLȤ餷ʸΥԡ
 *   board_url_p˳Ǽ롣
 * ȤͿ줿bbs_type_pNULLʾ:
 *   URLǼ줿ڡбǽʷǼĤΤΤäݤ
 *   tableˤ̤Ͽʾ硢ļ̤Ȥ餷ͤ
 *   bbs_type_p˳Ǽ롣
 * ȤͿ줿thread_id_pfrom_pto_pNULLʾ:
 *   URLǼ줿ڡбǽʷǼĤΤΤäݤ硢
 *   ID(thread_id)䡢쥹ϰϻƬ(from)(to)򤽤줾
 *   thread_id_pfrom_pto_p˳Ǽ롣thread_id_p˳ǼΤ
 *   IDʸΥԡ
 */
gboolean
ochusha_bbs_table_check_url(OchushaBBSTable *table, const char *url,
			    OchushaBulletinBoard **board_p,
			    char **board_url_p, OchushaBBSType *bbs_type_p,
			    char **thread_id_p,
			    unsigned int *from_p, unsigned int *to_p)
{
  char *server;
  char *abs_path;
  char *path_pos;
  OchushaBulletinBoard *board = NULL;
  OchushaBBSType bbs_type = OCHUSHA_BBS_TYPE_UNKNOWN;
  char *thread_id = NULL;
  int from = 0;
  int to = 0;
  char board_url[PATH_MAX];
  char tmp_buf[PATH_MAX];

  g_return_val_if_fail(OCHUSHA_IS_BBS_TABLE(table) && url != NULL, FALSE);

  board_url[0] = '\0';

  server = ochusha_utils_url_extract_http_server(url);
  abs_path = ochusha_utils_url_extract_http_absolute_path(url);

#if 0
  fprintf(stderr, "ochusha_bbs_table_check_url()\n");
  fprintf(stderr, "  url: %s\n", url);
  fprintf(stderr, "  server: %s\n", server);
  fprintf(stderr, "  abs_path: %s\n", abs_path);
#endif

  if (server == NULL || abs_path == NULL || *abs_path != '/')
    goto done;

  path_pos = abs_path + 1;
  if (strncmp(path_pos, "test/read.cgi", 13) == 0)
    {
      bbs_type = OCHUSHA_BBS_TYPE_2CH;
      if (path_pos[13] == '/')
	{
	  char *slash_pos = strchr(path_pos + 14, '/');
	  if (slash_pos != NULL)
	    {
	      char tmp_char = slash_pos[1];
	      slash_pos[1] = '\0';
	      if (snprintf(board_url, PATH_MAX, "http://%s%s", server,
			   path_pos + 13) < PATH_MAX)
		{
#if 0
		  fprintf(stderr, "  board_url: %s\n", board_url);
#endif
		  board = ochusha_bbs_table_lookup_board_by_url(table,
								board_url);
		  slash_pos[1] = tmp_char;
		  parse_thread_path(slash_pos + 1, &thread_id, &from, &to);
		}
	      else
		board_url[0] = '\0';
	    }
	}
#if ENABLE_REGEXP
      else if (path_pos[13] == '?')
	{
	  char *board_id = NULL;
	  if (parse_cgi_query(path_pos + 14, &board_id, &thread_id,
			      &from, &to))
	    {
	      if (snprintf(board_url, PATH_MAX, "http://%s/%s/", server,
			   board_id) < PATH_MAX)
		board = ochusha_bbs_table_lookup_board_by_url(table,
							      board_url);
	      else
		board_url[0] = '\0';
#if 0
	      fprintf(stderr, "  board_url: %s\n", board_url);
#endif
	    }
	  if (board_id != NULL)
	    G_FREE(board_id);
	}
#endif
    }
  else if (g_ascii_strcasecmp(server, "jbbs.shitaraba.com") == 0)
    {
      /* JBBSФ̯ʴ*/
      char *tmp_pos = strstr(path_pos, "bbs/read.cgi");
      bbs_type = OCHUSHA_BBS_TYPE_JBBS_SHITARABA;
      if (tmp_pos == NULL)
	{
	  /* URLȤƤ
	   * http://jbbs.shitaraba.com/computer/351/
	   * ߤʤΤ⤢롣
	   */
	  tmp_pos = strchr(path_pos, '/');
	  if (tmp_pos != NULL)
	    tmp_pos = strchr(tmp_pos + 1, '/');
	  if (tmp_pos != NULL)
	    {
	      char tmp_char = tmp_pos[1];
	      tmp_pos[1] = '\0';
	      if (snprintf(board_url, PATH_MAX, "http://%s/%s/", server,
			   path_pos) < PATH_MAX)
		board = ochusha_bbs_table_lookup_board_by_url(table,
							      board_url);
	      else
		board_url[0] = '\0';
	      tmp_pos[1] = tmp_char;
	    }
	}
      else
	{
	  /*
	   * URLȤƤϰʲ2ĤηΤʤ¾ˤ⤢뤫⡣
	   */
	  if (tmp_pos[12] == '/')
	    {
	      /*
	       * http://jbbs.shitaraba.com/bbs/read.cgi/computer/351/1040452877/109
	       * URL
	       */
	      char *slash_pos = strchr(tmp_pos + 1, '/');
	      if (slash_pos != NULL)
		slash_pos = strchr(slash_pos + 1, '/');	/* 2ܤ'/'õ*/

	      if (slash_pos != NULL)
		{
		  /* (tmp_pos + 12)slash_posޤǤʸ
		   * "/computer/351/"ʬˤʤ
		   */
		  char tmp_char = slash_pos[1];
		  slash_pos[1] = '\0';
		  if (snprintf(board_url, PATH_MAX, "http://%s%s", server,
			       tmp_pos + 12) < PATH_MAX)
		    {
		      board = ochusha_bbs_table_lookup_board_by_url(table,
								    board_url);
		      slash_pos[1] = tmp_char;
		      parse_thread_path(slash_pos + 1, &thread_id, &from, &to);
		    }
		  else
		    board_url[0] = '\0';
		}
	    }
#if ENABLE_REGEXP
	  else if (tmp_pos[12] == '?')
	    {
	      /*
	       * http://jbbs.shitaraba.com/computer/bbs/read.cgi?BBS=351&KEY=1040452877&START=109&END=109&NOFIRST=TRUE
	       * URL
	       */
	      char *board_id = NULL;
	      if (parse_cgi_query(tmp_pos + 13, &board_id, &thread_id,
				  &from, &to))
		{
		  *tmp_pos = '\0';
		  if (snprintf(board_url, PATH_MAX, "http://%s%s%s/",
			       server, abs_path, board_id) < PATH_MAX)
		    board = ochusha_bbs_table_lookup_board_by_url(table,
								  board_url);
		  else
		    board_url[0] = '\0';
		}
	      if (board_id != NULL)
		G_FREE(board_id);
	    }
#endif
	}
    }
#if ENABLE_REGEXP
  else if (strncmp(path_pos, "bbs/read.pl?", 12) == 0)
    {
      char *board_id = NULL;
      bbs_type = OCHUSHA_BBS_TYPE_MACHIBBS;
      if (parse_cgi_query(path_pos + 12, &board_id, &thread_id,
			  &from, &to))
	{
	  if (snprintf(board_url, PATH_MAX, "http://%s/%s/", server, board_id)
	      < PATH_MAX)
	    board = ochusha_bbs_table_lookup_board_by_url(table, board_url);
	  else
	    board_url[0] = '\0';
#if 0
	  fprintf(stderr, "  board_url: %s\n", board_url);
#endif
	}
      if (board_id != NULL)
	G_FREE(board_id);
    }
  else if (strncmp(path_pos, "read.cgi?", 9) == 0)
    {
      char *board_id = NULL;
      bbs_type = OCHUSHA_BBS_TYPE_MITINOKU;
      if (parse_cgi_query(path_pos + 12, &board_id, &thread_id,
			  &from, &to))
	{
	  if (snprintf(board_url, PATH_MAX, "http://%s/%s/", server, board_id)
	      < PATH_MAX)
	    board = ochusha_bbs_table_lookup_board_by_url(table, board_url);
	  else
	    board_url[0] = '\0';
#if 0
	  fprintf(stderr, "  board_url: %s\n", board_url);
#endif
	}
      if (board_id != NULL)
	G_FREE(board_id);
    }
#endif
  else if (strncmp(path_pos, "bbs/read.cgi", 12) == 0)
    {
      if (path_pos[12] == '/')
	{
	  char *slash_pos = strchr(path_pos + 13, '/');
	  if (slash_pos != NULL)
	    {
	      int len = slash_pos - (path_pos + 12);
	      memcpy(tmp_buf, path_pos + 12, len);
	      tmp_buf[len] = '\0';
	      if (snprintf(board_url, PATH_MAX, "http://%s%s", server,
			   tmp_buf) < PATH_MAX)
		{
		  board = ochusha_bbs_table_lookup_board_by_url(table,
								board_url);
		  parse_thread_path(slash_pos + 1, &thread_id, &from, &to);
		  bbs_type = OCHUSHA_BBS_TYPE_JBBS;	/* ̤ƥ */
		}
	      else
		board_url[0] = '\0';
#if 0
	      fprintf(stderr, "  board_url: %s\n", board_url);
#endif
	    }
	}
#if ENABLE_REGEXP
      else if (path_pos[12] == '?')
	{
	  char *board_id = NULL;
	  if (parse_cgi_query(path_pos + 13, &board_id, &thread_id,
			      &from, &to))
	    {
	      if (snprintf(board_url, PATH_MAX, "http://%s/%s/", server,
			   board_id) < PATH_MAX)
		board = ochusha_bbs_table_lookup_board_by_url(table,
							      board_url);
	      else
		board_url[0] = '\0';
#if 0
	      fprintf(stderr, "  board_url: %s\n", board_url);
#endif
	    }
	  if (board_id != NULL)
	    G_FREE(board_id);
	}
#endif
    }
  else
    {
      /* 2chäݤġġ*/
      char *slash_pos = strchr(path_pos, '/');
      if (slash_pos != NULL)
	{
	  char tmp_char = slash_pos[1];
	  slash_pos[1] = '\0';
	  snprintf(board_url, PATH_MAX, "http://%s%s", server, abs_path);
	  slash_pos[1] = tmp_char;
	  board = ochusha_bbs_table_lookup_board_by_url(table, board_url);

	  if (board != NULL)
	    {
	      parse_thread_path(slash_pos + 1, &thread_id, &from, &to);
	    }
	  else
	    board_url[0] = '\0';
	}
#if 0
      fprintf(stderr, "  board_url: %s\n", board_url);
#endif
    }

  if (board != NULL)
    bbs_type = board->bbs_type;

 done:
  if (server != NULL)
    G_FREE(server);

  if (abs_path != NULL)
    G_FREE(abs_path);

  if (board_p != NULL)
    *board_p = board;

  if (board_url_p != NULL)
    {
      if (board_url[0] != '\0')
	*board_url_p = G_STRDUP(board_url);
      else
	*board_url_p = NULL;
    }

  if (bbs_type_p != NULL)
    *bbs_type_p = bbs_type;

  if (thread_id_p != NULL)
    *thread_id_p = thread_id;
  else if (thread_id != NULL)
    G_FREE(thread_id);

  if (from_p != NULL)
    *from_p = from;

  if (to_p != NULL)
    *to_p = to;

  return board_url[0] != '\0';
}


typedef enum
{
  SAX_INITIAL,
  SAX_OCHUSHA,
  SAX_BOARDLIST,
  SAX_CATEGORY,
  SAX_CATEGORY_ATTRIBUTE,
  SAX_CATEGORY_ATTRIBUTE_BOOLEAN,
  SAX_CATEGORY_ATTRIBUTE_INT,
  SAX_CATEGORY_ATTRIBUTE_STRING,
  SAX_CATEGORY_BOARD,
  SAX_CATEGORY_BOARD_ATTRIBUTE,
  SAX_CATEGORY_BOARD_ATTRIBUTE_BOOLEAN,
  SAX_CATEGORY_BOARD_ATTRIBUTE_INT,
  SAX_CATEGORY_BOARD_ATTRIBUTE_STRING,
  SAX_ACCEPTED,
  SAX_ERROR
} SAXState;


typedef struct _SAXContext
{
  SAXState state;
  OchushaBBSTable *table;

  char *current_attr_name;
  char *current_attr_val;

  GSList *board_list;

  GHashTable *category_attributes;
  GHashTable *board_attributes;
} SAXContext;


static void
board_list_free(GSList *board_list)
{
  g_slist_foreach(board_list, (GFunc)g_object_unref, NULL);
  g_slist_free(board_list);
}


static void
cleanup_sax_context(SAXContext *context)
{
  if (context->current_attr_name != NULL)
    {
      G_FREE(context->current_attr_name);
      context->current_attr_name = NULL;
    }

  if (context->current_attr_val != NULL)
    {
      G_FREE(context->current_attr_val);
      context->current_attr_val = NULL;
    }

  if (context->board_list != NULL)
    {
      board_list_free(context->board_list);
      context->board_list = NULL;
    }

  if (context->category_attributes != NULL)
    {
      g_hash_table_destroy(context->category_attributes);
      context->category_attributes = NULL;
    }

  if (context->board_attributes != NULL)
    {
      g_hash_table_destroy(context->board_attributes);
      context->board_attributes = NULL;
    }
}


#if TRACE_MEMORY_USAGE
static void
trace_free(gpointer pointer)
{
  G_FREE(pointer);
}
#define TRACE_FREE	trace_free
#else
#define TRACE_FREE	g_free
#endif


static void
startElementHandler(void *context, const xmlChar *name, const xmlChar **atts)
{
  SAXContext *sax_context = (SAXContext *)context;

  switch (sax_context->state)
    {
    case SAX_INITIAL:
      if (strcmp(name, "ochusha") == 0)
	{ sax_context->state = SAX_OCHUSHA; return; }
      break;	/* 顼 */

    case SAX_OCHUSHA:
      if (strcmp(name, "boardlist") == 0)
	{
	  sax_context->category_attributes
	    = g_hash_table_new_full(g_str_hash, g_str_equal,
				    TRACE_FREE, TRACE_FREE);
	  sax_context->board_attributes
	    = g_hash_table_new_full(g_str_hash, g_str_equal,
				    TRACE_FREE, TRACE_FREE);
	  sax_context->state = SAX_BOARDLIST;
	  return;
	}
      break;	/* 顼 */

    case SAX_CATEGORY:
    case SAX_CATEGORY_BOARD:
      if (strcmp(name, "attribute") == 0
	  && atts != NULL && strcmp(atts[0], "name") == 0)
	{
	  switch (sax_context->state)
	    {
	    case SAX_CATEGORY:
	      sax_context->state = SAX_CATEGORY_ATTRIBUTE;
	      break;

	    case SAX_CATEGORY_BOARD:
	      sax_context->state = SAX_CATEGORY_BOARD_ATTRIBUTE;
	      break;

	    default:
	      fprintf(stderr, "Wrong implementation found.\n");
	      abort();
	    }
	  if (sax_context->current_attr_val != NULL)
	    {
#if DEBUG_SAX_HANDLER
	      fprintf(stderr, "unexpected attribute found: %s=%s\n",
		      sax_context->current_attr_name,
		      sax_context->current_attr_val);
#endif
	      G_FREE(sax_context->current_attr_name);
	      G_FREE(sax_context->current_attr_val);
	      sax_context->current_attr_name = NULL;
	      sax_context->current_attr_val = NULL;
	      break;	/* 顼 */
	    }
	  sax_context->current_attr_name = G_STRDUP(atts[1]);
	  return;
	}
      else if (sax_context->state == SAX_CATEGORY
	       && strcmp(name, "board") == 0)
	{ sax_context->state = SAX_CATEGORY_BOARD; return; }
      break;	/* 顼 */

    case SAX_CATEGORY_ATTRIBUTE:
    case SAX_CATEGORY_BOARD_ATTRIBUTE:
      if (atts != NULL && strcmp(atts[0], "val") == 0)
	{
	  /* int/booleanβǽ */
	  if (strcmp(name, "int") == 0)
	    {
	      switch (sax_context->state)
		{
		case SAX_CATEGORY_ATTRIBUTE:
		  sax_context->state = SAX_CATEGORY_ATTRIBUTE_INT;
		  break;

		case SAX_CATEGORY_BOARD_ATTRIBUTE:
		  sax_context->state = SAX_CATEGORY_BOARD_ATTRIBUTE_INT;
		  break;

		default:
		  fprintf(stderr, "Wrong implementation found.\n");
		  abort();
		}
	    }
	  else if (strcmp(name, "boolean") == 0)
	    {
	      switch (sax_context->state)
		{
		case SAX_CATEGORY_ATTRIBUTE:
		  sax_context->state = SAX_CATEGORY_ATTRIBUTE_BOOLEAN;
		  break;

		case SAX_CATEGORY_BOARD_ATTRIBUTE:
		  sax_context->state = SAX_CATEGORY_BOARD_ATTRIBUTE_BOOLEAN;
		  break;

		default:
		  fprintf(stderr, "Wrong implementation found.\n");
		  abort();
		}
	    }
	  else
	    {
#if DEBUG_SAX_HANDLER
	      fprintf(stderr, "element unexpected in state(%d) found: %s\n",
		      sax_context->state, name);
#endif
	      break;	/* 顼 */
	    }

	  if (sax_context->current_attr_val != NULL)
	    {
	      /* 顼ǤɤΤġ̯ */
#if DEBUG_SAX_HANDLER
	      fprintf(stderr,
		      "attribute %s=\"%s\" is overwritten by \"%s\"!\n",
		      sax_context->current_attr_name,
		      sax_context->current_attr_val, atts[1]);
#endif
	      G_FREE(sax_context->current_attr_val);
	    }

	  sax_context->current_attr_val = G_STRDUP(atts[1]);
	  return;
	}
      else if (strcmp(name, "string") == 0)
	{
	  switch (sax_context->state)
	    {
	    case SAX_CATEGORY_ATTRIBUTE:
	      sax_context->state = SAX_CATEGORY_ATTRIBUTE_STRING;
	      return;

	    case SAX_CATEGORY_BOARD_ATTRIBUTE:
	      sax_context->state = SAX_CATEGORY_BOARD_ATTRIBUTE_STRING;
	      return;

	    default:
	      break;	/* 顼 */
	    }
#if DEBUG_SAX_HANDLER
	  fprintf(stderr, "string element is unexpected in state(%d).\n",
		  sax_context->state);
#endif
	}
      break;	/* 顼 */

    case SAX_BOARDLIST:
      if (strcmp(name, "category") == 0)
	{
	  if (sax_context->board_list != NULL)
	    {
#if DEBUG_SAX_HANDLER
	      fprintf(stderr, "free board_list found!\n");
#endif
	      board_list_free(sax_context->board_list);
	    }
	  sax_context->board_list = NULL;
	  sax_context->state = SAX_CATEGORY;
	  return;
	}
      break;	/* 顼 */

    case SAX_CATEGORY_ATTRIBUTE_STRING:
    case SAX_CATEGORY_ATTRIBUTE_BOOLEAN:
    case SAX_CATEGORY_BOARD_ATTRIBUTE_STRING:
    case SAX_CATEGORY_BOARD_ATTRIBUTE_INT:
    case SAX_ACCEPTED:
    case SAX_ERROR:
      break;	/* 顼 */

    default:
      fprintf(stderr, "startHandler is called in unknown state: %d\n",
	      sax_context->state);
    }
  sax_context->state = SAX_ERROR;
}


static gboolean
hash_table_cleanup_func(gpointer key, gpointer value, gpointer unused)
{
  return TRUE;
}


static void
hash_table_cleanup(GHashTable *hash_table)
{
  g_hash_table_foreach_remove(hash_table, hash_table_cleanup_func, NULL);
}


static void
endElementHandler(void *context, const xmlChar *name)
{
  SAXContext *sax_context = (SAXContext *)context;

  switch (sax_context->state)
    {
    case SAX_OCHUSHA:
      if (strcmp(name, "ochusha") == 0)
	{ sax_context->state = SAX_ACCEPTED; return; }
      break;	/* 顼 */

    case SAX_CATEGORY_ATTRIBUTE:
    case SAX_CATEGORY_BOARD_ATTRIBUTE:
      if (strcmp(name, "attribute") == 0)
	{
	  GHashTable *hash_table;
	  switch (sax_context->state)
	    {
	    case SAX_CATEGORY_ATTRIBUTE:
	      hash_table = sax_context->category_attributes;
	      sax_context->state = SAX_CATEGORY;
	      break;

	    case SAX_CATEGORY_BOARD_ATTRIBUTE:
	      hash_table = sax_context->board_attributes;
	      sax_context->state = SAX_CATEGORY_BOARD;
	      break;

	    default:
	      fprintf(stderr, "Wrong implementation found.\n");
	      abort();
	    }
#if DEBUG_SAX_HANDLER_NEW
	  fprintf(stderr, "%s = \"%s\"\n", sax_context->current_attr_name,
		  sax_context->current_attr_val);
#endif
	  g_hash_table_insert(hash_table,
			      sax_context->current_attr_name,
			      sax_context->current_attr_val);
	  sax_context->current_attr_name = NULL;
	  sax_context->current_attr_val = NULL;
	  return;
	}
      break;	/* 顼 */

    case SAX_CATEGORY_ATTRIBUTE_STRING:
    case SAX_CATEGORY_BOARD_ATTRIBUTE_STRING:
      if (strcmp(name, "string") == 0)
	{
	  switch (sax_context->state)
	    {
	    case SAX_CATEGORY_ATTRIBUTE_STRING:
	      sax_context->state = SAX_CATEGORY_ATTRIBUTE;
	      break;

	    case SAX_CATEGORY_BOARD_ATTRIBUTE_STRING:
	      sax_context->state = SAX_CATEGORY_BOARD_ATTRIBUTE;
	      break;

	    default:
	      fprintf(stderr, "Wrong implementation found.\n");
	      abort();
	    }
	  /* NOTE:
	   * stringattributeƤSAXѡԹ()ǡĤ
	   * ޤǤheapˤʤΤǤstrdup
	   * SAXѡԹȤΤϡΥץŪˤ1ʸǤ٤
	   * ƤSAXѡŪʣǤȤƸƤޤ礬ꡢ
	   * Ϣ뤹뤿ΰХåեɬפǤ뤫顣
	   */
	  if (sax_context->current_attr_val != NULL)
	    sax_context->current_attr_val
	      = G_STRDUP(sax_context->current_attr_val);
	  else
	    sax_context->current_attr_val = G_STRDUP("");
	  return;
	}
      break;	/* 顼 */

    case SAX_BOARDLIST:
      if (strcmp(name, "boardlist") == 0)
	{
	  g_hash_table_destroy(sax_context->category_attributes);
	  sax_context->category_attributes = NULL;

	  g_hash_table_destroy(sax_context->board_attributes);
	  sax_context->board_attributes = NULL;

	  sax_context->state = SAX_OCHUSHA;
	  return;
	}
      break;	/* 顼 */

    case SAX_CATEGORY:
      if (strcmp(name, "category") == 0)
	{
	  const gchar *category_name
	    = g_hash_table_lookup(sax_context->category_attributes, "name");
#if DEBUG_SAX_HANDLER
	  fprintf(stderr, "closing category\n");
#endif
	  if (sax_context->board_list != NULL)
	    {
	      OchushaBBSTable *table = sax_context->table;
	      OchushaBoardCategory *category
		= ochusha_bbs_table_lookup_category(table, category_name);

	      if (category == NULL)
		category = ochusha_board_category_new(category_name);
	      else
		g_object_ref(G_OBJECT(category));

#if DEBUG_LIBOCHUSHA_MORE
	      fprintf(stderr, "emitting read_boardlist_element for category.\n");
#endif
	      g_signal_emit_by_name(G_OBJECT(category),
				    "read_boardlist_element",
				    sax_context->category_attributes);
#if DEBUG_LIBOCHUSHA_MORE
	      fprintf(stderr, "emitting boardlist_read_category_element.\n");
#endif
	      g_signal_emit(G_OBJECT(table),
			    bbs_table_signals[BOARDLIST_READ_CATEGORY_ELEMENT_SIGNAL],
			    0,
			    category,
			    sax_context->category_attributes);

	      ochusha_board_category_append_boards(category,
						   sax_context->board_list);
	      sax_context->board_list = NULL;
	      ochusha_bbs_table_add_category(table, category);
	      g_object_unref(G_OBJECT(category));
#if DEBUG_SAX_HANDLER
	      fprintf(stderr, "adding category: %s\n", category_name);
#endif
	    }
	  hash_table_cleanup(sax_context->category_attributes);
	  sax_context->state = SAX_BOARDLIST;
	  return;
	}
      break;	/* 顼 */

    case SAX_CATEGORY_ATTRIBUTE_BOOLEAN:
      if (strcmp(name, "boolean") == 0)
	{ sax_context->state = SAX_CATEGORY_ATTRIBUTE; return; }
      break;	/* 顼 */

    case SAX_CATEGORY_ATTRIBUTE_INT:
      if (strcmp(name, "int") == 0)
	{ sax_context->state = SAX_CATEGORY_ATTRIBUTE; return; }
      break;	/* 顼 */

    case SAX_CATEGORY_BOARD:
      if (strcmp(name, "board") == 0)
	{
	  gchar *board_name
	    = g_hash_table_lookup(sax_context->board_attributes, "name");
	  char *base_url
	    = g_hash_table_lookup(sax_context->board_attributes, "base_url");
	  OchushaBBSTable *table = sax_context->table;
	  OchushaBulletinBoard *board
	    = ochusha_bbs_table_lookup_board_by_url(table, base_url);
#if DEBUG_SAX_HANDLER
	  fprintf(stderr, "closing category/board\n");
#endif
	  if (board == NULL)
	    {
	      /*
	       * MEMO: deserializeݥȡǤbbs_typeΤߤˤꡢ
	       *       OchushaBulletinBoard֥Ȥconcreteʥ饹
	       *       Ƥ롣
	       */
	      int bbs_type = ochusha_utils_get_attribute_int(
						sax_context->board_attributes,
						"bbs_type");

	      switch (bbs_type)
		{
		case OCHUSHA_BBS_TYPE_2CH:
		case OCHUSHA_BBS_TYPE_2CHLIKE_EUCJP:
		case OCHUSHA_BBS_TYPE_JBBS:
		case OCHUSHA_BBS_TYPE_MACHIBBS:
		case OCHUSHA_BBS_TYPE_JBBS_SHITARABA:
		case OCHUSHA_BBS_TYPE_MITINOKU:
		  board = ochusha_board_2ch_new(board_name, base_url);
		  break;

		default:
		  /* board = NULL;  XXX ȤꤢĤ̵ */
		  break;
		}

	      if (board != NULL)
		{
		  ochusha_bulletin_board_set_bbs_type(board, bbs_type);
#if DEBUG_SAX_HANDLER
		  fprintf(stderr, "adding board: %s\n", board_name);
#endif

		  g_signal_emit_by_name(G_OBJECT(board),
					"read_boardlist_element",
					sax_context->board_attributes);
		  g_signal_emit(G_OBJECT(table),
				bbs_table_signals[BOARDLIST_READ_BOARD_ELEMENT_SIGNAL],
				0,
				board,
				sax_context->board_attributes);

		  ochusha_bbs_table_add_board(table, board);
		  g_object_unref(G_OBJECT(board));
		}
	    }

	  if (board != NULL)
	    sax_context->board_list
	      = g_slist_append(sax_context->board_list, board);

	  hash_table_cleanup(sax_context->board_attributes);
	  sax_context->state = SAX_CATEGORY;
	  return;
	}
      break;	/* 顼 */

    case SAX_CATEGORY_BOARD_ATTRIBUTE_BOOLEAN:
      if (strcmp(name, "boolean") == 0)
	{ sax_context->state = SAX_CATEGORY_BOARD_ATTRIBUTE; return; }
      break;	/* 顼 */

    case SAX_CATEGORY_BOARD_ATTRIBUTE_INT:
      if (strcmp(name, "int") == 0)
	{ sax_context->state = SAX_CATEGORY_BOARD_ATTRIBUTE; return; }
      break;	/* 顼 */

    case SAX_INITIAL:
    case SAX_ACCEPTED:
    case SAX_ERROR:
      break;	/* 顼 */

    default:
#if DEBUG_SAX_HANDLER
      fprintf(stderr, "endHandler called in unknown state: %d.\n",
	      sax_context->state);
#endif
      break;	/* 顼 */
    }
#if DEBUG_SAX_HANDLER
  fprintf(stderr, "Invalid document: </%s> appeared in state=%d\n",
	  name, sax_context->state);
#endif
  sax_context->state = SAX_ERROR;
}


#define STRING_BUFFER_LENGTH	1024
static void
charactersHandler(void *context, const xmlChar *ch, int len)
{
  SAXContext *sax_context = (SAXContext *)context;

  if (sax_context->state == SAX_CATEGORY_ATTRIBUTE_STRING
      || sax_context->state == SAX_CATEGORY_BOARD_ATTRIBUTE_STRING)
    {
      static char buffer[STRING_BUFFER_LENGTH] = "";
      int value_len;
      if (sax_context->current_attr_val == NULL)
	buffer[0] = '\0';

      value_len = strlen(buffer);
      if (value_len + len >= STRING_BUFFER_LENGTH)
	len = STRING_BUFFER_LENGTH - value_len - 1;

      strncat(buffer, ch, len);
      sax_context->current_attr_val = buffer;
    }
}


static void
nopHandler(void *context)
{
}


static xmlEntityPtr
getEntityHandler(void *context, const xmlChar *name)
{
  return NULL;
}


gboolean
ochusha_bbs_table_read_boardlist_xml(OchushaBBSTable *table,
				     OchushaConfig *config)
{
  SAXContext context =
    {
      SAX_INITIAL,	/* state */
      table,		/* table */
      NULL, NULL,	/* current_attr_name, current_attr_val */
      NULL,		/* board_list */
      NULL,		/* category_attributes */
      NULL		/* board_attributes */
    };
  xmlSAXHandler sax_handler;
  char *pathname;
  g_return_val_if_fail(OCHUSHA_IS_BBS_TABLE(table) && config != NULL, FALSE);

  pathname = ochusha_config_find_file(config, OCHUSHA_BOARDLIST_XML);
  if (pathname == NULL)
    return FALSE;

  memset(&sax_handler, 0, sizeof(xmlSAXHandler));
#if LIBXML_VERSION >= 20600
  xmlSAX2InitDefaultSAXHandler(&sax_handler, TRUE);
#else
  initxmlDefaultSAXHandler(&sax_handler, TRUE);
#endif

  sax_handler.getEntity = getEntityHandler;
  sax_handler.startDocument = nopHandler;
  sax_handler.endDocument = nopHandler;
  sax_handler.startElement = startElementHandler;
  sax_handler.endElement = endElementHandler;
#if LIBXML_VERSION >= 20600
  sax_handler.startElementNs = NULL;
  sax_handler.endElementNs = NULL;
#endif
  sax_handler.characters = charactersHandler;

  xmlSAXUserParseFile(&sax_handler, &context, pathname);

  cleanup_sax_context(&context);

  if (context.state == SAX_ACCEPTED)
    {
      G_FREE(pathname);
      return TRUE;
    }

  fprintf(stderr, "%s is unacceptable as ochusha's boardlist.\n", pathname);
  return FALSE;
}


typedef struct _WriteBoardlistArgs
{
  OchushaBBSTable *table;
  FILE *boardlist_xml;
} WriteBoardlistArgs;


static void
write_board(gpointer data, gpointer user_data)
{
  OchushaBulletinBoard *board = OCHUSHA_BULLETIN_BOARD(data);
  WriteBoardlistArgs *args = (WriteBoardlistArgs *)user_data;
  FILE *boardlist_xml = args->boardlist_xml;

  if (board->killed)
    return;

  fprintf(boardlist_xml, "      <board>\n");

  g_signal_emit_by_name(G_OBJECT(board),
			"write_boardlist_element",
			boardlist_xml);
#if DEBUG_LIBOCHUSHA_MORE
  fprintf(stderr, "table=%p, board=%p, boardlist_xml=%p\n",
	  args->table, board, boardlist_xml);
#endif
  g_signal_emit(G_OBJECT(args->table),
		bbs_table_signals[BOARDLIST_WRITE_BOARD_ELEMENT_SIGNAL],
		0,
		board,
		boardlist_xml);

  fprintf(boardlist_xml, "      </board>\n");
}


static void
write_category(gpointer data, gpointer user_data)
{
  OchushaBoardCategory *category = OCHUSHA_BOARD_CATEGORY(data);
  WriteBoardlistArgs *args = (WriteBoardlistArgs *)user_data;
  FILE *boardlist_xml = args->boardlist_xml;

  if (category->killed)
    return;

  fprintf(boardlist_xml, "    <category>\n");

  g_signal_emit_by_name(G_OBJECT(category),
			"write_boardlist_element",
			boardlist_xml);
  g_signal_emit(G_OBJECT(args->table),
		bbs_table_signals[BOARDLIST_WRITE_CATEGORY_ELEMENT_SIGNAL],
		0,
		category,
		boardlist_xml);

  g_slist_foreach(category->board_list, write_board, args);

  fprintf(boardlist_xml, "    </category>\n");
}


gboolean
ochusha_bbs_table_write_boardlist_xml(OchushaBBSTable *table,
				      OchushaConfig *config)
{
  WriteBoardlistArgs args;
  FILE *boardlist_xml;
  int fd;

  g_return_val_if_fail(OCHUSHA_IS_BBS_TABLE(table) && config != NULL, FALSE);
  g_return_val_if_fail(config->home != NULL, FALSE);

  fd = ochusha_config_open_file(config, OCHUSHA_BOARDLIST_XML,
				O_WRONLY | O_TRUNC | O_CREAT);
  if (fd < 0)
    {
      fprintf(stderr, "Couldn't open \"%s/%s\" to write.\n",
	      config->home, OCHUSHA_BOARDLIST_XML);
      return FALSE;
    }

  boardlist_xml = fdopen(fd, "w");
  if (boardlist_xml == NULL)
    {
      close(fd);
      fprintf(stderr, "Couldn't open fd to write.\n");
      return FALSE;
    }

  fprintf(boardlist_xml, "<?xml version=\"1.0\"?>\n");
  fprintf(boardlist_xml, "<ochusha>\n");
  fprintf(boardlist_xml, "  <boardlist>\n");

  args.table = table;
  args.boardlist_xml = boardlist_xml;

  g_slist_foreach(table->category_list, write_category, &args);

  fprintf(boardlist_xml, "  </boardlist>\n");
  fprintf(boardlist_xml, "</ochusha>\n");

  return fclose(boardlist_xml) == 0;
}
